/* PSPP - computes sample statistics.
   Copyright (C) 1997, 1998 Free Software Foundation, Inc.
   Written by Ben Pfaff <blp@gnu.org>.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
   02111-1307, USA. */

#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "heap.h"

/* Creates a heap at *H with an initial capacity of M_ELEM
   elements.  Returns nonzero only if successful. */
int
heap_create (heap ** h, int m_elem)
{
  *h = malloc (sizeof (heap) + (m_elem - 1) * sizeof (heap_elem));
  if (*h == NULL)
    return 0;
  (*h)->n_elem = 0;
  (*h)->m_elem = m_elem;
  return 1;
}

/* Destroys the heap at *H. */
void
heap_destroy (heap ** h)
{
  free (*h);
  *h = NULL;
}

/* Inserts into heap *H an element having index INDEX and key KEY.
   Returns nonzero only if successful. */
int
heap_insert (heap ** h, int index, int key)
{
  int i, j;

  assert (h && *h);
  if ((*h)->n_elem >= (*h)->m_elem)
    {
      *h = realloc (*h, (sizeof (heap) + ((*h)->m_elem * 2 - 1)
			 * sizeof (heap_elem)));
      if (*h == NULL)
	return 0;
      (*h)->m_elem *= 2;
    }

  /* Knuth's Algorithm 5.2.3-16.  Step 1. */
  j = (*h)->n_elem + 1;

  for (;;)
    {
      /* Step 2. */
      i = j / 2;

      /* Step 3. */
      if (i == 0 || (*h)->elem[i - 1].key <= key)
	{
	  (*h)->elem[j - 1].index = index;
	  (*h)->elem[j - 1].key = key;
	  (*h)->n_elem++;
	  return 1;
	}

      /* Step 4. */
      (*h)->elem[j - 1] = (*h)->elem[i - 1];
      j = i;
    }
}

/* Deletes the first element in the heap and returns its index, or -1
   if the heap is empty. */
int
heap_delete (heap **hp, int *key)
{
  /* Knuth's Algorithm 5.2.3H-19. */
  heap *h = *hp;
  int first, K, R, l, r, i, j;

  if (h->n_elem == 0)
    return -1;
  first = h->elem[0].index;
  if (key)
    *key = h->elem[0].key;
  K = h->elem[h->n_elem - 1].key;
  R = h->elem[h->n_elem - 1].index;
  l = 1;
  r = h->n_elem - 1;

  /* H3. */
  j = 1;

H4:
  i = j;
  j *= 2;
  if (j == r)
    goto H6;
  else if (j > r)
    goto H8;

/* H5. */
  if (h->elem[j - 1].key > h->elem[j].key)
    j++;

H6:
  if (K <= h->elem[j - 1].key)
    goto H8;

/* H7. */
  h->elem[i - 1] = h->elem[j - 1];
  goto H4;

H8:
  h->elem[i - 1].key = K;
  h->elem[i - 1].index = R;

  h->n_elem--;
  return first;
}

/* Returns the number of elements in heap H. */
int
heap_size (heap ** h)
{
  return (*h)->n_elem;
}

#if GLOBAL_DEBUGGING
/* Checks that a heap is really a heap. */
void
heap_verify (const heap **h)
{
  int j;

  for (j = 1; j <= (*h)->n_elem; j++)
    {
      if (j / 2 >= 1 && (*h)->elem[j / 2 - 1].key > (*h)->elem[j - 1].key)
	printf (_("bad ordering of keys %d and %d\n"), j / 2 - 1, j - 1);
    }
}

/* Dumps out the heap on stdout. */
void
heap_dump (const heap **h)
{
  int j;

  printf (_("Heap contents:\n"));
  for (j = 1; j <= (*h)->n_elem; j++)
    {
      int partner;
      if (j / 2 >= 1)
	partner = (*h)->elem[j / 2 - 1].key;
      else
	partner = -1;
      printf ("%6d-%5d", (*h)->elem[j - 1].key, partner);
    }
}
#endif /* GLOBAL_DEBUGGING */

#if STANDALONE
/* To perform a fairly thorough test of the heap routines, define
   STANDALONE to nonzero then compile this file by itself. */

/* Compares the second elements of the integer arrays at _A and _B and
   returns a strcmp()-type result. */
int
compare_int2 (const void *_a, const void *_b)
{
  int *a = (int *) _a;
  int *b = (int *) _b;

  return a[1] - b[1];
}

/* Test routine. */
int
main (void)
{
  heap *h;
  int i;
  int array[2048][2];

  srand (time (0));
  heap_create (&h, 16);
  for (i = 0; i < 2048; i++)
    {
      int j;

    gen_random:
      array[i][0] = rand () % 32768;
      array[i][1] = rand () % 32768;
      for (j = 0; j < i; j++)
	if (array[j][1] == array[i][1])
	  goto gen_random;
      heap_insert (&h, array[i][0], array[i][1]);
      heap_verify (&h);
    }
  qsort (array, 2048, sizeof (int[2]), compare_int2);
  /*heap_dump(&h); */
  for (i = 0; i < 2048; i++)
    {
      int top = heap_delete (&h);
      assert (top != -1);
      printf ("%6d", top);
      fflush (stdout);
      assert (top == array[i][0]);
      heap_verify (&h);
    }
  heap_destroy (&h);

  return 0;
}
#endif
