/* tclsave.c */

/* Vis5D version 4.3 */

/*
Vis5D system for visualizing five dimensional gridded data sets
Copyright (C) 1990 - 1997 Bill Hibbard, Johan Kellum, Brian Paul,
Dave Santek, and Andre Battaiola.

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 1, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
 * Tcl-based save and restore.
 */



#include <stdio.h>
#include <string.h>
#include "globals.h"
#include "graphics.h"
#include "gui.h"





/*
 * Return the parameters which describe a colorbar curve.
 * If one or more color table entries were 'drawn' by the user, rather
 * than computed from the parameters, return 1.  Return 0 if the user
 * has not drawn the curves.
 * Input:  index - context index
 *         graphic - one of:  VIS5D_ISOSURF, VIS5D_CHSLICE, VIS5D_CVSLICE,
 *                   VIS5D_VOLUME, VIS5D_TRAJ or VIS5D_TOPO.
 *         var - variable number if type isn't VIS5D_TOPO
 * Output: params - pointer to address of parameter array
 */
static int get_colorbar_params( int index, int graphic, int var,
                                float params[] )
{
   float *p;
   int i;
   unsigned int temptable[256], *table;
   int same;

   vis5d_get_color_table_params( index, graphic, var, &p );
   for (i=0; i<7; i++) params[i] = p[i];

   /*
    * This is tricky.  Compare the graphic's color table to one computed
    * from its parameters.  If they're different return 1, else return 0.
    * This is used by the SAVE function to determine if all the color
    * table entries have to be written or just the parameters which
    * describe the curves.
    */
   vis5d_get_color_table_address( index, graphic, var, &table );
   vis5d_color_table_recompute( temptable, 256, params, 1, 1 );
   same = 1;
   for (i=0;i<255;i++) {
      if (table[i]!=temptable[i]) {
         same = 0;
         break;
      }
   }

   if (same) {
      return 0;
   }
   else {
      return 1;
   }
}





/*
 * Save current graphics and colors to the 'savefile' as a Tcl script.
 * Input:  savefile - filename to save to.
 * Return:  0 for success,
 *          VIS5D_BAD_VALUE if unable to open file
 *          VIS5D_FAIL if error while writing file.
 */
int tcl_save( int index, char *savefile )
{
   FILE *f;
   int time, var, i, k, set;
   int numtimes, numvars;
   float r, g, b, a;
   char varname[20];

   vis5d_get_numtimes( index, &numtimes );
   vis5d_get_numvars( index, &numvars );

   f = fopen(savefile,"w");
   if (!f) {
      return VIS5D_BAD_VALUE;
   }

   /* Prolog */
   fprintf(f,"#Vis5D 4.3 Tcl save file\n\n");


   /* misc colors */
   fprintf(f,"\n#Box color\n");
   vis5d_get_color( index, VIS5D_BOX, 0, &r, &g, &b, &a );
   fprintf(f,"vis5d_set_color $ctx VIS5D_BOX 0 %5.3f %5.3f %5.3f %5.3f\n",
           r,g,b,a );

   fprintf(f,"\n#Map color\n");
   vis5d_get_color( index, VIS5D_MAP, 0, &r, &g, &b, &a );
   fprintf(f,"vis5d_set_color $ctx VIS5D_MAP 0 %5.3f %5.3f %5.3f %5.3f\n",
           r,g,b,a );

   fprintf(f,"\n#Background color\n");
   vis5d_get_color( index, VIS5D_BACKGROUND, 0, &r, &g, &b, &a );
   fprintf(f,
           "vis5d_set_color $ctx VIS5D_BACKGROUND 0 %5.3f %5.3f %5.3f %5.3f\n",
           r,g,b,a );

   /* Text labels */
   fprintf(f,"\n#Text labels\n");
   {
      int i = 1;
      int x, y;
      char label[1000];
      while (vis5d_get_label( index, i, &x, &y, label )==0) {
         fprintf(f,"vis5d_make_label $ctx %d %d \"%s\"\n", x, y, label );
         i++;
      }
   }

   /* View matrix */
   fprintf(f,"\n#Viewing matrix\n");
   {
      float mat[4][4];
      int i, j;
      vis5d_get_matrix( index, mat );
      fprintf(f,"vis5d_set_matrix $ctx {");
      for (i=0;i<4;i++) {
         for (j=0;j<4;j++) {
            fprintf(f," %g", mat[i][j] );
         }
      }
      fprintf(f," }\n");
   }

   /* Camera */
   fprintf(f,"\n#Camera\n");
   {
      int perspec;
      float front, zoom;
      vis5d_get_camera( index, &perspec, &front, &zoom );
      fprintf(f,"vis5d_set_camera $ctx %d %g %g\n", perspec, front, zoom );
   }

   /* Cloned and computed physical variables */
   fprintf(f,"\n#Cloned or computed variables\n");
   for (var=0; var<numvars; var++) {
      int type;
      vis5d_get_var_type( index, var, &type );
      if (type==VIS5D_CLONE) {
         int vartoclone;
         char origname[20], clonename[20];
         vis5d_get_var_name( index, var, origname );
         vis5d_get_var_info( index, var, (void*) &vartoclone );
         vis5d_get_var_name( index, vartoclone, clonename );
         fprintf(f,"vis5d_make_clone_variable $ctx \"%s\" \"%s\"\n",
                 origname, clonename );
      }
      else if (type==VIS5D_EXT_FUNC) {
         char funcname[100];
         vis5d_get_var_info( index, var, (void*) funcname );
         fprintf(f,"vis5d_compute_ext_func $ctx \"%s\"\n", funcname );
      }
      else if (type==VIS5D_EXPRESSION) {
         char expr[100];
         vis5d_get_var_info( index, var, (void*) expr );
         fprintf(f,"vis5d_make_expr_var $ctx \"%s\"\n", expr );
      }
   }

   /* Isosurfaces */
   fprintf(f,"\n#Isosurfaces\n");
   for (var=0; var<numvars; var++) {
      float isolevel, min, max;
      int colorvar;
      vis5d_get_isosurface( index, var, &isolevel );
      vis5d_get_var_range( index, var, &min, &max );
      vis5d_get_isosurface_color_var( index, var, &colorvar );
      vis5d_get_var_name( index, var, varname );
      if (isolevel!=min) {
         fprintf(f,"vis5d_set_isosurface $ctx \"%s\" %g\n", varname, isolevel);
         if (colorvar>-1) {
            char colorvarname[20];
            vis5d_get_var_name( index, colorvar, colorvarname );
            fprintf(f,"vis5d_set_isosurface_color_var $ctx \"%s\" \"%s\"\n",
                    varname, colorvarname );
         }
         /* command to recompute the isosurface */
         fprintf(f,"vis5d_make_isosurface $ctx VIS5D_ALL_TIMES \"%s\" 0\n",
                 varname );
      }
   }

   /* Horizontal contour slices */
   fprintf(f,"\n#Horizontal contour slices\n");
   for (var=0;var<numvars;var++) {
      float interval, low, high, level;
      vis5d_get_hslice( index, var, &interval, &low, &high, &level );
      vis5d_get_var_name( index, var, varname );
      fprintf(f,"vis5d_set_hslice $ctx \"%s\" %g %g %g %g\n", varname,
              interval, low, high, level );
   }

   /* Vertical contour slices */
   fprintf(f,"\n#Vertical contour slices\n");
   for (var=0;var<numvars;var++) {
      float interval, low, high, r0, c0, r1, c1;
      vis5d_get_vslice( index, var, &interval, &low, &high, &r0,&c0, &r1,&c1 );
      vis5d_get_var_name( index, var, varname );
      fprintf(f,"vis5d_set_vslice $ctx \"%s\" %g %g %g  %g %g  %g %g\n",
              varname, interval, low, high, r0,c0, r1,c1 );
   }

   /* Horizontal colored slices */
   fprintf(f,"\n#Horizontal colored slices\n");
   for (var=0;var<numvars;var++) {
      float level;
      vis5d_get_chslice( index, var, &level );
      vis5d_get_var_name( index, var, varname );
      fprintf(f,"vis5d_set_chslice $ctx \"%s\" %g\n", varname, level );
   }
   
   /* Vertical colored slices */
   fprintf(f,"\n#Vertical colored slices\n");
   for (var=0;var<numvars;var++) {
      float level, r0, c0, r1, c1;
      vis5d_get_cvslice( index, var, &r0, &c0, &r1, &c1 );
      vis5d_get_var_name( index, var, varname );
      fprintf(f,"vis5d_set_cvslice $ctx \"%s\" %g %g %g %g\n",
              varname, r0, c0, r1, c1 );
   }

   /* Horizontal wind vector slices */
   fprintf(f,"\n#Horizontal wind vector slices\n");
   for (i=0;i<VIS5D_WIND_SLICES;i++) {
      float density, scale, level;
      vis5d_get_hwindslice( index, i, &density, &scale, &level );
      fprintf(f,"vis5d_set_hwindslice $ctx %d  %g %g %g\n", i, density,
              scale, level );
   }

   /* Vertical wind vector slices */
   fprintf(f,"\n#Vertical wind vector slices\n");
   for (i=0;i<VIS5D_WIND_SLICES;i++) {
      float density, scale, r0, c0, r1, c1;
      vis5d_get_vwindslice( index, i, &density, &scale, &r0, &c0, &r1, &c1 );
      fprintf(f,"vis5d_set_vwindslice $ctx %d  %g %g  %g %g %g %g\n", i,
              density, scale, r0, c0, r1, c1 );
   }

   /* Horizontal wind stream slices */
   fprintf(f,"\n#Horizontal stream slices\n");
   for (i=0;i<VIS5D_WIND_SLICES;i++) {
      float density, level;
      vis5d_get_hstreamslice( index, i, &density, &level );
      fprintf(f,"vis5d_set_hstreamslice $ctx %d  %g %g\n", i, density,
              level );
   }

   /* Vertical wind stream slices */
   fprintf(f,"\n#Vertical stream slices\n");
   for (i=0;i<VIS5D_WIND_SLICES;i++) {
      float density, r0, c0, r1, c1;
      vis5d_get_vstreamslice( index, i, &density, &r0, &c0, &r1, &c1 );
      fprintf(f,"vis5d_set_vstreamslice $ctx %d  %g %g %g %g %g\n", i,
              density, r0, c0, r1, c1 );
   }

   /* Trajectories */
   fprintf(f,"\n#Trajectories\n");
   {
      float prevstep = 0.0, prevlength = 0.0;
      int prevribbon = 0;
      int numtraj = vis5d_get_num_traj( index );
      for (i=0;i<numtraj;i++) {
         float row, col, lev, step, length;
         int timestep, group, ribbon;
         vis5d_get_traj_info( index, i, &row, &col, &lev, &timestep,
                              &step, &length, &group, &ribbon );
         if (i==0 || step!=prevstep || length!=prevlength
             || ribbon!=prevribbon) {
            fprintf(f,"vis5d_set_traj $ctx %g %g %d\n", step, length, ribbon );
         }
         prevstep = step;
         prevlength = length;
         prevribbon = ribbon;
         
         fprintf(f,"vis5d_make_traj $ctx %g %g %g %d %d\n", row, col, lev,
                 timestep, group );
      }
   }


   /* Isosurface colors */
   fprintf(f,"\n#Isosurface colors\n");
   for (var=0;var<numvars;var++) {
      vis5d_get_color( index, VIS5D_ISOSURF, var, &r, &g, &b, &a );
      vis5d_get_var_name( index, var, varname );
      fprintf(f,
        "vis5d_set_color $ctx VIS5D_ISOSURF \"%s\" %5.3f %5.3f %5.3f %5.3f\n",
        varname, r,g,b,a );
   }

   /* HSlice colors */
   fprintf(f,"\n#Horizontal contour slice colors\n");
   for (var=0;var<numvars;var++) {
      vis5d_get_color( index, VIS5D_HSLICE, var, &r, &g, &b, &a );
      vis5d_get_var_name( index, var, varname );
      fprintf(f,
          "vis5d_set_color $ctx VIS5D_HSLICE \"%s\" %5.3f %5.3f %5.3f %5.3f\n",
          varname, r,g,b,a );
   }

   /* VSlice colors */
   fprintf(f,"\n#Vertical contour slice colors\n");
   for (var=0;var<numvars;var++) {
      vis5d_get_color( index, VIS5D_VSLICE, var, &r, &g, &b, &a );
      vis5d_get_var_name( index, var, varname );
      fprintf(f,
         "vis5d_set_color $ctx VIS5D_VSLICE \"%s\" %5.3f %5.3f %5.3f %5.3f\n",
         varname, r,g,b,a );
   }

   /* Colored HSlice colors */
   fprintf(f,"\n#Horizontal colored slice tickmark colors\n");
   for (var=0;var<numvars;var++) {
      vis5d_get_color( index, VIS5D_CHSLICE, var, &r, &g, &b, &a );
      vis5d_get_var_name( index, var, varname );
      fprintf(f,
        "vis5d_set_color $ctx VIS5D_CHSLICE \"%s\" %5.3f %5.3f %5.3f %5.3f\n",
        varname, r,g,b,a );
   }

   /* Colored VSlice colors */
   fprintf(f,"\n#Vertical colored slice tickmark colors\n");
   for (var=0;var<numvars;var++) {
      vis5d_get_color( index, VIS5D_CVSLICE, var, &r, &g, &b, &a );
      vis5d_get_var_name( index, var, varname );
      fprintf(f,
       "vis5d_set_color $ctx VIS5D_CVSLICE \"%s\" %5.3f %5.3f %5.3f %5.3f\n",
       varname, r,g,b,a );
   }

   /* HWind colors */
   fprintf(f,"\n#Horizontal wind slice colors\n");
   for (i=0;i<VIS5D_WIND_SLICES;i++) {
      vis5d_get_color( index, VIS5D_HWIND, i, &r, &g, &b, &a );
      fprintf(f,
              "vis5d_set_color $ctx VIS5D_HWIND %d %5.3f %5.3f %5.3f %5.3f\n",
              i, r,g,b,a );
   }

   /* VWind colors */
   fprintf(f,"\n#Vertical wind slice colors\n");
   for (i=0;i<VIS5D_WIND_SLICES;i++) {
      vis5d_get_color( index, VIS5D_VWIND, i, &r, &g, &b, &a );
      vis5d_get_var_name( index, var, varname );
      fprintf(f,
              "vis5d_set_color $ctx VIS5D_VWIND %d %5.3f %5.3f %5.3f %5.3f\n",
              i, r,g,b,a );
   }

   /* Horizontal Stream colors */
   fprintf(f,"\n#Horizontal stream slice colors\n");
   for (i=0;i<VIS5D_WIND_SLICES;i++) {
      vis5d_get_color( index, VIS5D_HSTREAM, i, &r, &g, &b, &a );
      fprintf(f,
              "vis5d_set_color $ctx VIS5D_HSTREAM %d %5.3f %5.3f %5.3f %5.3f\n",
              i, r,g,b,a );
   }

   /* Vertical Stream colors */
   fprintf(f,"\n#Vertical stream slice colors\n");
   for (i=0;i<VIS5D_WIND_SLICES;i++) {
      vis5d_get_color( index, VIS5D_VSTREAM, i, &r, &g, &b, &a );
      fprintf(f,
              "vis5d_set_color $ctx VIS5D_VSTREAM %d %5.3f %5.3f %5.3f %5.3f\n",
              i, r,g,b,a );
   }

   /* Trajectory colors */
   fprintf(f,"\n#Trajectory colors\n");
   for (i=0;i<VIS5D_TRAJ_SETS;i++) {
      int colorvar;
      vis5d_get_color( index, VIS5D_TRAJ, i, &r, &g, &b, &a );
      fprintf(f,"vis5d_set_color $ctx VIS5D_TRAJ %d %5.3f %5.3f %5.3f %5.3f\n",
              i, r,g,b,a );
      vis5d_get_trajectory_color_var( index, i, &colorvar );
      if (colorvar>=0) {
         char varname[20];
         vis5d_get_var_name( index, colorvar, varname );
         fprintf(f,"vis5d_set_trajectory_color_var $ctx %d \"%s\"\n",
                 i, varname );
      }
   }  



   /* Isosurface color tables */
   fprintf(f,"\n#Isosurface color tables\n");
   for (var=0;var<numvars;var++) {
      unsigned int *ctable;
      char varname[20];
      float params[8];

      vis5d_get_var_name( index, var, varname );

      k = get_colorbar_params( index, VIS5D_ISOSURF, var, params );

      fprintf(f,"vis5d_set_color_table_params $ctx VIS5D_ISOSURF");
      fprintf(f," \"%s\" %.3f %.3f %.3f %.3f\n", varname, params[0], params[1],
              params[2], params[3] );

      if (k) {
         /* the color table can't be described by the parameters alone */
         /* save each individual table entry */
         vis5d_get_color_table_address( index, VIS5D_ISOSURF, var, &ctable );
         for (i=0;i<256;i++) {
            fprintf(f,"vis5d_set_color_table_entry $ctx VIS5D_ISOSURF");
            fprintf(f," \"%s\" %d %d %d %d %d\n", varname, i, 
                    UNPACK_RED(ctable[i]), UNPACK_GREEN(ctable[i]),
                    UNPACK_BLUE(ctable[i]), UNPACK_ALPHA(ctable[i]) );
         }
         fprintf(f,"\n");
      }
   }         


   /* Horizontal color slice color tables */
   fprintf(f,"\n#Horizontal color slice color tables\n");
   for (var=0;var<numvars;var++) {
      unsigned int *ctable;
      char varname[20];
      float params[8];

      vis5d_get_var_name( index, var, varname );

      k = get_colorbar_params( index, VIS5D_CHSLICE, var, params );

      fprintf(f,"vis5d_set_color_table_params $ctx VIS5D_CHSLICE");
      fprintf(f," \"%s\" %.3f %.3f %.3f %.3f\n", varname, params[0], params[1],
              params[2], params[3] );

      if (k) {
         /* the color table can't be described by the parameters alone */
         /* save each individual table entry */
         vis5d_get_color_table_address( index, VIS5D_CHSLICE, var, &ctable );
         for (i=0;i<256;i++) {
            fprintf(f,"vis5d_set_color_table_entry $ctx VIS5D_CHSLICE");
            fprintf(f," \"%s\" %d %d %d %d %d\n", varname, i, 
                    UNPACK_RED(ctable[i]), UNPACK_GREEN(ctable[i]),
                    UNPACK_BLUE(ctable[i]), UNPACK_ALPHA(ctable[i]) );
         }
         fprintf(f,"\n");
      }
   }         

   /* Vertical color slice color tables */
   fprintf(f,"\n#Vertical color slice color tables\n");
   for (var=0;var<numvars;var++) {
      unsigned int *ctable;
      char varname[20];
      float params[8];

      vis5d_get_var_name( index, var, varname );

      k = get_colorbar_params( index, VIS5D_CVSLICE, var, params );

      fprintf(f,"vis5d_set_color_table_params $ctx VIS5D_CVSLICE");
      fprintf(f," \"%s\" %.3f %.3f %.3f %.3f\n", varname, params[0], params[1],
              params[2], params[3] );

      if (k) {
         /* the color table can't be described by the parameters alone */
         /* save each individual table entry */
         vis5d_get_color_table_address( index, VIS5D_CVSLICE, var, &ctable );
         for (i=0;i<256;i++) {
            fprintf(f,"vis5d_set_color_table_entry $ctx VIS5D_CVSLICE");
            fprintf(f," \"%s\" %d %d %d %d %d\n", varname, i, 
                    UNPACK_RED(ctable[i]), UNPACK_GREEN(ctable[i]),
                    UNPACK_BLUE(ctable[i]), UNPACK_ALPHA(ctable[i]) );
         }
         fprintf(f,"\n");
      }
   }         

   /* Volume color tables */
   fprintf(f,"\n#Volume color tables\n");
   for (var=0;var<numvars;var++) {
      unsigned int *ctable;
      char varname[20];
      float params[8];

      vis5d_get_var_name( index, var, varname );

      k = get_colorbar_params( index, VIS5D_VOLUME, var, params );

      fprintf(f,"vis5d_set_color_table_params $ctx VIS5D_VOLUME");
      fprintf(f," \"%s\" %.3f %.3f %.3f %.3f\n", varname, params[0], params[1],
              params[2], params[3] );

      if (k) {
         /* the color table can't be described by the parameters alone */
         /* save each individual table entry */
         vis5d_get_color_table_address( index, VIS5D_VOLUME, var, &ctable );
         for (i=0;i<256;i++) {
            fprintf(f,"vis5d_set_color_table_entry $ctx VIS5D_VOLUME");
            fprintf(f," \"%s\" %d %d %d %d %d\n", varname, i, 
                    UNPACK_RED(ctable[i]), UNPACK_GREEN(ctable[i]),
                    UNPACK_BLUE(ctable[i]), UNPACK_ALPHA(ctable[i]) );
         }
         fprintf(f,"\n");
      }
   }         

   /* Trajectory color tables */
   fprintf(f,"\n#Trajectory color tables\n");
   for (var=0;var<numvars;var++) {
      unsigned int *ctable;
      float params[8];

      vis5d_get_var_name( index, var, varname );
      k = get_colorbar_params( index, VIS5D_TRAJ, var, params );

      fprintf(f,"vis5d_set_color_table_params $ctx VIS5D_TRAJ");
      fprintf(f," \"%s\" %.3f %.3f %.3f %.3f\n", varname, params[0], params[1],
              params[2], params[3] );

      if (k) {
         /* the color table can't be described by the parameters alone */
         /* save each individual table entry */
         vis5d_get_color_table_address( index, VIS5D_TRAJ, var, &ctable );
         for (i=0;i<256;i++) {
            fprintf(f,"vis5d_set_color_table_entry $ctx VIS5D_TRAJ");
            fprintf(f," \"%s\" %d %d %d %d %d\n", varname, i, 
                    UNPACK_RED(ctable[i]), UNPACK_GREEN(ctable[i]),
                    UNPACK_BLUE(ctable[i]), UNPACK_ALPHA(ctable[i]) );
         }
         fprintf(f,"\n");
      }
   }         


   /* Topography color tables */
   fprintf(f,"\n#Topography color tables\n");
   for (var=-1;var<numvars;var++) {
      unsigned int *ctable;
      float params[8];

      if (var>=0) {
         vis5d_get_var_name( index, var, varname );
         k = get_colorbar_params( index, VIS5D_TOPO, var, params );
      }
      else {
         int j;
         for (j=0; j<8; j++) params[j] = 0;
         strcpy( varname, "-1" );
         k = 1;
      }
      fprintf(f,"vis5d_set_color_table_params $ctx VIS5D_TOPO");
      fprintf(f," \"%s\" %.3f %.3f %.3f %.3f\n", varname, params[0],
                  params[1], params[2], params[3] );

      if (k) {
         /* the color table can't be described by the parameters alone */
         /* save each individual table entry */
         vis5d_get_color_table_address( index, VIS5D_TOPO, var, &ctable );
         for (i=0;i<256;i++) {
            fprintf(f,"vis5d_set_color_table_entry $ctx VIS5D_TOPO");
            fprintf(f," \"%s\" %d %d %d %d %d\n", varname, i, 
                    UNPACK_RED(ctable[i]), UNPACK_GREEN(ctable[i]),
                    UNPACK_BLUE(ctable[i]), UNPACK_ALPHA(ctable[i]) );
         }
         fprintf(f,"\n");
      }
   }         

   {
      int colorvar;
      vis5d_get_topo_color_var( index, &colorvar );
      fprintf(f, "\n");
      fprintf(f, "vis5d_set_topo_color_var $ctx %d\n", colorvar );
   }


   fprintf(f, "\nvis5d_draw_frame $ctx\n" );

   fclose(f);
   return 0;
}


