/*
 * iplink.c		"ip link".
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 * Authors:	Alexey Kuznetsov, <kuznet@ms2.inr.ac.ru>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <fcntl.h>
#include <net/if.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>
#include <sys/ioctl.h>

#include "utils.h"

static void usage(void)
{
	fprintf(stderr, "Usage: ip link set DEVICE { up | down | arp { on | off } |\n");
	fprintf(stderr, "	                     multicast { on | off } | txqueuelen PACKETS }\n");
	fprintf(stderr, "       ip link show [ DEVICE ]\n");
	exit(-1);
}

static int do_chflags(char *dev, __u32 flags, __u32 mask)
{
	struct ifreq ifr;
	int fd;
	int err;

	strcpy(ifr.ifr_name, dev);
	fd = socket(AF_INET, SOCK_DGRAM, 0);
	err = ioctl(fd, SIOCGIFFLAGS, &ifr);
	if (err) {
		perror("ioctl");
		close(fd);
		return -1;
	}
	if ((ifr.ifr_flags^flags)&mask) {
		ifr.ifr_flags &= ~mask;
		ifr.ifr_flags |= mask&flags;
		err = ioctl(fd, SIOCSIFFLAGS, &ifr);
		if (err)
			perror("ioctl");
	}
	close(fd);
	return err;
}

static int set_qlen(char *dev, int qlen)
{
	struct ifreq ifr = {0};
	int s;

	s = socket(AF_INET, SOCK_STREAM, 0);
	if (s < 0) { 
		perror("socket");
		return -1;
	}

	strcpy(ifr.ifr_name, dev); 
	ifr.ifr_qlen = qlen; 
	if (ioctl(s, SIOCSIFTXQLEN, &ifr) < 0) {
		perror("SIOCGIFXQLEN");
		return -1;
	}
	close(s);

	return 0; 
}

static int do_set(int argc, char **argv)
{
	char *dev = NULL;
	__u32 mask = 0;
	__u32 flags = 0;
	int qlen = -1; 

	while (argc > 0) {
		if (strcmp(*argv, "up") == 0) {
			mask |= IFF_UP;
			flags |= IFF_UP;
		} else if (strcmp(*argv, "down") == 0) {
			mask |= IFF_UP;
			flags &= ~IFF_UP;
		} else if (matches(*argv, "txqueuelen") == 0 ||
			   matches(*argv, "txqlen") == 0) {
			if (qlen != -1) {
				fprintf(stderr, "Duplicate \"txqueuelen\"\n");
				return -1;
			}
			NEXT_ARG();
			if (get_integer(&qlen,  *argv, 0)) {
				fprintf(stderr, "Invalid \"txqueuelen\" value\n");
				return -1;
			}
		} else if (strcmp(*argv, "multicast") == 0) {
			NEXT_ARG();
			mask |= IFF_MULTICAST;
			if (strcmp(*argv, "on") == 0) {
				flags |= IFF_MULTICAST;
			} else if (strcmp(*argv, "off") == 0) {
				flags &= ~IFF_MULTICAST;
			} else
				usage();
		} else if (strcmp(*argv, "arp") == 0) {
			NEXT_ARG();
			mask |= IFF_NOARP;
			if (strcmp(*argv, "on") == 0) {
				flags &= ~IFF_NOARP;
			} else if (strcmp(*argv, "off") == 0) {
				flags |= IFF_NOARP;
			} else
				usage();
		} else if (strcmp(*argv, "dev") == 0) {
			NEXT_ARG();
			if (dev)
				usage();
			dev = *argv;
		} else {
			if (dev)
				usage();
			dev = *argv;
		}
		argc--; argv++;
	}

	if (!dev)
		usage();
	if (qlen != -1) { 
		if (set_qlen(dev, qlen) < 0) return -1; 
	}

	return do_chflags(dev, flags, mask);
}

static int do_show(int argc, char **argv)
{
	return ipaddr_list(argc, argv);
}

int do_iplink(int argc, char **argv)
{
	if (argc > 0) {
		if (matches(*argv, "set") == 0)
			return do_set(argc-1, argv+1);
		if (matches(*argv, "show") == 0 ||
		    matches(*argv, "lst") == 0 ||
		    matches(*argv, "list") == 0)
			return do_show(argc-1, argv+1);
	} else
		return do_show(0, NULL);

	usage();
}
