/*
 * ll_map.c
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 *
 * Authors:	Alexey Kuznetsov, <kuznet@ms2.inr.ac.ru>
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <syslog.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <string.h>

#include "libnetlink.h"
#include "ll_map.h"

struct idxmap
{
	struct idxmap * next;
	int		index;
	char		name[16];
};

static struct idxmap * idxmap;

int ll_remember_index(struct sockaddr_nl *who, struct nlmsghdr *n, void *arg)
{
	struct idxmap **map = (struct idxmap**)arg;
	struct ifinfomsg *ifi = NLMSG_DATA(n);
	struct idxmap *im;
	struct rtattr *tb[IFLA_MAX+1];

	if (map == NULL)
		map = &idxmap;

	if (n->nlmsg_type != RTM_NEWLINK)
		return 0;

	if (n->nlmsg_len < NLMSG_LENGTH(sizeof(ifi)))
		return -1;

	memset(tb, 0, sizeof(tb));
	parse_rtattr(tb, IFLA_MAX, IFLA_RTA(ifi), IFLA_PAYLOAD(n));
	if (tb[IFLA_IFNAME] == NULL)
		return 0;

	im = malloc(sizeof(*im));
	if (im == NULL)
		return 0;

	im->next = *map;
	im->index = ifi->ifi_index;
	strcpy(im->name, RTA_DATA(tb[IFLA_IFNAME]));
	*map = im;
	return 0;
}

char *ll_index_to_name(int idx)
{
	struct idxmap *im;
	static char nbuf[64];

	if (idx == 0)
		return "*";
	for (im = idxmap; im; im = im->next)
		if (im->index == idx)
			return im->name;
	sprintf(nbuf, "if%d", idx);
	return nbuf;
}

int ll_name_to_index(char *name)
{
	struct idxmap *im;

	if (name == NULL)
		return 0;
	for (im = idxmap; im; im = im->next)
		if (strcmp(im->name, name) == 0)
			return im->index;
	return 0;
}

int ll_init_map(struct rtnl_handle *rth)
{
	if (rtnl_wilddump_request(rth, AF_UNSPEC, RTM_GETLINK) < 0) {
		perror("cannot send dump request");
		exit(1);
	}

	if (rtnl_dump_filter(rth, ll_remember_index, &idxmap, NULL, NULL) < 0) {
		fprintf(stderr, "dump terminated\n");
		exit(1);
	}
	return 0;
}
