
/***

log.c - the iptraf logging facility
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This software is open source; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed WITHOUT ANY WARRANTY; without even the
implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
See the GNU General Public License in the included COPYING file for
details.

***/

#include <curses.h>
#include <panel.h>
#include <time.h>
#include <string.h>
#include <netinet/in.h>
#include <linux/if_ether.h>
#include <net/if_arp.h>
#include "dirs.h"
#include "error.h"
#include "ifstats.h"
#include "serv.h"
#include "hostmon.h"

void openlogerr()
{
    int resp;
    errbox("Unable to open log file", ANYKEY_MSG, &resp);
}

void opentlog(FILE ** fd, char *logfilename)
{
    *fd = fopen(logfilename, "a");

    if (*fd == NULL)
	openlogerr();
}

void genatime(time_t now, char *atime)
{
    strncpy(atime, ctime(&now), 26);
    atime[strlen(atime) - 1] = '\0';
}

void writelog(int logging, FILE * fd, char *msg)
{
    char atime[28];

    if (logging) {
	genatime(time((time_t *) NULL), atime);
	fprintf(fd, "%s: %s\n", atime, msg);
    }
}

void writegstatlog(struct iftab *table, unsigned long nsecs, FILE * fd)
{
    struct iflist *ptmp = table->head;
    char atime[26];

    genatime(time((time_t *) NULL), atime);
    fprintf(fd, "\n*** General interface statistics log generated %s\n\n", atime);

    while (ptmp != NULL) {

	fprintf(fd, "%s: %llu total, %llu IP, %llu non-IP, %lu IP checksum errors",
		ptmp->ifname, ptmp->total, ptmp->iptotal, ptmp->noniptotal, ptmp->badtotal);

	if (nsecs > 5) {
	    fprintf(fd, ", average activity %.2f Kbits/s", (float) (ptmp->br * 8 / 1000) / (float) nsecs);
	    fprintf(fd, ", peak activity %.2f Kbits/s", ptmp->peakrate);
	    fprintf(fd, ", last 5-second activity %.2f kbits/s", ptmp->rate);
	}
	fprintf(fd, "\n");

	ptmp = ptmp->next_entry;
    }

    fprintf(fd, "\n%lu seconds running time\n", nsecs);
}

void writedstatlog(char *ifname, float activity, float pps,
		   float peakactivity, float peakpps,
		   struct iftotals *ts, unsigned long nsecs,
		   FILE * fd)
{
    char atime[26];
    int i;
    
    genatime(time((time_t *) NULL), atime);

    fprintf(fd, "\n*** Detailed statistics for interface %s, generated %s\n\n",
	    ifname, atime);

    fprintf(fd, "Total: %llu packets, %llu bytes\n", ts->total, ts->bytestotal);
    fprintf(fd, "IP: %llu packets, %llu bytes\n", ts->iptotal, ts->ipbtotal);
    fprintf(fd, "TCP: %llu packets, %llu bytes\n", ts->tcptotal, ts->tcpbtotal);
    fprintf(fd, "UDP: %llu packets, %llu bytes\n", ts->udptotal, ts->udpbtotal);
    fprintf(fd, "ICMP: %llu packets, %llu bytes\n", ts->icmptotal, ts->icmpbtotal);
    fprintf(fd, "Other IP: %llu packets, %llu bytes\n", ts->othtotal, ts->othbtotal);
    fprintf(fd, "Non-IP: %llu packets, %llu bytes\n\n", ts->noniptotal, ts->nonipbtotal);

    fprintf(fd, "Packet Size Counts\n");

    for (i = 0; i <= 15; i++)
        fprintf(fd, "%u to %u bytes: %lu\n", ts->brackets[i].floor,
        				   ts->brackets[i].ceil,
        				   ts->brackets[i].count);
        
    if (nsecs > 5) {
	fprintf(fd, "Average activity: %.2f kbits/s, %.2f packets/s\n",
		((float) (ts->bytestotal * 8 / 1000) / (float) nsecs),
		((float) (ts->total) / (float) nsecs));

	fprintf(fd, "Peak activity: %.2f kbits/s, %.2f packets/s\n",
		peakactivity, peakpps);

	fprintf(fd, "Last 5-second interface activity: %.2f kbits/s, %.2f packets/s\n",
		activity, pps);
    }
    fprintf(fd, "IP checksum errors: %lu\n\n", ts->badtotal);
    fprintf(fd, "Running time: %lu seconds\n", nsecs);
}

void writeutslog(struct portlistent *list, unsigned long nsecs, FILE * fd)
{
    char atime[26];
    struct portlistent *ptmp = list;

    genatime(time((time_t *) NULL), atime);

    fprintf(fd, "\n*** TCP/UDP traffic log, generated %s\n\n", atime);

    while (ptmp != NULL) {
	if (ptmp->protocol == IPPROTO_TCP)
	    fprintf(fd, "TCP/%s: ", ptmp->servname);
	else
	    fprintf(fd, "UDP/%s: ", ptmp->servname);

	fprintf(fd, "%llu packets, %llu bytes total; ", ptmp->count, ptmp->bcount);
	fprintf(fd, "%llu packets, %llu bytes incoming; ", ptmp->icount, ptmp->ibcount);
	fprintf(fd, "%llu packets, %llu bytes outgoing\n\n", ptmp->ocount, ptmp->obcount);

	ptmp = ptmp->next_entry;
    }

    fprintf(fd, "\nRunning time: %lu seconds\n", nsecs);
}

void writeethlog(struct ethtabent *list, unsigned long nsecs, FILE * fd)
{
    char atime[26];
    struct ethtabent *ptmp = list;

    genatime(time((time_t *) NULL), atime);

    fprintf(fd, "\n*** LAN traffic log, generated %s\n\n", atime);

    while (ptmp != NULL) {
	if (ptmp->type == 0) {
	    if (ptmp->un.desc.linktype == ARPHRD_ETHER)
	        fprintf(fd, "\nEthernet address: %s", ptmp->un.desc.ascaddr);
 	    else if (ptmp->un.desc.linktype == ARPHRD_FDDI)
	        fprintf(fd, "\nFDDI address: %s", ptmp->un.desc.ascaddr);
	        
	    if (ptmp->un.desc.withdesc)
	        fprintf(fd, " (%s)", ptmp->un.desc.desc);
	
	    fprintf(fd, "\n");
	} else {
	    fprintf(fd, "\tIncoming total %llu packets, %llu bytes; %llu IP packets\n",
		    ptmp->un.figs.inpcount, ptmp->un.figs.inbcount, ptmp->un.figs.inippcount);
	    fprintf(fd, "\tOutgoing total %llu packets, %llu bytes; %llu IP packets\n",
		    ptmp->un.figs.outpcount, ptmp->un.figs.outbcount, ptmp->un.figs.outippcount);

	    fprintf(fd, "\tAverage rates: %.2f kbits/s incoming, %.2f kbits/s outgoing\n",
	     (float) (ptmp->un.figs.inbcount * 8 / 1000) / (float) nsecs,
	    (float) (ptmp->un.figs.outbcount * 8 / 1000) / (float) nsecs);

	    if (nsecs > 5)
		fprintf(fd, "\tLast 5-second rates: %.2f kbits/s incoming, %.2f kbits/s outgoing\n",
			ptmp->un.figs.inrate, ptmp->un.figs.outrate);
	}

	ptmp = ptmp->next_entry;
    }

    fprintf(fd, "\nRunning time: %lu seconds\n", nsecs);
}
