//
// File:	ParserTest.java
// Package:	gov.llnl.babel
// Revision:	$Revision: 4434 $
// Modified:	$Date: 2005-03-17 09:05:29 -0800 (Thu, 17 Mar 2005) $
// Description:	main test code for the front-end sidl parser
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel;

import gnu.getopt.Getopt;
import gnu.getopt.LongOpt;
import gov.llnl.babel.parsers.sidl.Parser;
import gov.llnl.babel.parsers.sidl.SIDLException;
import gov.llnl.babel.repository.Repository;
import gov.llnl.babel.repository.RepositoryException;
import gov.llnl.babel.repository.RepositoryFactory;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.url.URLUtilities;
import java.io.InputStream;
import java.io.IOException;
import java.net.URL;
import java.util.Iterator;
import java.util.Set;

/**
 * This class is the main test driver for the sidl parser.  Command line
 * options are --help, --input=URI (input repository), --output=URI (output
 * repository), and --repository=URI (input and output repository).  The
 * URIs following the command line options are parsed as sidl input files.
 */
public class ParserTest {
   /*
    * Command line arguments for GNU getopt.
    */
   private static final LongOpt[] s_lopts = {
      new LongOpt("help",       LongOpt.NO_ARGUMENT,       null, 'h'),
      new LongOpt("input",      LongOpt.REQUIRED_ARGUMENT, null, 'i'),
      new LongOpt("output",     LongOpt.REQUIRED_ARGUMENT, null, 'o'),
      new LongOpt("repository", LongOpt.REQUIRED_ARGUMENT, null, 'r')
   };
   private static final String s_opts = "hi:o:r:";
   private static final String s_name = "gov.llnl.babel.ParserTest";

   /**
    * Main entry point.  Each of the command line arguments is evaluated and
    * then each of the remaining URIs is parsed as a sidl file.
    */
   public static void main(String args[]) {
      Repository repository = null;
      URLUtilities.enableHTTPS();

      /*
       * Parse the input arguments using GNU getopt.
       */
      Getopt g = new Getopt(s_name, args, s_opts, s_lopts);
      g.setOpterr(true);
      try {
         int c;
         while ((c = g.getopt()) != -1) {
            switch (c) {
            case 'h':
               printUsage();
               break;
            case 'i':
               SymbolTable.getInstance().addSymbolResolver(
                  RepositoryFactory.getInstance().createRepository(
                     g.getOptarg()));
               break;
            case 'o':
               repository = RepositoryFactory.getInstance().createRepository(
                  g.getOptarg());
               break;
            case 'r':
               repository = RepositoryFactory.getInstance().createRepository(
                  g.getOptarg());
               SymbolTable.getInstance().addSymbolResolver(repository);
               break;
            default:
               printUsage();
               break;
            }
         }
      } catch (RepositoryException ex) {
         System.err.println("Babel: Error: Repository Exception");
         System.err.println(ex.getMessage());
         System.exit(1);
      }

      /*
       * Read the remaining files on the command line.
       */
      for (int i = g.getOptind(); i < args.length; i++) {
         parseURL(URLUtilities.expandURL(args[i]));
      }

      /*
       * List all new or modified symbols to the standard output.
       */
      Set symbols = SymbolTable.getInstance().getModifiedSymbolNames();
      for (Iterator i = symbols.iterator(); i.hasNext(); ) {
         System.out.println("New or modified symbol \""
            + ((SymbolID) i.next()).getSymbolName()
            + "\"");
      }

      /*
       * If the repository is not null, then write the modified symbols.
       */
      if (repository != null) {
         try {
            repository.writeSymbols(symbols);
         } catch (RepositoryException ex) {
	    System.err.println("Babel: Error: Repository Exception");
            System.err.println(ex.getMessage());
            System.exit(1);
         }
      }
   }

   /**
    * Print command line options to standard output.
    */
   private static void printUsage() {
      System.out.println(
         "Command line options:");
      System.out.println(
         "   -h     | --help           Print command line help");
      System.out.println(
         "   -i URI | --input=URI      Add repository to search path");
      System.out.println(
         "   -o URI | --output=URI     Set symbol output repository");
      System.out.println(
         "   -r URI | --repository=URI Set repository as input and output");
      System.exit(1);
   }

   /**
    * Parse a URL and add the symbols to the global scope.
    */
   private static void parseURL(String url) {
      System.out.println("Babel: Parsing URL \"" + url + "\"...");

      try {
         URL u = new URL(url);
         InputStream s = u.openStream();

         Parser parser = new Parser(s);
         parser.setSourceURL(url);
         String[] warnings = parser.beginParse();
         s.close();

         if (warnings != null) {
            for (int w = 0; w < warnings.length; w++) {
               System.err.println("Babel: Warning: " + warnings[w]);
            }
         }
      } catch (SIDLException ex) {
	  System.err.println("Babel: Error: SIDLException");
         System.err.println(ex.getMessage());
         System.exit(1);
      } catch (IOException ex) {
	  System.err.println("Babel: Error: IOException");
         System.err.println(ex.getMessage());
         System.exit(1);
      }
   }
}
