//
// File:        SkelSource.java
// Package:     gov.llnl.babel.backend.c
// Revision:    @(#) $Revision: 4462 $
// Date:        $Date: 2005-03-23 11:29:24 -0800 (Wed, 23 Mar 2005) $
// Description: generate C skeleton based for a particular class/interface
//
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.c;

import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.c.C;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;

/**
 * <p>
 * Class <code>SkelSource</code> generates a C skeleton source file to the
 * language writer output stream. The skeleton provides the glue between
 * the independent object representation (IOR) and the developer's C
 * implementation of the class.
 * </p>
 * <p>
 * The skeleton source contains free functions to fill the entry point
 * vector and optionally the static entry point vector. These functions
 * are what the IOR requires.
 * </p>
 * <p>
 * The skeleton source contains free functions to get/set the private
 * data pointer from the IOR. These functions are what the implementation
 * source requires. For the base class (i.e. the class without a parent
 * class), the skeleton also include a function to destroy the object.
 * This function is required for the implemention of
 * <code>deleteRef</code>.
 * </p>
 */
public class SkelSource {

  private static final String s_ensureOrderConstant[] = {
    "sidl_general_order",
    "sidl_column_major_order",
    "sidl_row_major_order"
  };

  private static final String s_self = Utilities.s_self;


  /**
   * Write the skeleton file for a particular class to the language writer
   * provided. The skeleton file is the glue between the IOR and the
   * developer's implementation of the class.
   *
   * @param cls    a skeleton file for this class will be written to
   *               <code>writer</code>
   * @param writer this is the output device to which the skeleton
   *               file will be written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public static void generateCode(Class cls, LanguageWriterForC writer)
    throws CodeGenerationException
  {
    final SymbolID id = cls.getSymbolID();
    writer.writeBanner(cls, C.getSkelFile(cls.getSymbolID()), false,
      CodeConstants.C_DESC_SKEL_PREFIX + id.getFullName());
    writer.generateInclude(IOR.getHeaderFile(cls.getSymbolID()), false);
    writer.generateInclude(C.getHeaderFile(cls.getSymbolID()), false);
    writer.printlnUnformatted("#include <stddef.h>");
   if(cls.hasOverwrittenMethods()) {
      writer.generateInclude(IOR.getHeaderFile(id), false);
      writer.printlnUnformatted("#ifdef SIDL_DYNAMIC_LIBRARY");
      writer.printlnUnformatted("#include <stdio.h>");
      writer.printlnUnformatted("#include <stdlib.h>");
      writer.printlnUnformatted("#include \"sidl_Loader.h\"");
      writer.printlnUnformatted("#endif");

      //function from the Impl file that sets the superEPV
      writer.println();
      writer.println("extern void " + C.getObjectName(id) + "__superEPV(");
      writer.println("struct " 
       + C.getObjectName(cls.getParentClass().getSymbolID()) + "__epv*);");

      StubSource.generateGetExternals(cls,writer);
    }
    writer.println();

    if (cls.getParentClass() == null) {
      writeIORCall(cls, C.getPrivateDestructor(id),
                   IOR.getBuiltinMethod(IOR.DELETE, id), writer);
    }

    ImplHeader.writeBuiltinDecls(writer,cls);
    ImplHeader.writeMethodDecls(writer, cls, true);
    ImplHeader.writeMethodDecls(writer, cls, false);

    writeSkelMethods(cls, writer);
    
    writeInitializeEPV(cls, writer);
    if (cls.hasStaticMethod(true)) {
      writeInitializeSEPV(cls, writer);
    }
    writeCallLoad(cls,writer);

    writeGetDataPointer(cls, writer);
    writeSetDataPointer(cls, writer);
  }

  /**
   * Write skeleton functions for methods that need them.  For the C
   * skeleton, only functions with array ordering specifications require
   * a skeleton method.
   *
   * @param  cls   the class whose skeleton methods are to be written.
   * @param writer the output device where the skeleton methods are to be
   *               written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private static void writeSkelMethods(Class cls, LanguageWriterForC writer) 
    throws CodeGenerationException
  {
    Iterator i = cls.getMethods(false).iterator();
    while (i.hasNext()){
      Method m = (Method)i.next();
      if (m.hasArrayOrderSpec() || m.hasRarray()) {
        if (IOR.supportInterceptors(cls)) {
          writeSkelMethod(cls.getSymbolID(), m, IOR.GENERIC_PRE_SUFFIX, writer);
          writeSkelMethod(cls.getSymbolID(), m, "", writer);
          writeSkelMethod(cls.getSymbolID(), m, IOR.GENERIC_POST_SUFFIX,writer);
        } else {
          writeSkelMethod(cls.getSymbolID(), m, "", writer);
        }
      }
    }
  }

  private static String getArgValue(Argument arg) {
    return  (arg.getMode() == Argument.IN) ? arg.getFormalName()
                                           : "*" + arg.getFormalName();
  }

  /**
   * Write a skeleton function for a method with an array ordering
   * specification.
   *
   * @param id	   the name of the class who owns the method.
   * @param m      the method itself.
   * @param suffix the method name suffix, if any, for the interceptor method.
   * @param writer the output device where the skeleton function
   *               will be written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private static void writeSkelMethod(SymbolID id, Method m, String suffix,
                                      LanguageWriterForC writer)
    throws CodeGenerationException
  {
    boolean isNormal     = suffix.equals("");
    Type    methodReturn = m.getReturnType();
    boolean useReturn    = isNormal && (methodReturn.getType() != Type.VOID);
    final String method_name = C.getMethodSkelName(id, m) + suffix;

    //Begin method signature
    writer.print("static ");
    if (isNormal) {
      writer.println(getReturnString(methodReturn));
    } else {
      writer.println("void");
    }
    writer.println(method_name + "(");
    writer.tab();
    C.generateArgumentList(writer, s_self, false, id, m, true, true, false, 
                           true, false, false, false);
    writer.backTab();
    writer.println(")");

    //Begin method body
    writer.println("{");
    writer.tab();
    if (useReturn) {
      writer.println(getReturnString(methodReturn) + " _return;");
      if (methodReturn.hasArrayOrderSpec()) {
        writer.println(getReturnString(methodReturn) + " _return_proxy;");
      }
    }
    List extArgs = Utilities.extendArgs(id, m, false);
    Iterator args = extArgs.iterator();
    HashMap index_args = m.getRarrayInfo();

    while (args.hasNext()) {
      Argument arg = (Argument)args.next();
      //declare ensured array and ensure it
      if (arg.hasArrayOrderSpec()) {
        final Type argType = arg.getType();
        writer.print(getReturnString(argType) + " " + arg.getFormalName() 
          + "_proxy");
        if (arg.getMode()!= Argument.OUT) {
          writer.println(" = " + C.getEnsureArray(argType.getArrayType())
            + "(" + getArgValue(arg) + ", " + argType.getArrayDimension() + ", "
            + s_ensureOrderConstant[argType.getArrayOrder()] + ");");
        } else {
          writer.println(" = " + C.NULL + ";");
        }
      }
      //declare rarray proxies.
      if(arg.getType().isRarray()) {
        Type argType = arg.getType();
        Type arrayType = argType.getArrayType();

        /*
        //This part creates a native array and initializes it
        String first_func_name = IOR.getArrayNameForFunctions(
                                       arrayType.getType()) + "_first";
        */

        writer.print(IOR.getArgumentWithFormal(arg, false, true, false) 
          + C.RAW_ARRAY_EXT + " = ");
        //writer.print(first_func_name+"(");

        writer.println(arg.getFormalName() + "_proxy->d_firstElement;");
        /*
        //This part creates the index variables and initializes THEM.
        int x = 0;
        for(Iterator r = argType.getArrayIndices().iterator(); r.hasNext(); ++x)
        {
          String index = (String) r.next();
          boolean contained = false;
          //Check to make sure this index has not already been created
          for(Iterator con = used_index_args.iterator(); con.hasNext();) {
            String loc_name = (String) con.next();
            if(loc_name.compareTo(index) == 0)
              contained = true;
          }
          if(!contained) {
            Argument index_arg = m.getArgumentByName(index);
            String upper_func_name = IOR.getArrayNameForFunctions(
                                          arrayType.getType()) + "_length";
            if(index_arg.getType().getType() == Type.LONG)
              writer.print("int64_t ");
            else
              writer.print("int32_t ");
            writer.print(index_arg.getFormalName() + " = " + upper_func_name 
              + "(");
            if(arg.getMode() == Argument.INOUT)
              writer.print("*");
            writer.println(arg.getFormalName() + "," + x + ");");
            //Add this index to the created list
            used_index_args.add(index_arg.getFormalName());

          }
          contained = false;
        }
        */
      }
    }
    //Create rarray index variables and initialize them.
    for(Iterator i = index_args.values().iterator(); i.hasNext();) {
      //This is more than a little ugly, but I all I want is the first
      //element off the collection of array dimension that define
      //the index's value.
      HashSet index_collection = (HashSet) i.next();
      Iterator tmp = index_collection.iterator();
      Method.RarrayInfo info = (Method.RarrayInfo) tmp.next();
      Argument index = info.index;
      Argument rarray = info.rarray;
      Type arrayType = rarray.getType().getArrayType();
      int dim = info.dim;

      String upper_func_name = "sidlLength";

      if(index.getType().getType() == Type.LONG)
        writer.print("int64_t ");
      else
        writer.print("int32_t ");
      writer.print(index.getFormalName() + " = " + upper_func_name + "(");
      writer.println(info.rarray.getFormalName() + "_proxy," + dim + ");");

    }

    args = extArgs.iterator();
    while (args.hasNext()) {
      Argument arg = (Argument)args.next();
      if (  arg.hasArrayOrderSpec() 
         && arg.getMode() == Argument.INOUT 
         && !arg.getType().isRarray() ) {
        writer.println(C.getDelRefArray(arg.getType().getArrayType()) 
          + "(" + getArgValue(arg) + ");");
      }
    }
    if (useReturn) {
      if (methodReturn.hasArrayOrderSpec()) {
        writer.println("_return_proxy =");
      } else {
        writer.println("_return =");
      }
      writer.tab();
    }
    //Write Impl call
    writer.println(C.getMethodImplName(id, m.getLongMethodName()) + suffix 
      + "(");
    writer.tab();
    List callArgs = Utilities.extendArgs(id, m, true);
    args = callArgs.iterator();
    while (args.hasNext()) {
      Argument arg = (Argument)args.next();
      if (arg.hasArrayOrderSpec() && !arg.getType().isRarray()) {
        if (Argument.IN != arg.getMode()) {
          writer.print("&");
        }
        writer.print(arg.getFormalName() + "_proxy");
      } else if (arg.getType().isRarray()) {
        writer.print(arg.getFormalName() + C.RAW_ARRAY_EXT);
      } else {
        writer.print(arg.getFormalName());
      }
      if (args.hasNext()) {
        writer.println(",");
      }
    }
    writer.println(");");
    writer.backTab();
    if (useReturn) {
       writer.backTab();
    }
    args = extArgs.iterator();
    while (args.hasNext()) {
      Argument arg = (Argument)args.next();
      if (arg.hasArrayOrderSpec() && !arg.getType().isRarray()) {
        final Type argType = arg.getType();
        if (arg.getMode() != Argument.IN) {
          writer.println(getArgValue(arg) + " = " 
            + C.getEnsureArray(argType.getArrayType()) + "(" 
            + arg.getFormalName() + "_proxy, " 
            + arg.getType().getArrayDimension() + ", " 
            + s_ensureOrderConstant[argType.getArrayOrder()] + ");");
        }
        writer.println(C.getDelRefArray(argType.getArrayType()) 
          + "(" + arg.getFormalName() + "_proxy);");
      }
      if(arg.getType().isRarray() && arg.getMode() == Argument.INOUT) {
        //We need to copy the rarray back into the sidl array
        String r_name = arg.getFormalName();
        String init_func_name = IOR.getArrayNameForFunctions(
                              arg.getType().getArrayType().getType()) + "_init";
        writer.println(init_func_name + "(" + r_name + C.RAW_ARRAY_EXT + ", *"
          + r_name + ", " + arg.getType().getArrayDimension() + ", (*" + r_name
          + ")->d_metadata.d_lower, (*" + r_name + ")->d_metadata.d_upper, (*" 
          + r_name + ")->d_metadata.d_stride);\n");
      }
    }
    if (useReturn) {
      if (methodReturn.hasArrayOrderSpec()) {
        writer.println("_return = " 
          + C.getEnsureArray(methodReturn.getArrayType()) + "(_return_proxy, "
          + methodReturn.getArrayDimension() + ", "
          + s_ensureOrderConstant[methodReturn.getArrayOrder()] + ");");
        writer.println(C.getDelRefArray(methodReturn.getArrayType()) 
          + "(_return_proxy);");
      }
      writer.println("return _return;");
    }
    writer.backTab();
    writer.println("}");
    writer.println();
  }

  /**
   * Generate an return string for the specified SIDL type.  Most
   * of the SIDL return strings are listed in the static structures defined
   * at the start of this class.  Symbol types and array types require
   * special processing.
   */
  private static String getReturnString(Type type)
    throws CodeGenerationException
  {
    return IOR.getReturnString(type, false, false);
  }

  /**
   * Write the support code to get the data pointer from the class.
   */
  private static void writeGetDataPointer(Class cls, LanguageWriter writer) {
    SymbolID id = cls.getSymbolID();
    writer.println(C.getDataName(id) + "*");
    writer.println(C.getDataGetName(id) + "(" + C.getObjectName(id) + " self)");
    writer.println("{");
    writer.tab();
    writer.println("return (" + C.getDataName(id) + "*)" 
      + "(self ? self->d_data : " + C.NULL + ");");
    writer.backTab();
    writer.println("}");
    writer.println();
  }

  /**
   * Write the support code to set the data pointer in the class.
   */
  private static void writeSetDataPointer(Class cls, LanguageWriter writer) {
    SymbolID id = cls.getSymbolID();
    writer.println("void " + C.getDataSetName(id) + "(");
    writer.tab();
    writer.println(C.getObjectName(id) + " self,");
    writer.println(C.getDataName(id) + "* data)");
    writer.backTab();
    writer.println("{");
    writer.tab();
    writer.println("if (self) {");
    writer.tab();
    writer.println("self->d_data = data;");
    writer.backTab();
    writer.println("}");
    writer.backTab();
    writer.println("}");
  }

  /**
   * Write a function to initialize entries in the entry point vector for
   * a particular class. This will generate an assignment statement for
   * each non-<code>static</code> method defined locally in the class.
   *
   * @param cls    the class for which an routine will be written.
   * @param writer the output writer to which the routine will be written.
   */
  private static void writeInitializeEPV(Class cls, LanguageWriterForC writer)
    throws CodeGenerationException
  {
    SymbolID id = cls.getSymbolID();
    writer.openCxxExtern();
    writer.println("void");
    writer.println(C.getSetEPVName(id) + "(" + IOR.getEPVName(id) + " *epv)");
    writer.println("{");
    writer.tab();
    initializeMethodPointer(writer, IOR.getBuiltinMethod(IOR.CONSTRUCTOR, id),
                            id);
    initializeMethodPointer(writer, IOR.getBuiltinMethod(IOR.DESTRUCTOR, id), 
                            id);
    if (IOR.supportAssertions(cls)) {
      initializeMethodPointer(writer, IOR.getBuiltinMethod(IOR.CHECK_ERROR, id),
                              id);
    }
    Iterator i = cls.getMethods(false).iterator();
    while (i.hasNext()) {
      Method m = (Method)i.next();
      initializeMethodPointer(writer, m, id);
    }
    writer.println();
    //if we have overwritten methods, initialize the super pointer
    if(cls.hasOverwrittenMethods()) {
      writer.println(C.getObjectName(id) + "__superEPV(_getExternals()->"
        + "getSuperEPV());");
    }
    writer.backTab();
    writer.println("}");
    writer.closeCxxExtern();
    writer.println();
  }

  /**
   * Write a function to call the _load builtin
   *
   * @param cls    the class for which an routine will be written.
   * @param writer the output writer to which the routine will be written.
   */
  private static void writeCallLoad(Class cls, LanguageWriterForC writer)
  {
    SymbolID id = cls.getSymbolID();
    String name = IOR.getSymbolName(id);
    writer.openCxxExtern();
    writer.println("void " + IOR.getCallLoadName(id) + "(void) { ");
    writer.tab();
    writer.println("impl_" + name + "__load();");
    writer.backTab();
    writer.println("}");
  }

  /**
   * For non-<code>static</code> methods, write an assignment statement to
   * initialize a particular membor of the entry point
   * vector. Initialization of <code>static</code> methods appears
   * elsewhere.
   *
   * @param writer     the output writer to which the assignment statement
   *                   is written.
   * @param m          a description of the method to initialize
   * @param id         the name of class that owns the method
   */
  private static void initializeMethodPointer(LanguageWriter writer,
                                              Method m, SymbolID id)
  {
    final String methodName = m.getLongMethodName();
    final String epv = "epv";
    switch (m.getDefinitionModifier()) {
    case Method.FINAL:
    case Method.NORMAL:
      String ename = IOR.getVectorEntry(methodName);
      String sname = C.getMethodSkelName(id, m);
      if (IOR.supportInterceptors(id) & (!IOR.isBuiltinMethod(methodName)) ) {
        writeMethodAssignment(writer, epv, ename + IOR.GENERIC_PRE_SUFFIX, 
                              sname + IOR.GENERIC_PRE_SUFFIX);
        writeMethodAssignment(writer, epv, ename, sname);
        writeMethodAssignment(writer, epv, ename + IOR.GENERIC_POST_SUFFIX, 
                              sname + IOR.GENERIC_POST_SUFFIX);
      } else {
        writeMethodAssignment(writer, epv, ename, sname);
      }
      break;
    case Method.ABSTRACT:
      writeMethodAssignment(writer, epv, IOR.getVectorEntry(methodName), 
                            C.NULL);
      break;
    default:
      /* do nothing */
      break;
    }
  }

  /**
   * Write a function to initialize entries in the static entry point vector
   * for a particular class. This will generate an assignment statement for
   * each <code>static</code> method defined locally in the class.
   *
   * @param cls    the class for which an routine will be written.
   * @param writer the output writer to which the routine will be written.
   */
  private static void writeInitializeSEPV(Class cls, LanguageWriterForC writer)
    throws CodeGenerationException
  {
    final String epv = "sepv";
    SymbolID id = cls.getSymbolID();
    writer.openCxxExtern();
    writer.println("void");
    writer.println(C.getSetSEPVName(id) + "(" + IOR.getSEPVName(id) 
      + " *" + epv + ")");
    writer.println("{");
    writer.tab();
    if (IOR.supportAssertions(cls)) {
      writeMethodAssignment(writer, epv, 
        IOR.getVectorEntry(IOR.getBuiltinName(IOR.CHECK_ERROR, true)), 
        C.getMethodSkelName(id, IOR.getBuiltinMethod(IOR.CHECK_ERROR, id, 
                                                     true)));
    }
    Iterator i = cls.getMethods(false).iterator();
    while (i.hasNext()) {
      Method m = (Method) i.next();
      if (m.isStatic()) {
        String name     = m.getLongMethodName();
        String skelName = C.getMethodSkelName(id,m);
        if (IOR.supportInterceptors(cls) && (!IOR.isBuiltinMethod(name,true))) {
          writeMethodAssignment(writer, epv, IOR.getVectorEntry(name 
            + IOR.GENERIC_PRE_SUFFIX), skelName + IOR.GENERIC_PRE_SUFFIX);
          writeMethodAssignment(writer, epv, IOR.getVectorEntry(name), 
                                skelName);
          writeMethodAssignment(writer, epv, IOR.getVectorEntry(name 
            + IOR.GENERIC_POST_SUFFIX), skelName + IOR.GENERIC_POST_SUFFIX);
        } else {
          writeMethodAssignment(writer, epv, IOR.getVectorEntry(name), 
                                skelName);
        }
      }
    }
    writer.backTab();
    writer.println("}");
    writer.closeCxxExtern();
    writer.println();
  }

  /**
   * Write the function assignment for the specified var and method name
   * to the specified value.
   *
   * @param writer     the output writer to which the assignment statement
   *                   is written.
   * @param var        the pointer variable representing the EPV.
   * @param mname      the method name.
   * @param value      the desired value, or RHS.
   */
  private static void writeMethodAssignment(LanguageWriter writer, String var, 
                                            String mname, String value)
  {
    writer.println(var + "->" + mname + " = " + value + ";");
  }
   

  /**
   * Create a wrapper function in the skeleton for a particular IOR method.
   *
   * @param cls       the class for which the routine is created.
   * @param funcName  the name of the wrapper function to be written.
   * @param iorMethod the description of the IOR method to be wrapped.
   * @param writer    the output writer to which the routine will be
   *                  written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  private static void writeIORCall(Class cls, String funcName, Method iorMethod,
                                   LanguageWriter writer)
    throws CodeGenerationException
  {
    final SymbolID id = cls.getSymbolID();
    List extArgs = Utilities.extendArgs(id, iorMethod, false);
    Iterator args = extArgs.iterator();
    writer.println("void");
    writer.print(funcName);
    writer.print("(");
    while (args.hasNext()) {
      Argument a = (Argument)args.next();
      writer.print(IOR.getArgumentWithFormal(a, false, false, false));
      if (args.hasNext()) {
        writer.print(", ");
      }
    }
    writer.println(") {");
    writer.tab();
    writer.println("if (self) {");
    writer.tab();
    writer.writeCommentLine("call the IOR method");
    writer.print("self->" + IOR.getEPVVar(IOR.PUBLIC_EPV) + "->");
    writer.print(IOR.getVectorEntry(iorMethod.getLongMethodName()));
    writer.print("(");
    args = extArgs.iterator();
    while (args.hasNext()){
      Argument a = (Argument)args.next();
      writer.print(a.getFormalName());
      if (args.hasNext()) {
        writer.print(", ");
      }
    }
    writer.println(");");
    writer.backTab();
    writer.println("}");
    writer.backTab();
    writer.println("}");
    writer.println();
  }
}
