//
// File:        StubSource.java
// Package:     gov.llnl.babel.backend.fortran
// Revision:    @(#) $Revision: 4434 $
// Description: Generate code to allow FORTRAN calls to BABEL
//
// Copyright (c) 2000-2003, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
//
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
//
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
//
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
//
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.fortran;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.c.C;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.writers.LanguageWriter;
import gov.llnl.babel.backend.writers.LanguageWriterForC;
import gov.llnl.babel.backend.writers.LanguageWriterForFortran;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Enumeration;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Class;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.Symbol;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.Type;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import gov.llnl.babel.backend.rmi.RMIStubSource;

/**
 * This class generates the C code that sits between a FORTRAN client and
 * the internal object representation (IOR) of a sidl object/interface.
 *
 * For each method, this generates a C function that will be called from
 * FORTRAN.  This C function massages the arguments from Fortran, calls
 * the IOR, massages the out values from the IOR call, and returns the
 * outgoing values to the Fortran caller.
 */
public class StubSource {

  /**
   * The argument name that holds the object/interface pointer in
   * a call to an object method. It's conceptually like the this pointer in
   * C++.  In FORTRAN, the object has to be passed explicitly.
   */
  public static final String s_self = "self";

  /**
   * The argument name that holds the return value of a method. The FORTRAN
   * binding treats all return values as an additional out parameter
   * tacked on to the end of the argument list.
   */
  public static final String s_return = "retval";

  /**
   * The argument name that holds the exception pointer which a method may
   * throw an exception.
   */
  public static final String s_exception = "exception";

  /**
   * An <code>#ifdef</code> to check whether character argument should be
   * treated like strings.
   */
  public static final String s_charCheck = "#ifdef " 
                                           + Fortran.getFortranPrefix() 
                                           + "_CHAR_AS_STRING";

  /**
   * This string is prepended to a proxy variable (a variable that takes the
   * place of another variable).  A proxy variable exists when the IOR and
   * Fortran representations are not directly compatible.
   */
  private static final String s_proxy = "_proxy_";

  /**
   * This string is prepended to a proxy variable (a variable that takes the
   * place of another variable).  A proxy variable exists when the IOR and
   * Fortran representations are not directly compatible.
   */
  private static final String s_proxyTwo = "_alt_";

  /**
   * The name of the variable holding the static entry point vector for the
   * class.
   */
  private static final String s_epv = "_epv";

  private static final boolean d_isF90 = Fortran.isFortran90();

  /**
   * A local cache of the names of the fundamental and base exception types.
   */
  private static final String s_exceptionFundamentalType = 
    BabelConfiguration.getBaseExceptionType();
  private static final String s_exceptionInterfaceType = 
    BabelConfiguration.getBaseExceptionInterface();

  /**
   * This writer controls where the generated C code goes.
   */
  private LanguageWriter d_writer;

  /**
   * This is a convenience utility function specifically for the generation
   * of super "Stub" functions in the Impl files. 
   * The output stream is not closed on exit.  A code
   * generation exception is thrown if an error is detected.
   *
   * @param methods is a collection of super methods to be generated.
   *
   * @param writer the output writer to which the stub source will
   *               be written. This will not be closed.
   *
   * @param cls The class in which these supers are to be generated      
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception. It can be caused by I/O trouble or
   *    violations of the data type invariants.
   */
  public static void generateSupers(Class cls, LanguageWriter writer)
    throws CodeGenerationException
  {
    StubSource source = new StubSource(writer);
    source.generateSupers(cls);
  }  

  public void generateSupers(Class cls) throws CodeGenerationException {
    generateSuperEPV(cls);
    generateGetIOR(cls.getSymbolID());
    generateSuperMethods(cls);
  }  

  /**
   * Create an object to generate the stub code for a sidl class/interface.
   * This is frequently called from {@link #generateCode(Symbol,
   * LanguageWriter) generateCode} rather than used directly.
   *
   * @param writer the stub code is generated to this output device.
   */
  public StubSource(LanguageWriter writer) {
    d_writer = writer;
  }

  /**
   * Write a comma and newline to <code>writer</code> iff
   * <code>needComma</code> is  <code>true</code>.  This always returns
   * <code>false</code>.
   *
   * @param writer    the device to which the comma should be output.
   * @param needComma If <code>true</code>, this method will write a
   *                  comma followed by a newline; otherwise, this
   *                  method takes no action.
   * @return <code>false</code> is always returned.
   */
  public static boolean comma(LanguageWriter writer, boolean needComma) {
    if (needComma) {
      writer.println(",");
    }
    return false;
  }

  /**
   * Write an argument declaration in C for an argument being passed in from
   * a FORTRAN caller or from C to a FORTRAN subroutine.
   *
   * @param writer    the place where the code is generated.
   * @param argName   the formal name of the argument.
   * @param argType   the type of the argument.
   * @param needComma whether a comma is needed or not.
   * @return <code>true</code> means a comma is needed before
   *         the next argument; <code>false</code> means a comma
   *         is not needed.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    the type of the argument is unsupported.
   */
  public static boolean declareArgument(LanguageWriter writer, String argName, 
                                        Type argType, boolean needComma)
    throws CodeGenerationException
  {
    final String pre = "fortran.StubSource.declareArgument: ";
    switch(argType.getDetailedType()) {
    case Type.VOID:
      // do nothing
      break;
    case Type.ARRAY:
      if (argType.isRarray()) {
        writer.print(Fortran.getFortranTypeInC(argType.getArrayType()) 
          + " *" + argName);
        needComma = true;
        break;
      }
      // fall through intended
    case Type.DCOMPLEX:
    case Type.DOUBLE:
    case Type.FCOMPLEX:
    case Type.FLOAT:
    case Type.BOOLEAN:
    case Type.ENUM:
    case Type.INT:
    case Type.LONG:
    case Type.CLASS:
    case Type.INTERFACE:
    case Type.OPAQUE:
      writer.print(Fortran.getFortranTypeInC(argType) + " *" + argName);
      needComma = true;
      break;
    case Type.STRING:
      writer.println(Fortran.getFortranPrefix() + "_String " + argName);
      writer.print(Fortran.getFortranPrefix() + "_STR_NEAR_LEN_DECL(" + argName 
        + ")");
      needComma = true;
      break;
    case Type.CHAR:
      writer.printlnUnformatted(s_charCheck);
      writer.println(Fortran.getFortranPrefix() + "_String " + argName);
      writer.println(Fortran.getFortranPrefix() + "_STR_NEAR_LEN_DECL(" 
        + argName + ")");
      writer.printlnUnformatted("#else");
      writer.println("char *" + argName);
      writer.printlnUnformatted("#endif");
      needComma = true;
      break;
    default:
      throw new CodeGenerationException(pre + "Unsupported Fortran argument "
                  + "type: " + argType.getTypeString() + " " + argName);
    }
    return needComma;
  }

  /**
   * Generate the compiler independent form of the function name.
   *
   * @param writer     the place where the symbol is written
   * @param methodName  the potentially mixed case form of the function
   *                    name.
   */
  private static void generateMethodSymbol(LanguageWriter writer,
                                           String methodName)
  {
    writer.println("void");
    writer.disableLineBreak();
    writer.println(Fortran.getFortranSymbol() + "(" + methodName.toLowerCase() 
      + ',' + methodName.toUpperCase() + ',' + methodName + ")");
    writer.println("(");
  }

  /**
   * Write declarations for each of the normal arguments for the
   * FORTRAN compatible method declaration.  This writes one argument
   * for each <code>Argument</code> in the iterator. This does not
   * write the extra arguments that are required by the FORTRAN compiler.
   * The FORTRAN compiler may pass string lengths as separate extra
   * argument.
   *
   * @param writer  where the arguments are declared.
   * @param i       a iterator to the collection of {@link
   *                gov.llnl.babel.symbols.Argument Arguments}.
   * @return <code>true</code> if a comma is needed before the
   *         next parameter declaration.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    a general catch all exception for problems generating the stub.
   */
  private static boolean declareNormalArguments(LanguageWriter writer,
                                                Iterator i)
    throws CodeGenerationException
  {
    boolean needComma = false;
    while (i.hasNext()) {
      needComma = comma(writer, needComma);
      Argument a = (Argument)i.next();
      needComma = 
        declareArgument(writer, a.getFormalName(), a.getType(), needComma);
    }
    return needComma;
  }
  
  /**
   * Write declarations for the extra arguments that are required by some
   * FORTRAN compilers for things like string lengths.  There may or may
   * not be any extra arguments.
   *
   * @param writer    where the arguments are declared.
   * @param i         a iterator to the collection of {@link
   *                gov.llnl.babel.symbols.Argument Arguments}.
   * @param needComma whether a comma is needed before the next argument.
   * @return <code>true</code> if a comma is needed before the
   *         next parameter declaration.
   */
  private static boolean declareExtraArguments(LanguageWriter writer,
                                               Iterator i, boolean needComma)
  {
    while (i.hasNext()) {
      Argument a = (Argument)i.next();
      final int t = a.getType().getDetailedType();
      if ((Type.STRING == t) || (Type.CHAR == t)) {
        if (needComma) {
          writer.println();
          needComma = false;
        }
        if (Type.CHAR == t) {
          writer.printlnUnformatted(s_charCheck);
        }
        writer.println(Fortran.getFortranPrefix() + "_STR_FAR_LEN_DECL(" 
          + a.getFormalName() + ")");
        if (Type.CHAR == t) {
          writer.printlnUnformatted("#endif");
        }
      }
    }
    return needComma;
  }

  /**
   * Generate the C signature for a FORTRAN subroutine to be called from C
   * or for a C function to be called from FORTRAN.  This uses a set of
   * preprocessor macros to handle the conventions of the FORTRAN compiler.
   *
   * @param writer     the place where the signature is written.
   * @param methodName the name of the function.
   * @param argument   a list of {@link gov.llnl.babel.symbols.Argument
   *                   Argument} objects.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    something went wrong -- probably an unsupported type.
   */
  public static void generateSignature(LanguageWriter writer,
                                       String methodName, List arguments)
    throws CodeGenerationException
  {
    generateMethodSymbol(writer, methodName);
    if (arguments.isEmpty()) {
      writer.print("void");
    } else {
      boolean needComma;
      writer.tab();
      needComma = declareNormalArguments(writer, arguments.iterator());
      needComma =
	declareExtraArguments(writer, arguments.iterator(), needComma);
      
      if (needComma) {
	writer.println();
      }
      writer.backTab();
    }
    writer.print(")");
    writer.enableLineBreak();
  }

  /**
   * Return <code>true</code> iff the type present requires a proxy.  A
   * proxy is required when the FORTRAN types is not directly compatible
   * with the C type for a particular sidl type.
   *
   * @param t   the sidl type description
   * @return <code>true</code> means that <code>t</code> requires a proxy;
   *  <code>false</code> means that <code>t</code> does not require a
   * proxy.
   */
  static public boolean hasProxy(Type t) {
    switch (t.getDetailedType()) {
    case Type.BOOLEAN:
    case Type.CHAR:
    case Type.OPAQUE:
    case Type.STRING:
    case Type.CLASS:
    case Type.INTERFACE:
    case Type.ENUM:
    case Type.ARRAY:
      return true;
    default:
      return false;
    }
  }

  /**
   * Return <code>true</code> if a particular sidl type is implemented using
   * a pointer type.
   * 
   * @param t    the sidl type description.
   * @return <code>true</code> means the type is implemented using a pointer
   * type; <code>false</code> means the type is not implemented using a
   * pointer type.
   */
  public static boolean isPointer(Type t) {
    switch(t.getDetailedType()) {
    case Type.ARRAY:
    case Type.STRING:
    case Type.OPAQUE:
    case Type.CLASS:
    case Type.INTERFACE:
      return true;
    default:
      return false;
    }
  }

  private static String getArrayStruct(Type type) throws CodeGenerationException
  {
    final String pointerType = getReturnString(type);
    return pointerType.substring(0, pointerType.length() - 1);
  }

  /**
   * Generate a return string for the specified SIDL type.  Most
   * of the SIDL return strings are listed in the static structures defined
   * at the start of this class.  Symbol types and array types require
   * special processing.
   */
  private static String getReturnString(Type type)
    throws CodeGenerationException
  {
    return IOR.getReturnString(type, true, false);
  }

  /**
   * Declare C variables to act as proxies (stand ins) for things whose
   * FORTRAN type is not directly compatible with the IOR.
   *
   * @param arguments a list of <code>Argument</code> objects.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   * a general catch all exception for something going wrong.
   */
  private void declareProxies(List arguments) throws CodeGenerationException {
    Iterator i = arguments.iterator();
    while (i.hasNext()) {
      Argument a    = (Argument)i.next();
      Type     t    = a.getType();
      String   name = a.getFormalName();
      if (hasProxy(t)) {
        if (t.isRarray()) { // rarray's have lots of stuff
          final int dimen = t.getArrayDimension();
          d_writer.println(getArrayStruct(t) + ' ' + s_proxyTwo + name + ';');
          d_writer.println(getReturnString(t) + ' ' + s_proxy + name 
            + " = &" + s_proxyTwo + name + ';');
          d_writer.println("int32_t " + name + "_lower[" + dimen + "], " 
            + name + "_upper[" + dimen + "], " + name + "_stride[" + dimen 
            + "];");
        } else {
          d_writer.print(getReturnString(t) + " " + s_proxy + name);
          if (isPointer(t)) {
            d_writer.println(" = NULL;");
          } else {
            d_writer.println(";");
          }
        }
      }
    }
  }

  /**
   * Write assignment statements (conceptually speaking) to copy material
   * from the actually incoming parameters to the corresponding proxy
   * variables.  In some cases, copying the incoming values requires
   * something more complex than a simple assignment statement.
   *
   * @param arguments  the list of all parameters.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   something went wrong while generating the statements.
   */
  private void copyIncomingValues(List arguments)
    throws CodeGenerationException
  {
    Iterator i = arguments.iterator();
    while (i.hasNext()) {
      Argument a    = (Argument)i.next();
      if (Argument.OUT != a.getMode()) {
        Type     t    = a.getType();
        String   name = a.getFormalName();
        switch (t.getDetailedType()) {
        case Type.BOOLEAN:
          d_writer.println(s_proxy + name + " = ((*" + name + " == " 
            + Fortran.getFortranPrefix() + "_TRUE) ? TRUE : FALSE);");
          break;
        case Type.CHAR:
          d_writer.printlnUnformatted(s_charCheck);
          d_writer.println(s_proxy + name + " = *" + Fortran.getFortranPrefix() 
            + "_STR(" + name + ");");
          d_writer.printlnUnformatted("#else");
          d_writer.println(s_proxy + name + " = *" + name + ";");
          d_writer.printlnUnformatted("#endif");
          break;
        case Type.STRING:
          d_writer.println(s_proxy + name + " =");
          d_writer.tab();
          d_writer.println("sidl_copy_fortran_str(" + Fortran.getFortranPrefix()
            + "_STR(" + name + "),");
          d_writer.tab();
          d_writer.println(Fortran.getFortranPrefix() + "_STR_LEN(" + name 
            + "));");
          d_writer.backTab();
          d_writer.backTab();
          break;
        case Type.ENUM:
          d_writer.println(s_proxy + name + " =");
          d_writer.tab();
          d_writer.println("(" + IOR.getEnumName(t.getSymbolID()) + ")*");
          d_writer.println(name + ";");
          d_writer.backTab();
          break;
        case Type.ARRAY:
          if (t.isRarray()) {
            Iterator indexVar = t.getArrayIndices().iterator();
            final int dimen = t.getArrayDimension();
            for(int j = 0; j < dimen && indexVar.hasNext(); ++j ) {
              d_writer.println(name + "_upper[" + j + "] = (*" 
                + indexVar.next().toString() + ")-1;");
            }
            d_writer.println(Fortran.getInitArray(t.getArrayType()) 
              + name + ", " + s_proxy + name + ", " + dimen + ", " + name 
              + "_lower, " + name + "_upper, " + name + "_stride);");
          } else {
            d_writer.println(s_proxy + name + " =");
            d_writer.tab();
            d_writer.println("(" + getReturnString(t) + ")");
            if (d_isF90) {
              d_writer.println("(ptrdiff_t)(" + name + "->d_ior);");
            } else {
              d_writer.println("(ptrdiff_t)(*" + name + ");");
            }
            d_writer.backTab();
          }
          break;
        case Type.OPAQUE:
        case Type.CLASS:
        case Type.INTERFACE:
          d_writer.println(s_proxy + name + " =");
          d_writer.tab();
          d_writer.println("(" + getReturnString(t) + ")");
          d_writer.println("(ptrdiff_t)(*" + name + ");");
          d_writer.backTab();
          break;
        }
      }
    }
  }

  /**
   * Copy values from the proxy variables to the actual parameters.  It is
   * assumed that the parameters are <code>out</code> or <code>inout</code>
   * parameters.
   *
   * @param arguments the list of all arguments.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   something went wrong while generating the statements.
   */
  private void copyOutgoingValues(List arguments)
    throws CodeGenerationException
  {
    final String pre = "fortran.StubSource.copyOutgoingValues: ";
    Iterator i = arguments.iterator();
    while (i.hasNext()) {
      Argument a = (Argument)i.next();
      String name = a.getFormalName();
      if (Argument.IN != a.getMode() && !name.equals(s_exception)) {
        Type   t    = a.getType();
        switch (t.getDetailedType()) {
        case Type.BOOLEAN:
          d_writer.println("*" + name + " = ((" + s_proxy + name
            + " == TRUE) ? " + Fortran.getFortranPrefix() + "_TRUE : " 
            + Fortran.getFortranPrefix() + "_FALSE);");
          break;
        case Type.ARRAY:
          if (t.isRarray()) break;
          if (d_isF90) {
            if (Fortran.hasDirectAccess(t)) {
              Type dataType = t.getArrayType();
              d_writer.println("if (sidl_" + dataType.getTypeString() 
                + "__array_convert2f90(" + s_proxy + name + ", " 
                + t.getArrayDimension() + ", " + name + ")) {");
              d_writer.tab();
              d_writer.writeCommentLine("Copy to contiguous column-order");
              d_writer.println(getReturnString(t) + " " + s_proxyTwo + name 
                + " =");
              d_writer.tab();
              d_writer.print(Fortran.getEnsureArray(dataType));
              d_writer.println(s_proxy + name + ", " + t.getArrayDimension() 
                + ",");
              d_writer.tab();
              d_writer.println("sidl_column_major_order);");
              d_writer.backTab();
              d_writer.backTab();
              d_writer.println(Fortran.getDelRefArray(dataType) + s_proxy 
                + name + ");");
              d_writer.println("if (sidl_" + dataType.getTypeString() 
                + "__array_convert2f90(" + s_proxyTwo + name + ", " 
                + t.getArrayDimension() + ", " + name + ")) {");
              d_writer.tab();
              d_writer.writeCommentLine("We're S.O.L.");
              d_writer.println("fprintf(stderr, \"convert2f90 failed: %p "
                + "%d\\n\", (void*)" + s_proxyTwo + name + ", " 
                + t.getArrayDimension() + ");");
              d_writer.println("exit(1); /*NOTREACHED*/");
              d_writer.backTab();
              d_writer.println("}");
              d_writer.backTab();
              d_writer.println("}");
            } else {
              d_writer.println(name + "->d_ior = (ptrdiff_t)" + s_proxy 
                + name + ";");
            }
          } else {
            d_writer.println("*" + name + " = (ptrdiff_t)" + s_proxy + name 
              + ";");
          }
          break;
        case Type.CLASS:
        case Type.INTERFACE:
        case Type.OPAQUE:
          d_writer.println("*" + name + " = (ptrdiff_t)" + s_proxy + name 
            + ";");
          break;
        case Type.CHAR:
          d_writer.printlnUnformatted(s_charCheck);
          d_writer.println("*" + Fortran.getFortranPrefix() + "_STR(" + name 
            + ") = " + s_proxy + name + ";");
          d_writer.printlnUnformatted("#else");
          d_writer.println("*" + name + " = " + s_proxy + name + ";");
          d_writer.printlnUnformatted("#endif");
          break;
        case Type.STRING:
          d_writer.println("sidl_copy_c_str(");
          d_writer.tab();
          d_writer.println(Fortran.getFortranPrefix() + "_STR(" + name + "),");
          d_writer.println(Fortran.getFortranPrefix() + "_STR_LEN(" + name 
            + "),");
          d_writer.println(s_proxy + name + ");");
          d_writer.backTab();
          break;
        case Type.ENUM:
          d_writer.println("*" + name + " = (int)");
          d_writer.tab();
          d_writer.println(s_proxy + name + ";");
          d_writer.backTab();
          break;
        case Type.SYMBOL:
          throw new CodeGenerationException(pre + "Unsupported Type: " 
                      + t.getTypeString());
        }
      }
    }
  }

  /**
   * Write the code to free any resources allocated by the stub function or
   * by the IOR that is no longer needed.
   *
   * @param arguments  the list of arguments for which resources may have
   *                   been allocated.
   */
  private void freeResources(List arguments) {
    Iterator i = arguments.iterator();
    while (i.hasNext()) {
      Argument a    = (Argument)i.next();
      if (Type.STRING == a.getType().getDetailedType()) {
        String name = a.getFormalName();
        d_writer.println("free((void *)" + s_proxy + name + ");");
      }
    }
  }

  /**
   * Write a declaration for the entry point vector pointer.
   *
   * @param isStatic <code>true</code> means the static method
   *                 entry point vector should be used, and
   *                 <code>false</code> means the object entry
   *                 point vector should be used.
   * @param id       the symbol whose EPV or SEPV is needed.
   */
  private void declareEntryPointVector(boolean isStatic, SymbolID id) {
    if (isStatic) {
      d_writer.println("const " + IOR.getSEPVName(id) + " *" + s_epv 
        + " = _getSEPV();");
    } else {
      d_writer.println(IOR.getEPVName(id) + " *" + s_epv + " = NULL;");
    }
  }
  
  /**
   *  Just generates the variable superEPV
   */
  private void generateSuperEPV(Class cls) {
    d_writer.println("static const struct " 
      + C.getObjectName(cls.getParentClass().getSymbolID()) 
      + "__epv* superEPV = NULL;");
    d_writer.println();
  }

  /**
   * Generates the super methods
   */
  private void generateSuperMethods(Class cls)  throws CodeGenerationException {
    SymbolID clsID = cls.getSymbolID();
    Collection methods = cls.getOverwrittenClassMethods();
    for (Iterator m = methods.iterator(); m.hasNext(); ) {
      Method method = (Method) m.next();
      extendAndGenerateSuper(cls, method, cls.getSymbolID(), false);
    }
  }

  /**
   * Generate the _getIOR function that provides access to the IOR functions
   * either through static linking or dynamic loading.
   */
  private void generateGetIOR(SymbolID id) {
    final String ext_name = IOR.getExternalName(id);
    d_writer.writeComment("Return pointer to internal IOR functions.", false);

    d_writer.println("static const " + ext_name + "* _getIOR(void)");
    d_writer.println("{");
    d_writer.tab();
    d_writer.println("static const " + ext_name + " *_ior = NULL;");
    d_writer.println("if (!_ior) {");
    d_writer.tab();
    if (BabelConfiguration.isSIDLBaseClass(id)) {
      d_writer.println("_ior = " + IOR.getExternalFunc(id) + "();");
    } else {
      d_writer.printlnUnformatted("#ifdef SIDL_STATIC_LIBRARY");
      d_writer.println("_ior = " + IOR.getExternalFunc(id) + "();");
      d_writer.printlnUnformatted("#else");
      d_writer.println("sidl_DLL dll = sidl_DLL__create();");
      d_writer.println("const " + ext_name + "*(*dll_f)(void);");
      d_writer.writeCommentLine("check global namespace for symbol first");
      d_writer.println("if (dll && sidl_DLL_loadLibrary(dll, \"main:\", TRUE, "
        + "FALSE)) {");
      d_writer.tab();
      d_writer.println("dll_f =");
      d_writer.tab();
      d_writer.print("(const " + ext_name + "*(*)(void)) ");
      d_writer.println("sidl_DLL_lookupSymbol(");
      d_writer.tab();
      d_writer.println("dll, \"" + IOR.getExternalFunc(id) + "\");");
      d_writer.backTab();
      d_writer.backTab();
      d_writer.println("_ior = (dll_f ? (*dll_f)() : NULL);");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println("if (dll) sidl_DLL_deleteRef(dll);");

      d_writer.println("if (!_ior) {");
      d_writer.tab();
      d_writer.println("dll = sidl_Loader_findLibrary(\"" + id.getFullName() 
        + "\",");
      d_writer.tab();
      d_writer.println("\"ior/impl\", sidl_Scope_SCLSCOPE,");
      d_writer.println("sidl_Resolve_SCLRESOLVE);");
      d_writer.backTab();
      d_writer.println("if (dll) {");
      d_writer.tab();
      d_writer.println("dll_f =");
      d_writer.tab();
      d_writer.print("(const " + ext_name + "*(*)(void)) ");
      d_writer.println("sidl_DLL_lookupSymbol(");
      d_writer.tab();
      d_writer.println("dll, \"" + IOR.getExternalFunc(id) + "\");");
      d_writer.backTab();
      d_writer.backTab();
      d_writer.println("_ior = (dll_f ? (*dll_f)() : NULL);");
      d_writer.println("sidl_DLL_deleteRef(dll);");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println("if (!_ior) {");
      d_writer.tab();
      d_writer.disableLineBreak();
      d_writer.println("fputs(\"Unable to find the implementation for "
        + id.getFullName() + "; please set SIDL_DLL_PATH\\n\", stderr);");
      d_writer.enableLineBreak();
      d_writer.println("exit(-1);");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.printlnUnformatted("#endif");
    }
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println("return _ior;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Return the pointer that provides access to the static EPV.
   */
  private void generateGetStaticEPV(SymbolID id) {
    String sepv_name = IOR.getSEPVName(id);
    d_writer.writeComment("Return pointer to static functions.", false);
    d_writer.println("static const " + sepv_name + "* _getSEPV(void)");
    d_writer.println("{");
    d_writer.tab();
    d_writer.println("static const " + sepv_name + " *_sepv = NULL;");
    d_writer.println("if (!_sepv) {");
    d_writer.tab();
    d_writer.println("_sepv = (*(_getIOR()->getStaticEPV))();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println("return _sepv;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  /**
   * Write the code to fetch the entry point vector needed for a method
   * call. If the case of a static EPV, it requires a function call, and
   * in the case of an object EPV, it requires dereferencing the IOR
   * pointer.
   *
   * @param isStatic <code>true</code> means it is a static method.
   * @param id       the name of the symbol who's EPV is needed.
   */
  private void getEntryPointVector(boolean isStatic, SymbolID id) {
    if (!isStatic) {
      d_writer.println(s_epv + " = " + s_proxy + s_self + "->d_epv;");
    }
  }

  /**
   * Write an argument to be passed to an IOR method call.  This is a helper
   * function for <code>makeMethodCall</code>.
   *
   * @param a           an Argument.
   * @param isInterface indicate whether then method is being called on
   *                    an object or interface.
   * @param needComma   whether a comma is needed before the next argument.
   * @return <code>true</code> means a comma is needed before the next
   *         argument; <code>false</code> means a comma is not needed.
   */
  private boolean passArgument(Argument a, boolean isInterface, 
                               boolean needComma)
  {
    String varName = a.getFormalName();
    if (!s_return.equals(varName)) {
      needComma = comma(d_writer, needComma);
      if (hasProxy(a.getType())) {
        if (Argument.IN != a.getMode()) {
          d_writer.print("&");
        }
        d_writer.print(s_proxy + varName);
        if (isInterface && s_self.equals(varName)) {
          d_writer.print("->d_object");
        }
      } else {
        if (Argument.IN == a.getMode()) {
          d_writer.print("*");
        }
        d_writer.print(varName);
      }
      needComma = true;
    }
    return needComma;
  }

  /**
   * Write the code to call the IOR.  This uses the original parameters
   * where possible and proxies otherwise.
   *
   * @param arguments   the list of arguments.  This list may have
   *                    implicitly defined arguments like the object, return
   *                    value and exception arguments.
   * @param m           the method to call on the IOR.
   * @param isInterface <code>true</code> means the method call of
   *                    an interface; <code>false</code> means the method
   *                    call to an object.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    something bad happened.
   */
  private void makeMethodCall(Extendable ext, List arguments, Method m,
                              boolean isInterface, boolean isSuper)
    throws CodeGenerationException
  {
    SymbolID id = ext.getSymbolID();
    Type returnType = m.getReturnType();
    Iterator i = arguments.iterator();
    boolean needComma = false;
    if (Type.VOID != returnType.getDetailedType()) {
      if (hasProxy(returnType)) {
        d_writer.print(s_proxy);
      } else {
        d_writer.print("*");
      }
      d_writer.println(s_return + " = ");
      d_writer.tab();
    }
    if(isSuper) {
      Class cls = (Class) ext;
      d_writer.println("(*(superEPV->"
        + IOR.getVectorEntry(m.getLongMethodName()) + "))(("
        + IOR.getObjectName(cls.getParentClass().getSymbolID()) + "*) ");
    } else {
      d_writer.println("(*(" + s_epv + "->" 
        + IOR.getVectorEntry(m.getLongMethodName()) + "))(");
    }
    d_writer.tab();
    while (i.hasNext()) {
      needComma = passArgument((Argument)i.next(), isInterface, needComma);
    }
    d_writer.println();
    d_writer.backTab();
    d_writer.println(");");
    if (Type.VOID != returnType.getDetailedType()) {
      d_writer.backTab();
    }
  }

  /**
   * If a sidl method, <code>m</code>, can thrown an exception, the stub
   * must check the exception argument. If the exception argument indicates
   * that an exception occured, it must be translated back to FORTRAN and
   * the other argument values should be ignored.
   *
   * @param m add exception handling if this sidl method can throw an 
   *              exception.
   */
  private void checkExceptionBlock(Method m) {
    if (!m.getThrows().isEmpty()) {
      d_writer.println("if (" + s_proxy + s_exception + ") {");
      d_writer.tab();
      d_writer.println("*" + s_exception + " = (ptrdiff_t)" + s_proxy 
        + s_exception + ";");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println("else {");
      d_writer.tab();
      d_writer.println("*" + s_exception + " = (ptrdiff_t)NULL;");
    }
  }

  /**
   * Write the end of the exception block if the sidl method can throw an
   * exception.
   *
   * @param m   close the exception block if this method can throw an
   *            exception.
   */
  private void endExceptionBlock(Method m) {
    if (!m.getThrows().isEmpty()) {
      d_writer.backTab();
      d_writer.println("}");
    }
  }

  /**
   * Generate the stub for a particular method. This generates the
   * signature, declares proxies, translates incoming values, makes the
   * call, and then does outgoing value processing.
   *
   * @param name      the name of the method. This may be different than the
   *                  original name of the method in <code>m</code>..
   * @param arguments the extended list of arguments. This has the argument
   *                  for <code>m</code> with extra arguments as needed for
   *                  the self pointer, the return value, and the exception
   *                  argument.
   * @param argumentsWithIndices argument plus any raw array index arguments
   * @param m         information about the argument to be written.
   * @param id        the name of the symbol who owns the method.
   * @param isInterface whether the symbol who owns the method is a class
   *                    or an interface.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  this exception provides feedback that something went wrong.
   */
  private void generateMethod(Extendable ext, String name, List arguments, 
                              List argumentsWithIndices, Method  m, SymbolID id,
                              boolean isInterface, boolean isSuper) 
    throws CodeGenerationException 
  {
    d_writer.writeComment(m, false);
    generateSignature(d_writer, name, argumentsWithIndices);
    d_writer.println();
    d_writer.println("{");
    d_writer.tab();
    declareEntryPointVector(m.isStatic(), id);
    declareProxies(arguments);
    copyIncomingValues(arguments);
    getEntryPointVector(m.isStatic(), id);
    makeMethodCall(ext, arguments, m, isInterface, isSuper);
    checkExceptionBlock(m);
    copyOutgoingValues(arguments);
    endExceptionBlock(m);
    freeResources(arguments);
    d_writer.backTab();
    d_writer.println("}");
  }

  static private Type convertRarrayToArray(Type array) {
    if (array.isRarray()) {
      return new Type(array.getArrayType(), array.getArrayDimension(),
                      array.getArrayOrder());
    }
    return array;
  }

  /**
   * Convert any rarray arguments to normal array arguments.
   */
  static public List convertRarrayToArray(List args) {
    ArrayList result  = new ArrayList(args.size());
    Iterator i = args.iterator();
    while (i.hasNext()) {
      Argument a = (Argument)i.next();
      if (a.getType().isRarray()) {
        result.add(new Argument(a.isCopy(), a.getMode(),
                                convertRarrayToArray(a.getType()),
                                a.getFormalName()));
      } else {
        result.add(a);
      }
    }
    return result;
  }

  /**
   * Add extra arguments to the original argument list of a method as needed
   * for the self pointer, the return value and the exception argument.
   * This makes these implicit arguments explicit and prevents having each
   * of these be a special case throughout the code.
   *
   * @param selfId   the name of the class/interface who owns the method.
   * @param m        the method whose argument list will be extended.
   * @param indices If true, get the argument list including rarray
   *                 indices.  Should be true for places that support rarrays.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception for problems in the code generation phase.
   */
  public static List extendArgs(SymbolID selfId, Method m, boolean indices) 
    throws CodeGenerationException
  {
    List origArgs = null;
    if(indices) {
      origArgs = m.getArgumentListWithIndices();
    } else { 
      origArgs = m.getArgumentListWithOutIndices();
    }

    ArrayList result = new ArrayList(origArgs.size() + 3);
    if (Method.STATIC != m.getDefinitionModifier()) {
      result.add(new Argument(false, Argument.IN, new Type(selfId), s_self));
    }
    result.addAll(origArgs);
    if (Type.VOID != m.getReturnType().getDetailedType()) {
      result.add(new Argument(false, Argument.OUT, m.getReturnType(), 
                              s_return));
    }
    if (!m.getThrows().isEmpty()) {
      Symbol ex = Utilities.lookupSymbol(s_exceptionFundamentalType);
      result.add(new Argument(false, Argument.OUT, 
                              new Type(ex.getSymbolID()), s_exception));
    }
    return result;
  }

  /**
   * Generate the expanded set of referenced <code>SymbolID</code>'s. This 
   * includes <code>sidl.BaseException</code> if any of the methods throws
   * an exception.
   *
   * @param ext       the class or interface to generate includes for.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception for problems in the code generation phase.
   */
  public static Set extendedReferences(Extendable ext)
    throws CodeGenerationException
  {
    Set result = new HashSet();
    for(Iterator i = ext.getMethods(true).iterator(); i.hasNext(); ) {
      Method method = (Method)i.next();
      result.addAll(method.getSymbolReferences());
      if (!method.getThrows().isEmpty()) {
        Symbol symbol = Utilities.lookupSymbol(s_exceptionInterfaceType);
        result.add(symbol.getSymbolID());
      }
    }
    return result;
  }

  /**
   * Generate a sequence of <code>#include</code. preprocessor directives
   * required by the stub.
   * 
   * @param writer    the output device where output is sent.
   * @param ext       the class or interface to generate includes for.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception for problems in the code generation phase.
   */
  public static void generateIncludes(LanguageWriterForC writer,
                                      Extendable ext) 
    throws CodeGenerationException
  {
    final SymbolID id = ext.getSymbolID();

    writer.printlnUnformatted("#include <stddef.h>");
    writer.printlnUnformatted("#include <stdlib.h>");
    writer.generateInclude("sidlfortran.h", false);
    if (d_isF90) {
      writer.generateInclude("sidlf90array.h", true);
    }
    writer.generateInclude("sidl_header.h", false);
    writer.generateInclude("sidl_interface_IOR.h", true);
    writer.printlnUnformatted("#include <stdio.h>");
    if (!BabelConfiguration.isSIDLBaseClass(id)) {
      writer.printlnUnformatted("#include \"babel_config.h\"");
      writer.printlnUnformatted("#ifdef SIDL_DYNAMIC_LIBRARY");
      writer.generateInclude("sidl_Loader.h", false);
      writer.printlnUnformatted("#endif");
    }
    writer.generateInclude(IOR.getHeaderFile(id), false);
    if (Fortran.needsAbbrev()) {
      writer.generateInclude(Fortran.getStubNameFile(id), false);
    }

    Set includes = extendedReferences(ext);
    includes.remove(id);

    for(Iterator i = includes.iterator(); i.hasNext(); ) {
      writer.generateInclude(IOR.getHeaderFile((SymbolID)i.next()), false);
    }
  }

  /**
   * Generate the extended argument list for a method and then generate the
   * stub code for the method.
   * 
   * @param m           the method to be generated.
   * @param id          the name of the class/interface who owns the method.
   * @param isInterface is the method owned by a class or interface.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception for problems in the code generation phase.
   */
  private void extendAndGenerate(Extendable ext, Method m, SymbolID id,
                                 boolean isInterface) 
    throws CodeGenerationException
  {
    String name = Fortran.getMethodStubName(id, m);
    List extendedArgs = extendArgs(id, m, false);
    List extendedArgsWithIndices = extendArgs(id, m, true);
    generateMethod(ext, name, extendedArgs, extendedArgsWithIndices,
                   m, id, isInterface, false);
    if (d_isF90 && m.hasRarray()) {
      extendedArgs = convertRarrayToArray(extendedArgs);
      generateMethod(ext, Fortran.getAltStubName(id, m),
                     extendedArgs, extendedArgs, m, id, isInterface, false);
    }
  }

  /**
   * Generate the extended argument list for a method and then generate the
   * stub code for the method.
   * 
   * @param m           the method to be generated.
   * @param id          the name of the class/interface who owns the method.
   * @param isInterface is the method owned by a class or interface.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception for problems in the code generation phase.
   */
  private void extendAndGenerateSuper(Class cls, Method m, SymbolID id,
                                      boolean isInterface) 
    throws CodeGenerationException
  {
    String name = Fortran.getMethodSuperName(id, m);
    List extendedArgs = extendArgs(id, m, false);
    List extendedArgsWithIndices = extendArgs(id, m, true);
    generateMethod(cls, name, extendedArgs, extendedArgsWithIndices,
                   m, id, isInterface, true);
  }

  /**
   * Generate the create method for a class.  The create method makes a new
   * wrapped instance of a class.
   *
   * @param id    the name of the class to write a creation method for.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *   a catch all exception for problems during the code generation phase.
   */
  private void generateCreateMethod(SymbolID id) throws CodeGenerationException
  {
    String newName = d_isF90 ? "_new" : "__create";
    String methodName = id.getFullName().replace('.', '_') + newName 
        + Fortran.getMethodSuffix();

    d_writer.writeComment("Constructor for the class.", false);
    d_writer.println("void");
    d_writer.disableLineBreak();
    d_writer.println(Fortran.getFortranSymbol() + "(" + methodName.toLowerCase()
      + ',' + methodName.toUpperCase() + ',' + methodName + ")");
    d_writer.enableLineBreak();
    d_writer.println("(");
    d_writer.tab();
    d_writer.println("int64_t *" + s_self);
    d_writer.backTab();
    d_writer.println(")");
    d_writer.println("{");
    d_writer.tab();
    d_writer.println("*self = (ptrdiff_t) (*(_getIOR()->createObject))();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  private void generateCast(SymbolID id) throws CodeGenerationException {
    Method m = Fortran.createCast(id);
    List extendedArgs = extendArgs(id, m, false);
    String name = Fortran.getMethodStubName(id, m);
    d_writer.writeComment(m, false);
    generateSignature(d_writer, name, extendedArgs);
    d_writer.println();
    d_writer.println("{");
    d_writer.tab();
    d_writer.println("struct sidl_BaseInterface__object  *_base =");
    d_writer.tab();
    d_writer.println("(struct sidl_BaseInterface__object *)(ptrdiff_t)*ref;");
    d_writer.backTab();
    d_writer.println("if (_base) {");
    d_writer.tab();
    d_writer.println("*retval = (ptrdiff_t)(");
    d_writer.tab();
    d_writer.println("*_base->d_epv->" 
      + IOR.getVectorEntry(IOR.getBuiltinName(IOR.CAST)) + ")(");
    d_writer.println("_base->d_object,");
    d_writer.println("\"" + id.getFullName() + "\");");
    d_writer.backTab();
    d_writer.backTab();
    d_writer.println("} else {");
    d_writer.tab();
    d_writer.println("*retval = 0;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  private void generateCastTwo(Extendable ext, SymbolID id, boolean isInterface)
    throws CodeGenerationException
  {
    Method m = IOR.getBuiltinMethod(IOR.CAST, id);
    List args = extendArgs(id, m, false);
    generateMethod(ext, Fortran.getMethodStubName(id, 
                                                  Fortran.createCastTwo(id)),
                   args, args, m, id, isInterface, false);
  }

  /**
   * This procedure writes the whole stub file for a class/interface.
   * It writes a banner, writes the includes needed, generates a create
   * method for classes, generates the cast methods, and then creates
   * stubs for all the methods.
   *
   * @param ext   a class/interface for which a stub file will be written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception for problems during the code generation phase.
   */
  private void generateExtendable(Extendable ext)
    throws CodeGenerationException
  {
    Iterator i  = ext.getMethods(true).iterator();
    final SymbolID id = ext.getSymbolID();
    final boolean isInterface = ext.isInterface();

    d_writer.writeBanner(ext, Fortran.getStubFile(id), false, 
                         CodeConstants.C_DESC_STUB_PREFIX + id.getFullName());
    d_writer.writeComment(ext, false);

    generateIncludes((LanguageWriterForC) d_writer, ext);
    d_writer.println();

    if (!ext.isInterface()) {
      generateGetIOR(id);
    }
    if (ext.hasStaticMethod(true)) {
      generateGetStaticEPV(id);
    }

    if (!ext.isAbstract()) {
      generateCreateMethod(id);
    }

    generateCast(id);
    generateCastTwo(ext, id, isInterface);
                         
    while (i.hasNext()) {
      d_writer.println();
      extendAndGenerate(ext, (Method)i.next(), id, isInterface);
    }

    d_writer.println();
    FortArrayMethods fam = new FortArrayMethods(id, false);
    fam.generateStub(d_writer);
  }

  /**
   * Generate a FORTRAN include file containing integer constants for the
   * members of an enumerated type.
   * 
   * @param enm an enumeration object to provide an include file for.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception to indicate problems during the code generation
   *  phase of the sidl processing.
   */
  public void generateEnum(Enumeration enm) throws CodeGenerationException {
    final SymbolID id = enm.getSymbolID();
    if (d_writer instanceof LanguageWriterForFortran) {
      Iterator i;
      d_writer.writeBanner(enm, Fortran.getEnumStubFile(id), false,
                           CodeConstants.C_DESC_STUB_PREFIX + id.getFullName());
      d_writer.println();
      d_writer.writeComment(enm, false);
      d_writer.println();
      i = enm.getEnumerators().iterator();
      while (i.hasNext()){
        String sym = (String)i.next();
        Comment cmt = enm.getEnumeratorComment(sym);
        d_writer.writeComment(cmt, true);
        d_writer.print(Fortran.getReturnString(new Type(id)));
        if (d_isF90) {
          d_writer.print(" :: ");
        } else {
          d_writer.print(" ");
        }
        d_writer.println(sym);
        d_writer.println("parameter (" + sym + " = " 
          + enm.getEnumeratorValue(sym) + ")");
        if (cmt != null) {
          d_writer.println();
        }
      }
    } else {
      d_writer.writeBanner(enm, Fortran.getEnumStubImpl(id), false,
                           CodeConstants.C_DESC_STUB_PREFIX + id.getFullName());
      ((LanguageWriterForC)d_writer).generateInclude("sidl_int_IOR.h", true);
      ((LanguageWriterForC)d_writer).generateInclude("sidlfortran.h", true);
      d_writer.printlnUnformatted("#include <stddef.h>");
      if (Fortran.needsAbbrev()) {
        ((LanguageWriterForC)d_writer).
          generateInclude(Fortran.getStubNameFile(id), false);
      }
      FortArrayMethods fam = new FortArrayMethods(id, true);
      fam.generateStub(d_writer);
    }
  }

  /**
   * Generate a C file to provide FORTRAN stubs for a sidl
   * object/interface. The stubs allow FORTRAN clients to make calls on
   * objects and interfaces or static methods.  No stub code is generated
   * enumerated types and packages.  Outside clients typically use 
   * {@link #generateCode generateCode} instead of calling this method
   * directly.
   *
   * @param symbol  the symbol for which stubs will be generated.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception to indicate problems during the code generation
   *  phase of the sidl processing.
   */
  public void generateCode(Symbol symbol) throws CodeGenerationException {
    final String pre = "fortran.StubSource.generateCode: ";
    switch(symbol.getSymbolType()) {
    case Symbol.CLASS:
    case Symbol.INTERFACE:
      if (d_writer instanceof LanguageWriterForC) {
        generateExtendable((Extendable)symbol);
        RMIStubSource.generateCode( symbol, (LanguageWriterForC) d_writer );
      } else {
        throw new CodeGenerationException(pre + "Extendable stub requires C "
                    + "language writer.");
      }
      break;
    case Symbol.ENUM:
      generateEnum((Enumeration)symbol);
      break;
    case Symbol.PACKAGE:
      break;
    default:
      throw new CodeGenerationException(pre + "Unsupported symbol type.");
    }
  }

  /**
   * Generate a C file to provide FORTRAN stubs for a sidl
   * object/interface. The stubs allow FORTRAN clients to make calls on
   * objects and interfaces or static methods.  No stub code is generated
   * enumerated types and packages.
   *
   * @param ext     the symbol for which stubs will be generated.
   * @param writer  the output device where the stub should be written.
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *  a catch all exception to indicate problems during the code generation
   *  phase of the sidl processing.
   */
  public static void generateCode(Symbol ext, LanguageWriter writer)
    throws CodeGenerationException
  {
    StubSource source = new StubSource(writer);
    source.generateCode(ext);
  }
}
