
package gov.llnl.babel.backend.jdk;

import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeSplicer;
import gov.llnl.babel.backend.FileManager;
import gov.llnl.babel.backend.writers.LanguageWriterForJava;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.SymbolUtilities;

import java.util.Iterator;
import java.util.List;
import java.util.Set;

/**
 * Create and write a source file for a Java extension class to wrap a 
 * BABEL extendable in a Java object. 
 */
public class JavaImplSource {
  
  private Extendable d_ext = null;
  private LanguageWriterForJava d_writer = null;
  private CodeSplicer d_splicer = null;
  private String d_classname = null;  // So files will be named with _Impl suffix, for naming collisions
  
  /**
   * Create an object capable of generating the source file for Java implementation
   *
   * @param ext   an interface or class symbol that needs source
   *              file for a Java extension class.
   */
  public JavaImplSource(Extendable ext) throws CodeGenerationException {
    if (ext == null) {
      throw new CodeGenerationException("Unexpected null extendable object");
    }
    
    d_ext= ext;
      
    SymbolID id = d_ext.getSymbolID();
    d_classname = Java.getJavaServerClassName(id); 
      
    int type = d_ext.getSymbolType();
    String filename = Java.getJavaImplSourceFile(id);
      
    // Set up the writer and the splicer
    try{
      // To generate _Impl file in correct directory
      FileManager.getInstance().setJavaStylePackageGeneration(true);
        
      d_splicer = FileManager.getInstance().getCodeSplicer(id, type, filename);
      d_writer = new LanguageWriterForJava((FileManager.getInstance()).createFile(id,
                                                                                  type,
                                                                                  "JAVAIMPL",
                                                                                  filename));
      FileManager.getInstance().setJavaStylePackageGeneration(false);
      d_writer.writeBanner(d_ext, filename,true, CodeConstants.C_DESC_IMPL_PREFIX + id.getFullName());
        
    } catch(java.io.IOException e){
      if (d_writer != null) {
        d_writer.close();
        d_writer = null;
      }
      throw new CodeGenerationException("IOException : " + e.getMessage() );
    }

  }
  
  /**
   * Writes Java implementation based on the provided symbol
   *
   * @param ext   an interface or class symbol that needs source
   *              file for a Java extension class.
   */
  public static void generateCode(Extendable ext) 
    throws CodeGenerationException
  {
    /** 
     * We don't generate Impl files for Interfaces or Abstact Classes
     */

    JavaImplSource source = new JavaImplSource(ext);
    source.generateCode();
    
  }    
  
  /**
   * Writes Java implimentation based on the symbol given to the constructor
   */
  public synchronized void generateCode() 
    throws CodeGenerationException 
  {
    writePackageImports();
    writeClassBeginning();
    writeCtorDtor();
    writeSIDLDefinedMethods();
    d_splicer.splice( d_ext.getSymbolID().getFullName() + "._misc",
                      d_writer, "miscellaneous");
    d_writer.println();
    writeClassEnd();
    checkSplicer();
	
    d_writer.close();	
  }



  /********************************************************************************************
   *                           Private support methods
   ********************************************************************************************/

  /**
   * Writes the necessary package and import statements needed
   */
  private void writePackageImports() {
    writePackage();
    writeImports();
    spliceImports();
  }
    
  private void writePackage() {
    String pkg = SymbolUtilities.getParentPackage(d_ext.getSymbolID().getFullName());
    d_writer.println("package " + pkg + ";");
    d_writer.println();
  }
    
  private void writeImports() {
    Iterator i = d_ext.getSymbolReferences().iterator();
    while (i.hasNext()) {
      SymbolID id = (SymbolID) i.next();
      if (! id.getFullName().equals(d_ext.getSymbolID().getFullName())) {
        d_writer.println("import " + id.getFullName() + ";");
      }
    }
    d_writer.println();
  }
    
  private void spliceImports() {
    d_splicer.splice(d_ext.getSymbolID().getFullName() + "._imports",
                     d_writer, "additional imports");
    d_writer.println();
  }

  /**
   * Writes class declaration and splicer block for private data
   */
  private void writeClassBeginning() {
    String name = d_ext.getSymbolID().getShortName();
    SymbolID id = d_ext.getSymbolID();
    String splicer_symbol_data = d_ext.getSymbolID().getFullName() + "._data";
    String splicer_symbol_load = d_ext.getSymbolID().getFullName() + "._load";
	
    d_writer.writeComment(d_ext, true);
    d_writer.println("public class " + d_classname + " extends " + d_ext.getSymbolID().getShortName());
    d_writer.println("{");
    d_writer.println();
    d_writer.tab();
    d_splicer.splice(splicer_symbol_data, d_writer,
                     "private data");
    d_writer.println();
    d_writer.println("static { ");
    d_splicer.splice(splicer_symbol_load, d_writer, 
                     "class initialization");
    d_writer.println("}");
    d_writer.println();
  }
    
   
    
  /**
   * Writes Java constructor and a "destructor", method that can be utilized to 
   * "destruct" a Java object.
   */
  private void writeCtorDtor() {
    String name = d_ext.getSymbolID().getShortName();
    // Constructor
    d_writer.writeComment("User defined constructor", true);
    d_writer.println("public " +d_classname + "(long IORpointer){");
    d_writer.tab();
    d_writer.println("super(IORpointer);");
    d_splicer.splice(d_ext.getSymbolID().getFullName() + "." + name,
                     d_writer, "constructor");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();

    // "destructor"
    d_writer.writeComment("User defined destructing method", true);
    d_writer.println("public void finalize() throws Throwable{");
    d_writer.tab(); 
    d_writer.println("super.finalize();");
    d_splicer.splice(d_ext.getSymbolID().getFullName() + "._dtor",
                     d_writer, "destructor");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();		     
  }
    
  /**
   * Writes both static and nonstatic methods defined in the sidl file
   */
  private void writeSIDLDefinedMethods() {
    List static_methods = (List) d_ext.getStaticMethods(false);
    if (static_methods.size() > 0) {
      d_writer.writeCommentLine("user defined static methods:");
      for (Iterator m = static_methods.iterator(); m.hasNext(); ) {
        Method method = (Method) m.next();
        generateMethod(method);
      }
    } 
    else {
      d_writer.writeCommentLine("user defined static methods: (none)");
    }
	
    d_writer.println();
    List nonstatic_methods = (List) d_ext.getNonstaticMethods(false);
    if (nonstatic_methods.size() > 0) {
      d_writer.writeCommentLine("user defined non-static methods:");
      for (Iterator m = nonstatic_methods.iterator(); m.hasNext(); ) {
        Method method = (Method) m.next();
        if (!method.isAbstract()) {
          generateMethod(method);
        }
      }
    } 
    else {
      d_writer.writeCommentLine("user defined non-static methods: (none)");
    }
    d_writer.println();
  }
    
  /**
   * Writes complete method header and comments for given parameter. 
   * Includes a default return if necessary.
   */
  private void generateMethod( Method method ) 
  {
    if ( method == null ) { return; }
	
    generateMethodBeginning( method );
	
    if (method.getArgumentList().size() > 0) {
      generateArgumentList(method);  
    } 
    else {
      d_writer.println(" () ");
    }

    if(method.getThrows().size() > 0) {
      generateThrowsList(method); 
    }
    generateMethodBody(method); 
  }


  private void generateMethodBeginning ( Method method )
  {
    d_writer.writeComment (method, true);
    d_writer.print("public ");
    if(method.isStatic()){
      d_writer.print("static ");
    }
    d_writer.print(Java.getJavaReturnType(method.getReturnType()) + " ");
    d_writer.print(Java.getJavaServerMethodName(method));
  }

  private void generateArgumentList( Method method )
  {
    if ( method == null ) { return; }
   
    List args = method.getArgumentList();

    d_writer.println(" (");
    d_writer.tab();
    for( Iterator a = args.iterator(); a.hasNext(); ) { 
      Argument arg = (Argument) a.next();
      d_writer.print("/*" + arg.getModeString() + "*/ " +
                     Java.getJavaServerArgument( arg ) + " " + 
                     arg.getFormalName());
      if ( a.hasNext() ) { 
        d_writer.println(",");
      }
    }
    d_writer.println(" ) ");
    d_writer.backTab();
  }
    
  private void generateThrowsList( Method method )  
  {
    if ( method == null ) { return; }	
 
    Set exceptions = method.getThrows();
	
    d_writer.print( "throws " );
    d_writer.tab();
    for( Iterator e = exceptions.iterator(); e.hasNext(); ) { 
      SymbolID id = (SymbolID) e.next();
      d_writer.print( Java.getFullJavaSymbolName(id) );
      if ( (!SymbolUtilities.isBaseException(id)) 
           && (SymbolTable.getInstance().lookupSymbol(id).isInterface()) ) {
            d_writer.print(".Wrapper");
          }
      if ( e.hasNext() ) { 
        d_writer.println(", ");
      }
    }
    d_writer.println();
    d_writer.backTab();
	
  }
    

  private void generateMethodBody ( Method method ) 
  {
    d_writer.println("{");
    d_writer.tab();
    d_splicer.splice(d_ext.getSymbolID().getFullName() + "." + method.getLongMethodName(),
                     d_writer, 
                     method.getShortMethodName(),
                     "return " + Java.getDefaultReturnValue(method) + ";");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();

  }

  
  /**
   * Closes out class
   */
  private void writeClassEnd() 
  {
    String name = d_ext.getSymbolID().getShortName();
    d_writer.backTab();
    d_writer.print("} "); // end of class
    d_writer.writeCommentLine("end class " + name );
    d_writer.println();
  }

  /**
   * Print any remaining symbols in the splicer
   */
  private void checkSplicer() 
  {
    if (d_splicer.hasUnusedSymbolEdits()) {
      d_writer.beginBlockComment(true);
      d_writer.println(CodeConstants.C_BEGIN_UNREFERENCED_METHODS);
      d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT1);
      d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT2);
      d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT3);
      d_writer.endBlockComment(true);
      d_splicer.outputUnusedSymbolEdits( d_writer.getPrintWriter() );
      d_writer.writeCommentLine(CodeConstants.C_END_UNREFERENCED_METHODS);
    }
  }
}
