//
// File:        RmiSkelSource.java
// Package:     gov.llnl.babel.backend.cxx
// Revision:    @(#) $Revision: 4434 $
// Date:        $Date: 2005-03-17 09:05:29 -0800 (Thu, 17 Mar 2005) $
// Description: Write Cxx extension header file for a BABEL extendable
// 
// This is typically directed by GenCxxClient.
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.rmi2;

import gov.llnl.babel.BabelConfiguration;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.rmi2.Cxx;
import gov.llnl.babel.backend.IOR;
import gov.llnl.babel.backend.LevelComparator;
import gov.llnl.babel.backend.writers.LanguageWriterForCxx;
import gov.llnl.babel.symbols.Argument;
import gov.llnl.babel.symbols.Comment;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.SymbolID;
import gov.llnl.babel.symbols.SymbolTable;
import gov.llnl.babel.symbols.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;

/**
 * Create and write a header for a Cxx C extension class to wrap a 
 * BABEL extendable in a Cxx object. The header has to expose a 
 * function to create a wrapped IOR, a function to check if a 
 * <code>PyObject</code> is an instance of this extension type, and
 * an import macro.
 */
public class RmiSkelSource {
  private Extendable d_ext = null;
  private LanguageWriterForCxx d_writer = null;
  private String d_self = null;
  
  /**
   * Create an object capable of generating the header file for a
   * BABEL extendable.
   *
   * @param ext   an interface or class symbol that needs a header
   *              file for a Cxx C extension class.
   */
  public RmiSkelSource(Extendable ext) {
    d_ext = ext;
  }
  
  /**
   * Generate the header file for the extendable with which this object was
   * created.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception for problems during the code
   *    generation phase.
   */
  public synchronized void generateCode() 
    throws CodeGenerationException 
  {
    String filename = Cxx.generateFilename( d_ext.getSymbolID(), 
                                            Cxx.FILE_ROLE_STUB, 
                                            Cxx.FILE_TYPE_CXX_SOURCE );
    //System.out.println("Create " + filename + "..." );

    if ( d_ext.isInterface() ) { 
      d_self = "d_self->d_object";
    } else { 
      d_self = "d_self";
    }

    try { 
      d_writer = Cxx.createSource( d_ext, Cxx.FILE_ROLE_STUB, "STUBSRCS" );
      d_writer.println();

      writeIncludes();
      writeRmiIncludes(); //Rmi
			writeNamespaceDebug();//Rmi

      writeUserDefinedMethods(); //Rmi 

      writeConstructors();

      writeCastingOperators();

      if ( !d_ext.isInterface()) {
        writeDynamicImplStuff();
      }
    } finally { 
      if (d_writer != null) {
        d_writer.close();
        d_writer = null;
      }
    }
  }

  private void writeIncludes() { 
    SymbolID id = d_ext.getSymbolID();
    d_writer.generateInclude( Cxx.generateFilename( d_ext.getSymbolID(), 
                                                    Cxx.FILE_ROLE_STUB, 
                                                    Cxx.FILE_TYPE_CXX_HEADER ),
                              true);
    d_writer.generateInclude( Cxx.generateFilename( "sidl.BaseInterface",
                                                    Cxx.FILE_ROLE_STUB, 
                                                    Cxx.FILE_TYPE_CXX_HEADER
                                                    ),
                              true );
    d_writer.generateInclude(  Cxx.generateFilename( "sidl.BaseClass",
                                                    Cxx.FILE_ROLE_STUB, 
                                                    Cxx.FILE_TYPE_CXX_HEADER
                                                    ),
                               true );
    if ( d_ext.hasExceptionThrowingMethod(true) ) { 
      d_writer.generateInclude( Cxx.generateFilename( "sidl.BaseException",
                                                    Cxx.FILE_ROLE_STUB, 
                                                    Cxx.FILE_TYPE_CXX_HEADER), 
                                true );
    }
    d_writer.generateInclude( "sidl_String.h", false );
    if (!BabelConfiguration.isSIDLBaseClass(id)) {
      d_writer.generateInclude("babel_config.h", false);
      d_writer.printlnUnformatted("#ifdef SIDL_DYNAMIC_LIBRARY");
      d_writer.printlnUnformatted("#include <stdio.h>");
      d_writer.printlnUnformatted("#include <stdlib.h>");
      d_writer.generateInclude("sidl_Loader.hh", false);
      d_writer.printlnUnformatted("#endif");
    }

    d_writer.println();
  } 

  private void writeConstructors() { 
    SymbolID id = d_ext.getSymbolID();
    final String extName = IOR.getExternalName(id);
    final String fullName = Cxx.getObjectName(id);
    final String fullNameWithoutLeadingColons = Cxx.getSymbolNameWithoutLeadingColons(id,"");
    final String name = id.getShortName();    
    final String ior_ptr = "ior_t*";

    if( !d_ext.isAbstract() ) { 
      d_writer.writeCommentLine("static constructor");
      d_writer.println(fullName);
      d_writer.println(fullNameWithoutLeadingColons + "::_create() {");
      d_writer.tab();
      d_writer.println( Cxx.getObjectName(id) + " self( (*_get_ext()->createObject)(), false );");
      d_writer.println("return self;");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println();
    }

    d_writer.writeCommentLine("default destructor");
    d_writer.println(fullNameWithoutLeadingColons + "::~" + name + " () {");
    d_writer.tab();
    d_writer.println("if ( d_self != 0 ) {");
    d_writer.tab();
    d_writer.println("deleteRef();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();

    d_writer.writeCommentLine("copy constructor");
    d_writer.println( fullNameWithoutLeadingColons + "::" + name + 
                      " ( const " + fullName + "& original ) {");
    d_writer.tab();
    d_writer.println("d_self = " + Cxx.constCast( ior_ptr, 
                                                "original.d_self" ) +
                     ";");
    d_writer.println("d_weak_reference = original.d_weak_reference;");
    d_writer.println("if (d_self != 0 ) {");
    d_writer.tab();
    d_writer.println("addRef();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();


    d_writer.writeCommentLine("assignment operator");
    d_writer.println( fullName + "&");
    d_writer.println( fullNameWithoutLeadingColons + "::operator=( const " + fullName + "& rhs ) {");
    d_writer.tab();
    d_writer.println("if ( d_self != rhs.d_self ) {");
    d_writer.tab();
    d_writer.println("if ( d_self != 0 ) {");
    d_writer.tab();
    d_writer.println("deleteRef();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println("d_self = " + Cxx.constCast( ior_ptr, 
                                                "rhs.d_self") + ";");
    d_writer.println("d_weak_reference = rhs.d_weak_reference;");
    d_writer.println("if ( d_self != 0 ) {");
    d_writer.tab();
    d_writer.println("addRef();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println("return *this;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  private void writeCastingOperators() { 
    SymbolID id = d_ext.getSymbolID();
    final String extName = IOR.getExternalName(id);
    final String fullName = Cxx.getObjectName(id);
    final String fullNameWithoutLeadingColons = Cxx.getSymbolNameWithoutLeadingColons(id,"");
    final String name = id.getShortName();   

 
    d_writer.writeCommentLine("conversion from ior to C++ class");
    d_writer.println( fullNameWithoutLeadingColons + "::" + name + 
                      " ( " + fullName + "::ior_t* ior ) ");
    d_writer.println("    : d_self( ior ), d_weak_reference(false) {");
    d_writer.tab();
    d_writer.println("if ( d_self != 0 ) {");
    d_writer.tab();
    d_writer.println("addRef();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();

    d_writer.writeCommentLine("Alternate constructor: does not call addRef()");
    d_writer.writeCommentLine("(sets d_weak_reference=isWeak)");
    d_writer.writeCommentLine("For internal use by Impls (fixes bug#275)");
    d_writer.println( fullNameWithoutLeadingColons + "::" + name +" ( " + fullName + 
                      "::ior_t* ior, bool isWeak ) ");
    d_writer.println("    : d_self( ior ), d_weak_reference(isWeak) { ");
    //d_writer.increaseTabLevel();
    //d_writer.println("// addRef();");
    //d_writer.decreaseTabLevel();
    d_writer.println("}");
    d_writer.println();


    d_writer.writeCommentLine("conversion from a StubBase");
    d_writer.println( fullNameWithoutLeadingColons + "::" + name + 
                      " ( const ::sidl::StubBase& base )");
    d_writer.println("{");
    d_writer.tab();
    d_writer.println(
      "d_self = " + Cxx.reinterpretCast(
        "ior_t*",
        "base._cast(\"" + id.getFullName() + "\")") + ";");
    d_writer.println("d_weak_reference = false;");
    d_writer.println("if (d_self != 0) {");
    d_writer.tab();
    d_writer.println("addRef();");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();

    d_writer.writeCommentLine("protected method that implements casting");
    d_writer.println("void* " + fullNameWithoutLeadingColons + "::_cast(const char* type) const");
    d_writer.println("{");
    d_writer.tab();
    d_writer.println("void* ptr = 0;");
    d_writer.println("if ( d_self != 0 ) {");
    d_writer.tab();
    d_writer.print("ptr = ");
    if (d_ext.isInterface()) {
      d_writer.print(Cxx.reinterpretCast(
        "void*",
        "(*d_self->d_epv->f__cast)(d_self->d_object, type)"));
    } else {
      d_writer.print(Cxx.reinterpretCast(
        "void*",
        "(*d_self->d_epv->f__cast)(d_self, type)"));
    }
    d_writer.println(";");
    d_writer.backTab();
    d_writer.println("}");    
    d_writer.println("return ptr;");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  private void writeDynamicImplStuff() throws CodeGenerationException { 
    SymbolID id = d_ext.getSymbolID();
    final String extName = IOR.getExternalName(id);
    final String fullName = Cxx.getObjectName(id);
    final String fullNameWithoutLeadingColons = Cxx.getSymbolNameWithoutLeadingColons(id,"");
    d_writer.writeCommentLine("Static data type");
    d_writer.println("const " + fullName + "::ext_t * " + fullNameWithoutLeadingColons +
                     "::s_ext = 0;");
    d_writer.println();

    d_writer.writeCommentLine("private static method to get static data type");
    d_writer.println("const " + fullName + "::ext_t *");
    d_writer.println( fullNameWithoutLeadingColons + "::_get_ext()");
    d_writer.println("  throw (::sidl::NullIORException)");
    d_writer.println("{");
    d_writer.tab();
    d_writer.println("if (! s_ext ) {");
    d_writer.tab();
    if (BabelConfiguration.isSIDLBaseClass(id)) { 
      d_writer.println("s_ext = " + IOR.getExternalFunc(id) + "();");
    } else {
      d_writer.printlnUnformatted("#ifdef SIDL_STATIC_LIBRARY");
      d_writer.println("s_ext = " + IOR.getExternalFunc(id) + "();");
      d_writer.printlnUnformatted("#else");
      d_writer.println("::sidl::DLL dll = ::sidl::DLL::_create();");
      d_writer.println("const ext_t *(*dll_f)(void);");
      d_writer.writeCommentLine("check global namespace for symbol first");
      d_writer.println("if (dll._not_nil() && dll.loadLibrary(\"main:\", TRUE, FALSE)) {");
      d_writer.tab();
      d_writer.println("dll_f =");
      d_writer.tab();
      d_writer.print("(const ext_t *(*)(void)) ");
      d_writer.println("dll.lookupSymbol(");
      d_writer.tab();
      d_writer.println("\"" + IOR.getExternalFunc(id) + "\");");
      d_writer.backTab();
      d_writer.backTab();
      d_writer.println("s_ext = (dll_f ? (*dll_f)() : NULL);");
      d_writer.backTab();
      d_writer.println("}");

      d_writer.println("if (!s_ext) {");
      d_writer.tab();
      d_writer.println("dll = ::sidl::Loader::findLibrary(\"" +
                       id.getFullName() + "\",");
      d_writer.tab();
      d_writer.println("\"ior/impl\", ::sidl::Scope_SCLSCOPE,");
      d_writer.println("::sidl::Resolve_SCLRESOLVE);");
      d_writer.backTab();
      d_writer.println("if (dll._not_nil()) {");
      d_writer.tab();
      d_writer.println("dll_f =");
      d_writer.tab();
      d_writer.print("(const ext_t *(*)(void)) ");
      d_writer.println("dll.lookupSymbol(");
      d_writer.tab();
      d_writer.println("\"" + IOR.getExternalFunc(id) + "\");");
      d_writer.backTab();
      d_writer.backTab();
      d_writer.println("s_ext = (dll_f ? (*dll_f)() : NULL);");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.backTab();
      d_writer.println("}");
      d_writer.println("if (!s_ext) {");
      d_writer.tab();
      d_writer.disableLineBreak();
      d_writer.println("throw ::sidl::NullIORException( ::std::string (");
      d_writer.tab();
      d_writer.println("\"cannot find implementation for " + 
		       id.getFullName() + 
		       "; please set SIDL_DLL_PATH\"");
      d_writer.backTab();
      d_writer.println("));");
      d_writer.enableLineBreak();
      d_writer.backTab();
      d_writer.println("}");
      d_writer.printlnUnformatted("#endif");
    }
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println("return s_ext;");
    d_writer.backTab();
    d_writer.println("}");   
    d_writer.println();    
  }

  private void writeUserDefinedMethods() throws CodeGenerationException { 
    d_writer.beginBoldComment();
    d_writer.println("User Defined Methods");
    d_writer.endBoldComment();

    Iterator m = null;
    m = d_ext.getStaticMethods(true).iterator();
    while (m.hasNext()) {
      Method method = (Method) m.next();
      generateMethodDispatch( method, "user defined static method" );
      d_writer.println();
    }

    m = d_ext.getNonstaticMethods(true).iterator();
    while (m.hasNext()) {
      Method method = (Method) m.next();
      generateMethodDispatch( method, "user defined non-static method." );
      d_writer.println();
    }

		writeRmiSkel(); //Rmi, rmi
		writeRmiInvocationHandler();//Rmi,rmi

    d_writer.beginBoldComment();
    d_writer.println("End User Defined Methods");
    d_writer.println("(everything else in this file is specific to");
    d_writer.println(" Babel's C++ bindings)");
    d_writer.endBoldComment();
  }

  private void generateMethodDispatch( Method m, String altcomment ) 
  throws CodeGenerationException { 
    if ( m == null ) { return; }
    final ArrayList vArgs = m.getArgumentList();
    final int nargs = vArgs.size();
    // everything through first "{"
    StringBuffer func_decl = new StringBuffer( nargs * 32 ); 
    // after first "{" but before "if (d_self != NULL )"
    StringBuffer init_stuff = new StringBuffer( nargs * 128 );
    // holds things before the ior_call
    StringBuffer pre_ior = new StringBuffer( nargs * 32 );
    // The actual IOR dispatch
    StringBuffer ior_call = new StringBuffer( nargs * 16 );
    // holds things after the ior_call
    StringBuffer post_ior = new StringBuffer( nargs * 128);
    // what to do if d_self == 0;
    StringBuffer retn_err = new StringBuffer( 128 );
    String extra_close_paren = "";

    String shortMethodName = m.getShortMethodName();
    String longMethodName = m.getLongMethodName();
    String className = d_ext.getSymbolID().getShortName();

    Comment comment = m.getComment();
    d_writer.writeComment( comment, altcomment );
    if ( shortMethodName.equals(className) ) { 
      System.out.println("WARNING: gov.llnl.babel.backend.rmi.RmiSkelSource: sidl / C++ conflict!");
      System.out.println("         methodName == className is not allowed in C++");
      System.out.println("         (this is restricted to constructors in C++)");
      System.out.println("         changing to " + className + "::f_" + shortMethodName + "()");
      shortMethodName = "f_" + shortMethodName;
    }

    Type return_type = m.getReturnType();
    SymbolID id = return_type.getSymbolID();
    /* if ( m.isStatic() ) {
     *   func_decl.append("static ");
     * } 
     */                   
    func_decl.append( Cxx.getReturnString( return_type ) );
    if ( return_type.getType() != Type.VOID ) { 
      init_stuff.append( Cxx.getCxxString( return_type ) );
      init_stuff.append( " _result;\n");
    }
    func_decl.append( "\n" );
    switch( return_type.getDetailedType() ) { 
    case Type.VOID:
      break;
    case Type.OPAQUE:
      ior_call.append("_result = ");
      retn_err.append("_result = 0;\n");
      break;
    case Type.BOOLEAN:
      pre_ior.append("sidl_bool _local_result;\n");
      ior_call.append("_local_result = ");
      post_ior.append("_result = _local_result;\n");
      retn_err.append("_result = false;\n");
      break;
    case Type.CHAR:
    case Type.INT:
    case Type.LONG:
    case Type.FLOAT:
    case Type.DOUBLE:
      ior_call.append("_result = ");
      retn_err.append("_result = 0;\n");
      break;
    case Type.ENUM:
      pre_ior.append( IOR.getEnumName(return_type.getSymbolID()) + " _local_result;\n" );
      ior_call.append("_local_result = ");
      post_ior.append("_result = (" + Cxx.getEnumName(return_type.getSymbolID()) + ")_local_result;\n");
      retn_err.append("//_result = 0;\n");
      break;
    case Type.FCOMPLEX:
      pre_ior.append("struct sidl_fcomplex _local_result;\n");
      ior_call.append("_local_result = ");
      post_ior.append("_result = ::std::complex<float>(_local_result.real,_local_result.imaginary);\n");
      retn_err.append("_result = 0;\n");
      break;
    case Type.DCOMPLEX:
      pre_ior.append("struct sidl_dcomplex _local_result;\n");
      ior_call.append("_local_result = ");
      post_ior.append("_result = ::std::complex<double>(_local_result.real, _local_result.imaginary);\n");
      retn_err.append("_result = 0;\n");
      break;
    case Type.STRING:
      pre_ior.append("char * _local_result;\n");
      ior_call.append("_local_result = ");
      // ior_call.append("const_cast<char *>(");
      // extra_close_paren = ")";
      post_ior.append("if (_local_result) {\n  _result = _local_result;\n  free( _local_result );\n}\n");
      retn_err.append("_result = \"\";\n");
      break;
    case Type.SYMBOL:
      throw new CodeGenerationException( "Type.SYMBOL should have been resolved to\n" + 
                                         "Type.ENUM, Type.CLASS, or Type.INTERFACE");
      //break;
    case Type.CLASS:
    case Type.INTERFACE:
      ior_call.append("_result = " + Cxx.getReturnString( return_type ) + "( ");
      extra_close_paren = ", false)";
      retn_err.append("_result = " + Cxx.getReturnString( return_type ) + "();\n");
      break;
    case Type.ARRAY:
      String iorArrayName;
      if ( return_type.getArrayType().isSymbol() ) { 
	iorArrayName = IOR.getArrayName( return_type.getArrayType().getSymbolID() );
      } else { 
         iorArrayName = "sidl_" + return_type.getArrayType().getTypeString() +
           "__array";
      }
      pre_ior.append(iorArrayName + "* _local_result;\n");
      ior_call.append("_local_result = ");
      post_ior.append("_result._set_ior(_local_result);\n");
      break;
    default:
      throw new CodeGenerationException("Unexpected User Defined Return Type" + 
                                        return_type.getType());
    }
    func_decl.append( Cxx.getMethodStubName( d_ext.getSymbolID(), 
                                             shortMethodName ) );
    func_decl.append( "( " );
     
    
    if ( m.isStatic() ) { 
      //      ior_call.append(fullName.replace('.','_') + "_");
      //ior_call.append("(" + IOR.getStaticsName( d_ext.getSymbolID() ) + "()->" + 
      // IOR.getVectorEntry(methodName) + ")( " );
      ior_call.append("( _get_sepv()->" +  IOR.getVectorEntry(longMethodName) + ")( ");
    } else { 
      ior_call.append( "(*(d_self->d_epv->f_" + m.getLongMethodName() + "))(" +
                       d_self );
      if ( nargs > 0 ) { ior_call.append(", "); }
    }
    // writeup the argument lists
    for( Iterator it = vArgs.iterator(); it.hasNext(); ) { 
      Argument arg = (Argument) it.next();
      Type type = arg.getType();
      int typeInt = type.getDetailedType();
      String argName = arg.getFormalName();
      String mode = "/* " + Cxx.argModeToString( arg ) + " */ ";
      int modeInt = arg.getMode();
      func_decl.append( Cxx.getArgumentString( arg ) );
      if ( it.hasNext() ) { 
        func_decl.append(", ");
      }
      ior_call.append( mode );
      switch ( typeInt ) { 
      case Type.OPAQUE:
        switch( modeInt ) { 
        case Argument.IN:
          ior_call.append( argName );
          break;
        case Argument.OUT:
          ior_call.append( "&" + argName );
          break;
        case Argument.INOUT:
          ior_call.append( "&" + argName );
          break;
        }
        break;
      case Type.BOOLEAN:
        switch( modeInt ) { 
        case Argument.IN:
          pre_ior.append("sidl_bool _local_" + argName + " = " + argName + 
                         ";\n");
          ior_call.append("_local_" + argName );          
          break;
        case Argument.OUT:
          pre_ior.append("sidl_bool _local_" + argName + ";\n");
          ior_call.append("&_local_" + argName);
          post_ior.append(argName + " = _local_" + argName + ";\n");
          break;
        case Argument.INOUT:
          pre_ior.append("sidl_bool _local_" + argName + " = " + argName + 
                         ";\n");
          ior_call.append("&_local_" + argName);
          post_ior.append(argName + " = _local_" + argName + ";\n");          
          break;
        }
        break;
      case Type.CHAR:
      case Type.INT:
      case Type.LONG:
      case Type.FLOAT:
      case Type.DOUBLE:
        switch( modeInt ) { 
        case Argument.IN:
          ior_call.append( argName );
          break;
        case Argument.OUT:
          ior_call.append( "&" + argName );
          break;
        case Argument.INOUT:
          ior_call.append( "&" + argName );
          break;
        }
        break;
      case Type.ENUM:
	  String ior_name = IOR.getEnumName(  type.getSymbolID() );  
	switch( modeInt ) { 
        case Argument.IN:
          ior_call.append( "(" + ior_name + ")" + argName );
          break;
        case Argument.OUT:
          ior_call.append( "(" + ior_name + "*)&" + argName );
          break;
        case Argument.INOUT:
          ior_call.append( "(" + ior_name + "*)&" + argName );
          break;
        }
        break;
      case Type.FCOMPLEX:
        switch( modeInt ) { 
        case Argument.IN:
          pre_ior.append("struct sidl_fcomplex _local_" + argName + " = {");
          pre_ior.append( argName + ".real(), ");
          pre_ior.append( argName + ".imag() } ; \n");
          ior_call.append("_local_" + argName );
          break;
        case Argument.OUT:
          pre_ior.append("struct sidl_fcomplex _local_" + argName + "; \n");
          ior_call.append("&_local_" + argName );
          post_ior.append( argName + " = ::std::complex<float>(_local_" +
                           argName + ".real, _local_" + argName + 
                           ".imaginary);\n");
          break;
        case Argument.INOUT:
          pre_ior.append("struct sidl_fcomplex _local_" + argName + " = {");
          pre_ior.append( argName + ".real(), ");
          pre_ior.append( argName + ".imag() } ; \n");
          ior_call.append("&_local_" + argName );
          post_ior.append( argName + " = ::std::complex<float>(_local_" +
                           argName + ".real, _local_" + argName + 
                           ".imaginary);\n");
          break;
        }
        break;
      case Type.DCOMPLEX:
        switch( modeInt ) { 
         case Argument.IN:
          pre_ior.append("struct sidl_dcomplex _local_" + argName + " = {");
          pre_ior.append( argName + ".real(), ");
          pre_ior.append( argName + ".imag() } ; \n");
          ior_call.append("_local_" + argName );
          break;
        case Argument.OUT:
          pre_ior.append("struct sidl_dcomplex _local_" + argName + "; \n");
          ior_call.append("&_local_" + argName );
          post_ior.append( argName + " = ::std::complex<double>(_local_" +
                           argName + ".real, _local_" + argName + 
                           ".imaginary);\n");
          break;
        case Argument.INOUT:
          pre_ior.append("struct sidl_dcomplex _local_" + argName + " = {");
          pre_ior.append( argName + ".real(), ");
          pre_ior.append( argName + ".imag() } ; \n");
          ior_call.append("&_local_" + argName );
          post_ior.append( argName + " = ::std::complex<double>(_local_" +
                           argName + ".real, _local_" + argName + 
                           ".imaginary);\n");
          break;
        }
        break;
      case Type.STRING:
        switch( modeInt ) { 
        case Argument.IN:
          ior_call.append( argName + ".c_str()" );
          break;
        case Argument.OUT:
          pre_ior.append("char * _local_" + argName + " = 0;\n");
          ior_call.append("&_local_" + argName );
          post_ior.append("if (_local_" + argName + ") {\n");
          post_ior.append( "  " + argName + " = _local_" + argName + ";\n");
          post_ior.append("  sidl_String_free( _local_" + argName + 
                          ");\n");
          post_ior.append("}\n");
          post_ior.append("else {\n");
          post_ior.append("  " + argName + " = \"\";\n");
          post_ior.append("}\n");
          break;
        case Argument.INOUT:
          pre_ior.append("char * _local_" + argName + " = sidl_String_strdup( " +
                         argName + ".c_str() );\n");
          ior_call.append("&_local_" + argName );
          post_ior.append("if (_local_" + argName + ") {\n");
          post_ior.append("  " + argName + " = _local_" + argName + ";\n");
          post_ior.append("  sidl_String_free( _local_" + argName + 
                          ");\n");
          post_ior.append("}\n");
          post_ior.append("else {\n");
          post_ior.append("  " + argName + " = \"\";\n");
          post_ior.append("}\n");
          break;
        }
        break; 
      case Type.SYMBOL:
        throw new CodeGenerationException( "Type.SYMBOL should have been resolved to\n" + 
                                           "Type.ENUM, Type.CLASS, or Type.INTERFACE");
        // break;
      case Type.INTERFACE:
      case Type.CLASS:
        id = type.getSymbolID();
        switch( modeInt ) { 
        case Argument.IN:
          ior_call.append( argName + "._get_ior()");
          break;
        case Argument.OUT:
          pre_ior.append( IOR.getObjectName(id) + "* _local_" + argName + 
                          ";\n" );
          ior_call.append( "&_local_" + argName );
	  post_ior.append("if ( " + argName + "._not_nil() ) {\n");
	  post_ior.append("  " + argName + ".deleteRef();\n"); 
	  post_ior.append("}\n");
	  post_ior.append( argName + "._set_ior( _local_" + argName + ");\n");
          break;
        case Argument.INOUT:
          pre_ior.append( IOR.getObjectName(id) + "* _local_" + argName + 
                          " = " + argName + "._get_ior();\n" );
	  pre_ior.append( argName + "._set_ior( 0 );\n");
          ior_call.append( "&_local_" + argName );
          post_ior.append( argName + "._set_ior( _local_" + argName + ");\n");
          break;
	}
        break;
     case Type.ARRAY:
       String iorArrayName;
       if ( type.getArrayType().isSymbol() ) { 
	   iorArrayName = IOR.getArrayName( type.getArrayType().getSymbolID() );
       } else { 
         iorArrayName = "sidl_" + type.getArrayType().getTypeString() + 
           "__array";
       }
       
       switch( modeInt ) { 
       case Argument.IN:
         ior_call.append( argName + "._get_ior()" );
         break;
       case Argument.OUT:
         pre_ior.append(iorArrayName + " *_local_" + argName + ";\n");
         ior_call.append("&_local_" + argName);
         post_ior.append(argName + "._set_ior(_local_" + argName + ");\n");
         break;
       case Argument.INOUT:
         pre_ior.append("if (" + argName + ") {\n");
         pre_ior.append("  " + argName + ".addRef();\n");
         pre_ior.append("}\n");
         pre_ior.append(iorArrayName + " *_local_" + argName + 
                        " = " + argName + "._get_ior();\n");
         ior_call.append("&_local_" + argName);
         post_ior.append(argName + "._set_ior(_local_" + argName + 
                         ");\n");  // sieze ownership of what's returned.
         break;
       }
       break;
       
      default:
        throw new CodeGenerationException("Unexpected User Defined Return Type" + 
                                          return_type.getType());
      }      
      if ( it.hasNext() ) { 
        ior_call.append(", ");
      }
    }
 
    // if method throws exceptions...
    if (m.getThrows().size()>0) { 
      pre_ior.append("sidl_BaseInterface__object * _exception = 0;\n");
      if ( nargs > 0 || (!m.isStatic())) { // if other args are listed.
        ior_call.append( ", " );
      }
      ior_call.append("&_exception");
    }

    // final details
    func_decl.append(" )\n");
    ior_call.append(" )" + extra_close_paren + ";\n");
    
    // finally dump everything out
    d_writer.println( func_decl.toString().trim() );
    Cxx.generateThrowsList( d_writer, m, true );
    d_writer.println();
    d_writer.println("{");
    d_writer.tab();
    if ( ! m.isStatic() ) { 
      d_writer.println("if ( d_self == 0 ) {");
      d_writer.tab();
      d_writer.println("throw ::sidl::NullIORException( ::std::string (");
      d_writer.tab();
      d_writer.disableLineBreak();
      d_writer.println( "\"" + 
			"Null IOR Pointer in \\\"" + 
			Cxx.getMethodStubName(d_ext.getSymbolID(),shortMethodName) +
			"()\\\"\"");
      d_writer.enableLineBreak();
      d_writer.backTab();
      d_writer.println("));");
      d_writer.backTab();
      d_writer.println("}");
    }
    d_writer.println( init_stuff.toString().trim() );
    if ( ! m.isStatic() ) { 
      if (shortMethodName.equals("addRef") ||
          shortMethodName.equals("deleteRef") ) { 
        d_writer.println("if ( !d_weak_reference ) {");
	d_writer.tab();
      }
    }
    d_writer.writeCommentLine("pack args to dispatch to ior");
    d_writer.println( pre_ior.toString().trim() );
    d_writer.writeCommentLine("dispatch to ior");
    d_writer.println( ior_call.toString().trim() );
    d_writer.writeCommentLine("unpack results and cleanup");
    if (m.getThrows().size()>0) { // if throws exception
      d_writer.println("if (_exception != 0 ) {");      
      d_writer.tab();
      d_writer.println("void * _p = 0;");
      Object [] exceptions = m.getThrows().toArray();
      Arrays.sort(exceptions, new LevelComparator(SymbolTable.getInstance()));
      
      for( int i=0; i<exceptions.length; ++i) {
        SymbolID exid = (SymbolID)exceptions[i];
        d_writer.println("if ( (_p=(*(_exception->d_epv->f__cast))(_exception, \"" +
                         exid.getFullName() + "\")) != 0 ) {");
        d_writer.tab();
        d_writer.println(IOR.getObjectName(exid) + " * _realtype = " + 
                         Cxx.reinterpretCast(IOR.getObjectName(exid)+"*" ,
                                             "_p") + ";");
        d_writer.writeCommentLine("Note: alternate constructor does not increment refcount.");
        d_writer.println("throw " + Cxx.getObjectName(exid) + "( _realtype, false );");
        d_writer.backTab();
        d_writer.println("}");
      }
      d_writer.backTab();
      d_writer.println("}");
    }

    d_writer.println( post_ior.toString().trim() );
    if ( ! m.isStatic() ) { 
      if (shortMethodName.equals("addRef") ||
          shortMethodName.equals("deleteRef") ) { 
	  if ( shortMethodName.equals("deleteRef") ) { 
	      d_writer.println("d_self = 0;");
	  }
	d_writer.backTab();
        d_writer.println("}");
      }
    }
    //if ( !m.isStatic() ) { 
    //  d_writer.decreaseTabLevel();
    //  d_writer.println( "} else {");
    //  d_writer.increaseTabLevel();
    //  d_writer.println( retn_err.toString().trim() );
    //  d_writer.decreaseTabLevel();
    //  d_writer.println("}");
    //}
    if ( return_type.getType() != Type.VOID ) { 
      d_writer.println("return _result;");
    }
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

//------------------------ rmi
	private void	writeRmiIncludes() {

		d_writer.println("");
    
//    d_writer.println("#include <unistd.h> ");
//    d_writer.println("#include <iostream> ");
//    d_writer.println("using namespace std; ");
    
    d_writer.println("#include <msl/debug.h> ");
    d_writer.println("#include <msl/Format.h> ");
    d_writer.println("#include <msl/Time.h> ");
    d_writer.println("#include <msl/Condvar.h> ");
    d_writer.println("#include <msl/Mutex.h> ");
    d_writer.println("#include <msl/Opt.h> ");
    d_writer.println("#include <msl/Def.h> ");
    d_writer.println("#include <msl/Sptr.h> ");
    d_writer.println("using namespace msl; ");
    d_writer.println();

    d_writer.println("#include <wsit/ArgList.h> ");
    d_writer.println("#include <proteus/Core.h> ");
    d_writer.println("#include <wsit/Invocation.h> ");
    d_writer.println("#include <wsit/InvocationHandler.h> ");
    d_writer.println("#include <proteus/Node.h> ");
    d_writer.println("#include <proteus/NodeId.h> ");
    d_writer.println("#include <wsit/PrimitiveArg_tmpl.h> ");
    d_writer.println("#include <wsit/QName.h> ");
    d_writer.println("#include <proteus/RAddress.h> ");
    d_writer.println("#include <proteus/RNodepoint.h> ");
    d_writer.println("#include <proteus/RRepertoireMap.h> ");
    d_writer.println("#include <proteus/RScheme.h> ");
    d_writer.println("#include <proteus/RSecurityContext.h> ");
    d_writer.println("#include <proteus/SAddress.h> ");
    d_writer.println("#include <wsit/Semiinvocation.h> ");
    d_writer.println("#include <proteus/SNodepoint.h> ");
    d_writer.println("#include <proteus/SRepertoireMap.h> ");
    d_writer.println("#include <proteus/SScheme.h> ");
    d_writer.println("#include <proteus/SSecurityContext.h> ");
    d_writer.println("#include <proteus/Selector.h> ");
    d_writer.println("#include <wsit/StringArg.h> ");
    d_writer.println("#include <wsit/ValueResult.h> ");
    d_writer.println("#include <wsit/VobjectArg.h> ");
    d_writer.println("#include <wsit/types.h> ");
    d_writer.println("using namespace proteus; ");
    d_writer.println();

    d_writer.println("#include \"BabelVobject.h\" ");
    d_writer.println();
	}

/**
  Generate definition of static function RmiSkel::_export().	
*/
	private void writeRmiSkel(){

    //String className = d_ext.getSymbolID().getShortName();
		SymbolID id = d_ext.getSymbolID();
    final String fullName = Cxx.getObjectName(id);
    final String fullNameWithoutLeadingColons = Cxx.getSymbolNameWithoutLeadingColons(id,"");

		//everything through first "{" 
    StringBuffer export_begin = new StringBuffer( 128 );
		// inside {} 
    StringBuffer export_call  = new StringBuffer( 128 );
		// end "}"
    StringBuffer export_end   = new StringBuffer( 128 );

    d_writer.writeCommentLine("return server address to clients");
		export_begin.append("::std::string "); 
		export_begin.append(fullNameWithoutLeadingColons); 
		export_begin.append("RmiSkel::_export("); 
		export_begin.append(fullNameWithoutLeadingColons); 
		export_begin.append(" obj) {"); 
    d_writer.println( export_begin.toString().trim() );

    d_writer.tab();

		export_call.append(fullNameWithoutLeadingColons); 
		export_call.append("InvocationHandler ih(obj);\n");

		//export_call.append("Def<Node> n(ih);\n");
		export_call.append("Node *n = new Node(&ih);\n");
		export_call.append("return (n->nodepoint())->toString();\n");
    d_writer.println( export_call.toString().trim() );

    d_writer.backTab();

		export_end.append("}"); 
    d_writer.println(export_end.toString().trim());

    d_writer.println(" ");

	}

/**
	Generate debug namespace
*/
	private void writeNamespaceDebug() {
    d_writer.println("namespace {");
    d_writer.tab();
    d_writer.println("const char *xyzzy_modname = __FILE__;");
    d_writer.println("const char *xyzzy_funcname = \"function\";");
    d_writer.backTab();
    d_writer.println("};\n");

	}

/**
  Generate definition for InvocationHandler class.	
*/

	private void writeRmiInvocationHandler() {

		SymbolID id = d_ext.getSymbolID();
    final String shortName = id.getShortName();    
    final String fullName = Cxx.getObjectName(id);
    final String fullNameWithoutLeadingColons = Cxx.getSymbolNameWithoutLeadingColons(id,"");

		//figure out the namespace for handle function 
		//int indx = fullNameWithoutLeadingColons.indexOf(shortName);
		//String myNameSpace = fullNameWithoutLeadingColons.substring(0,indx);


		// constructor
    StringBuffer ih_begin = new StringBuffer( 128 );
    StringBuffer ih_debug = new StringBuffer( 128 );
    StringBuffer ih_call  = new StringBuffer( 128 );
    StringBuffer ih_end   = new StringBuffer( 128 );

		ih_call.append(fullNameWithoutLeadingColons); 
		ih_call.append("InvocationHandler::"); 
		ih_call.append(shortName); 
		ih_call.append("InvocationHandler("); 
		ih_call.append(fullName); 
		//ih_call.append(fullNameWithoutLeadingColons); 
		ih_call.append(" &hw): obj(hw) { }\n"); 
    d_writer.println(ih_call.toString().trim() );


		// handle() function
    d_writer.println();
    d_writer.println("wsit::Result_sptr " + fullNameWithoutLeadingColons  
			+"InvocationHandler::handle(const wsit::Invocation &inv) {");

    d_writer.tab();

		ih_debug.append("const char *const xyzzy_funcname = \" "); 
		ih_call.append(fullNameWithoutLeadingColons); 
		ih_debug.append("InvocationHandler::handle\";");
    d_writer.println(ih_debug.toString().trim() );
    d_writer.println();

    d_writer.println("msl_dpf(\"Got an invocation: \"); ");
    d_writer.println("msl_dpf(\"Interface: \" << inv.interface); ");
    d_writer.println("msl_dpf(\"Method: \" << inv.semiinvocation.method); ");
    d_writer.println("msl_dpf(\"n args: \" <<  inv.semiinvocation.argList->size()); ");
    d_writer.println("msl_dpf(\"Args: \"); ");
    d_writer.println();

    d_writer.println("msl_check(inv.semiinvocation.argList != Null); ");
    d_writer.println("msl_check((*inv.semiinvocation.argList)[0] != Null);");
    d_writer.println();

    d_writer.println("msl_dpf(sptr_dynamic_cast<const wsit::PrimitiveArg_tmpl<int> >((*inv.semiinvocation.argList)[0])->value());");
    d_writer.println();

    d_writer.println("msl_dpf(sptr_dynamic_cast<const wsit::PrimitiveArg_tmpl<double> >((*inv.semiinvocation.argList)[1])->value());");
    d_writer.println();
    
    d_writer.println("wsit::Vobject_sptr vo = sptr_dynamic_cast<const wsit::VobjectArg>((*inv.semiinvocation.argList)[2])->vobject();"); 
    d_writer.println();

    d_writer.println("Sptr<const BabelVobject> bvo = sptr_dynamic_cast<const BabelVobject>(vo);");
    d_writer.println();

    d_writer.println("hello::TestObj vobj = bvo->m_object; ");
    d_writer.println();
    d_writer.writeCommentLine("Prepare a result to send back. It has an empty string for the name, ");
    d_writer.writeCommentLine("so represents the return value. ");
    d_writer.println("msl_dpf(\"call getMsg: \"); ");
    d_writer.println("wsit::StringArg_sptr r1(NewTag, wsit::QName(\"\"), obj.getMsg(vobj)); ");
    d_writer.println("wsit::ArgList_sptr al(NewTag, r1); ");
    d_writer.println("msl_dpf(\"getMsg call done \"); ");
    d_writer.println();
    d_writer.println("return new wsit::ValueResult(al); ");

    d_writer.backTab();
    d_writer.println("} ");

		//
	}
}
