//
// File:        RmiStubSource.java
// Package:     gov.llnl.babel.backend.cxx
// Revision:    @(#) $Revision: 4454 $
// Date:        $Date: 2005-03-22 14:18:30 -0800 (Tue, 22 Mar 2005) $
// Description: Write Cxx extension header file for a BABEL extendable
// 
// This is typically directed by GenCxxClient.
// Copyright (c) 2000-2001, The Regents of the University of Calfornia.
// Produced at the Lawrence Livermore National Laboratory.
// Written by the Components Team <components@llnl.gov>
// UCRL-CODE-2002-054
// All rights reserved.
// 
// This file is part of Babel. For more information, see
// http://www.llnl.gov/CASC/components/. Please read the COPYRIGHT file
// for Our Notice and the LICENSE file for the GNU Lesser General Public
// License.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License (as published by
// the Free Software Foundation) version 2.1 dated February 1999.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the IMPLIED WARRANTY OF
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the terms and
// conditions of the GNU Lesser General Public License for more details.
// 
// You should have recieved a copy of the GNU Lesser General Public License
// along with this program; if not, write to the Free Software Foundation,
// Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

package gov.llnl.babel.backend.rmi2;

import gov.llnl.babel.backend.CodeConstants;
import gov.llnl.babel.backend.CodeGenerationException;
import gov.llnl.babel.backend.CodeSplicer;
import gov.llnl.babel.backend.rmi2.Cxx;
import gov.llnl.babel.backend.FileManager;
import gov.llnl.babel.backend.Utilities;
import gov.llnl.babel.backend.writers.LanguageWriterForCxx;
import gov.llnl.babel.backend.writers.LineRedirector;
import gov.llnl.babel.symbols.Extendable;
import gov.llnl.babel.symbols.Method;
import gov.llnl.babel.symbols.SymbolID;
import java.util.Iterator;
import java.util.List;

/**
 * Create and write a header for a Cxx C extension class to wrap a 
 * BABEL extendable in a Cxx object. The header has to expose a 
 * function to create a wrapped IOR, a function to check if a 
 * <code>PyObject</code> is an instance of this extension type, and
 * an import macro.
 */
public class RmiStubSource {
  private Extendable d_ext = null;
  private LanguageWriterForCxx d_writer = null;
  private CodeSplicer d_splicer = null;
  
  /**
   * Create an object capable of generating the header file for a
   * BABEL extendable.
   *
   * @param ext   an interface or class symbol that needs a header
   *              file for a Cxx C extension class.
   */
  public RmiStubSource(Extendable ext) {
    d_ext = ext;
  }
  
  /**
   * Generate the header file for the extendable with which this object was
   * created.
   *
   * @exception gov.llnl.babel.backend.CodeGenerationException
   *    this is a catch all exception for problems during the code
   *    generation phase.
   */
  public synchronized void generateCode() 
    throws CodeGenerationException 
  {
    final SymbolID id = d_ext.getSymbolID();
    final int type= d_ext.getSymbolType();
    String filename = Cxx.generateFilename( id, 
                                            Cxx.FILE_ROLE_IMPL,
                                            Cxx.FILE_TYPE_CXX_SOURCE );
    //System.out.println("Create " + filename + "..." );
    
    try { 
      d_splicer = FileManager.getInstance().
        getCodeSplicer(id, type, filename);
      d_writer = Cxx.createSource( d_ext, Cxx.FILE_ROLE_IMPL, "IMPLSRCS");
      d_splicer.setLineRedirector( (LineRedirector) d_writer );
      writeIncludes();

      writeCtorDtor();

      writeSIDLDefinedMethods();

      d_splicer.splice( d_ext.getSymbolID().getFullName() + "._misc", 
                        d_writer, "miscellaneous code");
      d_writer.println();

      if (d_splicer.hasUnusedSymbolEdits()) {
        d_writer.printlnUnformatted("#error File has unused splicer blocks.");
        d_writer.beginBlockComment(true);
        d_writer.println(CodeConstants.C_BEGIN_UNREFERENCED_METHODS);
        d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT1);
        d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT2);
        d_writer.println(CodeConstants.C_UNREFERENCED_COMMENT3);
        d_writer.endBlockComment(true);
        d_splicer.outputUnusedSymbolEdits( d_writer.getPrintWriter() );
        d_writer.writeCommentLine(CodeConstants.C_END_UNREFERENCED_METHODS);
      }
    } catch ( java.io.IOException ex) { 
      throw new CodeGenerationException("IOException : " + ex.getMessage() );
    } finally { 
      if (d_writer != null) {
        d_writer.close();
        d_writer = null;
      }
    }
  }

  private void writeIncludes() { 
    d_writer.generateInclude( Cxx.generateFilename( d_ext.getSymbolID(), 
                                                    Cxx.FILE_ROLE_IMPL,  
                                                    Cxx.FILE_TYPE_CXX_HEADER),
                              false );
    d_writer.println();
    writeRmiIncludes();
    d_writer.println();
    d_splicer.splice( d_ext.getSymbolID().getFullName() + "._includes", 
                      d_writer, "additional includes or code" );
    d_writer.println();
  }                   
  
  private void writeCtorDtor() {

    d_writer.writeCommentLine("user defined constructor");
    d_writer.print("void " + Utilities.replace(d_ext.getSymbolID().getFullName(),".","::"));
    d_writer.println("_impl::_ctor() {");
    d_writer.tab();
    d_splicer.splice( d_ext.getSymbolID().getFullName() + "._ctor", 
                      d_writer, "construction details");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();

    d_writer.writeCommentLine("user defined destructor");
    d_writer.print("void " + Utilities.replace(d_ext.getSymbolID().getFullName(),".","::"));
    d_writer.println("_impl::_dtor() {");
    d_writer.tab();
    d_splicer.splice( d_ext.getSymbolID().getFullName() + "._dtor",
                      d_writer, "destruction details");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();
  }

  private void writeSIDLDefinedMethods() throws CodeGenerationException { 
    List static_methods = (List) d_ext.getStaticMethods(false);
    if ( static_methods.size() > 0 ) { 
      d_writer.writeCommentLine("user defined static methods:");
      for (Iterator m = static_methods.iterator(); m.hasNext(); ) {
        Method method = (Method) m.next();
        generateMethod(method);
      }
    } else { 
      d_writer.writeCommentLine("user defined static methods: (none)");
    }
    
    d_writer.println();
    List nonstatic_methods = (List) d_ext.getNonstaticMethods(false);
    if ( nonstatic_methods.size() > 0 ) { 
      d_writer.writeCommentLine("user defined non-static methods:");
      for (Iterator m = nonstatic_methods.iterator(); m.hasNext(); ) {
        Method method = (Method) m.next();
	if ( !method.isAbstract() ) { 	    
	    generateMethod( method );
	}
      }
    } else { 
      d_writer.writeCommentLine("user defined non-static methods: (none)");
    }
    d_writer.println();

  }

  private void generateMethod( Method method ) 
    throws CodeGenerationException 
  { 
    if ( method == null ) { return ; }

    d_writer.writeComment ( method, true );

    d_writer.println( Cxx.getReturnString( method.getReturnType() ) );
    d_writer.print(Utilities.replace(d_ext.getSymbolID().getFullName(),".","::"));
    d_writer.print("_impl::");
    d_writer.print( method.getShortMethodName() );
    
    if ( method.getArgumentList().size() > 0 ) { 
      d_writer.println(" (");
      d_writer.tab();
      Cxx.generateArgumentList( d_writer, method );
      d_writer.println(" ) ");
      d_writer.backTab();
      Cxx.generateThrowsList( d_writer, method, false );
      d_writer.println("{");
    } else { 
      d_writer.println(" () ");      
      Cxx.generateThrowsList( d_writer, method, false);
      d_writer.println();
      d_writer.println("{");
    }
    d_writer.tab();
    generateRmiStub();  // invoke proteus
    d_splicer.splice( d_ext.getSymbolID().getFullName() + "." +
                      method.getLongMethodName(), 
                      d_writer, "implementation");
    d_writer.backTab();
    d_writer.println("}");
    d_writer.println();    
  }

//-------------------------------- Rmi part
  private void writeRmiIncludes() { 

	d_writer.println("#include <msl/debug.h>");
	d_writer.println("#include <msl/Format.h>");
	d_writer.println("#include <msl/Time.h>");
	d_writer.println("#include <msl/Condvar.h>");
	d_writer.println("#include <msl/Mutex.h>");
	d_writer.println("#include <msl/Opt.h>");
	d_writer.println("#include <msl/Def.h>");

	d_writer.println("#include <wsit/ArgList.h>");
	d_writer.println("#include <wsit/PrimitiveArg_tmpl.h>");
	d_writer.println("#include <proteus/Core.h>");
	d_writer.println("#include <wsit/Invocation.h>");
	d_writer.println("#include <wsit/InvocationHandler.h>");
	d_writer.println("#include <proteus/Node.h>");
	d_writer.println("#include <proteus/NodeId.h>");
	d_writer.println("#include <proteus/RAddress.h>");
	d_writer.println("#include <proteus/RNodepoint.h>");
	d_writer.println("#include <proteus/RRepertoireMap.h>");
	d_writer.println("#include <proteus/RScheme.h>");
	d_writer.println("#include <proteus/RSecurityContext.h>");
	d_writer.println("#include <proteus/SAddress.h>");
	d_writer.println("#include <wsit/Semiinvocation.h>");
	d_writer.println("#include <proteus/SNodepoint.h>");
	d_writer.println("#include <proteus/SRepertoireMap.h>");
	d_writer.println("#include <proteus/SScheme.h>");
	d_writer.println("#include <proteus/SSecurityContext.h>");
	d_writer.println("#include <proteus/Selector.h>");
	d_writer.println("#include <wsit/StringArg.h>");
	d_writer.println("#include <wsit/ValueResult.h>");
	d_writer.println("#include <wsit/VobjectArg.h>");
	d_writer.println("#include <wsit/types.h>");

//  d_writer.println("#include \"BabelVobject.h\"");
	d_writer.println("#include <BabelVobject.h>");

	d_writer.println("using namespace msl;");
	d_writer.println("using namespace proteus;");
	d_writer.println("using namespace wsit;");

	d_writer.println("extern const char *DestAddr; ");

	d_writer.println("namespace {");

	d_writer.println("  const char *xyzzy_modname = \"helloWorld-client-impl\";");
	d_writer.println("  const char *xyzzy_funcname = \"\";");
	d_writer.println("}");
  }

/**
 Generate Rmi stub for client to invoke  proteus rmi calls	
*/
  private void generateRmiStub() {  // invoke proteus

	d_writer.println();
	d_writer.println("Def<Node> n; ");

	d_writer.println();
	d_writer.println("assert(DestAddr != NULL); ");
	d_writer.println("Sptr<SScheme> ss(NewTag, Sptr<SNodepoint>(NewTag, DestAddr)); ");

	d_writer.println();
	d_writer.println("PrimitiveArg_tmpl<int>::def a1(QName(\"{}<NONE>\"), 1); ");
	d_writer.println("PrimitiveArg_tmpl<double>::def a2(QName(\"{}<NONE>\"), 3.14159); ");

	d_writer.println();
	d_writer.println("Def<BabelVobject> bvo(obj); ");
	d_writer.println("VobjectArg_def a3(QName(\"{}<NONE>\"), bvo); ");

	d_writer.println();
	d_writer.println("PrimitiveArg_tmpl<int64_t>::def a4(QName(\"{}<NONE>\"), 9000000000LL); ");
	d_writer.println("PrimitiveArg_tmpl<float>::def a5(QName(\"{}<NONE>\"), 3.12); ");
	d_writer.println("PrimitiveArg_tmpl<char>::def a6(QName(\"{}<NONE>\"), 'c'); ");
	d_writer.println("Def<ArgList> al(a1, a2, a3, a4, a5, a6); // argument list ");

	d_writer.println("Semiinvocation si(QName(\"{}getMsg\"), al);");
	//d_writer.println("Invocation inv(\"Interface\", si); ");
	d_writer.println("Invocation inv(QName(\"{}Interface\"), si); ");

	d_writer.println();
	d_writer.writeCommentLine("Invoke.");
	d_writer.println("Result_sptr res(n->invoke(ss, inv)); ");

	//d_writer.println("msl_dpf(\"Rmi call done.\"); ");

	d_writer.println();
	d_writer.writeCommentLine("Assume it's not an exception for now.");
	d_writer.println("ValueResult_sptr vr(sptr_dynamic_cast<const ValueResult>(res)); ");

	d_writer.println();
	d_writer.println("msl_dpf(\"Result: \" " + 
      "<< sptr_dynamic_cast<const StringArg >(( *vr->argList())[0])->value()); ");

	d_writer.println();
    d_writer.println(" return sptr_dynamic_cast<const StringArg >"+
             "((*vr->argList())[0])->value();\n");

  }

}
