#ifndef __RSTRING_H__
#define __RSTRING_H__ 1
/* 
   elmo - ELectronic Mail Operator

   Copyright (C) 2002, 2003, 2004 rzyjontko

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; version 2.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  

   ----------------------------------------------------------------------

   Please read it carefully before using this module!

   1. 
   This implements very simple (and thus fast) module that simplifies
   operations on arrays of null-terminated strings.  Array's last element
   is NULL pointer -- this means that
   ptr->array[ptr->count] == NULL;

   2.
   Pointers in array are not copies of originals.  They point to original
   string and if you accidentaly free that pointer (or memory block) you
   will get segfault as soon as you try to dereference this pointer (this
   is an optimistic scenario).  Be aware of what can happen if you
   accidentaly overwrite data in string...

   3.
   Removal of any element may result in diffrent succession of elements in
   array so I would not depend on it.

   4.
   Splitted string IS MODIFIED.  If you accidentaly remove first element
   and did not save somewhere pointer to splitted string you won't be able
   to free this string.

   5.
   If you split string twice with the same delimiter/regexp second array
   will have exactly 1 element (because original string will get \0 after
   first match).  It is quite possible, that you will try to free that
   memory block twice...

   6.
   If you put dynamically allocated strings into array you have to care
   yourself about freeing them.  This module won't do it for you.  It only
   frees memory used for its own purpose.

   7.
   rstring made by splitting string is created with rstring_create, so you
   should call rstring_delete when it is not needed any more or you'll get
   a memory leak.

   8.
   This is not a C++ class and you are encouraged to use structure
   implementation.  I wrote these functions only to make it easier to
   operate on (char **) objects.

   It is very simple to iterate through array like this:

   char **txt;
   for (txt = ptr->array; *txt; txt++){
   // do the work
   }

*/
/*****************************************************************************
 *    INTERFACE REQUIRED HEADERS
 ****************************************************************************/

#include <stdio.h>

#include "memchunk.h"

/*****************************************************************************
 *    INTERFACE DEFINITIONS / ENUMERATIONS / SIMPLE TYPEDEFS
 ****************************************************************************/

#ifdef __GNUC__
# define FMT __attribute__ ((format (printf, 2, 3)))
#else
# define FMT
#endif

/****************************************************************************
 *    INTERFACE CLASS PROTOTYPES / EXTERNAL CLASS REFERENCES
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE STRUCTURES / UTILITY CLASSES
 ****************************************************************************/

struct rstring {
        char     **array;
        int        size;
        int        count;
        unsigned   allocated_first : 1;
        unsigned   allocated_all   : 1;
};


typedef struct rstring rstring_t;

/****************************************************************************
 *    INTERFACE DATA DECLARATIONS
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE FUNCTION PROTOTYPES
 ****************************************************************************/

#ifdef __cplusplus
extern "C" {
#endif

extern rstring_t *rstring_create (void);
extern rstring_t *rstring_create_size (int size);
extern void       rstring_delete (rstring_t *ptr);
extern void       rstring_add (rstring_t *ptr, char *txt);
extern void       rstring_sprintf (rstring_t *ptr, const char *fmt, ...) FMT;
extern void       rstring_remove (rstring_t *ptr, unsigned index);
extern void       rstring_shrink (rstring_t *ptr);
extern rstring_t *rstring_split (char *txt, const char *delimiter);
extern rstring_t *rstring_split_re (char *txt, const char *re);
extern rstring_t *rstring_split_lines (char *txt, size_t line_len);
extern rstring_t *rstring_join (rstring_t *x, rstring_t *y);
extern rstring_t *rstring_copy (rstring_t *x);
extern void       rstring_free_strings (rstring_t *x);
extern void       rstring_free_firstn_strings (rstring_t *x, int n);
extern void       rstring_dump (memchunk_t *memchunk, rstring_t *x);
extern rstring_t *rstring_read (memchunk_t *memchunk);
extern char      *rstring_flatten (rstring_t *ptr, const char *delim);

#ifdef __cplusplus
}
#endif

/****************************************************************************
 *    INTERFACE OBJECT CLASS DEFINITIONS
 ****************************************************************************/
/****************************************************************************
 *    INTERFACE TRAILING HEADERS
 ****************************************************************************/
/****************************************************************************
 *
 *    END HEADER rstring.h
 *
 ****************************************************************************/
#endif
