# -*- coding: utf-8 -*-

# Copyright (c) 2003 - 2006 Detlev Offenbach <detlev@die-offenbachs.de>
#

"""
Module implementing a TestResult derivative for the eric3 debugger.
"""

import select
import traceback
from unittest import TestResult


from DebugProtocol import *


class DCTestResult(TestResult):
    """
    A TestResult derivative to work with eric3's debug client.
    
    For more details see unittest.py of the standard python distribution.
    """
    def __init__(self, parent):
        """
        Constructor
        
        @param parent The parent widget.
        """
        TestResult.__init__(self)
        self.parent = parent
        
    def addFailure(self, test, err):
        """
        Method called if a test failed.
        
        @param test Reference to the test object
        @param err The error traceback
        """
        TestResult.addFailure(self, test, err)
        tracebackLines = apply(traceback.format_exception, err + (10,))
        self.parent.write('%s%s\n' % (ResponseUTTestFailed,
            str((str(test), tracebackLines))))
        
    def addError(self, test, err):
        """
        Method called if a test errored.
        
        @param test Reference to the test object
        @param err The error traceback
        """
        TestResult.addError(self, test, err)
        tracebackLines = apply(traceback.format_exception, err + (10,))
        self.parent.write('%s%s\n' % (ResponseUTTestErrored,
            str((str(test), tracebackLines))))
        
    def startTest(self, test):
        """
        Method called at the start of a test.
        
        @param test Reference to the test object
        """
        TestResult.startTest(self, test)
        self.parent.write('%s%s\n' % (ResponseUTStartTest,
            str((str(test), test.shortDescription()))))

    def stopTest(self, test):
        """
        Method called at the end of a test.
        
        @param test Reference to the test object
        """
        TestResult.stopTest(self, test)
        self.parent.write('%s\n' % ResponseUTStopTest)
        
        # ensure that pending input is processed
        rrdy, wrdy, xrdy = select.select([self.parent.readstream],[],[], 0.01)

        if self.parent.readstream in rrdy:
            self.parent.readReady(self.parent.readstream.fileno())
