/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: engine.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <time.h>
#include <limits.h>
#include <mntent.h>

#include <fullengine.h>
#include "engine.h"
#include "dload.h"
#include "faulthdlr.h"
#include "handlemgr.h"
#include "internalAPI.h"
#include "memman.h"
#include "commit.h"
#include "discover.h"
#include "message.h"


/*----------------------------------------------------------------------------+
+                                                                             +
+                                  GLOBAL DATA                                +
+                                                                             +
+-----------------------------------------------------------------------------*/

int evms_block_dev_handle = 0;      /* handle to EVMS block device */

dlist_t PluginList           = NULL;
dlist_t DiskList             = NULL;
dlist_t SegmentList          = NULL;
dlist_t ContainerList        = NULL;
dlist_t RegionList           = NULL;
dlist_t EVMSObjectList       = NULL;
dlist_t VolumeList           = NULL;
dlist_t KillSectorList       = NULL;
dlist_t SoftVolumeDeleteList = NULL;
dlist_t HardVolumeDeleteList = NULL;
dlist_t VolumeRemoveList     = NULL;
dlist_t VolumeDataList       = NULL;

BOOLEAN changes_pending   = FALSE;

engine_mode_t engine_mode = ENGINE_CLOSED;

debug_level_t debug_level = DEFAULT;


char * log_file_name = DEFAULT_LOG_FILE;
int    log_file = 0;
u_char log_buf[LOG_BUF_SIZE];

/* Forward references */
static int ensure_dev_node(ADDRESS object,
                           TAG     object_tag,
                           uint    object_size,
                           ADDRESS object_handle,
                           ADDRESS parameters);

/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                                PRIVATE SUBROUTINES                                   +
+                                                                                      +
+-------------------------------------------------------------------------------------*/

char * build_archive_log_name(char * log_name, int index) {
    char * archive_log_file_name;
    char * pch1;
    char * pch2;

    /*
     * Get memory to build the archive file names.  Get enough for the name,
     * three bytes for the ".nn" that will be inserted/appended, plus one for
     * the null terminator.
     */
    archive_log_file_name = malloc(strlen(log_name) + 4);

    if (archive_log_file_name != NULL) {

        strcpy(archive_log_file_name, log_name);

        /*
         * The ".nn" will be inserted ahead of the last '.' in the file name.
         * If the file name doesn't have a '.' in it, the .nn will be
         * appended at the end.
         */
        pch1 = strrchr(archive_log_file_name, '.');
        if (pch1 == NULL) {
            pch1 = archive_log_file_name + strlen(archive_log_file_name);
        }

        /* Add/insert the ".nn". */
        *(pch1++) = '.';
        sprintf(pch1, "%d", index);

        /* Copy the remainder of the file name, if it exists. */
        pch2 = strrchr(log_name, '.');
        if (pch2 != NULL) {
            strcat(pch1, pch2);
        }
    }

    return archive_log_file_name;
}


#define MAX_LOG_ARCHIVES 10

static int start_logging(char * filename) {
    int rc = 0;
    int i;
    char * old_archive_log_name;
    char * new_archive_log_name;

    if (log_file == 0) {

        /* Roll back the archive log files. */
        old_archive_log_name = build_archive_log_name(filename, MAX_LOG_ARCHIVES);
        unlink(old_archive_log_name);

        for (i = MAX_LOG_ARCHIVES; i > 1; i--) {
            new_archive_log_name = build_archive_log_name(filename, i - 1);

            rename(new_archive_log_name, old_archive_log_name);

            free(old_archive_log_name);
            old_archive_log_name = new_archive_log_name;
        }

        rename(filename, old_archive_log_name);

        log_file = open(filename, O_CREAT | O_TRUNC | O_WRONLY, 0664);

        if (log_file < 0) {
            rc = log_file;
            log_file = 0;
        }

    } else {
        rc = EEXIST;
    }

    return rc;
}


static int stop_logging() {
    int rc = 0;

    if (log_file > 0) {
        close(log_file);
        log_file = 0;
    } else {
        rc = ENOENT;
    }

    return rc;
}


/*
 * Put a time stamp in the buffer.
 * This function assumes the buffer has enough room for the timestamp.
 */
void timestamp(char * buf, size_t len) {
    time_t t;
    size_t stamp_len;

    time(&t);

    strftime(buf, len, "%b %d %H:%M:%S ", localtime(&t));

    stamp_len = strlen(buf);

    gethostname(buf + stamp_len, len - stamp_len);

    strcat(buf, " ");
}


int engine_write_log_entry(debug_level_t level,
                           char        * fmt,
                           ...) {
    int rc = 0;
    int len;
    va_list args;

    if (level <= debug_level) {
        if (log_file > 0) {
            timestamp(log_buf, LOG_BUF_SIZE);

            strcat(log_buf, "Engine: ");
            len = strlen(log_buf);

            va_start(args, fmt);
            len += vsprintf(log_buf + strlen(log_buf), fmt, args);
            va_end(args);

            if (write(log_file, log_buf, len) < 0) {
                rc = errno;
            }

        } else {
            rc = ENOENT;
        }
    }

    return rc;
}


static int create_evms_dlists(void) {

    int error = ENOMEM;

    LOG_PROC_ENTRY();

    PluginList = CreateList();
    if (PluginList!=NULL) {

        DiskList = CreateList();
        if (DiskList!=NULL) {

            SegmentList = CreateList();
            if (SegmentList != NULL) {
                ContainerList = CreateList();
                if (ContainerList!=NULL) {

                    RegionList = CreateList();
                    if (RegionList != NULL) {
                        EVMSObjectList = CreateList();
                        if (EVMSObjectList!=NULL) {

                            VolumeList = CreateList();
                            if (VolumeList!=NULL) {

                                KillSectorList = CreateList();
                                if (KillSectorList!=NULL) {

                                    SoftVolumeDeleteList = CreateList();
                                    if (SoftVolumeDeleteList != NULL) {

                                        HardVolumeDeleteList = CreateList();
                                        if (HardVolumeDeleteList != NULL) {

                                            VolumeRemoveList = CreateList();
                                            if (VolumeRemoveList != NULL) {

                                                error = 0;
                                            }
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    LOG_PROC_EXIT_INT(error);
    return error;
}


static void free_list_items(dlist_t list) {

    int rc = 0;
    uint size;
    TAG tag;
    ADDRESS obj = NULL;

    rc = BlindGetObject(list,
                        &size,
                        &tag,
                        NULL,
                        FALSE,
                        &obj);

    while ((rc == DLIST_SUCCESS) && (obj != NULL)) {
        switch (tag) {
            case DISK_TAG:
                engine_free_logical_disk((storage_object_t *) obj);
                break;

            case SEGMENT_TAG:
                engine_free_segment((storage_object_t *) obj);
                break;

            case REGION_TAG:
                engine_free_region((storage_object_t *) obj);
                break;

            case EVMS_OBJECT_TAG:
                engine_free_evms_object((storage_object_t *) obj);
                break;

            case CONTAINER_TAG:
                engine_free_container((storage_container_t *) obj);
                break;

            case VOLUME_TAG:
                {
                    logical_volume_t * vol = (logical_volume_t *) obj;

                    destroy_handle(vol->app_handle);
                    engine_unregister_name(vol->name);

                    /*
                     * The volume may also be on the SoftVolumeDeleteList and/or
                     * the HardVolumeDeleteList.  If we are processing the
                     * VolumeList, remove the volume from the
                     * SoftVolumeDeleteList and the HardVolumeDeleteList, if it
                     * is on either of those lists.
                     */
                    if (list == VolumeList) {
                        DeleteObject(SoftVolumeDeleteList, vol);
                        DeleteObject(HardVolumeDeleteList, vol);
                    }
                }

                /*
                 * Fall through to remove the logical_volume from
                 * the list and free its memory.
                 */

            default:
                /*
                 * Don't know what the object is.  Remove the item from the
                 * list and free its memory
                 */
                DeleteObject(list, obj);
                free(obj);
                break;
        }

        /* Get another object from the list. */
        rc = BlindGetObject(list,
                            &size,
                            &tag,
                            NULL,
                            FALSE,
                            &obj);
    }
}


static void destroy_evms_dlists(void) {

    LOG_PROC_ENTRY();

    if (PluginList != NULL) {
        DestroyList(&PluginList, TRUE);
    }

    if (DiskList != NULL) {
        free_list_items(DiskList);
        DestroyList(&DiskList, TRUE);
    }

    if (SegmentList != NULL) {
        free_list_items(SegmentList);
        DestroyList(&SegmentList, TRUE);
    }

    if (ContainerList != NULL) {
        free_list_items(ContainerList);
        DestroyList(&ContainerList, TRUE);
    }

    if (RegionList != NULL) {
        free_list_items(RegionList);
        DestroyList(&RegionList, TRUE);
    }

    if (EVMSObjectList != NULL) {
        free_list_items(EVMSObjectList);
        DestroyList(&EVMSObjectList, TRUE);
    }

    if (VolumeList != NULL) {
        free_list_items(VolumeList);
        DestroyList(&VolumeList, TRUE);
    }

    if (KillSectorList != NULL) {
        DestroyList(&KillSectorList, TRUE);
    }

    if (SoftVolumeDeleteList != NULL) {
        free_list_items(SoftVolumeDeleteList);
        DestroyList(&SoftVolumeDeleteList, TRUE);
    }

    if (HardVolumeDeleteList != NULL) {
        free_list_items(HardVolumeDeleteList);
        DestroyList(&HardVolumeDeleteList, TRUE);
    }

    if (VolumeRemoveList != NULL) {
        free_list_items(VolumeRemoveList);
        DestroyList(&VolumeRemoveList, TRUE);
    }

    if (VolumeDataList != NULL) {
        DestroyList(&VolumeDataList, TRUE);
    }

    LOG_PROC_EXIT_VOID();
}


/*
 *  Called to open the EVMS Common Device Driver so we can send ioctl
 *  commands to the EVMS kernel runtime.
 *
 *  Input:  nothing
 *
 *  Output: rc > 0 if successful, rc = errno if any errors occur
 *
 */

int open_evms_block_dev( void ) {
    dev_t devt;
    int rc = 0;
    struct stat statbuf;
    struct flock lockinfo = {0};

    LOG_PROC_ENTRY();

    /* Only open if we haven't done so already. */
    if (evms_block_dev_handle == 0) {

        /* check for the /dev/evms subdir, and create if it does not exist.
         *
         * If devfs is running and mounted on /dev, these checks will all pass,
         * so a new node will not be created.
         */
        devt = makedev(EVMS_MAJOR, 0) ;
        rc = stat(EVMS_DEV_NODE_PATH, &statbuf);
        if (rc) {
            if (errno == ENOENT) {
                /* dev node does not exist. */
                rc = mkdir(EVMS_DEV_NODE_PATH, (S_IFDIR | S_IRWXU |
                                                S_IRGRP | S_IXGRP |
                                                S_IROTH | S_IXOTH));
            } else {
                LOG_CRITICAL("Problem with EVMS dev directory.  Error code from stat() is %d\n\n", errno);
            }

        } else {
            if (!(statbuf.st_mode & S_IFDIR)) {
                rc = unlink(EVMS_DEV_NODE_PATH);
                if (!rc) {
                    rc = mkdir(EVMS_DEV_NODE_PATH, (S_IFDIR | S_IRWXU |
                                                    S_IRGRP | S_IXGRP |
                                                    S_IROTH | S_IXOTH));
                }
            }
        }

        /*
         * Check for the /dev/evms/block_device node, and create if it does not
         * exist.
         */
        rc = stat(EVMS_DEVICE_NAME, &statbuf);
        if (rc) {
            if (errno == ENOENT) {
                /* dev node does not exist */
                rc = mknod(EVMS_DEVICE_NAME, (S_IFBLK | S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP), devt);
            } else {
                LOG_CRITICAL("Problem with EVMS block device node directory.  Error code form stat() is %d\n\n", errno);
            }

        } else {
            if (!(statbuf.st_mode & S_IFBLK)) {
                rc = unlink(EVMS_DEVICE_NAME);
                if (!rc) {
                    rc = mknod(EVMS_DEVICE_NAME, (S_IFBLK | S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP), devt);
                }
            }
        }

        evms_block_dev_handle = open(EVMS_DEVICE_NAME, O_RDWR|O_NONBLOCK);

        if (evms_block_dev_handle > 0) {

            /*
             * Lock our access to the Engine.  Figure out whether to lock for
             * reading or for writing based on the engine_mode.
             */
            if (engine_mode == ENGINE_READONLY) {
                lockinfo.l_type = F_RDLCK;
            } else {
                lockinfo.l_type = F_WRLCK;
            }

            /* Lock the whole thing */
            lockinfo.l_whence = SEEK_SET;
            lockinfo.l_start = 0;
            lockinfo.l_len = 0;

            rc = fcntl(evms_block_dev_handle, F_SETLK, &lockinfo);

            if (rc != 0) {
                char buf1[PATH_MAX + 1];

                /*
                 * We could not lock the Engine.  It must be in use by some
                 * other process.  Try to find out what process it is and tell
                 * the user.
                 */
                rc = fcntl(evms_block_dev_handle, F_GETLK, &lockinfo);

                if (rc == 0) {
                    char buf2[PATH_MAX + 1];
                    int bytes;

                    /*
                     * The file name for the pid is symlinked at
                     * /proc/{pid}/exe
                     */
                    sprintf(buf1, "/proc/%d/exe", lockinfo.l_pid);

                    bytes = readlink(buf1, buf2, sizeof(buf2));

                    if (bytes > 0) {
                        /*
                         * readlink() does not null terminate the returned
                         * file name.
                         */
                        buf2[bytes] = '\0';

                        engine_user_message(NULL, NULL, "The EVMS Engine is currently in use by process %d (%s).\n", lockinfo.l_pid, buf2);

                    } else {
                        /*
                         * Couldn't get the name for the process ID.
                         * Log a message with just the PID.
                         */
                        engine_user_message(NULL, NULL, "The EVMS Engine is currently in use by process %d.\n", lockinfo.l_pid);
                    }

                } else {
                    /*
                     * Couldn't get the lockinfo of the locking process.
                     * Log a generic message.
                     */
                    engine_user_message(NULL, NULL, "The EVMS Engine is currently in use by another process.\n");
                }

                close(evms_block_dev_handle);
                evms_block_dev_handle = EACCES;
            }
        }
    }

    LOG_PROC_EXIT_INT(evms_block_dev_handle);
    return evms_block_dev_handle;
}


void close_evms_block_dev(void) {
    if (evms_block_dev_handle > 0) {
        close(evms_block_dev_handle);
        evms_block_dev_handle = 0;
    }
}


/*
 * Get the kernel volume data for each volume in the system and put it on the
 * VolumeDataList.
 */
int get_kernel_volume_data() {
    int rc = 0;
    int status;

    LOG_PROC_ENTRY();

    if (VolumeDataList != NULL) {
        DestroyList(&VolumeDataList, TRUE);
    }

    VolumeDataList = CreateList();
    if (VolumeDataList != NULL) {
        evms_user_minor_t minor_info;

        /* Run a loop getting a minor code from the EVMS kernel. */

        minor_info.command = EVMS_FIRST_VOLUME;
        minor_info.status = EVMS_VOLUME_VALID;

        status = ioctl(evms_block_dev_handle, EVMS_GET_MINOR, &minor_info);

        /* Set the command to "next volume" for the loop below. */
        minor_info.command = EVMS_NEXT_VOLUME;

        while ((status == 0) && (rc == 0) && (minor_info.status == EVMS_VOLUME_VALID)) {
            evms_volume_data_t * vol_data;

            LOG_DEBUG("Got minor number %d.\n", minor_info.minor);

            /*
             * Allocate a volume data structure and have the kernel fill in the
             * volume data for the minor number.
             */

            vol_data = (evms_volume_data_t *) malloc(sizeof(evms_volume_data_t));
            if (vol_data != NULL) {
                vol_data->minor = minor_info.minor;
                vol_data->status = 0;   /* Not used, but zeroed for safety. */

                status = ioctl(evms_block_dev_handle, EVMS_GET_VOLUME_DATA, vol_data);

                if ((status == 0) && (vol_data->status == 0)) {
                    ADDRESS handle;

                    LOG_DEBUG("Minor number %d is for volume %s.\n", minor_info.minor, vol_data->volume_name);

                    /* Put the volume data structure into our own list. */

                    rc = InsertObject(VolumeDataList,
                                      sizeof(evms_volume_data_t),
                                      vol_data,
                                      VOLUME_DATA_TAG,
                                      NULL,
                                      AppendToList,
                                      FALSE,
                                      &handle);

                    if (rc == DLIST_SUCCESS) {
                        /* Get the next minor number. */
                        status = ioctl(evms_block_dev_handle, EVMS_GET_MINOR, &minor_info);

                    } else {
                        LOG_WARNING("Error code %d when inserting a volume_data structure into the VolumeDataList.\n", rc);
                    }

                } else {
                    LOG_WARNING("Error getting volume data for minor %d.  status is %d.  errno is %d.  vol_data.status is %d.\n", minor_info.minor, status, errno, vol_data->status);
                }

            } else {
                LOG_CRITICAL("Error allocating memory for a evms_volume_data structure.\n");
                rc = ENOMEM;
            }
        }

        if (status != 0) {
            rc = -errno;
            LOG_WARNING("Error from ioctl to get kernel volume minor number.  status is %d.  errno is %d.\n", status, errno);
        }

    } else {
        LOG_CRITICAL("Error allocating memory for the VolumeDataList.\n");
        rc = ENOMEM;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Check if the Engine is currently allowing read access.  The Engine must be
 * open and a commit must not be in progress.
 */
int check_engine_read_access() {

    int rc = 0;

    LOG_PROC_ENTRY();

    if ((engine_mode == ENGINE_CLOSED) ||
        commit_in_progress) {
        if (engine_mode == ENGINE_CLOSED) {
            LOG_ERROR("The Engine is not open.\n");
        }
        if (commit_in_progress) {
            LOG_ERROR("The Engine is currently committing changes.\n");
        }
        rc = EACCES;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Check if the Engine is allowing write access, i.e., data structures, etc.
 * can be updated.  The Engine must be opened in a mode that allows writing
 * and a commit must not be in progress.
 */
int check_engine_write_access() {

    int rc = 0;

    LOG_PROC_ENTRY();

    if ((engine_mode == ENGINE_CLOSED) ||
        (engine_mode == ENGINE_READONLY) ||
        commit_in_progress) {
        if (engine_mode == ENGINE_CLOSED) {
            LOG_ERROR("The Engine is not open.\n");
        } else {
            if (engine_mode == ENGINE_READONLY) {
                LOG_ERROR("The Engine is open for reading only.\n");
            }
        }
        if (commit_in_progress) {
            LOG_ERROR("The Engine is currently committing changes.\n");
        }
        rc = EACCES;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*-------------------------------------------------------------------------------------+
+                                                                                      +
+                                 General Interface                                    +
+                                                                                      +
+  Subroutines:  Open, Close, ChangesPending, Rediscover, getting and setting debug    +
+                levels                                                                +
+                                                                                      +
+-------------------------------------------------------------------------------------*/

void evms_get_api_version(evms_version_t * version) {
    version->major      = ENGINE_API_MAJOR_VERSION;
    version->minor      = ENGINE_API_MINOR_VERSION;
    version->patchlevel = ENGINE_API_PATCH_LEVEL;
}


static int check_kernel_ioctl_version(void) {

    int rc = 0;
    int status;
    evms_version_t kernel_ioctl_version;

    LOG_PROC_ENTRY();

    status = ioctl(evms_block_dev_handle, EVMS_GET_IOCTL_VERSION, &kernel_ioctl_version);

    if (status == 0) {
        if (kernel_ioctl_version.major != EVMS_IOCTL_INTERFACE_MAJOR) {
            rc = ENOSYS;

        } else {
            if (kernel_ioctl_version.minor < EVMS_IOCTL_INTERFACE_MINOR) {
                rc = ENOSYS;

            } else {
                if (kernel_ioctl_version.minor == EVMS_IOCTL_INTERFACE_MINOR) {
                    if (kernel_ioctl_version.patchlevel < EVMS_IOCTL_INTERFACE_PATCHLEVEL) {
                        rc = ENOSYS;
                    }
                }
            }
        }

        if (rc != 0) {
            engine_user_message(NULL, NULL, "The EVMS kernel does not supply a version of its ioctl interface that is compatible with this Engine.\n");
            engine_user_message(NULL, NULL, "The Engine requires EVMS kernel ioctl version %d.%d.%d.\n", \
                                EVMS_IOCTL_INTERFACE_MAJOR, EVMS_IOCTL_INTERFACE_MINOR, EVMS_IOCTL_INTERFACE_PATCHLEVEL);
            engine_user_message(NULL, NULL, "The kernel's EVMS ioctl version number is %d.%d.%d.\n", \
                                kernel_ioctl_version.major, kernel_ioctl_version.minor, kernel_ioctl_version.patchlevel);
        } else {
            LOG_DEBUG("The Engine requires EVMS kernel ioctl version %d.%d.%d.\n", \
                      EVMS_IOCTL_INTERFACE_MAJOR, EVMS_IOCTL_INTERFACE_MINOR, EVMS_IOCTL_INTERFACE_PATCHLEVEL);
            LOG_DEBUG("The kernel's EVMS ioctl version number is %d.%d.%d.\n", \
                      kernel_ioctl_version.major, kernel_ioctl_version.minor, kernel_ioctl_version.patchlevel);
        }

    } else {
        /* The ioctl failed.  Get the error number. */
        rc = errno;
    }

    LOG_PROC_EXIT_INT(rc)
    return rc;
}


int evms_open_engine(engine_mode_t                mode,
                     engine_message_functions_t * callbacks,
                     debug_level_t                level,
                     char                       * log_name) {
    int rc = 0;

    ui_callbacks = callbacks;

    debug_level = level;

    if (log_name != NULL) {
        log_file_name = log_name;
    }

    start_logging(log_file_name);

    LOG_PROC_ENTRY();

    if ((mode != ENGINE_READONLY) &&
        (mode != ENGINE_READWRITE)) {
        LOG_ERROR("Open mode of %d is not valid.\n", mode);

        LOG_PROC_EXIT_INT(EINVAL);
        return EINVAL;
    }

    if (engine_mode == ENGINE_CLOSED) {

        engine_mode = mode;

        /* Initialize the random number generator. */
        srand(time(NULL) + getpid());

        /* Make sure we can access the EVMS kernel code. */
        rc = open_evms_block_dev();

        if (rc > 0) {

            rc = check_kernel_ioctl_version();

            if (rc == 0) {
                /* Install signal handlers. */
                install_signal_handlers();

                /* Create the EVMS linked lists. */
                rc = create_evms_dlists();

                if (rc == 0) {
                    /*
                     * Get information on all the kernel exported EVMS
                     * volumes.
                     */
                    rc = get_kernel_volume_data();

                    /* Initialize the Handle Manager. */
                    if (rc == 0) {

                        /*
                         * Make sure the dev nodes are in sync with the
                         * kernel's view of the volumes.
                         */
                        evms_verbose_level_t verbose = NO_MESSAGES;

                        ForEachItem(VolumeDataList,
                                    ensure_dev_node,
                                    &verbose,
                                    TRUE);

                        if (initialize_handle_manager()) {

                            /* Load the EVMS plug-ins. */
                            rc = load_plugins(PluginList);

                            /* Discover what's in the disk system */
                            if (rc == 0) {
                                rc = do_discovery();

                                if (rc == 0) {
                                    /*
                                     * If the Engine is open for read only
                                     * access, close the kernel device handle.
                                     * This lets any other instance of the
                                     * Engine open for write (exclusive).
                                     */
                                    if (mode == ENGINE_READONLY) {
                                        close_evms_block_dev();
                                    }

                                } else {
                                    /*
                                     * Discovery failed.
                                     * Unload the plug-ins.
                                     */
                                    unload_plugins(PluginList);
                                }

                            } else {
                                LOG_DEBUG("Return code from load_plugins is %d.\n", rc);
                            }

                        } else {
                            LOG_CRITICAL("Handle Manager failed to initialize.\n");
                        }

                    } else {
                        LOG_CRITICAL("get_kernel_volume_data failed with return code %d.\n", rc);
                    }

                    if (rc != 0) {
                        /*
                         * Something went wrong.
                         * Destroy the lists we created.
                         */
                        destroy_evms_dlists();
                    }

                } else {
                    LOG_CRITICAL("create_evms_dlists failed with return code %d.\n", rc);
                }
            }

            if (rc != 0) {
                /* Something went wrong.  Close the EVMS block device. */
                close_evms_block_dev();
            }
        }

        if (rc != 0) {
            /* Something went wrong.  Mark the Engine closed. */
            engine_mode = ENGINE_CLOSED;
        }

    } else {
        rc = EACCES;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


int evms_close_engine(void) {

    LOG_PROC_ENTRY();

    if (engine_mode != ENGINE_CLOSED) {
        engine_mode = ENGINE_CLOSED;
        unload_plugins(PluginList);
        destroy_evms_dlists();
        destroy_all_handles();
        clear_name_registry();
        close_evms_block_dev();
        remove_signal_handlers();

    } else {
        LOG_DEBUG("The Engine is already closed.\n");
    }

    LOG_PROC_EXIT_INT(0);

    stop_logging();

    return 0;
}


BOOLEAN evms_changes_pending(void) {

    LOG_PROC_ENTRY();

    LOG_PROC_EXIT_BOOLEAN(changes_pending);
    return changes_pending;
}


/*
 * evms_rediscover only forces a kernel rediscover.  It does not make the
 * Engine go through discovery again.
 */
int evms_rediscover(void) {

    int rc = 0;

    LOG_PROC_ENTRY();

    rc = kernel_rediscover();

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Get the Engine's current setting of the debug level.
 */
int evms_get_debug_level(debug_level_t * level) {

    int rc = 0;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();

    if (rc == 0) {
        *level = debug_level;
    }

    LOG_PROC_EXIT_INT(0)
    return 0;
}


/*
 * Set the Engine's debug level.
 */
int evms_set_debug_level(debug_level_t level) {

    int rc = 0;

    LOG_PROC_ENTRY();

    /*
     * People that opened the Engine for read-only are allowed to set their
     * debug level.
     */
    rc = check_engine_read_access();

    if (rc == 0) {

        if ((level >= CRITICAL) &&
            (level <= EVERYTHING)) {
            debug_level = level;

        } else {
            LOG_ERROR("Debug level is out of range.  It must be between %d and %d, inclusive.\n", CRITICAL, EVERYTHING);
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc)
    return(rc);
}


/*
 * Get the EVMS kernel's current setting of its info level.
 * This API can be called without having the Engine open.
 */
int evms_get_kernel_info_level(debug_level_t * level) {

    int rc = 0;
    int status;

    LOG_PROC_ENTRY();

    if (evms_block_dev_handle >= 0) {
        if (evms_block_dev_handle > 0) {

            /* Call the kernel to get the info level. */
            status = ioctl(evms_block_dev_handle, EVMS_GET_INFO_LEVEL, level);

            /* If something went wrong with the ioctl, return the error code. */
            if (status) {
                rc = errno;
            }

        } else {
            /*
             * The Engine does not have the kernel block device open.
             * Open it.
             */
            rc = open_evms_block_dev();

            if (rc == 0) {

                /* Call the kernel to get the info level. */
                status = ioctl(evms_block_dev_handle, EVMS_GET_INFO_LEVEL, level);

                /* If something went wrong with the ioctl, return the error code. */
                if (status) {
                    rc = errno;
                }

                close_evms_block_dev();
            }
        }

    } else {
        /*
         * The Engine previously attempted to open the EVMS kernel block device
         * and got an error.  evms_block_dev_handle contains the error code.
         */
        rc = evms_block_dev_handle;
    }

    LOG_PROC_EXIT_INT(rc)
    return rc;
}

/*
 * Set the EVMS kernel's info level.
 */
int evms_set_kernel_info_level(debug_level_t level) {

    int rc = 0;
    int status;

    LOG_PROC_ENTRY();

    /*
     * The user must have opened the Engine for readwrite.  Since only one
     * writer is allowed at a time, this guarantees that no other Engine process
     * has the ability to change the kernel's info level.
     */
    rc = check_engine_write_access();

    if (rc == 0) {

        if ((level >= CRITICAL) &&
            (level <= EVERYTHING)) {

            /* Call the kernel to set the info level. */
            status = ioctl(evms_block_dev_handle, EVMS_SET_INFO_LEVEL, &level);

            /* If something went wrong with the ioctl, return the error code. */
            if (status) {
                rc = errno;
            }

        } else {
            LOG_ERROR("Info level is out of range.  It must be between %d and %d, inclusive.\n", CRITICAL, EVERYTHING);
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc)
    return rc;
}


/*-----------------------------------------------------------------------------+
+                                                                              +
+                              Logging Functions                               +
+                                                                              +
+-----------------------------------------------------------------------------*/

int evms_write_log_entry(debug_level_t level,
                         char        * module_name,
                         char        * fmt,
                         ...) {
    int rc = 0;
    int len;
    va_list args;

    rc = check_engine_read_access();
    if (rc == 0) {
        if (level <= debug_level) {
            if (log_file > 0) {
                timestamp(log_buf, LOG_BUF_SIZE);
                strcat(log_buf, module_name);
                strcat(log_buf, ":  ");
                len = strlen(log_buf);

                va_start(args, fmt);
                len += vsprintf(log_buf + strlen(log_buf), fmt, args);
                va_end(args);

                if (write(log_file, log_buf, len) < 0) {
                    rc = errno;
                }

            } else {
                rc = ENOENT;
            }
        }
    }

    return rc;
}


/*-----------------------------------------------------------------------------+
+                                                                              +
+                             Some List Handlers                               +
+                                                                              +
+-----------------------------------------------------------------------------*/

/*
 * Make sure this object has an app_handle.
 */
int ensure_app_handle(void * object, object_type_t object_type, object_handle_t * app_handle) {
    int rc = 0;

    LOG_PROC_ENTRY();

    if (*app_handle == 0) {
        LOG_DEBUG("Create a handle for a thing of type %d.\n", object_type);
        rc = create_handle(object,
                           object_type,
                           app_handle);
        if (rc == 0) {
            LOG_DEBUG("Handle is %d.\n", *app_handle);
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Add the app_handle for a given object to a handle_array_t.
 * The parameters of this function are structured so that it can be called by
 * ForEachItem().
 */
int make_handle_entry(ADDRESS Object, TAG Tag, uint ObjectSize, ADDRESS ObjectHandle, ADDRESS Parameters) {
    int rc = 0;

    handle_array_t * ha = (handle_array_t *) Parameters;

    LOG_PROC_ENTRY();

    LOG_DEBUG("Current number of entries in handle array:  %d.\n", ha->count);
    switch (Tag) {
        case PLUGIN_TAG:
            {
                plugin_record_t * pPlugRec = (plugin_record_t *) Object;

                LOG_DEBUG("Add entry for plug-in %s.\n", pPlugRec->short_name);
                rc = ensure_app_handle(Object,
                                       PLUGIN,
                                       &pPlugRec->app_handle);
                if (rc == 0) {
                    ha->handle[ha->count] = pPlugRec->app_handle;
                    ha->count++;
                }
            }
            break;

        case DISK_TAG:
        case SEGMENT_TAG:
        case REGION_TAG:
        case EVMS_OBJECT_TAG:
            {
                storage_object_t * pObj = (storage_object_t *) Object;

                LOG_DEBUG("Add entry for storage object %s.\n", pObj->name);
                rc = ensure_app_handle(Object,
                                       pObj->object_type,
                                       &pObj->app_handle);
                if (rc == 0) {
                    ha->handle[ha->count] = pObj->app_handle;
                    ha->count++;
                }
            }
            break;

        case CONTAINER_TAG:
            {
                storage_container_t * pCon = (storage_container_t *) Object;

                LOG_DEBUG("Add entry for container %s.\n", pCon->name);
                rc = ensure_app_handle(Object,
                                       CONTAINER,
                                       &pCon->app_handle);
                if (rc == 0) {
                    ha->handle[ha->count] = pCon->app_handle;
                    ha->count++;
                }
            }
            break;

        case VOLUME_TAG:
            {
                logical_volume_t * pVol = (logical_volume_t *) Object;

                LOG_DEBUG("Add entry for volume %s.\n", pVol->name);
                rc = ensure_app_handle(Object,
                                       VOLUME,
                                       &pVol->app_handle);
                if (rc == 0) {
                    ha->handle[ha->count] = pVol->app_handle;
                    ha->count++;
                }
            }
            break;

        default:
            /* It's nothing for which we hand out app handles.  Skip it. */
            LOG_WARNING("Attempt to make an app handle for an object of tag %ld.\n", Tag);
            break;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Make a user (it has a Engine memory block header) array of handles
 * (handle_array_t) for the objects in a dlist_t.
 */
int make_user_handle_array(dlist_t list, handle_array_t * * ha) {
    int rc = 0;
    uint count;
    uint size;

    LOG_PROC_ENTRY();

    rc = GetListSize(list, &count);

    if (rc == 0) {
        LOG_DEBUG("Number of objects in the list:  %d\n", count);
        if (count > 1) {
            size = sizeof(handle_array_t) + ((count -1) * sizeof(object_handle_t));
        } else {
            size = sizeof(handle_array_t);
        }

        *ha = alloc_app_struct(size, NULL);
        if (*ha != NULL) {
            rc = ForEachItem(list,
                             make_handle_entry,
                             *ha,
                             TRUE);
        } else {
            rc = ENOMEM;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Get a list of handles for plug-ins, optionally filtering by plug-in type.
 */
int evms_get_plugin_list(evms_plugin_code_t    type,
                         plugin_search_flags_t flags,
                         handle_array_t    * * plugin_handle_list) {
    int rc = 0;
    dlist_t plugin_list;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();
    if (rc == 0) {

        if (plugin_handle_list != NULL) {
            rc = engine_get_plugin_list(type, flags, &plugin_list);

            if (rc == 0) {
                rc = make_user_handle_array(plugin_list, plugin_handle_list);

                DestroyList(&plugin_list, FALSE);
            }

        } else {
            LOG_DEBUG("User specified NULL pointer for plugin_handle_list. I'm not doing anything since there is nowhere to put the results.\n");
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Get the handle for a plug-in with a given ID.
 */
int evms_get_plugin_by_ID(plugin_id_t       plugin_ID,
                          object_handle_t * plugin_handle) {
    int rc = 0;
    plugin_record_t * pPlugRec;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();
    if (rc == 0) {
        rc = engine_get_plugin_by_ID(plugin_ID, &pPlugRec);

        if (rc == 0) {
            rc = ensure_app_handle(pPlugRec,
                                   PLUGIN,
                                   &pPlugRec->app_handle);

            if (rc == 0) {
                *plugin_handle = pPlugRec->app_handle;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Get the handle for a plug-in with a given short name.
 */
int evms_get_plugin_by_name(char            * plugin_name,
                            object_handle_t * plugin_handle) {
    int rc = 0;

    plugin_record_t * pPlugRec;

    LOG_PROC_ENTRY();

    rc = check_engine_read_access();
    if (rc == 0) {
        rc = engine_get_plugin_by_name(plugin_name, &pPlugRec);

        if (rc == 0) {
            rc = ensure_app_handle(pPlugRec,
                                   PLUGIN,
                                   &pPlugRec->app_handle);

            if (rc == 0) {
                *plugin_handle = pPlugRec->app_handle;
            }
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/********************************************/
/* Functions for updating the dev node tree */
/********************************************/

/*
 * Check if a volume is mounted.  Return the name of the mount point if
 * requested (mount_name != NULL).
 */
BOOLEAN is_mounted(char * volume_name, char * * mount_name) {

    BOOLEAN result = FALSE;
    FILE *          mount_records;  /* Pointer for system's mount records */
    struct mntent * mount_entry;    /* Holds data for entry in mount list */

    LOG_PROC_ENTRY();

    /*
     * If the caller wants a mount name, initialize it to NULL in case we don't
     * find a mount point for the volume.
     */
    if (mount_name != NULL) {
        *mount_name = NULL;
    }

    mount_records = setmntent(MOUNTED, "r");

    if (mount_records == NULL) {

        /*
         * Unable to access list of mounted devices in /etc/mtab !
         * Attempt to open /proc/mounts for access.
         */
        mount_records = setmntent("/proc/mounts", "r");
    }

    if (mount_records != NULL) {

        /* Scan the mount entries to see if this volume is mounted. */
        while ((!result) && ((mount_entry = getmntent(mount_records)) != NULL)) {
            if (strcmp(mount_entry->mnt_fsname, volume_name) == 0) {
                result = TRUE;

                /* If the user wants to know the mount point, make a copy */
                if (mount_name != NULL) {
                    *mount_name = strdup(mount_entry->mnt_dir);
                }
            }
        }

        /* Close the stream. */
        endmntent(mount_records);

    } else {
        LOG_WARNING("Could not obtain a list of mounted devices from neither /proc/mounts nor " MOUNTED ".\n");
    }

    /* Check if the volume is mounted as swap. */
    if (!result) {
        FILE * proc_swaps;

        proc_swaps = fopen("/proc/swaps", "r");
        if (proc_swaps != NULL) {
            char buffer[EVMS_VOLUME_NAME_SIZE+1];
            int volume_name_len = strlen(volume_name);

            while (!result && (fgets(buffer, EVMS_VOLUME_NAME_SIZE+1, proc_swaps) != NULL)) {

                if (strncmp(buffer, volume_name, volume_name_len) == 0) {
                    result = TRUE;

                    /* If the user wants to know the mount point, make a copy */
                    if (mount_name != NULL) {
                        *mount_name = strdup("swap");
                    }
                }
            }

            fclose(proc_swaps);

        } else {
            LOG_WARNING("Could not open /proc/swaps.\n");
        }
    }

    LOG_PROC_EXIT_BOOLEAN(result);
    return result;
}


/*
 * Make a directory in the EVMS dev node tree.
 * This function is a utility function for make_evms_dev_node() below.
 */
static int make_evms_dir_entry(char * dir_name, evms_verbose_level_t verbose) {
    char name_buf[EVMS_VOLUME_NAME_SIZE];
    char * tmp_ptr = name_buf;
    struct stat statbuf;
    int rc = 0;

    LOG_PROC_ENTRY();

    /* See if this directory exists */
    rc = stat(dir_name, &statbuf);
    if (rc != 0) {
        if (errno == ENOENT) {
            /*
             * This directory doesn't exist.  Peel off the top-most directory
             * name, and try again.
             */
            strcpy(name_buf, dir_name);
            tmp_ptr = strrchr(name_buf, '/');

            if (tmp_ptr != NULL) {
                *tmp_ptr = 0;

                /* Call myself recursively to make the parent directory. */
                rc = make_evms_dir_entry(name_buf, verbose);
                if (rc == 0) {
                    rc = mkdir(dir_name, (S_IFDIR | S_IRWXU |
                                          S_IRGRP | S_IXGRP |
                                          S_IROTH | S_IXOTH));

                    if (rc != 0) {
                        LOG_WARNING("mkdir(%s) failed with error code %d.\n", dir_name, rc);
                        if (verbose >= ERRORS_ONLY) {
                            fprintf(stderr, "mkdir(%s) failed with error code %d.\n", dir_name, rc);
                        }
                    }
                }

            } else {
                /*
                 * Either we were passed a relative path name, or the root
                 * file system doesn't exist.
                 */
                rc = ENODEV;
            }

        } else {
            LOG_WARNING("stat(%s) failed with error code %d.\n", dir_name, errno);
            if (verbose >= ERRORS_ONLY) {
                fprintf(stderr, "stat(%s) failed with error code %d.\n", dir_name, errno);
            }
        }

    } else {
        /* Make sure this is a directory */
        if (!(statbuf.st_mode & S_IFDIR)) {
            LOG_ERROR("%s is a non-directory file\n", dir_name);
            if (verbose >= ERRORS_ONLY) {
                fprintf(stderr, "%s is a non-directory file\n", dir_name);
            }
            rc = EINVAL;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Make a dev node for a volume.
 */
static int make_evms_dev_node(char * volume_name, uint minor_number, evms_verbose_level_t verbose) {
    dev_t devt;
    char name_buf[EVMS_VOLUME_NAME_SIZE];
    char * tmp_ptr = name_buf;
    int rc = 0;

    LOG_PROC_ENTRY();

    /* Make any directory entries that are necessary */
    strcpy(name_buf, volume_name);
    tmp_ptr = strrchr(name_buf, '/');
    *tmp_ptr = '\0';

    rc = make_evms_dir_entry(name_buf, verbose);
    if (rc == 0) {
        devt = makedev(EVMS_MAJOR, minor_number) ;
        rc = mknod(volume_name, S_IFBLK | S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH, devt);
        if (rc == 0) {
            LOG_DEBUG("Made a node for %s (minor number %d).\n", volume_name, minor_number);
            if (verbose >= ERRORS_AND_INFO) {
                printf("Made a node for %s (minor number %d).\n", volume_name, minor_number);
            }

        } else {
            LOG_WARNING("mknod(%s) failed with error code %d.\n", volume_name, rc);
            if (verbose >= ERRORS_ONLY) {
                fprintf(stderr, "mknod(%s) failed with error code %d.\n", volume_name, rc);
            }
        }

    } else {
        LOG_WARNING("Failure making directory %s (%d)\n", name_buf, rc);
        if (verbose >= ERRORS_ONLY) {
            fprintf(stderr, "Failure making directory %s (%d)\n", name_buf, rc);
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Check if a dev node exists for a volume and that is has the correct minor
 * number.
 * Returns:
 * 0        dev node exists with the correct minor number
 * ENOENT   dev node does not exist
 * EEXIST   dev node exists but has the wrong minor number
 * other    some unexpected error code from stat()
 */
int hasa_dev_node(char * volume_name, uint minor) {

    int rc = 0;
    struct stat statbuf;

    LOG_PROC_ENTRY();

    rc = stat(volume_name, &statbuf);

    if (rc == 0) {

        /* dev node exists. Check if it has the right minor number. */
        dev_t devt;

        devt = makedev(EVMS_MAJOR, minor);
        if (statbuf.st_rdev != devt) {

            /* dev node exists but it has the wrong minor number. */
            rc = EEXIST;
        }

    } else {
        /* stat() failed.  Return the error code. */
        rc = errno;
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Given a kernel volume data structure, make sure there is a dev node for the
 * volume.
 * This function has its parameters structured so that it can be called by the
 * ForEachItem() dlist processor.
 */
static int ensure_dev_node(ADDRESS object,
                           TAG     object_tag,
                           uint    object_size,
                           ADDRESS object_handle,
                           ADDRESS parameters) {

    int rc = 0;
    evms_volume_data_t * vol_data = (evms_volume_data_t *) object;
    evms_verbose_level_t * verbose = (evms_verbose_level_t *) parameters;

    /* Safety check */
    if (object_tag == VOLUME_DATA_TAG) {
        rc = hasa_dev_node(vol_data->volume_name, vol_data->minor);

        switch (rc) {
            case 0:
                if (*verbose >= ERRORS_AND_INFO) {
                    printf("Device node for %s (minor number %d) already exists and is correct.\n", vol_data->volume_name, vol_data->minor);
                }
                break;

            case ENOENT:
                /* /dev node does not exist. */
                rc = make_evms_dev_node(vol_data->volume_name, vol_data->minor, *verbose);
                if (rc) {
                    LOG_WARNING("Failure when making device node for %s.  Return code was %d.\n", vol_data->volume_name, rc);
                    if (*verbose >= ERRORS_ONLY) {
                        fprintf(stderr, "Failure when making device node for %s.  Return code was %d.\n", vol_data->volume_name, rc);
                    }
                }
                break;

            case EEXIST:
                /*
                 * Dev-node minor doesn't match kernel.  Remove and recreate.
                 */
                if (*verbose >= ERRORS_AND_INFO) {
                    printf("The minor number for %s is out of sync with the kernel.  Deleting and recreating.\n", vol_data->volume_name);
                }
                rc = unlink(vol_data->volume_name);
                if (rc == 0) {
                    rc = make_evms_dev_node(vol_data->volume_name, vol_data->minor, *verbose);
                    if (rc != 0) {
                        LOG_WARNING("Failure when making device node for %s.  Return code was %d.\n", vol_data->volume_name, rc);
                    }

                } else {
                    LOG_WARNING("unlink(%s) failed with error code %d.\n", vol_data->volume_name, errno);
                    if (*verbose >= ERRORS_ONLY) {
                        fprintf(stderr, "unlink(%s) failed with error code %d.\n", vol_data->volume_name, errno);
                    }
                }
                break;

            default:
                /*
                 * The stat() command returned an unexpected error when
                 * checking for the dev node.
                 */
                LOG_WARNING("stat(%s) failed with error code %d.\n", vol_data->volume_name, errno);
                if (*verbose >= ERRORS_ONLY) {
                    fprintf(stderr, "stat(%s) failed with error code %d.\n", vol_data->volume_name, errno);
                }
                break;
        }
    }

    return DLIST_SUCCESS;
}


/*
 * Check if a given volume name matches the volume name in a volume data
 * record obtained from the kernel.
 * This function has its parameters structured so that it can be called by
 * the ForEachItem() dlist processor.
 * This function returns 0 if the name does not match, EEXIST if the name does
 * match.  A zero return code will allow ForEachItem() to continue processing
 * the list.  A non-zero return code stops ForEachItem(), which is what we want
 * to do if the volume name matches.
 */
static int is_kernel_volume(ADDRESS object,
                            TAG     object_tag,
                            uint    object_size,
                            ADDRESS object_handle,
                            ADDRESS parameters) {
    int rc = 0;
    evms_volume_data_t * vol_data = (evms_volume_data_t *) object;
    char * vol_name = (char *) parameters;

    LOG_PROC_ENTRY();

    /* Safety check */
    if (object_tag == VOLUME_DATA_TAG) {
        if (strcmp(vol_name,vol_data->volume_name) == 0) {
            rc = EEXIST;
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}


/*
 * Given a dev node, delete it if it is not currently exported as a volume by
 * the kernel.
 */
static void remove_dev_node_if_not_kernel_volume(char * dev_node, evms_verbose_level_t verbose) {

    int rc;

    /*
     * Check if we have a kernel volume that matches the one in the dev tree.
     */
    rc = ForEachItem(VolumeDataList,
                     is_kernel_volume,
                     dev_node,
                     TRUE);
    /*
     * is_kernel_volume() returns 0 when it doesn't find a match.  Returning 0
     * enables ForEachItem() to continue processing the list.
     * is_kernel_volume() returns non-zero when it finds a match, which aborts
     * the ForEachItem() which halts the search.  Getting 0 from ForEachItem()
     * means a kernel volume with the same name was not found.
     */
    if (rc == 0) {

        if (!is_mounted(dev_node, NULL)) {
            LOG_DEBUG("Removing unused device node %s.\n", dev_node);
            if (verbose >= ERRORS_AND_INFO) {
                printf("Removing unused device node %s.\n", dev_node);
            }

            rc = unlink(dev_node);

            if (rc != 0) {
                LOG_WARNING("unlink(%s) failed with error code %d.\n", dev_node, errno);
                if (verbose >= ERRORS_ONLY) {
                    fprintf(stderr, "unlink(%s) failed with error code %d.\n", dev_node, errno);
                }
            }

        } else {
            LOG_WARNING("Volume %s is not exported by the kernel but its device node cannot be deleted because it is mounted.\n", dev_node);
            if (verbose >= ERRORS_ONLY) {
                fprintf(stderr, "Volume %s is not exported by the kernel but its device node cannot be deleted because it is mounted.\n", dev_node);
            }
        }
    }
}


/*
 * Given a directory in the /dev/evms tree Remove any nodes and directories that
 * are not currently exported as volumes by the kernel.  If this directory ends
 * up having no nodes, then the directory is removed.
 * This function calls itself recursively to prune subdirectories that are in
 * this directory.
 */
static void prune_dev_tree(char * dir_name, evms_verbose_level_t verbose) {

    int   rc;
    DIR * pDir;
    char  full_name[EVMS_VOLUME_NAME_SIZE + 1];
    uint  name_offset;

    strcpy(full_name, dir_name);
    strcat(full_name, "/");
    name_offset = strlen(full_name);

    pDir = opendir(dir_name);
    if (pDir != NULL) {
        BOOLEAN dir_is_empty;
        struct dirent * pDirent = readdir(pDir);

        while (pDirent != NULL) {

            if ((strcmp(pDirent->d_name, ".") != 0) &&
                (strcmp(pDirent->d_name, "..") != 0)) {

                /* Build the full path to the directory entry. */
                strcpy(full_name + name_offset, pDirent->d_name);

                if (pDirent->d_type == DT_DIR) {
                    prune_dev_tree(full_name, verbose);
                } else {
                    remove_dev_node_if_not_kernel_volume(full_name, verbose);
                }
            }

            pDirent = readdir(pDir);
        }

        /* Check if the directory is empty. */
        dir_is_empty = TRUE;

        rewinddir(pDir);

        pDirent = readdir(pDir);
        while ((pDirent != NULL) && dir_is_empty) {
            if ((strcmp(pDirent->d_name, ".") != 0) &&
                (strcmp(pDirent->d_name, "..") != 0)) {
                dir_is_empty = FALSE;
            }

            pDirent = readdir(pDir);
        }

        closedir(pDir);

        /* Remove the directory if it is empty. */
        if (dir_is_empty) {

            LOG_DEBUG("Removing empty directory %s.\n", dir_name);
            if (verbose >= ERRORS_AND_INFO) {
                printf("Removing empty directory %s.\n", dir_name);
            }
            rc = rmdir(dir_name);

            if (rc != 0) {
                LOG_WARNING("rmdir(%s) failed with error code %d.\n", dir_name, errno);
                if (verbose >= ERRORS_ONLY) {
                    fprintf(stderr, "rmdir(%s) failed with error code %d.\n", dir_name, errno);
                }
            }
        }
    }
}


/*
 * Remove any nodes and directories in the /dev/evms tree that are not
 * currently exported as volumes by the kernel.
 */
static void remove_unused_dev_nodes(evms_verbose_level_t verbose) {

    DIR           * pDir;
    struct dirent * pDirent;
    char            full_name[EVMS_VOLUME_NAME_SIZE + 1];
    uint            name_offset;

    strcpy(full_name, DEV_PATH "/" EVMS_DIR_NAME);
    strcat(full_name, "/");
    name_offset = strlen(full_name);

    pDir = opendir(DEV_PATH "/" EVMS_DIR_NAME);
    if (pDir != NULL) {

        pDirent = readdir(pDir);
        while (pDirent != NULL) {

            if ((strcmp(pDirent->d_name, ".") != 0) &&
                (strcmp(pDirent->d_name, "..") != 0) &&
                (strcmp(pDirent->d_name, EVMS_DEV_NAME) != 0)) {

                /* Build the full path to the directory entry. */
                strcpy(full_name + name_offset, pDirent->d_name);

                if (pDirent->d_type == DT_DIR) {
                    prune_dev_tree(full_name, verbose);
                } else {
                    remove_dev_node_if_not_kernel_volume(full_name, verbose);
                }
            }

            pDirent = readdir(pDir);
        }

        closedir(pDir);
    }
}


/*
 * Check if devfs is installed.
 */
BOOLEAN is_devfs_installed() {

    int rc = 0;
    struct stat statbuf;

    LOG_PROC_ENTRY();

    rc = stat("/dev/.devfsd", &statbuf);

    LOG_PROC_EXIT_BOOLEAN((rc == 0));
    return(rc == 0);
}


/*
 * Update the /dev/evms tree so that its nodes reflect the volumes that
 * are exported by the EVMS kernel.  This command will remove nodes and
 * directories that are no longer exported by the EVMS kernel.  It will
 * create/update the nodes in the tree so that there is a node with the
 * correct minor number for each volume exported by the EVMS kernel.
 * This API can be run without calling evms_open_engine() first.
 * The verbose_level says how much information to display.
 */
int evms_update_evms_dev_tree(evms_verbose_level_t verbose) {

    int rc = 0;
    engine_mode_t prev_engine_mode = engine_mode;

    LOG_PROC_ENTRY();

    /*
     * Check for devfs.  If devfs is installed, then the dev tree is always
     * up to date and there is nothing to do.
     */

    if (!is_devfs_installed()) {

        /* Does this process already have the EVMS block device open? */
        if (evms_block_dev_handle == 0) {

            /*
             * This process does not have the EVMS block device opened.
             * Either the process didn't open the Engine at all or it opened
             * it in read-only mode (which closes the block device when it
             * is not in use).  Open the block device.
             */
            open_evms_block_dev();

            if (evms_block_dev_handle >= ERRORS_ONLY) {

                /*
                 * Mark the Engine open for read-only mode.  The Engine will
                 * either be closed or already opened in read only mode.  If the
                 * Engine was opened in readwrite mode we wouldn't be here
                 * because the block device would already be open.  We saved the
                 * previous Engine mode above (closed or read-only).
                 */
                engine_mode = ENGINE_READWRITE;
            }
        }

        if (evms_block_dev_handle > 0) {

            /*
             * Get a list of volumes from the kernel point of view if we don't
             * have one already.
             */
            if (VolumeDataList == NULL) {
                rc = get_kernel_volume_data();
            }

            if (rc == 0) {

                /* Prune out unused dev nodes. */
                if (verbose >= ERRORS_AND_INFO) {
                    printf("Removing unused EVMS device nodes and directories...\n");
                }

                remove_unused_dev_nodes(verbose);

                /*
                 * Make sure there is a dev node for each kernel volume and
                 * that its minor number is correct.
                 */
                if (verbose >= ERRORS_AND_INFO) {
                    printf("Making device nodes for EVMS kernel volumes...\n");
                }

                ForEachItem(VolumeDataList,
                            ensure_dev_node,
                            &verbose,
                            TRUE);
            }

            /*
             * If the Engine mode is read-only that means we opened the EVMS
             * block device.  Close it and restore the Engine mode.
             */
            if (engine_mode == ENGINE_READONLY) {
                close_evms_block_dev();
                engine_mode = prev_engine_mode;
            }

        } else {
            /*
             * We couldn't open the EVMS block device.  evms_block_dev_handle
             * contains the error code.
             */
            rc = evms_block_dev_handle;
        }

    } else {
        LOG_DEBUG("devfs is installed.  I have no work to do.\n");
        if (verbose >= ERRORS_AND_INFO) {
            printf("devfs is installed.  I have no work to do.\n");
        }
    }

    LOG_PROC_EXIT_INT(rc);
    return rc;
}

