/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: discovery.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>

#include "ptables.h"
#include "defsegmgr.h"
#include "discovery.h"
#include "segs.h"
#include "checks.h"
#include "commit.h"
#include "display.h"
#include "os2dlat.h"
#include "sn.h"
#include "embedded.h"
#include "bsd.h"
#include "unixware.h"
#include "solarisX86.h"
#include "seg_geometry.h"



/*
 *  Called to see if the drive has a MS-DOS partitioning scheme. This is
 *  a simple MBR+EBR chain ... and possibly ... OS2 dlat sectors.
 */
static BOOLEAN isa_disk_with_msdos_partitions( LOGICALDISK *ld, Master_Boot_Record  *mbr )
{
    int rc=-1;
    DISK_PRIVATE_DATA *disk_pdata = get_disk_private_data(ld);
    geometry_t  guess;


    LOGENTRY();

    // first off ... look for msdos signature
    if ( has_msdos_signature( mbr )==FALSE) {
        LOG_DEBUG("disk %s has no msdos signature\n", ld->name);
        LOGEXIT();
        return FALSE;
    }


    /*
     * Check for a disk that hasnt been partitioned yet. We will not
     * claim such a disk unless explicitly assigned to the disk at
     * a later time by an ASSIGN task.
     */
    if ( ptable_has_partition_records(ld,mbr) == TRUE ) {

        // Look for chain of valid OS/2 dlat sectors
        if ( isa_os2_partitioned_disk(ld, mbr, 0, 0)== TRUE ) {

            disk_pdata->flags |= DISK_HAS_OS2_DLAT_TABLES;

        }

        // test if the disk is using LBA addressing for the partition records
        if ( disk_uses_lba_addressing(ld) == TRUE ) {

            disk_pdata->flags |= DISK_USES_LBA_ADDRESSING;

        }

        // All that remains is to check the partition tables
        if ( disk_pdata->flags & DISK_USES_LBA_ADDRESSING ) {
            rc = isa_valid_partition_table_chain(ld, mbr, 0, 0, FALSE, 0, TRUE);  // final call cuz not using CHS
        }
        else {
            rc = isa_valid_partition_table_chain(ld, mbr, 0, 0, FALSE, 0, FALSE); // not final cuz we can try alternalte
                                                                                  // geometry for the drive.
        }

        if ( rc == EAGAIN ) {   // if we failed and can try again ...

            rc = seg_geometry_guess(ld, &guess );
            if (rc == 0) {

                // success, so switch to new geometry
                disk_pdata->geometry.cylinders         = guess.cylinders;
                disk_pdata->geometry.sectors_per_track = guess.sectors_per_track;
                disk_pdata->geometry.heads             = guess.heads;

                POPUP_MSG( NULL,
                           NULL,
                           "\nWarning, using an alternate geometry (Cylinders, Heads, Sectors) for drive %s.\n"
                           "\nThe kernel reported drive geometry is: C= %lld H= %d S= %d\n"
                           "\nThe partition records report a geometry of: C= %lld H= %d S= %d\n"
                           "\nUsing the alternate geometry reported by the partition records.\n",
                           ld->name,
                           ld->geometry.cylinders,
                           ld->geometry.heads,
                           ld->geometry.sectors_per_track,
                           disk_pdata->geometry.cylinders,
                           disk_pdata->geometry.heads,
                           disk_pdata->geometry.sectors_per_track );

            }

            else if (rc == EPROTO) {

                // partial success, had to ignore CHS errors
                disk_pdata->geometry.cylinders         = guess.cylinders;
                disk_pdata->geometry.sectors_per_track = guess.sectors_per_track;
                disk_pdata->geometry.heads             = guess.heads;

                rc = 0;
            }


            // last attempt will be to use LBA addressing for the drive
            if (rc==-1) {

                disk_pdata->flags |= DISK_USES_LBA_ADDRESSING;

                rc = isa_valid_partition_table_chain(ld, mbr, 0, 0, FALSE, 0, TRUE);
                if ( rc == 0 ) {

                    disk_pdata->flags |= DISK_HAS_FORCED_LBA_ADDRESSING;

                    POPUP_MSG( NULL,
                               NULL,
                               "\nWarning, forced to use LBA addressing on drive %s due to inconsistant geometry.\nChanges will not be committed on this drive.\n",
                               ld->name);

                }

            }

        }

    }
    else {
        LOG_INFO("disk %s is not partitioned\n",ld->name);

         // Look for chain of valid OS/2 dlat sectors
        if ( isa_os2_partitioned_disk(ld, mbr, 0, 0)== TRUE ) {

            disk_pdata->flags |= DISK_HAS_OS2_DLAT_TABLES;

        }

        rc = 0; // return success and claim this disk that doesnt
                // seem to have any partitions on it yet.
    }


    LOGEXIT();

    if (rc == 0) {
        return TRUE;
    }
    else {
        return FALSE;
    }

}






/*
 *  Called by get_DiskSegments() when an extended partition is encountered.
 *  during disk segment discovery.
 *
 *  Part of walking a drives partition tables. This routine will walk an
 *  EBR chain, hanging DOS logical drives off the evms logical disk.
 *
 *  A DOS logical drives consists of an EBR sector and a logical partition.
 *  The EBR will actually consume the entire track because the partitions
 *  start on track boundaries.
 *
 *  This routine will hang the following segments off the logical disk:
 *
 *  EBR Meta Data Seg - describes the EBR track
 *  Evms Data Seg     - describes a logical partition
 *
 */
static int get_logical_drives(  LOGICALDISK           *ld,                  // drive
                                Extended_Boot_Record  *boot_sector,         // EBR boot sector buffer
                                DISKSEG               *ebr )                // EBR DISKSEG
{
    struct plugin_functions_s  *dft;
    Partition_Record     *part   = NULL;

    DISKSEG              *log_part_seg = NULL;
    SEG_PRIVATE_DATA     *log_part_pdata = NULL;

    SEG_PRIVATE_DATA     *ebr_pdata = (SEG_PRIVATE_DATA *) ebr->private_data;

    DISK_PRIVATE_DATA    *disk_pdata = get_disk_private_data(ld);

    DISKSEG              *new_ebr_seg;
    SEG_PRIVATE_DATA     *new_ebr_seg_pdata;

    Extended_Boot_Record *ebr_buffer;
    lba_t                 ebr_lba;

    int                   i;
    int                   rc;



    LOGENTRY();

    // get disk manager function table
    dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
    if (dft==NULL) {
        LOGEXIT();
        return ENODEV;
    }

    // hang logical partitions first ...
    for (i=0; i<4; i++) {
        part = &boot_sector->Partition_Table[i];

        if (  ( isa_null_partition_record( part )==TRUE ) ||
              ( isa_ebr_partition_record(part) == TRUE ) ) {
            continue;
        }

        log_part_seg = build_diskseg_from_partition_record(ld, part, ebr, i, FALSE );
        if (log_part_seg==NULL) {
            LOGEXIT();
            return ENOMEM;
        }

        log_part_pdata = (SEG_PRIVATE_DATA *) log_part_seg->private_data;

        // increment count of logical drives
        ++disk_pdata->logical_drive_count;

        // find partition number for this logical partition, they
        // are numbered 5-n
        log_part_pdata->part_number = ebr_pdata->ebr_number + 5;

        // hang seg off disk
        if ( insert_diskseg_into_list(ld->parent_objects, log_part_seg) == NULL ) {
            free_disk_segment(log_part_seg);
            LOGEXIT();
            return ENOMEM;
        }

        // if the logical partition is offset from its EBR then shrink the
        // EBR segment to a single sector ... like other partitioning tools.
        if ( (log_part_seg->start - ebr->start) > disk_pdata->geometry.sectors_per_track ){
            ebr->size = 1;
        }

    }


    //   Now follow the EBR chain. Look for an extended partition record
    //   that chains us to another logical drive.
    for (i=0; i<4; i++) {

        part = &boot_sector->Partition_Table[i];

        if ( isa_ebr_partition_record(part) == TRUE ) {
            break;
        }
    }

    // did we find an extended partition record ?
    if ( isa_ebr_partition_record(part) == TRUE )  {

        /*
         * add the starting LBA of this disks extended partitoin to get a
         * disk relative starting lba for the logical drive.
         */
        ebr_lba = DISK_TO_CPU32(START_LBA(part)) + disk_pdata->extd_partition_lba;

        // malloc a buffer and read the Extended Boot Record
        ebr_buffer = (Extended_Boot_Record *) malloc(ld->geometry.bytes_per_sector);
        if (ebr_buffer) {

            rc = dft->read(ld, ebr_lba, 1, (void *) ebr_buffer );
            if ( rc == 0 ) {

                // Look for BUGS in the EBR chain ...
                //
                // (1) Check if this extended partition record points to a null ptable
                //     and nuke the terminal extended partition record, finishing the
                //     logical drive discovery.
                //
                // (2) Check if this extended partition record points to a ptable that
                //     -ONLY- chains to another ebr and nuke the -DO NOTHING- ebr and
                //     continue following the chain.
                //
                if ( ptable_has_partition_records( ld, ebr_buffer ) == TRUE ) {

                    if ( ptable_has_data_partition_record( ebr_buffer ) == TRUE) {

                        new_ebr_seg = build_ebr_disk_segment( ld, part, ebr, ebr_lba, i, FALSE );
                        if (new_ebr_seg == NULL) {
                            LOGEXIT();
                            return ENOMEM;
                        }

                        new_ebr_seg_pdata = (SEG_PRIVATE_DATA *)new_ebr_seg->private_data;
                        new_ebr_seg_pdata->ebr_number = ebr_pdata->ebr_number + 1;

                        if ( insert_diskseg_into_list(ld->parent_objects, new_ebr_seg) == NULL ) {
                            free_disk_segment(new_ebr_seg);
                            LOGEXIT();
                            return ENOMEM;
                        }

                        // follow the EBR chain with the new ebr
                        rc = get_logical_drives(  ld, ebr_buffer, new_ebr_seg );
                    }
                    else {

                        // follow the EBR chain ... loosing the DO_NOTHING ebr
                        rc = get_logical_drives(  ld, ebr_buffer, ebr );
                    }

                }


            }

            free(ebr_buffer);
        }
        else {
            rc = ENOMEM;
        }
    }
    else {
        rc = 0;
    }

    LOGEXITRC();
    return rc;
}





/*
 *  Called by segment discovery code.
 *
 *  After learning that we own the partitioning scheme on the specified
 *  drive, we now need to walk the drive's partition tables and create
 *  DISKSEG structs for each partition record that we find.
 *
 *  This routine will hang the following segments off the logical disk:
 *
 *  MBR MetaData - describes the MBR track
 *  EBR MetaData - describes an EBR track
 *  Data         - could be a primary or logical partition
 *  FreeSpace    - a segment that describes an unallocated region on the disk
 *
 *  This routine is called after examining the disk and determining
 *  that we own responsibility for the partitioning scheme on the disk.
 *  So, we know the partitioning is Valid and need not do very exact
 *  checking on its layout while creating evms segment objects.
 *
 */
static int get_disk_segments(  LOGICALDISK         *ld,                  // drive
                               Master_Boot_Record  *boot_sector )        // master boot sector
{
    struct plugin_functions_s  *dft;
    DISK_PRIVATE_DATA          *disk_pdata    = get_disk_private_data(ld);
    Partition_Record           *part = NULL;
    DISKSEG                    *mbr  = NULL;
    DISKSEG                    *pri_part_seg;
    SEG_PRIVATE_DATA           *ebr_pdata;
    Extended_Boot_Record       *ebr_buffer;
    DISKSEG                    *ebr;
    int                         i,rc;


    LOGENTRY();

    // get disk manager function table
    dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
    if (dft==NULL) {
        LOGEXIT();
        return ENODEV;
    }

    // hang a MBR meta data segment and DLAT off the logical drive
    mbr = build_mbr_disk_segment( ld );
    if (mbr == NULL) {
        LOGEXIT();
        return ENOMEM;
    }
    else {

        if ( insert_diskseg_into_list(ld->parent_objects, mbr) == NULL ) {
            free_disk_segment(mbr);
            LOGEXIT();
            return ENOMEM;
        }

    }

    // create segment objects for primary partitions ...
    for (i=0; i<4; i++) {

        part = &boot_sector->Partition_Table[i];

        if (  ( isa_null_partition_record( part )==TRUE ) ||
              ( isa_ebr_partition_record(part) == TRUE ) ) {
            continue;
        }

        pri_part_seg = build_diskseg_from_partition_record(ld, part, mbr, i, TRUE );
        if (pri_part_seg==NULL) {
            LOGEXIT();
            return ENOMEM;
        }

        // get partition number for this primary
        ((SEG_PRIVATE_DATA *)pri_part_seg->private_data)->part_number = i+1;

        // insert into segment list
        if ( insert_diskseg_into_list(ld->parent_objects, pri_part_seg) == NULL ) {
            free_disk_segment(pri_part_seg);
            LOGEXIT();
            return ENOMEM;
        }

    }

    // look for an extended partition record ...
    for (i=0; i<4; i++) {

        part = &boot_sector->Partition_Table[i];

        if ( isa_ebr_partition_record(part) == TRUE ) {
            break;
        }

    }

    // if we found an extended partition then try to follow the ebr chain ...
    if ( isa_ebr_partition_record(part) == TRUE )  {

        // save extended partition information
        disk_pdata->flags                 |= DISK_HAS_EXTENDED_PARTITION;
        disk_pdata->extd_partition_size    = DISK_TO_CPU32(NR_SECTS(part));
        disk_pdata->extd_partition_lba     = DISK_TO_CPU32(START_LBA(part));
        disk_pdata->extd_partition_end_lba = DISK_TO_CPU32(START_LBA(part)) + DISK_TO_CPU32(NR_SECTS(part)) - 1;
        disk_pdata->extd_partition_sys_ind = SYS_IND(part);

        // malloc a buffer and read the Extended Boot Sector
        ebr_buffer = (Extended_Boot_Record *) malloc(ld->geometry.bytes_per_sector);
        if (ebr_buffer) {

            rc = dft->read( ld, DISK_TO_CPU32(START_LBA(part)), 1,(void *) ebr_buffer );
            if ( rc == 0 ) {

                // look for logical drives in the extended partition ...
                if ( ptable_has_partition_records( ld, ebr_buffer ) == TRUE ) {

                    // build the extended partition segment object
                    ebr = build_ebr_disk_segment( ld, part, mbr, (lba_t) DISK_TO_CPU32(START_LBA(part)), i, FALSE );
                    if (ebr == NULL) {
                        LOG_ERROR("error, unable to build extd partition segment object\n");
                        LOGEXIT();
                        return ENOMEM;
                    }

                    // extended partition is sequence number 0 in ebr chain
                    ebr_pdata = (SEG_PRIVATE_DATA *)ebr->private_data;
                    ebr_pdata->ebr_number = 0;

                    // insert into segment list
                    if ( insert_diskseg_into_list(ld->parent_objects, ebr) == NULL ) {
                        free_disk_segment(ebr);
                        LOGEXIT();
                        return ENOMEM;
                    }

                    // now discover logical drives found in the extended partition
                    rc = get_logical_drives(  ld,                     // disk
                                              ebr_buffer,             // ebr boot sector
                                              ebr );                  // ebr DISKSEG *
                }
            }

            free(ebr_buffer);
        }
        else {
            rc = ENOMEM;
        }
    }
    else {
        rc = 0;
    }


    // create segment objects for embedded partitions ...
    for (i=0; (i<4)&&(rc==0); i++) {

        part = &boot_sector->Partition_Table[i];

        switch ( SYS_IND(part) ) {

            case BSD_PARTITION:
            case NETBSD_PARTITION:
            case OPENBSD_PARTITION:
                rc = do_bsd_partition_discover(ld, part);
                break;
            case UNIXWARE_PARTITION:
                rc = do_unixware_partition_discover(ld, part);
                break;
            case SOLARIS_X86_PARTITION:
                rc = do_solaris_x86_partition_discover(ld, part);
                break;
            default:
                break;
        }

    }

    // look for guid partitioned disk and warn user
    if ( has_guid_partition_table( boot_sector )==TRUE) {

        POPUP_MSG(NULL,NULL,"\nWarning, disk %s seems to have a GUID Partition Table. Making "
                            "any changes to this disk may damage the GPT partitions.",
                            ld->name );
    }


    // If successful, find all the freespace on the disk and then
    // make sure the ebr metadata segments are chained properly.
    if (rc == 0) {

        rc = find_freespace_on_disk( ld );

        if (rc == 0) {
            rc = fixup_EBR_Chain( ld );
        }
    }

    LOGEXITRC();
    return rc;
}

/*
 *  Called when we are aborting discovery.  Called with a dlist
 *  of segment objects we have created, the purpose of this routine
 *  is simply to provide a convenient means of discarding the work
 *  done prior to aborting discovery.  By returning TRUE for any
 *  object we created, the object will be pruned from the dlist.
 */
static BOOLEAN prune_our_objects_from_list( ADDRESS   Object,
                                            TAG       ObjectTag,
                                            uint      ObjectSize,
                                            ADDRESS   ObjectHandle,
                                            ADDRESS   Parameters,
                                            BOOLEAN  *FreeMemory,
                                            uint     *Error )
{
    *FreeMemory = FALSE;           // tells dlist not to free any memory
    *Error      = DLIST_SUCCESS;   // tells dlist we were successful

    // prune any storage object belonging to us
    if ( ((DISKSEG *)Object)->plugin == Seg_My_PluginRecord_Ptr ) {

        free_disk_segment((DISKSEG *) Object);

        return TRUE;               // tells dlist to prune the object
    }

    return FALSE;                  // tells dlist not to prune
}



/*
 *   Called by SEG_Discover() ... once for each item in the input_object dlist
 *
 *   Walks the partition table of the disk and if it is our partitioning scheme we will
 *   create storage object segments for the partition table records.
 */
int SegmentDiscovery( void *       Object,
                      TAG          ObjectTag,
                      u_int32_t    ObjectSize,
                      void *       ObjectHandle,
                      void *       Parameters )
{
    LOGICALDISK                *ld;
    DISK_PRIVATE_DATA          *disk_pdata;
    Master_Boot_Record         *mbr = NULL;
    BOOLEAN                     success=FALSE;
    int                         rc;
    void                       *handle;
    TAG                         tag;
    uint                        segcount;
    dlist_t                     output_objects;
    struct plugin_functions_s  *dft;
    BOOLEAN                     acceptable_object = FALSE;

    LOGENTRY();
    LOG_DEBUG("examining object %s\n", ((storage_object_t *)Object)->name );

    // pickup the parms
    ld               = (LOGICALDISK *) Object;
    output_objects   = ((discovery_parm_block_t *)Parameters)->output_objects;



    // we only work on DISK storage objects so filter out other storage objects that
    // may have been in the dlist.
    //
    // add checks for plugin record and data seg types in order to support
    // msdos partitions on s390.
    //
    if ( ld->object_type == DISK ) {
        acceptable_object = TRUE;
    }
    else if ( ( ld->plugin != Seg_My_PluginRecord_Ptr ) &&
              ( ld->object_type == SEGMENT && ld->data_type == DATA_TYPE )) {
        acceptable_object = TRUE;
    }

    if ( acceptable_object == FALSE ){

        // get this storage objects tag so we can do the InsertObject()
        switch (ld->object_type) {

        case REGION:
            tag = REGION_TAG;
            break;
        case SEGMENT:
            tag = SEGMENT_TAG;
            break;
        case EVMS_OBJECT:
            tag = EVMS_OBJECT_TAG;
            break;
        case CONTAINER:
            tag = CONTAINER_TAG;
            break;
        case VOLUME:
            tag = VOLUME_TAG;
            break;
        default:
            return DLIST_BAD;
        }

        // just copy object over to the output dlist
        rc = InsertObject ( (dlist_t)          output_objects,
                            (int)              sizeof(storage_object_t),
                            (void *)           ld,
                            (TAG)              tag,
                            (void *)           NULL,
                            (Insertion_Modes)  AppendToList,
                            (BOOLEAN)          TRUE,
                            (void **)         &handle );


        LOG_DEBUG("object is not acceptable\n" );
        LOGEXIT();
        return 0;
    }

    // get disk manager function table
    dft = (struct plugin_functions_s *)ld->plugin->functions.plugin;
    if (dft==NULL) {
        POPUP_MSG(NULL, NULL, "\nerror ... no device manager function table for drive %s\n", ld->name);
        LOGEXIT();
        return 0;
    }

    // malloc i/o buffer
    mbr = (Master_Boot_Record *) malloc(ld->geometry.bytes_per_sector);
    if ( mbr == NULL) {
        POPUP_MSG(NULL, NULL, "\nerror, unable to allocate memory for i/o buffer for reading MBR off drive %s\n", ld->name);
        LOGEXIT();
        return 0;
    }

    // read MBR sector off drive
    rc = dft->read( ld, 0, 1, (void *) mbr );
    if ( rc ) {
        POPUP_MSG(NULL, NULL, "\nerror, i/o error reading MBR off drive %s\n", ld->name);
        free(mbr);
        LOGEXIT();
        return 0;
    }
    else {
        // create disk private data area for this drive
        if ( create_disk_private_data(ld) ) {
            POPUP_MSG(NULL, NULL, "\nerror, unable to malloc a sector sized buffer for reading sectors off drive %s\n", ld->name );
            free(mbr);
            LOGEXIT();
            return 0;
        }
        else {
            disk_pdata = get_disk_private_data(ld);
        }
    }


    if ( isa_disk_with_msdos_partitions(ld, mbr ) == TRUE ) {

        // create evms segment storage objects from mbr/ebr chain
        rc = get_disk_segments(ld, mbr );

        if (rc == DLIST_SUCCESS) {

            rc = GetListSize( ld->parent_objects, &segcount );
            if (rc) segcount=0;

            if (segcount > 0) {

                LOG_INFO( "Discovery, Success ...I found partitions and created %d segments for disk: %s\n",
                          segcount,
                          ld->name);

                DisplayDiskSegmentList(ld);

                // copy storage objects to the output dlist
                rc = CopyList( output_objects, ld->parent_objects, AppendToList );

                // mark segment discovery as successful
                if (rc==DLIST_SUCCESS) {

                    // return number of segments found
                    *((discovery_parm_block_t *)Parameters)->object_count += segcount;

                    success = TRUE;
                }

            }

        }

    }
    else {   // errors but if it has an msdos signature we will need to do an assign
             // operation and popup further messages as needed.

        DISKSEG *freespace=NULL;
        DISKSEG *mbrseg=NULL;
        char * choices[] = {"Yes", "No", NULL};
        char * reminder[] = {"Continue",NULL};
        int answer = 0;     // Initialize to Revert

        if ( has_msdos_signature(mbr) == TRUE ) {

            // create freespace on the disk
            rc = find_freespace_on_disk( ld );
            if ( rc == 0) {

                // now get the freespace segment we just created
                freespace = get_first_freespace_seg_in_list( ld->parent_objects );
                if ( freespace ) {

                    // create an MBR storage object for the new disk
                    rc = create_mbr_For_Disk( ld, "no name", FALSE );
                    if (rc == 0) {

                        DisplayDiskSegmentList(ld);

                        // copy storage objects to the output dlist
                        rc = CopyList( output_objects, ld->parent_objects, AppendToList );

                        // mark segment discovery as successful
                        if (rc==DLIST_SUCCESS) {

                            // return number of segments found ... 2 ... mbr seg plus freespace seg
                            *((discovery_parm_block_t *)Parameters)->object_count += 2;


                             POPUP_MSG( &answer,
                                        choices,
                                 "\nErrors were found with the partition information on drive %s.\n\n"
                                 "Due to the errors, the drive will appear as though all the partitions were removed, with just an mbr "
                                 "and freespace segments showing.\n\n"
                                 "You can keep this change by answering YES to the following question, but you will later need to commit "
                                 "this change when you exit from evms. Committing this change will cause any existing partition information "
                                 "to be discarded and an empty partition table created on the drive.\n\n"
                                 "If you choose NO, you should exit evms and correct the problem on the drive.\n\n"
                                 "Question: Would you like to mark the drive dirty so that the partitions can be discarded?\n",
                                 ld->name);

                            if (answer == 0) {
                                SegEngFncs->set_changes_pending();
                            }
                            else {
                                freespace->flags &= ~SOFLAG_DIRTY;
                                mbrseg = get_mbr_from_seglist(ld->parent_objects);
                                if (mbrseg) mbrseg->flags &= ~SOFLAG_DIRTY;

                                POPUP_MSG( &answer,
                                           reminder,
                                           "\n\nDrive %s will not be marked dirty but you must remember not to make any changes to "
                                           "this drive before correcting problems with the partitions.\n\n"
                                           "Choose continue when ready.\n",
                                           ld->name );
                            }

                            success = TRUE;

                        }

                    }

                }

            }

        }

    }

    if (success == FALSE) {

        LOG_INFO("Discovery, Failure ... not my disk or no partitions\n");

        // remove any storage objects we may have created
        PruneList(ld->parent_objects, prune_our_objects_from_list, NULL);

        // place the disk storage object back onto the discovery dlist
        rc = InsertObject ( (dlist_t)          output_objects,
                            (int)              sizeof(storage_object_t),
                            (void *)           ld,
                            (TAG)              DISK_TAG,
                            (void *)           NULL,
                            (Insertion_Modes)  AppendToList,
                            (BOOLEAN)          TRUE,
                            (void **)         &handle );

        // get rid of the disk private data area we built
        delete_disk_private_data(ld);

    }


    free(mbr);
    LOGEXIT();
    return DLIST_SUCCESS;
}
