/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: segoptions.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>

#include "ptables.h"
#include "defsegmgr.h"
#include "segs.h"
#include "checks.h"
#include "discovery.h"
#include "commit.h"
#include "display.h"
#include "os2dlat.h"
#include "sn.h"
#include "segoptions.h"

#define  NEED_SOLARISX86_NAMES
#define  NEED_BSD_NAMES
#define  NEED_UNIXWARE_NAMES
#include "bsd.h"
#include "solarisX86.h"
#include "unixware.h"

static const struct seg_partition_type_info_s partition_types[] =
{
    {"FAT16", 6},
    {"HPFS",  7},
    {"Linux", 0x83},
    {"Linux Swap", 0x82},
    {OTHER_TYPENAME, 0},
    {NULL, 0}                    /* end-of-list */
};


static const struct disk_type_info_s disk_types[] =
{
    {"Linux"},
    {"OS/2"},
    {NULL}                    /* end-of-list */
};



/*
 *  Returns TRUE if the specified Drive Letter is valid letter C-Z
 */
static BOOLEAN  isa_valid_drive_letter( char  *drive_letter )
{

    LOGENTRY();

    if ( drive_letter ) {

       if ( (*(drive_letter) >= 'C') && ( *(drive_letter) <= 'Z') ) {
           LOGEXIT();
           return TRUE;
       }
    }

    LOGEXIT();
    return FALSE;
}

/*
 *  Returns the 1st object found in a DLIST.
 */
static  storage_object_t *  get_first_object_in_list( dlist_t list )
{
    int               rc;
    storage_object_t *obj;
    storage_object_t *object_from_list = NULL;
    uint              size;
    TAG               tag;

    LOGENTRY();

    rc = GoToStartOfList( list );
    if ( rc == DLIST_SUCCESS ) {

        rc = BlindGetObject( list, &size, &tag, NULL, FALSE, (void **)&obj );
        if ( rc == DLIST_SUCCESS ) {

            object_from_list = obj;

        }

    }

    LOGEXIT();
    return  object_from_list;
}


/*
 *  Returns the current object in the specified DLIST.  This call should
 *  be preceeded by the get_first_object_in_list() call or the caller
 *  should position the DLIST current pointer themselves.  Note, this call
 *  will not modify anything in the DLIST, including the CURRENT_ITEM ptr
 *  and so it will not advance in the DLIST.
 */
static  storage_object_t *  get_object_from_list( dlist_t list )
{
    int rc;
    storage_object_t *obj;
    storage_object_t *object_from_list = NULL;
    uint size;
    TAG  tag;

    LOGENTRY();

    rc = BlindGetObject( list, &size, &tag, NULL, FALSE, (void **)&obj );
    if (rc == DLIST_SUCCESS) {
        object_from_list = obj;
    }

    LOGEXIT();
    return  object_from_list;
}



/*
 *  Called to initialize the ASSIGN options in the task context
 *  for the specified LOGICALDISK.
 */
static int initialize_assign_option_descriptors( LOGICALDISK *ld, task_context_t * context)
{
    int rc=EINVAL;


    LOGENTRY();

    if ( ld && context ) {

        // there should be exactly SEG_ASSIGN_OPTION_COUNT options
        if ( context->option_descriptors->count == SEG_ASSIGN_OPTION_COUNT) {

            // Disk Type by name
            context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].flags = 0;
            strcpy (context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].value.s, "Linux");

            // Disk Name
            memset( context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].value.s, 0, DISK_NAME_SIZE );
            context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;

            rc = 0;

        }
        else {
            LOG_ERROR("error, wrong number of assign options.\n");
            rc = EINVAL;
        }

    }


    LOGEXITRC();
    return rc;
}




/*
 *  Called to initialize the CREATE segment options in the task context
 *  for the specified FREESPACE segment and LOGICALDISK.
 */
static int initialize_create_option_descriptors( LOGICALDISK *ld, DISKSEG *freespace, task_context_t * context)
{
    int                 rc=EINVAL;
    lba_t               Start;
    lba_t               End;
    sector_count_t      MaxSize;
    sector_count_t      CylinderSize=0;
    DISK_PRIVATE_DATA  *disk_pdata;
    BOOLEAN             OS2_Disk = FALSE;
    DISKSEG            *mbr;

    LOGENTRY();

    disk_pdata = get_disk_private_data( ld );
    if (disk_pdata) {
        if (disk_pdata->signature == DEFAULT_SEG_MGR_PDATA_SIGNATURE) {

            // we should be able to get the cylinder size if there are no problems with disk geometry
            CylinderSize = get_cylinder_size(ld);
            if (CylinderSize ) {

                // there should be exactly SEG_CREATE_OPTION_COUNT options
                if ( context->option_descriptors->count == SEG_CREATE_OPTION_COUNT) {
                    rc = 0;
                }
                else {
                    LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
                }

            }
            else {
                LOG_ERROR("error, failed to get the cylinder size for disk %s\n", ld->name);
            }

        }
    }

    if (rc == 0) {

        // check if disk has dlats
        if ( disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {
            OS2_Disk = TRUE;
        }

        // determine defaults based on geometry
        Start    = freespace->start;
        End      = rounddown_to_cylinder_boundary( ld, freespace->start + freespace->size ) - 1;
        MaxSize  = End - Start + 1;

        // if MAX_USEABLE_SIZE is at least 1 cylinder in size ... well minus a track ...
        // then things are Ok ... init the default options.  this lets me pickup the
        // first cylinder on the disk that has an mbr in the 1st track. it also lets
        // me pickup freespace at the front of an extd partition when the 1st sector
        // is being used to anchor the ebr chain.
        if ( MaxSize >= CylinderSize-disk_pdata->geometry.sectors_per_track) {

            if (rc == 0) {

                // Segment Size
                if ( MaxSize >= CylinderSize ) {
                    context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].flags = 0;
                    context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->min.ui64       = CylinderSize;
                    context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64       = MaxSize;
                    context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = CylinderSize;
                    context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64 = CylinderSize;
                }
                else {  // cuz we track allign start of partition, the freespace can be less than a cylinder
                    context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].flags = 0;
                    context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->min.ui64       = MaxSize;
                    context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64       = MaxSize;
                    context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = 0;
                    context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64 = MaxSize;
                }

                // Sector Offset to start of Partition
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].flags = EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64 = 0;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->min.ui64       = 0;
                context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64       = MaxSize-CylinderSize;


                // Partition Type by name
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].flags = EVMS_OPTION_FLAGS_NOT_REQUIRED;
                if (OS2_Disk == TRUE) {
                    strcpy (context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].value.s, "FAT16");
                }
                else {
                    strcpy (context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].value.s, "Linux");
                }

                // Partition Type
                context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].flags = EVMS_OPTION_FLAGS_ADVANCED;
                if (OS2_Disk == TRUE) {
                    context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].value.ui8 = 0x06;
                }
                else {
                   context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].value.ui8 = 0x83;
                }

                // Make Partition Bootable ?
                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].flags = 0;
                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.bool = FALSE;

                // Make Primary Partition ?
                mbr = get_mbr_from_seglist(ld->parent_objects);

                if ( disk_has_extended_partition( ld ) == TRUE ) {

                    if ( freespace->start < disk_pdata->extd_partition_lba ) {
                        context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.bool = TRUE;
                    }
                    else if ( seg_is_within_or_adjacant_to_extended_partition( ld, freespace ) == TRUE ) {
                        context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.bool = FALSE;
                    }
                    else {
                        context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.bool = TRUE;
                    }

                }
                else {  // no extd partition yet. check if mbr ptable has room for
                        // another primary partition record. if not ... then we will
                        // create the extd partition.
                    if (get_first_unused_ptable_entry(ld->parent_objects, mbr)==-1) {
                        context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.bool = FALSE;
                    }
                    else {
                        context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.bool = TRUE;
                    }
                }

                context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].flags = 0;


                // Partition Name
                if (OS2_Disk == TRUE) {
                    context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
                    memset( context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].value.s, 0, PARTITION_NAME_SIZE );
                }
                else {
                    context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE ;
                }

                // Volume Name
                if (OS2_Disk == TRUE) {
                    context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE | EVMS_OPTION_FLAGS_NOT_REQUIRED;
                    memset(context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].value.s, 0, VOLUME_NAME_SIZE);
                }
                else {
                    context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE ;
                }

                // OS/2 Drive Letter
                if (OS2_Disk == TRUE) {
                    context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE | EVMS_OPTION_FLAGS_NOT_REQUIRED;
                    memset( context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].value.s, 0, DRIVE_LETTER_SIZE);
                }
                else {
                    strcpy(context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].value.s, " ");
                    context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
                }

                rc = 0;
            }

        }
        else {
            rc = EINVAL;
            LOG_ERROR("error trying to create a segment out of something smaller than a cylinder.\n");
        }

    }


    LOGEXITRC();
    return rc;
}



/*
 *  Called to initialize the EXPAND segment options in the task context
 *  for the specified FREESPACE segment and LOGICALDISK.
 */
static int initialize_expand_option_descriptors( LOGICALDISK *ld, DISKSEG *freespace, task_context_t * context)
{
    int                rc=EINVAL;
    int64_t            MaxSize=0;
    int64_t            MinSize=0;
    sector_count_t     CylinderSize=0;
    DISK_PRIVATE_DATA *disk_pdata;


    LOGENTRY();

    disk_pdata = get_disk_private_data( ld );
    if (disk_pdata) {

        if (disk_pdata->signature == DEFAULT_SEG_MGR_PDATA_SIGNATURE) {

            // we should be able to get the cylinder size if there are no problems with disk geometry
            CylinderSize = get_cylinder_size(ld);
            if (CylinderSize ) {

                // there should be exactly SEG_EXPAND_OPTION_COUNT options
                if ( context->option_descriptors->count == SEG_EXPAND_OPTION_COUNT) {

                    rc = 0;

                }
                else {
                    LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
                }

            }
            else {
                LOG_ERROR("error, failed to get the cylinder size for disk %s\n", ld->name);
            }

        }
    }

    if (rc == 0) {

        // make sure we can actually expand the DATA segment by at least 1 cylinder
        if ( freespace->size >= CylinderSize ) {

LOG_DEBUG("freespace= %s   size= %lld\n", freespace->name, freespace->size );

            // get minimum and maximum sizes for expanding this DATA segment
            MaxSize  = freespace->size - (freespace->size % CylinderSize);
            MinSize  = CylinderSize;

LOG_DEBUG("Max= %lld   Min= %lld\n", MaxSize, MinSize );

            if ( MinSize >= CylinderSize ) {

                // ask engine how much we can expand by
                while (MaxSize > 0) {

                    sector_count_t DeltaSize = MaxSize;

                    rc = SegEngFncs->can_expand_by(context->object, &DeltaSize);
                    if (rc == 0) {
                        break;
                    }
                    else {
                        MaxSize -= CylinderSize;
                    }
                }

                if (MaxSize > 0) {

                    context->option_descriptors->count = SEG_EXPAND_OPTION_COUNT;
LOG_DEBUG("Setting constraints ... Max= %lld   Min= %lld\n", MaxSize, MinSize );
                    context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range->min.ui64       = MinSize;
                    context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range->max.ui64       = MaxSize;
                    context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = CylinderSize;
                    context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].value.ui64 = MinSize;
                    context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].flags = 0;

                    rc = 0;
                }
                else {
                    rc = EINVAL;
                    LOG_ERROR("error, engine said segment could not be expanded\n");
                }

            }
        }
        else {
            rc = EINVAL;
            LOG_ERROR("error trying to expand segment into freespace < 1 cylinder.\n");
        }

    }


    LOGEXITRC();
    return rc;
}




/*
 *  Called to initialize the SHRINK segment options in the task context
 *  for the specified FREESPACE segment and LOGICALDISK.
 */
static int initialize_shrink_option_descriptors( LOGICALDISK *ld, DISKSEG *seg, task_context_t * context)
{
    int                rc=EINVAL;
    sector_count_t     MaxSize=0;
    lba_t              MaxEnd=0;
    lba_t              MinEnd=0;
    sector_count_t     MinSize=0;
    sector_count_t     CylinderSize=0;
    DISK_PRIVATE_DATA *disk_pdata;
    sector_count_t     BiggestDelta=0;
    sector_count_t     SmallestDelta=0;

    LOGENTRY();

    disk_pdata = get_disk_private_data( ld );
    if (disk_pdata) {

        if (disk_pdata->signature == DEFAULT_SEG_MGR_PDATA_SIGNATURE) {

            // we should be able to get the cylinder size if there are no problems with disk geometry
            CylinderSize = get_cylinder_size(ld);
            if (CylinderSize ) {

                // there should be exactly SEG_SHRINK_OPTION_COUNT options
                if ( context->option_descriptors->count == SEG_SHRINK_OPTION_COUNT) {
                    rc = 0;
                }
                else {
                    LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
                }

            }
            else {
                LOG_ERROR("error, failed to get the cylinder size for disk %s\n", ld->name);
            }

        }
    }

    if (rc == 0) {

        // we shrink partition sizes ... keeping cylinder allignment
        CylinderSize = get_cylinder_size( ld );

        // get minimum and maximum sizes for shrinking this DATA segment
        if ( seg->size > CylinderSize ) {

            MaxEnd   = roundup_to_cylinder_boundary( ld, seg->start + (seg->size - CylinderSize) - 63 );
            MaxSize  = MaxEnd - seg->start + 1;

            MinEnd   = roundup_to_cylinder_boundary(ld, seg->start + 63 );
            MinSize  = MinEnd - seg->start + 1;

            BiggestDelta  = seg->size - MinSize;
            SmallestDelta = seg->size - MaxSize;

            // ask engine how much we can shrink by ... when the
            // engine says stop ... check if the engine will allow
            // any shrink at all.
            do {
                sector_count_t DeltaSize = BiggestDelta;

                rc = SegEngFncs->can_shrink_by( seg, &DeltaSize );

                if (rc == 0) {
                    break;   // engine said Ok to shrink by BiggestDelta
                }

                BiggestDelta -= CylinderSize;

            } while ( BiggestDelta >= SmallestDelta ); // continue till engine says Ok or we cant shrink at all

        }
        else {  // cant shrink this segment !
            rc = EINVAL;
        }


        // make sure we can actually shrink the DATA segment
        if ( rc == 0 ) {

            context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range->min.ui64       = SmallestDelta;
            context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range->max.ui64       = BiggestDelta;
            context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = CylinderSize;
            context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].flags = 0;
            context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].value.ui64 = SmallestDelta;

            rc = 0;

        }


    }


    LOGEXITRC();
    return rc;
}


/*
 *  Called by init_task() to allocate the assign option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_assign_option_descriptors( task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();

    if (context) {

        context->option_descriptors->count = SEG_ASSIGN_OPTION_COUNT ;      // must set count


        // Disk Type
        context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
        context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].constraint.list =
             SegEngFncs->engine_alloc(sizeof(value_list_t) + sizeof(value_t) * (sizeof(disk_types)/sizeof(struct disk_type_info_s)));

        if (context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].constraint.list)
        {
            int i=0;

            while (disk_types[i].name)
            {
                SET_STRING_FIELD(context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].constraint.list->value[i].s, disk_types[i].name);
                i++;
            }

            context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].constraint.list->count = i;
        }
        else
        {
            rc = ENOMEM;
            LOGEXITRC();
            return rc;
        }

        context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].constraint_type = EVMS_Collection_List;
        context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].name, SEG_CREATE_OPTION_TYPENAME_NAME );
        context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].size = MAX_TYPENAME_SIZE;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].tip, "Choose between OS/2 and Linux disk types. An OS/2 disk will maintain extra metadata that is used by OS/2.");
        SET_STRING_FIELD( context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].title, "Disk Type");
        context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].type = EVMS_Type_String;
        context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].value.s = SegEngFncs->engine_alloc (MAX_TYPENAME_SIZE+1);

        if (context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].value.s)
            strcpy (context->option_descriptors->option[SEG_ASSIGN_OPTION_TYPENAME_INDEX].value.s, "Linux");
        else
        {
            rc = ENOMEM;
            LOGEXITRC();
            return rc;
        }


         // Disk Name
        context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE ;
        context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].constraint.list = NULL;
        context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].constraint_type = EVMS_Collection_None;
        context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].name, SEG_ASSIGN_OPTION_DISKNAME_NAME );
        context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].size = DISK_NAME_SIZE;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].tip, "OS/2 requires that you choose a name for the disk that is no more than 20 characters.");
        SET_STRING_FIELD( context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].title, "Disk Name" );
        context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].type = EVMS_Type_String;
        context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].value.s = SegEngFncs->engine_alloc(DISK_NAME_SIZE+1);
        if (context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].value.s) {
            memset(context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].value.s, 0, DISK_NAME_SIZE );
            rc = 0;
        }
        else {
            rc = ENOMEM;
        }

     }


    LOGEXITRC();
    return rc;
}



/*
 *  Called by init_task() to allocate the create option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_create_option_descriptors( task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();

    if (context) {

        context->option_descriptors->count = SEG_CREATE_OPTION_COUNT ;      // must set count


        // Segment Size
        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.list = NULL;
        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range = SegEngFncs->engine_alloc( sizeof(value_range_t) );
        if (context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range==NULL) {
            LOGEXIT();
            return ENOMEM;
        }
        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint_type                  = EVMS_Collection_Range;
        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->min.ui64       = 0;
        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64       = 0xffffffff;
        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->increment.ui64 = 1;

        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].name, SEG_CREATE_OPTION_SIZE_NAME );
        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].size = sizeof(u_int64_t);
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].tip,"This option allows you to set the size of the segment that you are creating." );
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].title, "Size" );
        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;
        context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64 = 0;

        // Sector Offset to start of Partition
        context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range = SegEngFncs->engine_alloc( sizeof(value_range_t) );
        if (context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range==NULL) {
            LOGEXIT();
            return ENOMEM;
        }
        context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint_type                  = EVMS_Collection_Range;
        context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->min.ui64       = 0;
        context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64       = 1;
        context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->increment.ui64 = 1;


        context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE | EVMS_OPTION_FLAGS_NO_UNIT_CONVERSION;
        context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].name, SEG_CREATE_OPTION_OFFSET_NAME );
        context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].size = sizeof(u_int64_t);
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].tip,
                          "Use this option if you dont want the segment you are creating to start at the beginning of freespace. "
                          "It allows you to specifiy the number of sectors to skip before creating the segment.");
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].title, "Offset");
        context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].type = EVMS_Type_Unsigned_Int64;
        context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].unit = EVMS_Unit_Sectors;
        context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64 = 0;

        // Partition Type
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].constraint.list =
             SegEngFncs->engine_alloc(sizeof(value_list_t) + sizeof(value_t) * (sizeof(partition_types)/sizeof(struct seg_partition_type_info_s)));

        if (context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].constraint.list)
        {
            int i=0;

            while (partition_types[i].name)
            {
                SET_STRING_FIELD(context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].constraint.list->value[i].s, partition_types[i].name);
                i++;
            }

            context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].constraint.list->count = i;
        }
        else
        {
            rc = ENOMEM;
            LOGEXITRC();
            return rc;
        }

        context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].constraint_type = EVMS_Collection_List;
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].name, SEG_CREATE_OPTION_TYPENAME_NAME );
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].size = MAX_TYPENAME_SIZE;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].tip, "Select from a list of common partition types");
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].title, "Partition Type");
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].type = EVMS_Type_String;
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].value.s = SegEngFncs->engine_alloc (MAX_TYPENAME_SIZE+1);

        if (context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].value.s)
            strcpy (context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].value.s, "Linux");
        else
        {
            rc = ENOMEM;
            LOGEXITRC();
            return rc;
        }

        // Partition Type
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].constraint.list = NULL;
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].constraint_type = EVMS_Collection_None;
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].name, SEG_CREATE_OPTION_TYPE_NAME );
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].size = sizeof(unsigned char);
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].tip, "The type of partition you are creating, e.g. 6 = FAT16");
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].title, "Partition Type Id");
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].type = EVMS_Type_Unsigned_Int8;
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].value.ui8 = 0x83;

        // Make Partition Bootable ?
        context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
        context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].constraint.list = NULL;
        context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].constraint_type = EVMS_Collection_None;
        context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].name, SEG_CREATE_OPTION_BOOTABLE_NAME );
        context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].size = sizeof(u_int32_t);
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].tip,
                          "Choose Yes if the primary partition should be marked active so that you can boot from it.");
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].title, "Bootable");
        context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].type = EVMS_Type_Boolean;
        context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.bool = FALSE;

        // Make Primary Partition ?
        context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
        context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].constraint.list = NULL;
        context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].constraint_type = EVMS_Collection_None;
        context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].name, SEG_CREATE_OPTION_PRIMARY_PARTITION_NAME );
        context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].size = sizeof(u_char);
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].tip,
                          "This option allows you to choose between creating a logical or primary partition. "
                          "Choose Yes if you would like a primary partition created.");
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].title, "Primary Partition" );
        context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].type = EVMS_Type_Boolean;
        context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.bool = FALSE;

        // Partition Name
        context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE ;
        context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].constraint.list = NULL;
        context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].constraint_type = EVMS_Collection_None;
        context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].name, SEG_CREATE_OPTION_PARTITIONNAME_NAME );
        context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].size = PARTITION_NAME_SIZE;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].tip, "OS/2 partition name");
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].title, "Partition Name" );
        context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].type = EVMS_Type_String;
        context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].value.s = SegEngFncs->engine_alloc(PARTITION_NAME_SIZE+1);
        if (context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].value.s) {
            memset( context->option_descriptors->option[SEG_CREATE_OPTION_PARTITIONNAME_INDEX].value.s, 0, PARTITION_NAME_SIZE );
        }
        else {
            rc = ENOMEM;
            LOGEXITRC();
            return rc;
        }

        // Volume Name
        context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
        context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].constraint.list = NULL;
        context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].constraint_type = EVMS_Collection_None;
        context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].name, SEG_CREATE_OPTION_VOLUMENAME_NAME );
        context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].size = VOLUME_NAME_SIZE;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].tip, "Name of OS/2 volume this segment will reside in");
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].title, "Volume Name" );
        context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].type = EVMS_Type_String;
        context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].value.s = SegEngFncs->engine_alloc(VOLUME_NAME_SIZE+1);
        if (context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].value.s) {
            memset( context->option_descriptors->option[SEG_CREATE_OPTION_VOLUMENAME_INDEX].value.s, 0, VOLUME_NAME_SIZE);
        }
        else {
            rc = ENOMEM;
            LOGEXITRC();
            return rc;
        }


        // OS/2 Drive Letter
        context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].constraint_type = EVMS_Collection_None;

        context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].flags = EVMS_OPTION_FLAGS_INACTIVE;
        context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].constraint_type = EVMS_Collection_None;
        context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].help = NULL;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].name, SEG_CREATE_OPTION_DRIVELETTER_NAME );
        context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].size = DRIVE_LETTER_SIZE;
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].tip, "OS/2 drive letter for this new segment");
        SET_STRING_FIELD( context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].title, "Drive Letter" );
        context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].type = EVMS_Type_String;
        context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].unit = EVMS_Unit_None;
        context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].value.s = SegEngFncs->engine_alloc(DRIVE_LETTER_SIZE+1);
        if (context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].value.s) {
            memset( context->option_descriptors->option[SEG_CREATE_OPTION_DRIVELETTER_INDEX].value.s, 0, DRIVE_LETTER_SIZE);
        }
        else {
            rc = ENOMEM;
            LOGEXITRC();
            return rc;
        }

        rc = 0;
    }


    LOGEXITRC();
    return rc;
}

/*
 *  Called by init_task() to allocate the expand option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int  allocate_expand_option_descriptors( task_context_t * context )
{
    int rc = EINVAL;
    storage_object_t *obj;
    LOGICALDISK      *ld=NULL;
    DISKSEG          *freespace=NULL;
    sector_count_t    CylinderSize;


    LOGENTRY();

    context->option_descriptors->count = 0;

    obj = context->object;
    if ( obj ) {

        if (obj->object_type == SEGMENT) {

            if (obj->data_type == DATA_TYPE) {

                ld = get_logical_disk( (DISKSEG *) obj );
                if (ld) {

                    // we increment partition sizes ... 1 cylinder at a time ... due to forced cyl allignment
                    CylinderSize = get_cylinder_size(ld);

                    // DATA segment better have freespace following it.
                    freespace =  get_freespace_following_seg( obj );
                    if (freespace == NULL) {
                        rc = EINVAL;
                        LOGEXITRC();
                        return rc;
                    }

                    // make sure we can actually expand the DATA segment by at least 1 cylinder
                    if ( freespace->size >= CylinderSize ) {

                        context->option_descriptors->count = SEG_EXPAND_OPTION_COUNT;

                        // Expanded segment size delta
                        context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.list = NULL;
                        context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range = SegEngFncs->engine_alloc( sizeof(value_range_t) );
                        if (context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint.range==NULL) {
                            LOGEXIT();
                            return ENOMEM;
                        }
                        context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].constraint_type                  = EVMS_Collection_Range;
                        context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].flags = 0;
                        context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].help = NULL;
                        SET_STRING_FIELD( context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].name, SEG_EXPAND_OPTION_SIZE_NAME );
                        context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].size = sizeof(u_int64_t);
                        SET_STRING_FIELD( context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].tip,
                                          "Use this option to specify how many sectors to add to the segment. "
                                          "Remember that segments grow by cylinder size amounts.");
                        SET_STRING_FIELD( context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].title, "Size");
                        context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
                        context->option_descriptors->option[SEG_EXPAND_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;

                        rc = 0;
                    }

                }

            }

        }

    }

    LOGEXITRC();
    return rc;

}


/*
 *  Called by init_task() to allocate the shrink option descriptor array and
 *  setup the individual option descriptors.  It does not initialize the
 *  descriptors.
 */
static int allocate_shrink_option_descriptors( task_context_t * context )
{
    int                rc = EINVAL;
    storage_object_t  *obj;
    LOGICALDISK       *ld;
    sector_count_t     CylinderSize;
    DISK_PRIVATE_DATA *disk_pdata;

    LOGENTRY();

    context->option_descriptors->count = 0;

    obj = context->object;

    if ( obj ) {

        if (obj->object_type == SEGMENT) {

            if (obj->data_type == DATA_TYPE) {

                ld = get_logical_disk( (DISKSEG *) obj );

                disk_pdata = get_disk_private_data(ld);

                if (disk_pdata) {

                    // we shrink partition sizes ... 1 cylinder at a time ... due to forced cyl allignment
                    CylinderSize = disk_pdata->geometry.sectors_per_track * disk_pdata->geometry.heads;

                    // make sure we can actually shrink the DATA segment while falling
                    // on a cylinder boundary.
                    if ( obj->size > CylinderSize ) {

                        context->option_descriptors->count = SEG_SHRINK_OPTION_COUNT ;

                        // Shrunken segment size
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.list = NULL;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range = SegEngFncs->engine_alloc( sizeof(value_range_t) );
                        if (context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint.range==NULL) {
                            LOGEXIT();
                            return ENOMEM;
                        }
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].constraint_type = EVMS_Collection_Range;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].flags = 0;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].help = NULL;
                        SET_STRING_FIELD( context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].name, SEG_SHRINK_OPTION_SIZE_NAME );
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].size = sizeof(u_int64_t);
                        SET_STRING_FIELD( context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].tip,
                                          "Use this option to specify how many sectors to remove from the segment. "
                                          "Remember that segments shrink by cylinder size amounts.");
                        SET_STRING_FIELD( context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].title, "Size" );
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].type = EVMS_Type_Unsigned_Int64;
                        context->option_descriptors->option[SEG_SHRINK_OPTION_SIZE_INDEX].unit = EVMS_Unit_Sectors;

                        rc = 0;
                    }

                }

            }

        }

    }

    LOGEXITRC();
    return rc;
}



/*
 *  Called to set and validate an option for a CREATE task.
 */
static int set_assign_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
    int rc = EINVAL;


    LOGENTRY();

    *effect = 0;


    switch (index) {

    case SEG_ASSIGN_OPTION_TYPENAME_INDEX:

            if ( strlen(value->s) > 0 ) {

                strcpy( context->option_descriptors->option[index].value.s, value->s);

                if ( strncmp(value->s, "OS/2",4)==0) {
                    context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].flags = 0;
                }
                else {
                    context->option_descriptors->option[SEG_ASSIGN_OPTION_DISKNAME_INDEX].flags=EVMS_OPTION_FLAGS_INACTIVE;
                }
                *effect = EVMS_Effect_Reload_Options;
                rc = 0;

            }
            else {
                rc = EINVAL;
            }
            break;


        case SEG_ASSIGN_OPTION_DISKNAME_INDEX:

            if ( (strlen(value->s) > 0) &&
                 (strlen(value->s) <= DISK_NAME_SIZE) ){

                rc = SegEngFncs->validate_name( value->s);

                if (rc == 0) {
                    strcpy( context->option_descriptors->option[index].value.s, value->s);
                }
            }
            else if ( strlen(value->s) > DISK_NAME_SIZE ) {
                char temp_name[DISK_NAME_SIZE+1];

                // get shorter disk name
                strncpy(temp_name, value->s, DISK_NAME_SIZE );

                // try and validate a shortened name
                if ( SegEngFncs->validate_name( temp_name) == 0) {
                    strcpy( context->option_descriptors->option[index].value.s, temp_name );
                    *effect = EVMS_Effect_Inexact;
                    rc = 0;
                }
                else {
                    rc = EINVAL;
                }
            }
            else {
                rc = EINVAL;
            }
            break;


        default:
            LOG_ERROR("index is unknown or unsupported\n");
            rc = EINVAL;
            break;
    }


    LOGEXITRC();
    return rc;
}

/*
 * Scans partition_types array for a name matching given
 * name in order to return corresponding partition type
 * id number.
 */
u_int8_t get_partition_type_number_from_name (char *name)
{
    int      i=0;
    u_int8_t type=0;

    while (partition_types[i].name)
    {
        if (strcasecmp (partition_types[i].name, name) == 0)
        {
            type = partition_types[i].number;
            break;
        }

        i++;
    }

    return type;
}

/*
 * Scans partition_types array for a number matching given
 * number in order to return corresponding partition type
 * name.
 */
char* get_partition_type_name_from_number (u_int8_t number)
{
    int   i=0;
    char *name=OTHER_TYPENAME;

    while (partition_types[i].name)
    {
        if (partition_types[i].number == number)
        {
            name = partition_types[i].name;
            break;
        }

        i++;
    }

    return name;
}

/*
 *  Called to set and validate an option for a CREATE task.
 */
static int set_create_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
    int                rc = EINVAL;
    storage_object_t  *obj=NULL;
    LOGICALDISK       *ld=NULL;
    DISK_PRIVATE_DATA *disk_pdata=NULL;
    sector_count_t     MaxSize = 0;
    sector_count_t     size = 0;
    lba_t              start_lba = 0;
    lba_t              end_lba = 0;
    lba_t              rounded_end_lba = 0;
    sector_count_t     CylinderSize=0;


    LOGENTRY();

    // init effect of set_option to nothing ...
    *effect = 0;

    obj = get_first_object_in_list( context->selected_objects );
    if (obj) {

        if (obj->data_type == META_DATA_TYPE) {
            rc = ENODEV;
            LOG_ERROR("error, i do not create segs from metadata type segments\n");
            LOGEXITRC();
            return rc;
        }

        // get disk info
        ld = get_logical_disk(obj);
        if (ld == NULL) {
            rc = ENODEV;
            LOG_ERROR("error, no disk info available\n");
            LOGEXITRC();
            return rc;
        }
        else {
            disk_pdata = get_disk_private_data(ld);
            if (disk_pdata == NULL) {
                rc = ENODATA;
                LOG_ERROR("error, no disk private data available\n");
                LOGEXITRC();
                return rc;
            }
        }


        CylinderSize = get_cylinder_size(ld);

        start_lba    = obj->start;
        end_lba      = obj->start + obj->size - 1;
        size         = obj->size;

        rounded_end_lba = rounddown_to_cylinder_boundary( ld, end_lba+1 ) - 1;

        MaxSize = rounded_end_lba - start_lba + 1;

    }
    else {
        rc = ENODEV;
        LOG_ERROR("error, no disk segment found in the selected_objects list\n");
        LOGEXITRC();
        return rc;
    }

    switch (index) {

        case SEG_CREATE_OPTION_SIZE_INDEX:
        {

            sector_count_t segoffset = context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64;

                // SIZE == 0 ... ERROR
                if ( value->ui64 == 0) {
                    *effect = EVMS_Effect_Inexact ;
                     rc = EINVAL;
                }
                // if OFFSET + SIZE < MAX_USEABLE_FREESPACE_SIZE ... then SIZE is OK
                else if ( value->ui64+segoffset <= MaxSize ) {

                    context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64 = MaxSize - value->ui64;
                    *effect = EVMS_Effect_Reload_Options;
                    context->option_descriptors->option[index].value.ui64 = value->ui64;
                    rc = 0;

                }
                // user is trying to create something that runs off end
                // of the useable area of the freespace segment.
                else {

                    if (segoffset > 0) { // OFFSET and SIZE both need to be examined
                                         // to determine how to fix things up.

                        if (segoffset > obj->size) {     // OFFSET is toooo big ... ERROR

                            context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].constraint.range->max.ui64 = MaxSize - value->ui64;
                            context->option_descriptors->option[SEG_CREATE_OPTION_OFFSET_INDEX].value.ui64 = MaxSize - value->ui64;
                            context->option_descriptors->option[index].value.ui64 = value->ui64;

                           *effect = EVMS_Effect_Reload_Options ;
                            rc = 0;

                        }
                        else {   // OFFSET is OK ... perhaps big ... so reduce SIZE
                                 // and let the user figure things out.

                            sector_count_t newsize = ((obj->size - segoffset)/CylinderSize)*CylinderSize;

                            context->option_descriptors->option[index].constraint.range->max.ui64 = newsize;
                            context->option_descriptors->option[index].value.ui64 = newsize;
                            *effect = EVMS_Effect_Reload_Options ;
                            rc = 0;
                        }

                    }
                    else {  // OFFSET == 0 ... so SIZE is the only option we need
                            // to adjust downward for user.
                        context->option_descriptors->option[index].value.ui64 = MaxSize;
                        *effect = EVMS_Effect_Inexact;
                        rc = 0;
                    }

                }


            }

            break;

        case SEG_CREATE_OPTION_OFFSET_INDEX:

            {
             sector_count_t segsize = context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].value.ui64;

                // if OFFSET + SIZE < MAX_USEABLE_FREESPACE_SIZE ... then OFFSET is OK
                if ( value->ui64+segsize <= MaxSize ) {

                    context->option_descriptors->option[SEG_CREATE_OPTION_SIZE_INDEX].constraint.range->max.ui64 = MaxSize - value->ui64;
                    context->option_descriptors->option[index].value.ui64 = value->ui64;
                    *effect = EVMS_Effect_Reload_Options ;
                    rc = 0;
                }
                else {

                    context->option_descriptors->option[index].constraint.range->max.ui64 = MaxSize - segsize;
                    context->option_descriptors->option[index].value.ui64 = MaxSize - segsize;
                    *effect = EVMS_Effect_Reload_Options  ;
                    rc = 0;
                }

            }

            break;

        case SEG_CREATE_OPTION_TYPENAME_INDEX:
            if (strcasecmp (value->s, OTHER_TYPENAME) == 0)
                rc = 0;
            else
            {
                u_int8_t type;

                type = get_partition_type_number_from_name (value->s);

                if (type)
                {
                    context->option_descriptors->option[SEG_CREATE_OPTION_TYPE_INDEX].value.ui8 = type;
                    *effect = EVMS_Effect_Reload_Options;
                    rc = 0;
                }
            }

            if (rc == 0)
                strncpy (context->option_descriptors->option[index].value.s, value->s, MAX_TYPENAME_SIZE);

            break;

        case SEG_CREATE_OPTION_TYPE_INDEX:
            if (value->uc != 0) {
                context->option_descriptors->option[index].value.ui8 = value->ui8;
                rc = 0;
            }
            else{
                if (disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES ) {
                    context->option_descriptors->option[index].value.ui8 = 6;
                }
                else {
                    context->option_descriptors->option[index].value.ui8 = 0x83;
                }
                *effect = EVMS_Effect_Inexact ;
                rc = 0;
            }

            if (rc == 0)
            {
                char *name;

                name = get_partition_type_name_from_number (context->option_descriptors->option[index].value.ui8);
                strncpy (context->option_descriptors->option[SEG_CREATE_OPTION_TYPENAME_INDEX].value.s, name, MAX_TYPENAME_SIZE);
                *effect |= EVMS_Effect_Reload_Options;
            }
            break;

        case SEG_CREATE_OPTION_BOOTABLE_INDEX:
            if ( ((value->uc == TRUE)||(value->uc==FALSE)) &&
                 ( context->option_descriptors->option[SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX].value.bool == TRUE )) {
                context->option_descriptors->option[index].value.bool = value->bool;
                rc = 0;
            }
            else {
                context->option_descriptors->option[index].value.bool = FALSE;
                *effect = EVMS_Effect_Inexact;
                rc = 0;
            }
            break;

        case SEG_CREATE_OPTION_PRIMARY_PARTITION_INDEX:

            if ((value->bool == TRUE)||(value->bool == FALSE)) {

                if ( ( value->bool == FALSE ) &&
                     ( disk_has_extended_partition( ld ) == TRUE ) &&
                     ( seg_is_within_or_adjacant_to_extended_partition( ld, obj ) == FALSE )) {

                    context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.bool = FALSE;
                    context->option_descriptors->option[index].value.bool = FALSE;
                    *effect = EVMS_Effect_Reload_Options ;
                    rc = 0;

                }
                else {

                    // dont let them try to create primary if mbr has no unused records
                    // and dont let them try to create a primary if freespace falls
                    // within the extd partition.
                    if (value->bool==TRUE) {

                        DISKSEG *mbr = get_mbr_from_seglist(ld->parent_objects);

                        if (mbr) {

                            if ( get_first_unused_ptable_entry(ld->parent_objects, mbr) == -1) {

                                context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.bool = FALSE;
                                context->option_descriptors->option[index].value.bool = FALSE;
                                *effect = EVMS_Effect_Reload_Options ;
                                rc = 0;

                            }
                            else {  // ok have room for primary so check if it lays
                                    // within the extd partition.
                                if (disk_has_extended_partition( ld ) == TRUE) {

                                    if ( ( obj->start >= disk_pdata->extd_partition_lba ) &&
                                         ( obj->start <= disk_pdata->extd_partition_end_lba ) ) {
                                        context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.bool = FALSE;
                                        context->option_descriptors->option[index].value.bool = FALSE;
                                        *effect = EVMS_Effect_Reload_Options ;
                                        rc = 0;
                                    }
                                    else {
                                        rc = 0;
                                    }
                                }
                                else {
                                    rc = 0;
                                }

                            }
                        }
                        else {
                            context->option_descriptors->option[index].value.bool = FALSE;
                            *effect = EVMS_Effect_Inexact;
                            rc = 0;
                        }

                    }
                    else {
                        context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.bool = FALSE;
                        *effect = EVMS_Effect_Reload_Options;
                        rc = 0;
                    }

                }

                if (rc==0) {
                    context->option_descriptors->option[index].value.bool = value->bool;
                }

            }
            else {  // invalid option value ... not TRUE or FALSE ... just do
                    // some fixups for recovery attempt.

                DISKSEG *mbr = get_mbr_from_seglist(ld->parent_objects);

                if (mbr) {

                    // not bootable
                    context->option_descriptors->option[SEG_CREATE_OPTION_BOOTABLE_INDEX].value.bool = FALSE;

                    // if it can fit in mbr ptable then make_primary == true
                    if ( get_first_unused_ptable_entry(ld->parent_objects, mbr) == -1) {
                        context->option_descriptors->option[index].value.bool = FALSE;
                    }
                    else {
                        context->option_descriptors->option[index].value.bool = TRUE;
                    }

                    // reload options and good rc
                    *effect = EVMS_Effect_Reload_Options ;
                    rc = 0;

                }
                else {
                    rc = EINVAL;  // no mbr for disk ... ERROR ... fail
                }

            }
            break;

        case SEG_CREATE_OPTION_PARTITIONNAME_INDEX:

            if ( (strlen(value->s) > 0) &&
                 (strlen(value->s) <= PARTITION_NAME_SIZE) ){

                char pname[PARTITION_NAME_SIZE+12];

                strcpy(pname, "os2_seg_");
                strncat(pname, value->s, PARTITION_NAME_SIZE );

                rc = SegEngFncs->validate_name( pname );

                if (rc == 0) {
                    strcpy( context->option_descriptors->option[index].value.s, value->s);
                }

            }
            else if ( strlen(value->s) > PARTITION_NAME_SIZE ) {
                char  temp_name[PARTITION_NAME_SIZE+1];

                // shorten name to allowed length
                strncpy(temp_name, context->option_descriptors->option[index].value.s, PARTITION_NAME_SIZE );

                // try and validate a shortened name
                if ( SegEngFncs->validate_name( temp_name) == 0) {
                    strcpy( context->option_descriptors->option[index].value.s, temp_name );
                    *effect = EVMS_Effect_Inexact;
                    rc = 0;
                }
                else {
                    rc = EINVAL;
                }

            }
            else {
                memset( context->option_descriptors->option[index].value.s, 0, PARTITION_NAME_SIZE );
                rc = EINVAL;
            }
            break;

        case SEG_CREATE_OPTION_VOLUMENAME_INDEX:
            if ( (strlen(value->s) > 0) &&
                 (strlen(value->s) <= VOLUME_NAME_SIZE) ){

                 rc = SegEngFncs->validate_name( value->s);

                 if (rc == 0) {
                     strcpy( context->option_descriptors->option[index].value.s, value->s);
                 }
                 else {
                     memset( context->option_descriptors->option[index].value.s, 0, VOLUME_NAME_SIZE );
                 }
            }
            else  if ( strlen(value->s) > VOLUME_NAME_SIZE ) {
                char  temp_name[VOLUME_NAME_SIZE+1];

                // shorten name to allowed length
                strncpy(temp_name, context->option_descriptors->option[index].value.s, VOLUME_NAME_SIZE );

                // try and validate a shortened name
                if ( SegEngFncs->validate_name( temp_name) == 0) {
                    strcpy( context->option_descriptors->option[index].value.s, temp_name );
                    *effect = EVMS_Effect_Inexact;
                    rc = 0;
                }
                else {
                    memset( context->option_descriptors->option[index].value.s, 0, VOLUME_NAME_SIZE);
                    rc = EINVAL;
                }
            }
            else {
                memset( context->option_descriptors->option[index].value.s, 0, VOLUME_NAME_SIZE);
                rc = 0;
            }
            break;

        case SEG_CREATE_OPTION_DRIVELETTER_INDEX:

            if ( strlen(value->s) == 0 ){
                memset( context->option_descriptors->option[index].value.s, 0, DRIVE_LETTER_SIZE);
                rc = 0;
            }
            else if ( strlen(value->s) > DRIVE_LETTER_SIZE ){
                memset( context->option_descriptors->option[index].value.s, 0, DRIVE_LETTER_SIZE);
                rc = EINVAL;
            }
            else {

                if ( isa_valid_drive_letter(value->s)==TRUE) {
                    strncpy( context->option_descriptors->option[index].value.s, value->s, 1);
                    rc = 0;
                }
                else {
                    memset( context->option_descriptors->option[index].value.s, 0, DRIVE_LETTER_SIZE);
                    rc = EINVAL;
                }

            }
            break;

        default:
            LOG_ERROR("error, index is unknown or unsupported\n");
            rc = EINVAL;
            break;
    }


    LOGEXITRC();
    return rc;
}


/*
 *  Called to set and validate an option for a EXPAND task.
 */
static int set_expand_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
    int               rc=EINVAL;
    storage_object_t *obj=NULL;
    LOGICALDISK      *ld=NULL;
    DISKSEG          *freespace=NULL;
    DISKSEG          *seg=NULL;
    sector_count_t    SectorsPerCylinder;
    lba_t             end_lba = 0;
    sector_count_t    expand_sectors=0;

    LOGENTRY();
    if (context && context->object && context->object->name) LOG_DEBUG("  segment= %s\n", context->object->name);

    if (index == SEG_EXPAND_OPTION_SIZE_INDEX ) {

        obj = get_first_object_in_list( context->selected_objects );
        if ( obj ) {

            if (obj->object_type == SEGMENT) {

                if (obj->data_type == FREE_SPACE_TYPE) {

                    seg = context->object;

                    ld = get_logical_disk( (DISKSEG *) seg );

                    if (ld) {

                        freespace = get_freespace_following_seg( seg );

                        if ( freespace == obj ) {

                            // calc the minimum number of sectors we need for a partition
                            SectorsPerCylinder = get_cylinder_size(ld);

                            // get expand sector count
                            expand_sectors = value->ui64;

LOG_DEBUG("Expand Option Value= %lld (sectors)\n", expand_sectors );

                            // we can only expand if in cylinder size amounts so the freespace
                            // following the expanding segment must be at least 1 cylinder.
                            // and caller should be asking for at least a cylinder
                            if ( ( freespace->size >= SectorsPerCylinder ) &&
                                 ( expand_sectors >= SectorsPerCylinder ) &&
                                 ( expand_sectors <= freespace->size ) ) {

                                // expand in cylinder size amounts
                                expand_sectors = (expand_sectors/SectorsPerCylinder)*SectorsPerCylinder;

                                // do cylinder allignment
                                end_lba = seg->start + seg->size + expand_sectors - 1;
                                if (ends_on_cylinder_boundary(ld, end_lba) == FALSE) {
                                    end_lba = roundup_to_cylinder_boundary( ld, end_lba );
                                }

                                // adjust downwards if too big ...
                                if ( end_lba > (freespace->start + freespace->size - 1) ) {
                                     end_lba = roundup_to_cylinder_boundary(ld, end_lba - SectorsPerCylinder );
                                }

                                // NOW ... test if we can expand the data seg into the freespace area
                                if ( ( end_lba > freespace->start ) &&
                                     ( end_lba <= freespace->start+freespace->size - 1) ) {

                                    // calc actual expand sector count
                                    expand_sectors = end_lba - freespace->start + 1;

                                    if (value->ui64 == expand_sectors) {
                                        *effect = 0;
                                    }
                                    else {
                                        *effect = EVMS_Effect_Inexact;
                                    }

                                    context->option_descriptors->option[index].value.ui64 = expand_sectors;
                                    rc = 0;
                                }
                                else {
                                    LOG_ERROR("error, cannot expand segment ... still passes end of freespace\n");
                                }
                            }
                            else {
                                LOG_ERROR("error, invalid expand sector count\n");
                            }
                        }
                        else {
                            LOG_ERROR("error, selected freespace does not immediately follow data segment\n");
                        }
                    }
                    else {
                        LOG_ERROR("error, no logical drive found for segment\n");
                    }
                }
                else {
                    LOG_ERROR("error, selected freespace is NOT a freespace segment\n");
                }
            }

        }

    }

    LOGEXITRC();
    return rc;
}


/*
 *  Called to set and validate an option for a SHRINK task.
 */
static int set_shrink_option( task_context_t * context,
                              u_int32_t        index,
                              value_t        * value,
                              task_effect_t  * effect )
{
    int               rc=EINVAL;
    storage_object_t *obj=NULL;
    LOGICALDISK      *ld=NULL;
    sector_count_t    SectorsPerCylinder;
    sector_count_t    shrink_sector_count=0;
    lba_t             end_lba=0;


    LOGENTRY();

    if (context && context->object && context->object->name) LOG_DEBUG("  segment= %s\n", context->object->name);

    if (index == SEG_SHRINK_OPTION_SIZE_INDEX ) {

        obj = get_first_object_in_list( context->selected_objects );
        if ( obj ) {

            if (obj->object_type == SEGMENT) {

                if (obj->data_type == DATA_TYPE) {

                    ld = get_logical_disk( (DISKSEG *) obj );

                    if ( ld ) {

                        // get option value
                        shrink_sector_count = value->ui64;

                        // calc the minimum number of sectors we need for a partition
                        SectorsPerCylinder = get_cylinder_size(ld);

                        // we can only shrink in cylinder size amounts so the shrink
                        // sectors must be >= 1 cylinder.
                        if ( ( shrink_sector_count < obj->size ) &&
                             ( shrink_sector_count >= SectorsPerCylinder ) ) {

                            // shrink in cylinder size amounts
                            shrink_sector_count = (shrink_sector_count/SectorsPerCylinder)*SectorsPerCylinder;

                            // do cylinder allignment
                            end_lba = obj->start + obj->size - shrink_sector_count - 1;
                            if (ends_on_cylinder_boundary(ld, end_lba) == FALSE) {
                                end_lba = rounddown_to_cylinder_boundary( ld, end_lba-1 ) - 1;
                            }

                            // adjust upwards if we shrunk too small ...
                            if ( end_lba <= obj->start ) {
                                end_lba = roundup_to_cylinder_boundary(ld, end_lba + 1 );
                            }

                            // final test if we can shrink the segment
                            if (  ( end_lba > obj->start ) &&
                                  ( end_lba < (obj->start+obj->size-1)) ) {

                                shrink_sector_count = obj->size - (end_lba - obj->start + 1);

                                if ( value->ui64 == shrink_sector_count ) {
                                    *effect = 0;
                                }
                                else {
                                    value->ui64 = shrink_sector_count;
                                    *effect = EVMS_Effect_Inexact;
                                }

                                context->option_descriptors->option[index].value.ui64 = shrink_sector_count;
                                rc = 0;
                            }
                            else {
                                LOG_ERROR("error, invalid shrink sector count\n");
                            }
                        }
                        else {
                            LOG_ERROR("error, invalid shrink sector count\n");
                        }
                    }
                    else {
                        LOG_ERROR("error, logical drive is missing for this segment\n");
                    }
                }
                else {
                    LOG_ERROR("error, segment is NOT a data segment\n");
                }

            }
            else {
                LOG_ERROR("error, not a SEGMENT object type\n");
            }

        }

    }

    LOGEXITRC();
    return rc;
}


/*
 *  Get the user selected object from the assign context and init the option descriptors
 *  as appropriate for the object.
 *
 *  Note ... if there is more than 1 object in the source object list we will simply
 *           take the first good object and decline the remainder.  That is the reason for
 *           the BOOLEAN flag found_good_object.
 */
static int set_assign_object( task_context_t * context,
                              dlist_t          declined_objects,
                              task_effect_t  * effect )
{
    int                 rc;
    storage_object_t   *obj;
    void               *handle;
    uint                segment_count;
    int                 found_good_object = FALSE;
    declined_object_t  *declined_object=NULL;


    LOGENTRY();

    rc = GoToStartOfList( context->selected_objects );

    while (rc == DLIST_SUCCESS) {

        obj = get_object_from_list( context->selected_objects );
        if (obj) {

            rc = GetListSize(obj->parent_objects, &segment_count);

            if ( ( found_good_object == FALSE) &&
                 ( obj->object_type  == DISK || obj->object_type == SEGMENT ) &&
                 ( obj->volume       == NULL ) &&
                 ( segment_count     == 0 ) ) {

                found_good_object = TRUE;

                rc = initialize_assign_option_descriptors(obj, context);

                if (rc == DLIST_SUCCESS) {
                    *effect |=  EVMS_Effect_Reload_Options;
                    rc = NextItem(context->selected_objects);
                }

            }
            else {

                declined_object = SegEngFncs->engine_alloc( sizeof(declined_object_t));

                if (declined_object) {

                    declined_object->object = obj;
                    declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                    rc = InsertObject ( declined_objects,
                                        sizeof(declined_object_t),
                                        declined_object,
                                        DECLINED_OBJECT_TAG,
                                        NULL,
                                        AppendToList,
                                        TRUE,
                                        &handle );
                    if (rc == DLIST_SUCCESS) {
                        *effect |=  EVMS_Effect_Reload_Objects;
                        rc = NextItem( context->selected_objects );
                    }
                    else {
                        SegEngFncs->engine_free(declined_object);
                    }

                }
                else {
                    LOG_ERROR("error, unable to malloc a declined object struct\n");
                    rc = ENOMEM;
                }

            }


        }
        else {
            rc = DLIST_END_OF_LIST;
        }

    };


    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY )||( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }


    LOGEXITRC();
    return rc;
}







/*
 *  Get the user selected object from the create context and init the option descriptors
 *  as appropriate for the object.
 *
 *  Note ... if there is more than 1 object in the source object list we will simply
 *           take the first good object and decline the remainder.  That is the reason for
 *           the BOOLEAN flag found_good_object.
 */
static int set_create_object( task_context_t * context,
                              dlist_t          declined_objects,
                              task_effect_t  * effect )
{
    int                 rc;
    storage_object_t   *obj;
    LOGICALDISK        *ld;
    void               *handle;
    int                 found_good_object = FALSE;
    declined_object_t  *declined_object=NULL;


    LOGENTRY();

    rc = GoToStartOfList( context->selected_objects );

    while (rc == DLIST_SUCCESS) {

        obj = get_object_from_list( context->selected_objects );
        if (obj) {

            if ( ( obj->data_type == FREE_SPACE_TYPE ) &&
                 ( obj->plugin    == Seg_My_PluginRecord_Ptr) &&
                 ( found_good_object == FALSE ) ) {

                ld = get_logical_disk(obj);

                if (ld) {

                    found_good_object = TRUE;
                    rc = initialize_create_option_descriptors(ld, obj, context);

                    if (rc == DLIST_SUCCESS) {
                        *effect |=  EVMS_Effect_Reload_Options;
                        rc = NextItem(context->selected_objects);
                    }

                }
                else {
                    rc = ENODEV;
                }

            }
            else {

                declined_object = SegEngFncs->engine_alloc( sizeof(declined_object_t));

                if (declined_object) {

                    declined_object->object = obj;
                    declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                    rc = InsertObject ( declined_objects,
                                        sizeof(declined_object_t),
                                        declined_object,
                                        DECLINED_OBJECT_TAG,
                                        NULL,
                                        AppendToList,
                                        TRUE,
                                        &handle );
                    if (rc == DLIST_SUCCESS) {
                        *effect |=  EVMS_Effect_Reload_Objects;
                        rc = NextItem( context->selected_objects );
                    }
                    else {
                        SegEngFncs->engine_free(declined_object);
                    }

                }
                else {
                    LOG_ERROR("unable to malloc a declined object struct\n");
                    rc = ENOMEM;
                }

            }


        }
        else {
            rc = DLIST_END_OF_LIST;
        }

    };


    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY )||( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }


    LOGEXITRC();
    return rc;
}



/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 *
 * Context->object == expansion point
 * Context->selected_objects->object == freespace segment
 * Context->acceptable_objects->object == freespace segment chosen during inittask time
 *
 */
static int set_expand_object( task_context_t * context,
                              dlist_t          declined_objects,
                              task_effect_t  * effect )
{
    int  rc;
    storage_object_t  *obj;
    storage_object_t  *freespace;
    void              *handle;
    LOGICALDISK       *ld;
    BOOLEAN            found_good_object = FALSE;
    declined_object_t *declined_object=NULL;


    LOGENTRY();

    *effect = 0;

    rc = GoToStartOfList( context->selected_objects );

    while (rc == DLIST_SUCCESS) {

        obj = get_object_from_list( context->selected_objects );
        if (obj) {

            if ( ( obj->data_type == FREE_SPACE_TYPE ) &&
                 ( obj->plugin    == Seg_My_PluginRecord_Ptr) &&
                 ( found_good_object == FALSE ) ) {

                freespace = get_freespace_following_seg( context->object );

                ld = get_logical_disk(obj);

                if (ld && freespace) {

                    found_good_object = TRUE;

                    // is selected object correct?
                    if (freespace == obj) {

                        rc = initialize_expand_option_descriptors(ld, obj, context);

                        if (rc == 0) {
                            *effect |= EVMS_Effect_Reload_Options;
                            rc = NextItem(context->selected_objects);
                        }

                    }
                    else {
                        LOG_ERROR("error, selected object is not freespace that follows the data segment\n");
                        rc = EINVAL;
                    }

                }
                else {
                    rc = ENODEV;
                }

            }
            else {

                declined_object = SegEngFncs->engine_alloc( sizeof(declined_object_t));

                if (declined_object) {

                    declined_object->object = obj;
                    declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                    rc = InsertObject ( declined_objects,
                                        sizeof(declined_object_t),
                                        declined_object,
                                        DECLINED_OBJECT_TAG,
                                        NULL,
                                        AppendToList,
                                        TRUE,
                                        &handle );

                    if (rc == DLIST_SUCCESS) {
                        *effect |=  EVMS_Effect_Reload_Objects;
                        rc = NextItem(context->selected_objects);
                    }
                    else {
                        SegEngFncs->engine_free(declined_object);
                    }

                }
                else {
                    LOG_ERROR("error, unable to malloc a declined object struct\n");
                    rc = ENOMEM;
                }

            }


        }
        else {
            rc = DLIST_END_OF_LIST;
        }

    };


    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY )||( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }


    LOGEXITRC();
    return rc;
}


/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int set_shrink_object( task_context_t * context,
                              dlist_t          declined_objects,
                              task_effect_t  * effect )
{
    int  rc;
    storage_object_t  *obj;
    void              *handle;
    LOGICALDISK       *ld;
    BOOLEAN            found_good_object = FALSE;
    declined_object_t *declined_object=NULL;


    LOGENTRY();

    *effect = 0;

    rc = GoToStartOfList( context->selected_objects );

    while (rc == DLIST_SUCCESS) {

        obj = get_object_from_list( context->selected_objects );
        if (obj) {

            if ( ( obj->data_type == DATA_TYPE ) &&
                 ( obj->plugin    == Seg_My_PluginRecord_Ptr) &&
                 ( found_good_object == FALSE ) ) {

                ld = get_logical_disk(obj);

                if (ld) {

                    found_good_object = TRUE;

                    rc = initialize_shrink_option_descriptors(ld, obj, context);

                    if (rc == DLIST_SUCCESS) {
                        *effect |=  EVMS_Effect_Reload_Options;
                        rc = NextItem(context->selected_objects);
                    }

                }
                else {
                    rc = ENODEV;
                }

            }
            else {

                declined_object = SegEngFncs->engine_alloc( sizeof(declined_object_t));

                if (declined_object) {

                    declined_object->object = obj;
                    declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                    rc = InsertObject ( declined_objects,
                                        sizeof(declined_object_t),
                                        declined_object,
                                        DECLINED_OBJECT_TAG,
                                        NULL,
                                        AppendToList,
                                        TRUE,
                                        &handle );

                    if (rc == DLIST_SUCCESS) {
                        *effect |=  EVMS_Effect_Reload_Objects;
                        rc = NextItem(context->selected_objects);
                    }
                    else {
                        SegEngFncs->engine_free(declined_object);
                    }

                }
                else {
                    LOG_ERROR("unable to malloc a declined object struct\n");
                    rc = ENOMEM;
                }

            }


        }
        else {
            rc = DLIST_END_OF_LIST;
        }

    };


    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY )||( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }


    LOGEXITRC();
    return rc;
}


/*
 * Validate the objects in the selected_objects dlist in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects dlist.
 * Modify the accepatble_objects dlist in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
int SEG_SetObjects( task_context_t * context,
                    dlist_t          declined_objects,
                    task_effect_t  * effect )
{

    int rc = EINVAL;

    LOGENTRY();

    if (context) {

        switch (context->action) {

        case EVMS_Task_Create:

            rc = set_create_object( context, declined_objects, effect );
            break;

        case  EVMS_Task_Assign_Plugin:

            rc = set_assign_object( context, declined_objects, effect );
            break;

        case EVMS_Task_Expand:

            rc = set_expand_object( context, declined_objects, effect );
            break;

        case EVMS_Task_Shrink:

            rc = set_shrink_object( context, declined_objects, effect );
            break;

        default:

            LOG_ERROR("context->action is unknown or unsupported\n");
            break;
        }
    }

    LOGEXITRC();
    return rc;
}



/*
 *  Called from get_acceptable_assign_objects, once for each disk
 *  storage object we got back from the kernel. The purpose is to weed out
 *  any logical disks that have already been claimed by segment managers.
 *  This is so we only lay down our partition scheme on disks that
 *  have unrecognized partition schemes and the user has deliberately
 *  requested the defsegmgr to assign itself to the DISK.
 *
 *
 */
static BOOLEAN prune_acceptable_disks( ADDRESS   Object,
                                       TAG       ObjectTag,
                                       uint      ObjectSize,
                                       ADDRESS   ObjectHandle,
                                       ADDRESS   Parameters,
                                       BOOLEAN  *FreeMemory,
                                       uint     *Error )
{
    storage_object_t  *obj = (storage_object_t *)Object;

    *FreeMemory = FALSE;           // tells dlist not to free any memory
    *Error      = DLIST_SUCCESS;   // tells dlist we were successful


    if (obj) {

        // actually, any object we got from the filter is Ok at this point
        // as long as we dont own it, meaning it is either a DISK object
        // or else a DATA SEGMENT object produced by another seg mgr.
        if ( obj->plugin != Seg_My_PluginRecord_Ptr ) {

            return FALSE;  // tells dlist NOT to prune segment from list of acceptable DISKS

        }

    }

    return TRUE;           // tells dlist to prune this object from acceptable DISK list
}


/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int get_acceptable_assign_objects( task_context_t * context )
{
    int   rc = EINVAL;
    uint  count;

    LOGENTRY();

    rc = GetListSize(context->acceptable_objects, &count);
    if (rc == 0) {

        if (count == 0) {

            rc = SegEngFncs->get_object_list( DISK | SEGMENT,
                                              DATA_TYPE,
                                              NULL,
                                              TOPMOST,
                                              &context->acceptable_objects );

            if (rc == DLIST_SUCCESS) {

                rc = GetListSize(context->acceptable_objects, &count);
                if (rc) {
                    count = 0;
                }

                if (count > 0) {
                    PruneList(context->acceptable_objects, prune_acceptable_disks, NULL);
                }
                else {
                    LOG_DEBUG("error, no storage objects returned by get_object_list call\n");
                    rc = ENODEV;
                }

            }

        }
        else {
            LOG_ERROR("context already has acceptable objects\n");
        }

    }

    LOGEXITRC();
    return rc;
}






/*
 *  Called from get_acceptable_create_objects, once for each freespace
 *  segment we got back from the kernel. The purpose is to weed out
 *  any FREESPACE segment that we cannot create a DATA segment from.
 *  This is so we only present FREESPACE segments that are truly acceptable
 *  for CREATE.
 */
static BOOLEAN prune_small_segs( ADDRESS   Object,
                                 TAG       ObjectTag,
                                 uint      ObjectSize,
                                 ADDRESS   ObjectHandle,
                                 ADDRESS   Parameters,
                                 BOOLEAN  *FreeMemory,
                                 uint     *Error )
{
    LOGICALDISK       *ld;
    DISK_PRIVATE_DATA *disk_pdata;
    DISKSEG           *seg = (DISKSEG *) Object;
    sector_count_t     CylinderSize;
    lba_t              Start;
    lba_t              End;
    sector_count_t     MaxSize;
    DISKSEG           *mbr;

    *FreeMemory = FALSE;           // tells dlist not to free any memory
    *Error      = DLIST_SUCCESS;   // tells dlist we were successful


    if (seg) {

        ld = get_logical_disk(seg);
        if (ld) {

            disk_pdata = get_disk_private_data(ld);
            mbr        = get_mbr_from_seglist(ld->parent_objects);

            CylinderSize = get_cylinder_size(ld);

            if ( (seg->start == disk_pdata->extd_partition_lba+1) ||
                 (seg->start <= disk_pdata->geometry.sectors_per_track) ){
                Start = seg->start;
            }
            else {
                Start = roundup_to_cylinder_boundary( ld, seg->start ) + 1;
            }

            if ( ends_on_cylinder_boundary( ld, seg->start + seg->size - 1 ) == TRUE ) {
                End = seg->start + seg->size - 1;
            }
            else if (seg->size > CylinderSize) {
                End = rounddown_to_cylinder_boundary( ld, seg->start + seg->size - 1 ) - 1;
            }
            else {
                End = Start; // force failure ... cuz we cant round down this seg to cyl bdy
            }

            if (Start < End) {

                MaxSize  = End - Start + 1;

                // if this seg is at least 1 cylinder (minus a track) in size then it is Ok
                // because they can start on track boundaries.
                if ( MaxSize >= CylinderSize - disk_pdata->geometry.sectors_per_track ) {

                    // we may have free space but are unable to create a segment on the
                    // disk due to not being able to create a partition record in the mbr.
                    if ( ( get_first_unused_ptable_entry(ld->parent_objects, mbr)==-1) &&
                         ( disk_has_extended_partition( ld ) == FALSE ) ) {
                        return TRUE;
                    }
                    else if ( ( get_first_unused_ptable_entry(ld->parent_objects, mbr)==-1) &&
                              ( disk_has_extended_partition( ld ) == TRUE ) &&
                              ( seg_is_within_or_adjacant_to_extended_partition( ld, seg ) == FALSE )) {
                        return TRUE;
                    }

                    return FALSE;  // tells dlist NOT to prune segment from list
                }
                else{
                    LOG_DEBUG("max size < cyl size\n");
                }

            }

        }

    }

    return TRUE;           // tells dlist to prune the segment from the list
}




/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int get_acceptable_create_objects( task_context_t * context )
{
    int   rc = EINVAL;
    uint  count;

    LOGENTRY();

    rc = GetListSize(context->acceptable_objects, &count);
    if (rc) {
        count = 0;
    }

    if (count == 0) {

        rc = SegEngFncs->get_object_list( SEGMENT,
                                          FREE_SPACE_TYPE,
                                          Seg_My_PluginRecord_Ptr,
                                          VALID_INPUT_OBJECT,
                                          &context->acceptable_objects );

        if (rc == DLIST_SUCCESS) {

            PruneList(context->acceptable_objects, prune_small_segs, NULL);

        }

    }
    else {
        LOG_ERROR("context already has acceptable objects\n");
    }

    LOGEXITRC();
    return rc;
}


/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int get_acceptable_expand_objects( task_context_t * context )
{
    int   rc = EINVAL;
    uint  count=0;
    storage_object_t *freespace;
    LOGICALDISK      *ld;
    sector_count_t    CylinderSize;
    lba_t             end_lba;
    void             *handle;
    SEG_PRIVATE_DATA *pdata;


    LOGENTRY();

    if ( ( context ) &&
         ( context->acceptable_objects ) &&
         ( context->selected_objects ) &&
         ( context->object ) ) {

        GetListSize(context->acceptable_objects, &count);

        pdata = (SEG_PRIVATE_DATA *)context->object->private_data;

        if ( ( count == 0 ) &&                            // cant start with any
             ( (pdata->flags & SEG_IS_EMBEDDED) == 0 )) {   // cant use embedded partitions

            freespace = get_freespace_following_seg( context->object );
            if ( freespace ) {

                ld = get_logical_disk(context->object);
                if (ld) {

                    CylinderSize = get_cylinder_size(ld);

                    if (freespace->size >= CylinderSize) {

                        end_lba = roundup_to_cylinder_boundary(ld, freespace->start + 1);

                        if ( end_lba <= freespace->start + freespace->size - 1) {

                            rc = InsertObject ( context->acceptable_objects,
                                                sizeof(storage_object_t),
                                                freespace,
                                                SEGMENT_TAG,
                                                NULL,
                                                AppendToList,
                                                TRUE,
                                                &handle );

                        }

                    }

                }

            }
        }
    }


    LOGEXITRC();
    return rc;
}



/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int get_acceptable_shrink_objects( task_context_t * context )
{
    int   rc = EINVAL;
    uint  count;
    void *handle=NULL;


    LOGENTRY();

    if ( (context) &&
         (context->object) &&
         (context->acceptable_objects) ) {

        rc = GetListSize(context->acceptable_objects, &count);
        if (rc == DLIST_SUCCESS) {

            if (count == 0) {

                // only acceptable object is the focus object for the shrink
                // task we are initializing.
                rc = InsertObject ( context->acceptable_objects,
                                    sizeof(storage_object_t),
                                    context->object,
                                    SEGMENT_TAG,
                                    NULL,
                                    AppendToList,
                                    TRUE,
                                    &handle );
            }

        }
        else {
            LOG_ERROR("error, context already has acceptable objects\n");
        }

    }

    LOGEXITRC();
    return rc;
}



/*
 *  Initialize a new task context by allocating the option descriptor
 *  array and by getting acceptable objects, from the engine, for the
 *  task context, i.e. create, shrink, expand, assign ...
 */
int SEG_InitTask(task_context_t * context)
{
    int rc = EINVAL;

    LOGENTRY();

    if (context) {

        switch (context->action) {

        case EVMS_Task_Create:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

            rc = allocate_create_option_descriptors( context );
            if (rc == 0) {
                rc = get_acceptable_create_objects( context );
            }

            break;

        case EVMS_Task_Assign_Plugin:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

            rc = allocate_assign_option_descriptors( context );
            if (rc == 0) {
                rc = get_acceptable_assign_objects( context );
            }

            break;


        case EVMS_Task_Expand:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

            rc = allocate_expand_option_descriptors( context );
            if (rc == 0) {
                rc = get_acceptable_expand_objects( context );
            }
            break;

        case EVMS_Task_Shrink:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

            rc = allocate_shrink_option_descriptors( context );
            if (rc == 0) {
                rc = get_acceptable_shrink_objects( context );
            }
            break;

        default:
            LOG_ERROR("error, context->action is unknown or unsupported\n");
            break;
        }
    }

    LOGEXITRC();
    return rc;
}



/*
 *  Returns count of options for specified task
 */
int SEG_GetOptionCount(task_context_t * task)
{
    int count;

    LOGENTRY();

    switch (task->action) {

    case EVMS_Task_Create:
        count = SEG_CREATE_OPTION_COUNT;
        break;

    case EVMS_Task_Assign_Plugin:
         count = SEG_ASSIGN_OPTION_COUNT;
         break;

    case EVMS_Task_Expand:
        count = SEG_EXPAND_OPTION_COUNT;
        break;

    case EVMS_Task_Shrink:
        count = SEG_SHRINK_OPTION_COUNT;
        break;

    default:
        count = 0;
        break;
    }

    LOGEXIT();
    return count;
}



/*
 *
 */
int SEG_SetOption( task_context_t * context,
                   u_int32_t        index,
                   value_t        * value,
                   task_effect_t  * effect )
{
    int rc=EINVAL;

    LOGENTRY();


    if (context ) {

        switch (context->action) {

        case EVMS_Task_Create:
            rc = set_create_option( context, index, value, effect );
            break;

        case EVMS_Task_Assign_Plugin:
            rc = set_assign_option( context, index, value, effect );
            break;

        case EVMS_Task_Expand:
            rc = set_expand_option( context, index, value, effect );
            break;

        case EVMS_Task_Shrink:
            rc = set_shrink_option( context, index, value, effect );
            break;

        default:
            LOG_ERROR("error, context->action is unknown or unsupported\n");
            break;
        }
    }

    LOGEXITRC();

    return rc;
}





/*
 * Returns segment specific information
 */
int SEG_GetInfo( storage_object_t  * object, char * name, extended_info_array_t * * info)
{
    int rc = EINVAL;
    extended_info_array_t   *Info;
    char                     drive_letter[8];
    char                     disk_name[EVMS_NAME_SIZE];
    char                     volume_name[EVMS_NAME_SIZE];
    char                     partition_name[EVMS_NAME_SIZE];
    LOGICALDISK             *ld;
    DISK_PRIVATE_DATA       *disk_pdata;

    LOGENTRY();

    // a measure of protection ...
    if (info == NULL) {
        LOGEXITRC();
        return rc;
    }

    rc    = ENOMEM;  // init to failed calloc
    *info = NULL;     // init to no info returned

    if ( object->object_type == SEGMENT ) {

        ld         = get_logical_disk( object );
        disk_pdata = get_disk_private_data(ld);

        Info = SegEngFncs->engine_alloc( sizeof(extended_info_array_t) + ( SEG_INFO_COUNT * sizeof(extended_info_t) ) );
        if (Info) {

            Info->count = 3;

            SET_STRING_FIELD( Info->info[SEG_INFO_NAME_INDEX].name, "Name" );
            SET_STRING_FIELD( Info->info[SEG_INFO_NAME_INDEX].title, "Name" );
            SET_STRING_FIELD( Info->info[SEG_INFO_NAME_INDEX].desc, "This is the partition name. It must be unique on the system.");
            Info->info[SEG_INFO_NAME_INDEX].type               = EVMS_Type_String;
            Info->info[SEG_INFO_NAME_INDEX].unit               = EVMS_Unit_None;
            SET_STRING_FIELD( Info->info[SEG_INFO_NAME_INDEX].value.s, object->name );
            Info->info[SEG_INFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[SEG_INFO_NAME_INDEX].group, 0, sizeof(group_info_t));

            SET_STRING_FIELD( Info->info[SEG_INFO_SIZE_INDEX].name, "Size" );
            SET_STRING_FIELD( Info->info[SEG_INFO_SIZE_INDEX].title, "Size" );
            SET_STRING_FIELD( Info->info[SEG_INFO_SIZE_INDEX].desc, "This is the size of the partition in sectors.");
            Info->info[SEG_INFO_SIZE_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[SEG_INFO_SIZE_INDEX].unit               = EVMS_Unit_None;
            Info->info[SEG_INFO_SIZE_INDEX].value.ui64         = object->size;
            Info->info[SEG_INFO_SIZE_INDEX].format             = EVMS_Format_Normal;
            Info->info[SEG_INFO_SIZE_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[SEG_INFO_SIZE_INDEX].group, 0, sizeof(group_info_t));

            SET_STRING_FIELD( Info->info[SEG_INFO_START_INDEX].name, "Start" );
            SET_STRING_FIELD( Info->info[SEG_INFO_START_INDEX].title, "Start LBA" );
            SET_STRING_FIELD( Info->info[SEG_INFO_START_INDEX].desc, "This is the sector offset of the partition on the disk, i.e. the logical block address of the first sector of the partition.");
            Info->info[SEG_INFO_START_INDEX].type               = EVMS_Type_Unsigned_Int64;
            Info->info[SEG_INFO_START_INDEX].unit               = EVMS_Unit_None;
            Info->info[SEG_INFO_START_INDEX].value.ui64         = object->start;
            Info->info[SEG_INFO_START_INDEX].format             = EVMS_Format_Normal;
            Info->info[SEG_INFO_START_INDEX].collection_type    = EVMS_Collection_None;
            memset( &Info->info[SEG_INFO_START_INDEX].group, 0, sizeof(group_info_t));


            // dont show the following info for METADATA and FREESPACE segments
            if ( object->data_type == DATA_TYPE ) {

              // embedded partition info
              if ( ((SEG_PRIVATE_DATA *)object->private_data)->flags & SEG_IS_EMBEDDED ) {

                   char  tag_string[EVMS_VOLUME_NAME_SIZE];
                   SEG_PRIVATE_DATA *pdata = (SEG_PRIVATE_DATA *)object->private_data;

                   SET_STRING_FIELD( Info->info[SEG_INFO_FLAGS_INDEX].name, "Flags" );
                   SET_STRING_FIELD( Info->info[SEG_INFO_FLAGS_INDEX].title, "Flag" );
                   SET_STRING_FIELD( Info->info[SEG_INFO_FLAGS_INDEX].desc, "This is the permission flags field from the partition.");
                   Info->info[SEG_INFO_FLAGS_INDEX].type               = EVMS_Type_Unsigned_Int32;
                   Info->info[SEG_INFO_FLAGS_INDEX].unit               = EVMS_Unit_None;
                   Info->info[SEG_INFO_FLAGS_INDEX].format             = EVMS_Format_Hex;
                   Info->info[SEG_INFO_FLAGS_INDEX].value.uc           = (unsigned char) pdata->permissions;
                   Info->info[SEG_INFO_FLAGS_INDEX].collection_type    = EVMS_Collection_None;
                   memset( &Info->info[SEG_INFO_FLAGS_INDEX].group, 0, sizeof(group_info_t));

                   Info->count += 1;

                   SET_STRING_FIELD( Info->info[SEG_INFO_TYPE_INDEX].name, "Type" );
                   SET_STRING_FIELD( Info->info[SEG_INFO_TYPE_INDEX].title, "Type" );
                   SET_STRING_FIELD( Info->info[SEG_INFO_TYPE_INDEX].desc, "Type of embedded partition.");
                   Info->info[SEG_INFO_TYPE_INDEX].type               = EVMS_Type_String;
                   Info->info[SEG_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
                   Info->info[SEG_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
                   memset( &Info->info[SEG_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));
                   if ( pdata->flags & SEG_IS_SOLARIS_X86_PARTITION ) {
                       SET_STRING_FIELD(Info->info[SEG_INFO_TYPE_INDEX].value.s, "solaris x86 partition");
                   }
                   else if ( pdata->flags & SEG_IS_UNIXWARE_PARTITION ) {
                       SET_STRING_FIELD(Info->info[SEG_INFO_TYPE_INDEX].value.s, "unixware partition");
                   }
                   else if ( pdata->flags & SEG_IS_BSD_PARTITION ) {
                       SET_STRING_FIELD(Info->info[SEG_INFO_TYPE_INDEX].value.s, "bsd partition");
                   }

                   Info->count += 1;

                   SET_STRING_FIELD( Info->info[SEG_INFO_SLICE_INDEX].name, "Slice" );
                   SET_STRING_FIELD( Info->info[SEG_INFO_SLICE_INDEX].title, "Slice" );
                   SET_STRING_FIELD( Info->info[SEG_INFO_SLICE_INDEX].desc, "This is the index of the slice in its partition table.");
                   Info->info[SEG_INFO_SLICE_INDEX].type               = EVMS_Type_Unsigned_Int32;
                   Info->info[SEG_INFO_SLICE_INDEX].unit               = EVMS_Unit_None;
                   Info->info[SEG_INFO_SLICE_INDEX].format             = EVMS_Format_Normal;
                   Info->info[SEG_INFO_SLICE_INDEX].value.uc           = (unsigned char) pdata->ptable_index;
                   Info->info[SEG_INFO_SLICE_INDEX].collection_type    = EVMS_Collection_None;
                   memset( &Info->info[SEG_INFO_SLICE_INDEX].group, 0, sizeof(group_info_t));

                   Info->count += 1;

                   SET_STRING_FIELD( Info->info[SEG_INFO_TAG_INDEX].name, "Tag" );
                   SET_STRING_FIELD( Info->info[SEG_INFO_TAG_INDEX].title, "Tag" );
                   SET_STRING_FIELD( Info->info[SEG_INFO_TAG_INDEX].desc, "This is the tag or file system type field from the partition.");
                   Info->info[SEG_INFO_TAG_INDEX].type               = EVMS_Type_String;
                   Info->info[SEG_INFO_TAG_INDEX].unit               = EVMS_Unit_None;
                   Info->info[SEG_INFO_TAG_INDEX].collection_type    = EVMS_Collection_None;
                   memset( &Info->info[SEG_INFO_TAG_INDEX].group, 0, sizeof(group_info_t));
                   if ( pdata->flags & SEG_IS_SOLARIS_X86_PARTITION ) {

                       if (pdata->tag < SOLARISX86_TAG_NAME_COUNT) {
                           sprintf(tag_string, "%d - %s", pdata->tag, solarisx86_names[pdata->tag] );
                       }
                       else {
                           sprintf(tag_string, "%d - unknown", pdata->tag );
                       }

                   }
                   else if ( pdata->flags & SEG_IS_UNIXWARE_PARTITION ) {

                       if (pdata->tag < UW_TAG_NAME_COUNT) {
                           sprintf(tag_string, "%d - %s", pdata->tag, uw_names[pdata->tag] );
                       }
                       else {
                           sprintf(tag_string, "%d - unknown", pdata->tag );
                       }

                   }
                   else if ( pdata->flags & SEG_IS_BSD_PARTITION ) {

                       if (pdata->tag < BSD_TAG_NAME_COUNT) {
                           sprintf(tag_string, "%d - %s", pdata->tag, bsd_names[pdata->tag] );
                       }
                       else {
                           sprintf(tag_string, "%d - unknown", pdata->tag );
                       }

                   }

                   SET_STRING_FIELD(Info->info[SEG_INFO_TAG_INDEX].value.s, tag_string );

                   Info->count += 1;


              }
              else {  // msdos partition info

                if ( disk_pdata->disk_name ) {
                    strcpy(disk_name, disk_pdata->disk_name );
                }
                else {
                    strcpy(disk_name, "n/a");
                }

                SET_STRING_FIELD( Info->info[SEG_INFO_FLAGS_INDEX].name, "Flag" );
                SET_STRING_FIELD( Info->info[SEG_INFO_FLAGS_INDEX].title, "Primary" );
                SET_STRING_FIELD( Info->info[SEG_INFO_FLAGS_INDEX].desc, "Logical or primary partition.");
                Info->info[SEG_INFO_FLAGS_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_INFO_FLAGS_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_INFO_FLAGS_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_FLAGS_INDEX].group, 0, sizeof(group_info_t));
                if ( ((SEG_PRIVATE_DATA *)object->private_data)->flags & SEG_IS_PRIMARY_PARTITION ) {
                    SET_STRING_FIELD(Info->info[SEG_INFO_FLAGS_INDEX].value.s, "Yes");
                }
                else {
                    SET_STRING_FIELD(Info->info[SEG_INFO_FLAGS_INDEX].value.s, "No");
                }

                Info->count += 1;

                SET_STRING_FIELD( Info->info[SEG_INFO_TYPE_INDEX].name, "Type" );
                SET_STRING_FIELD( Info->info[SEG_INFO_TYPE_INDEX].title, "Type" );
                SET_STRING_FIELD( Info->info[SEG_INFO_TYPE_INDEX].desc, "Type of partition, e.g. 0x83 = Linux EXT2 Partition");
                Info->info[SEG_INFO_TYPE_INDEX].type               = EVMS_Type_Unsigned_Char;
                Info->info[SEG_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_INFO_TYPE_INDEX].format             = EVMS_Format_Hex;
                Info->info[SEG_INFO_TYPE_INDEX].value.uc           = (unsigned char) ((SEG_PRIVATE_DATA *)object->private_data)->sys_id;
                Info->info[SEG_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;

                SET_STRING_FIELD( Info->info[SEG_INFO_BOOT_INDEX].name, "BootInd" );
                SET_STRING_FIELD( Info->info[SEG_INFO_BOOT_INDEX].title, "Flags" );
                SET_STRING_FIELD( Info->info[SEG_INFO_BOOT_INDEX].desc, "This is the flag field from the partition record.");
                Info->info[SEG_INFO_BOOT_INDEX].type               = EVMS_Type_Unsigned_Char;
                Info->info[SEG_INFO_BOOT_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_INFO_BOOT_INDEX].format             = EVMS_Format_Hex;
                Info->info[SEG_INFO_BOOT_INDEX].value.uc           = (unsigned char) ((SEG_PRIVATE_DATA *)object->private_data)->boot_ind;
                Info->info[SEG_INFO_BOOT_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_BOOT_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;

                if (  disk_pdata->flags & DISK_HAS_OS2_DLAT_TABLES )  {

                SET_STRING_FIELD( Info->info[SEG_INFO_DSKNAME_INDEX].name, "DiskName" );
                SET_STRING_FIELD( Info->info[SEG_INFO_DSKNAME_INDEX].title, "Disk Name" );
                SET_STRING_FIELD( Info->info[SEG_INFO_DSKNAME_INDEX].desc, "OS/2 disks are assigned names that are stored on the disk for persistance");
                Info->info[SEG_INFO_DSKNAME_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_INFO_DSKNAME_INDEX].unit               = EVMS_Unit_None;
                SET_STRING_FIELD( Info->info[SEG_INFO_DSKNAME_INDEX].value.s, disk_name );
                Info->info[SEG_INFO_DSKNAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_DSKNAME_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;


                if (((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Drive_Letter != 0x00) {
                    drive_letter[0] = ((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Drive_Letter;
                    drive_letter[1] = '\0';
                }
                else {
                    strcpy(drive_letter, "n/a" );
                }

                if (strlen(((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Volume_Name)) {
                    strcpy(volume_name, ((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Volume_Name );
                }
                else {
                    strcpy(volume_name, "n/a");
                }

                if (strlen(((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Partition_Name)) {
                    strcpy(partition_name, ((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Partition_Name );
                }
                else {
                    strcpy(partition_name, "n/a");
                }


                SET_STRING_FIELD( Info->info[SEG_INFO_PNAME_INDEX].name, "Name" );
                SET_STRING_FIELD( Info->info[SEG_INFO_PNAME_INDEX].title, "OS/2 Name" );
                SET_STRING_FIELD( Info->info[SEG_INFO_PNAME_INDEX].desc, "OS/2 partition name.");
                Info->info[SEG_INFO_PNAME_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_INFO_PNAME_INDEX].unit               = EVMS_Unit_None;
                SET_STRING_FIELD( Info->info[SEG_INFO_PNAME_INDEX].value.s, partition_name );
                Info->info[SEG_INFO_PNAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_PNAME_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;

                SET_STRING_FIELD( Info->info[SEG_INFO_VNAME_INDEX].name, "Volume Name" );
                SET_STRING_FIELD( Info->info[SEG_INFO_VNAME_INDEX].title, "Volume Name" );
                SET_STRING_FIELD( Info->info[SEG_INFO_VNAME_INDEX].desc, "OS/2 assigns volume names to Compatibility volumes and LVM volumes.");
                Info->info[SEG_INFO_VNAME_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_INFO_VNAME_INDEX].unit               = EVMS_Unit_None;
                SET_STRING_FIELD( Info->info[SEG_INFO_VNAME_INDEX].value.s, volume_name );
                Info->info[SEG_INFO_VNAME_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_VNAME_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;

                SET_STRING_FIELD( Info->info[SEG_INFO_DRVLTR_INDEX].name, "Drive Letter" );
                SET_STRING_FIELD( Info->info[SEG_INFO_DRVLTR_INDEX].title, "Drive Letter");
                SET_STRING_FIELD( Info->info[SEG_INFO_DRVLTR_INDEX].desc, "OS/2 volumes are given a drive letter assignment, e.g. C:");
                Info->info[SEG_INFO_DRVLTR_INDEX].type               = EVMS_Type_String;
                Info->info[SEG_INFO_DRVLTR_INDEX].unit               = EVMS_Unit_None;
                SET_STRING_FIELD( Info->info[SEG_INFO_DRVLTR_INDEX].value.s, drive_letter );
                Info->info[SEG_INFO_DRVLTR_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_DRVLTR_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;

                SET_STRING_FIELD( Info->info[SEG_INFO_VSN_INDEX].name, "Volume SN" );
                SET_STRING_FIELD( Info->info[SEG_INFO_VSN_INDEX].title, "Volume SN" );
                SET_STRING_FIELD( Info->info[SEG_INFO_VSN_INDEX].desc, "OS/2 volume serial number");
                Info->info[SEG_INFO_VSN_INDEX].type               = EVMS_Type_Unsigned_Int32;
                Info->info[SEG_INFO_VSN_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_INFO_VSN_INDEX].format             = EVMS_Format_Hex;
                Info->info[SEG_INFO_VSN_INDEX].value.ui32         = ((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Volume_Serial_Number;
                Info->info[SEG_INFO_VSN_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_VSN_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;

                SET_STRING_FIELD( Info->info[SEG_INFO_PSN_INDEX].name, "Partition SN" );
                SET_STRING_FIELD( Info->info[SEG_INFO_PSN_INDEX].title, "Partition SN" );
                SET_STRING_FIELD( Info->info[SEG_INFO_PSN_INDEX].desc, "OS/2 partition serial number");
                Info->info[SEG_INFO_PSN_INDEX].type               = EVMS_Type_Unsigned_Int32;
                Info->info[SEG_INFO_PSN_INDEX].unit               = EVMS_Unit_None;
                Info->info[SEG_INFO_PSN_INDEX].format             = EVMS_Format_Hex;
                Info->info[SEG_INFO_PSN_INDEX].value.ui32         = ((SEG_PRIVATE_DATA *)object->private_data)->dla_entry->Partition_Serial_Number;
                Info->info[SEG_INFO_PSN_INDEX].collection_type    = EVMS_Collection_None;
                memset( &Info->info[SEG_INFO_PSN_INDEX].group, 0, sizeof(group_info_t));

                Info->count += 1;

                } //end of OS2 disk
              } //end of msdos info
            } //end of data partition

            *info = Info;

            rc = 0;
        }
        else {
            LOG_ERROR("unable to malloc memory for extended info array\n");
        }
    }

    LOGEXITRC();
    return rc;
}


/*
 *
 */
int SEG_GetPluginInfo( char * descriptor_name, extended_info_array_t * * info )
{
    int rc = EINVAL;
    extended_info_array_t   *Info;
    char                     version_string[64];
    char                     required_version_string[64];


    LOGENTRY();

    // a measure of protection ...
    if (info == NULL) {
        LOGEXITRC();
        return rc;
    }

    rc    = ENOMEM;  // init to failed calloc
    *info = NULL;     // init to no info returned

    Info = SegEngFncs->engine_alloc( sizeof(extended_info_array_t) + (SEG_PLUGIN_INFO_COUNT*sizeof(extended_info_t))  );
    if (Info) {

        Info->count = SEG_PLUGIN_INFO_COUNT;

        sprintf(version_string, "%d.%d.%d",
                MAJOR_VERSION,
                MINOR_VERSION,
                PATCH_LEVEL );

        sprintf(required_version_string, "%d.%d.%d",
                Seg_My_PluginRecord_Ptr->required_api_version.major,
                Seg_My_PluginRecord_Ptr->required_api_version.minor,
                Seg_My_PluginRecord_Ptr->required_api_version.patchlevel );

        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].name, "ShortName" );
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].title, "Short Name" );
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].desc, "A short name given to this plugin.");
        Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].type               = EVMS_Type_String;
        Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].value.s, Seg_My_PluginRecord_Ptr->short_name );
        Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[SEG_PLUGIN_INFO_SNAME_INDEX].group, 0, sizeof(group_info_t));

        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].name, "LongName" );
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].title, "Long Name" );
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].desc, "A longer and more descriptive name for this plugin.");
        Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].type               = EVMS_Type_String;
        Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].value.s, Seg_My_PluginRecord_Ptr->long_name );
        Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[SEG_PLUGIN_INFO_LNAME_INDEX].group, 0, sizeof(group_info_t));

        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].name, "Type" );
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].title, "Plugin Type" );
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].desc, "There are various types of plugins; each responsible for some kind of storage object.");
        Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].type               = EVMS_Type_String;
        Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].value.s, "Partition Manager" );
        Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[SEG_PLUGIN_INFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].name, "Version" );
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].title, "Plugin Version" );
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].desc, "This is the version number of the plugin.");
        Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].type               = EVMS_Type_String;
        Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].value.s, version_string );
        Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[SEG_PLUGIN_INFO_VERSION_INDEX].group, 0, sizeof(group_info_t));

        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_REQVERSION_INDEX].name, "Required Version" );
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_REQVERSION_INDEX].title, "Required Version" );
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_REQVERSION_INDEX].desc, "This is the version of the engine that the plugin requires. It will not run on older versions of the Engine.");
        Info->info[SEG_PLUGIN_INFO_REQVERSION_INDEX].type               = EVMS_Type_String;
        Info->info[SEG_PLUGIN_INFO_REQVERSION_INDEX].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[SEG_PLUGIN_INFO_REQVERSION_INDEX].value.s, required_version_string );
        Info->info[SEG_PLUGIN_INFO_REQVERSION_INDEX].collection_type    = EVMS_Collection_None;
        memset( &Info->info[SEG_PLUGIN_INFO_REQVERSION_INDEX].group, 0, sizeof(group_info_t));

        *info = Info;

        rc = 0;
    }


    LOGEXITRC();
    return rc;
}
