/*
 *
 *   Copyright (c) International Business Machines  Corp., 2000
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: ormoptions.c
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>
#include <linux/evms/evms_user.h>
#include <linux/genhd.h>

#include "os2regmgr.h"
#include "orm_options.h"




/*
 *  Returns the current object in the specified DLIST.  This call should
 *  be preceeded by the get_first_object_in_list() call or the caller
 *  should position the DLIST current pointer themselves.  Note, this call
 *  will not modify anything in the DLIST, including the CURRENT_ITEM ptr
 *  and so it will not advance in the DLIST.
 */
static  storage_object_t *  get_object_from_list( dlist_t list )
{
    int rc;
    storage_object_t *obj;
    storage_object_t *object_from_list = NULL;
    uint size;
    TAG  tag;

    LOGENTRY();

    rc = BlindGetObject( list, &size, &tag, NULL, FALSE, (void **)&obj );
    if (rc == DLIST_SUCCESS) {
        object_from_list = obj;
    }

    LOGEXIT();
    return  object_from_list;
}



/*
 *  Returns count of options for specified task
 */
int ORM_GetOptionCount(task_context_t * task)
{
    int count;

    LOGENTRY();

    switch (task->action) {

    case EVMS_Task_Create:
        count = ORM_CREATE_OPTION_COUNT;
        break;

    case EVMS_Task_Expand:
        count = ORM_EXPAND_OPTION_COUNT;
        break;

    case EVMS_Task_Shrink:
        count = ORM_SHRINK_OPTION_COUNT;
        break;

    default:
        count = 0;
        break;
    }

    LOGEXIT();
    return count;
}




static int allocate_create_option_descriptors( task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();

    //
    //  If things look Ok it means we have a default size for the create operation and now
    //  simply need to initialize options.
    //
    if ( context ) {

        context->option_descriptors->count = ORM_CREATE_OPTION_COUNT ;      // must set count

    }

    rc = 0;
    LOGEXITRC();
    return rc;
}

/*
static int initialize_create_option_descriptors( storage_object_t *obj, task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();

    // there should be exactly ORM_CREATE_OPTION_COUNT options
    if ( context->option_descriptors->count != ORM_CREATE_OPTION_COUNT) {
        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
        LOGEXITRC();
        return rc;
    }

    rc = 0;
    LOGEXITRC();
    return rc;
}
*/


static int allocate_expand_option_descriptors( task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();

    //
    //  If things look Ok it means we have a default size for the create operation and now
    //  simply need to initialize options.
    //
    if ( context ) {

        context->option_descriptors->count = ORM_EXPAND_OPTION_COUNT ;      // must set count

    }

    rc = 0;
    LOGEXITRC();
    return rc;
}

/*
static int initialize_expand_option_descriptors( storage_object_t *obj, task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();

    // there should be exactly ORM_EXPAND_OPTION_COUNT options
    if ( context->option_descriptors->count != ORM_EXPAND_OPTION_COUNT) {
        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
        LOGEXITRC();
        return rc;
    }

    rc = 0;
    LOGEXITRC();
    return rc;
}
*/


static int allocate_shrink_option_descriptors( task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();

    //
    //  If things look Ok it means we have a default size for the create operation and now
    //  simply need to initialize options.
    //
    if ( context ) {

        context->option_descriptors->count = ORM_SHRINK_OPTION_COUNT ;      // must set count

    }

    rc = 0;
    LOGEXITRC();
    return rc;
}

/*
static int initialize_shrink_option_descriptors( storage_object_t *obj, task_context_t * context )
{
    int rc = EINVAL;


    LOGENTRY();

    // there should be exactly ORM_SHRINK_OPTION_COUNT options
    if ( context->option_descriptors->count != ORM_SHRINK_OPTION_COUNT) {
        LOG_ERROR("error, wrong number of option descriptors ... count= %d\n", context->option_descriptors->count);
        LOGEXITRC();
        return rc;
    }

    rc = 0;

    LOGEXITRC();
    return rc;
}
*/




/*
 *  Get the user selected object from the create context and init the option descriptors
 *  as appropriate for the object.
 *
 *  Note ... if there is more than 1 object in the source object list we will simply
 *           take the first good object and decline the remainder.  That is the reason for
 *           the BOOLEAN flag found_good_object.
 */
static int set_create_object( task_context_t * context,
                              dlist_t          declined_objects,
                              task_effect_t  * effect )
{
    int                 rc;
    storage_object_t   *obj;
    void               *handle;
    declined_object_t *declined_object=NULL;


    LOGENTRY();

    rc = GoToStartOfList( context->selected_objects );

    while (rc == DLIST_SUCCESS) {

        obj = get_object_from_list( context->selected_objects );

        if (obj) {

            LOG_ERROR("declining an object found in context->selected_objects, object name= %s\n", obj->name);

            declined_object = ORM_EngFncs->engine_alloc( sizeof(declined_object_t));

            if (declined_object) {

                declined_object->object = obj;
                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                rc = InsertObject ( declined_objects,
                                    sizeof(declined_object_t),
                                    declined_object,
                                    DECLINED_OBJECT_TAG,
                                    NULL,
                                    AppendToList,
                                    TRUE,
                                    &handle );
                if (rc == DLIST_SUCCESS) {
                    *effect |=  EVMS_Effect_Reload_Objects;
                    rc = NextItem( context->selected_objects );
                }
                else {
                    ORM_EngFncs->engine_free(declined_object);
                }

            }
            else {
                LOG_ERROR("unable to malloc a declined object struct\n");
                rc = ENOMEM;
            }

        }
        else {
            rc = DLIST_END_OF_LIST;
        }

    };


    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY )||( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }


    LOGEXITRC();
    return rc;
}


/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int set_expand_object( task_context_t * context,
                              dlist_t          declined_objects,
                              task_effect_t  * effect )
{
    int  rc;
    storage_object_t  *obj;
    void              *handle;
    declined_object_t *declined_object=NULL;


    LOGENTRY();


    rc = GoToStartOfList( context->selected_objects );

    while (rc == DLIST_SUCCESS) {

        obj = get_object_from_list( context->selected_objects );
        if (obj) {

            LOG_ERROR("declining an object found in context->selected_objects, object name= %s\n", obj->name);

            declined_object = ORM_EngFncs->engine_alloc( sizeof(declined_object_t));

            if (declined_object) {

                declined_object->object = obj;
                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                rc = InsertObject ( declined_objects,
                                    sizeof(declined_object_t),
                                    declined_object,
                                    DECLINED_OBJECT_TAG,
                                    NULL,
                                    AppendToList,
                                    TRUE,
                                    &handle );

                if (rc == DLIST_SUCCESS) {
                    *effect |=  EVMS_Effect_Reload_Objects;
                    rc = NextItem(context->selected_objects);
                }
                else {
                    ORM_EngFncs->engine_free(declined_object);
                }

            }
            else {
                LOG_ERROR("unable to malloc a declined object struct\n");
                rc = ENOMEM;
            }

        }
        else {
            rc = DLIST_END_OF_LIST;
        }

    };


    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY )||( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }



    LOGEXITRC();
    return rc;
}



/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int set_shrink_object( task_context_t * context,
                              dlist_t          declined_objects,
                              task_effect_t  * effect )
{
    int  rc;
    storage_object_t  *obj;
    void              *handle;
    declined_object_t *declined_object=NULL;


    LOGENTRY();

    *effect = 0;

    rc = GoToStartOfList( context->selected_objects );

    while (rc == DLIST_SUCCESS) {

        obj = get_object_from_list( context->selected_objects );
        if (obj) {

            LOG_ERROR("declining an object found in context->selected_objects, object name= %s\n", obj->name);

            declined_object = ORM_EngFncs->engine_alloc( sizeof(declined_object_t));

            if (declined_object) {

                declined_object->object = obj;
                declined_object->reason = rc; // REMINDER ... need to pickup reason codes the UI understands

                rc = InsertObject ( declined_objects,
                                    sizeof(declined_object_t),
                                    declined_object,
                                    DECLINED_OBJECT_TAG,
                                    NULL,
                                    AppendToList,
                                    TRUE,
                                    &handle );

                if (rc == DLIST_SUCCESS) {
                    *effect |=  EVMS_Effect_Reload_Objects;
                    rc = NextItem(context->selected_objects);
                }
                else {
                    ORM_EngFncs->engine_free(declined_object);
                }

            }
            else {
                LOG_ERROR("unable to malloc a declined object struct\n");
                rc = ENOMEM;
            }
        }
        else {
            rc = DLIST_END_OF_LIST;
        }

    };


    // throw away errors like ... DLIST_END_OF_LIST ... which are not real errors
    if ( ( rc == DLIST_EMPTY )||( rc == DLIST_END_OF_LIST )) {
        rc = DLIST_SUCCESS;
    }


    LOGEXITRC();
    return rc;
}




/*
 * Validate the objects in the selected_objects dlist in the task context.
 * Remove from the selected objects lists any objects which are not
 * acceptable.
 *
 * For unacceptable objects, create a declined_handle_t structure with the
 * reason why it is not acceptable, and add it to the declined_objects dlist.
 * Modify the accepatble_objects dlist in the task context as necessary
 * based on the selected objects and the current settings of the options.
 *
 * Modify any option settings as necessary based on the selected objects.
 * Return the appropriate task_effect_t settings if the object list(s),
 * minimum or maximum objects selected, or option settings have changed.
 */
int ORM_SetObjects( task_context_t * context,
                    dlist_t          declined_objects,
                    task_effect_t  * effect )
{

    int rc = EINVAL;

    LOGENTRY();

    if (context) {

        switch (context->action) {

        case EVMS_Task_Create:

            rc = set_create_object( context, declined_objects, effect );
            break;

        case EVMS_Task_Expand:

            rc = set_expand_object( context, declined_objects, effect );
            break;

        case EVMS_Task_Shrink:

            rc = set_shrink_object( context, declined_objects, effect );
            break;

        default:

            LOG_ERROR("context->action is unknown or unsupported\n");
            break;
        }
    }

    LOGEXITRC();
    return rc;
}




static int SetCreateOption( task_context_t * context,
                            u_int32_t        index,
                            value_t        * value,
                            u_int32_t      * info )
{
    int rc = EINVAL;

    LOGENTRY();


    LOGEXITRC();
    return rc;
}



static int SetExpandOption( task_context_t * context,
                            u_int32_t        index,
                            value_t        * value,
                            u_int32_t      * info )
{
    int rc = EINVAL;

    LOGENTRY();


    LOGEXITRC();
    return rc;
}


static int SetShrinkOption( task_context_t * context,
                            u_int32_t        index,
                            value_t        * value,
                            u_int32_t      * info )
{
    int rc = EINVAL;

    LOGENTRY();


    LOGEXITRC();
    return rc;
}


/*
 *
 */
int ORM_SetOption( task_context_t * context,
                   u_int32_t        index,
                   value_t        * value,
                   u_int32_t      * info )
{
    int rc=EINVAL;

    LOGENTRY();

    if (context == NULL) {

        switch (context->action) {

        case EVMS_Task_Create:
            rc = SetCreateOption( context, index, value, info );
            break;

        case EVMS_Task_Expand:
            rc = SetExpandOption( context, index, value, info );
            break;

        case EVMS_Task_Shrink:
            rc = SetShrinkOption( context, index, value, info );
            break;

        default:
            rc = EINVAL;
            break;
        }
    }

    LOGEXITRC();

    return rc;
}




/*
 *  Called from get_acceptable_expand_objects
 */
static BOOLEAN prune_noncreatable_objects( ADDRESS   Object,
                                           TAG       ObjectTag,
                                           uint      ObjectSize,
                                           ADDRESS   ObjectHandle,
                                           ADDRESS   Parameters,
                                           BOOLEAN  *FreeMemory,
                                           uint     *Error )
{

    *FreeMemory = FALSE;           // tells dlist not to free any memory
    *Error      = DLIST_SUCCESS;   // tells dlist we were successful

    if (Object) {

        return FALSE;

    }

    return TRUE;           // tells dlist to prune the segment from the list
}


/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int get_acceptable_create_objects( task_context_t * context )
{
    int   rc = EINVAL;
    uint  count;

    LOGENTRY();

    rc = GetListSize(context->acceptable_objects, &count);
    if (rc == DLIST_SUCCESS) {

        if (count == 0) {

            rc = ORM_EngFncs->get_object_list( SEGMENT,
                                               FREE_SPACE_TYPE,
                                               NULL,
                                               VALID_INPUT_OBJECT,
                                               &context->acceptable_objects );

            if (rc == DLIST_SUCCESS) {

                PruneList(context->acceptable_objects, prune_noncreatable_objects, NULL);

            }

        }
        else {
            LOG_ERROR("context already has acceptable objects\n");
        }
    }


    LOGEXITRC();
    return rc;
}




/*
 *  Called from get_acceptable_expand_objects
 */
static BOOLEAN prune_unexpandable_objects( ADDRESS   Object,
                                           TAG       ObjectTag,
                                           uint      ObjectSize,
                                           ADDRESS   ObjectHandle,
                                           ADDRESS   Parameters,
                                           BOOLEAN  *FreeMemory,
                                           uint     *Error )
{

    *FreeMemory = FALSE;           // tells dlist not to free any memory
    *Error      = DLIST_SUCCESS;   // tells dlist we were successful

    if (Object) {

        return FALSE;

    }

    return TRUE;           // tells dlist to prune the segment from the list
}

/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int get_acceptable_expand_objects( task_context_t * context )
{
    int   rc = EINVAL;
    uint  count;

    LOGENTRY();

    rc = GetListSize(context->acceptable_objects, &count);
    if (rc) {
        count = 0;
    }

    if (count == 0) {

        rc = ORM_EngFncs->get_object_list( EVMS_OBJECT,
                                           DATA_TYPE,
                                           &ORM_PluginRecord,
                                           VALID_INPUT_OBJECT,
                                           &context->acceptable_objects );

        if (rc == DLIST_SUCCESS) {

            PruneList(context->acceptable_objects, prune_unexpandable_objects, NULL);

        }

    }
    else {
        LOG_ERROR("context already has acceptable objects\n");
    }

    LOGEXITRC();
    return rc;
}



/*
 *  Called from get_acceptable_shrink_objects
 */
static BOOLEAN prune_unshrinkable_objects( ADDRESS   Object,
                                           TAG       ObjectTag,
                                           uint      ObjectSize,
                                           ADDRESS   ObjectHandle,
                                           ADDRESS   Parameters,
                                           BOOLEAN  *FreeMemory,
                                           uint     *Error )
{

    *FreeMemory = FALSE;           // tells dlist not to free any memory
    *Error      = DLIST_SUCCESS;   // tells dlist we were successful

    if (Object) {

        return FALSE;

    }

    return TRUE;           // tells dlist to prune the segment from the list
}


/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
static int get_acceptable_shrink_objects( task_context_t * context  )
{
    int   rc = EINVAL;
    uint  count;


    LOGENTRY();

    rc = GetListSize(context->acceptable_objects, &count);
    if (rc) {
        count = 0;
    }

    if (count == 0) {

        rc = ORM_EngFncs->get_object_list( SEGMENT,
                                           DATA_TYPE,
                                           &ORM_PluginRecord,
                                           VALID_INPUT_OBJECT,
                                           &context->acceptable_objects );

        if (rc == DLIST_SUCCESS) {

            PruneList(context->acceptable_objects, prune_unshrinkable_objects, NULL);

        }

    }
    else {
        LOG_ERROR("context already has acceptable objects\n");
    }


    LOGEXITRC();
    return rc;
}



/*
 * Validate the objects in the source and target dlists in the context
 * record.  Remove from the selected objects lists any objects which are no
 * longer acceptable.  Return all acceptable objects in the parameter
 * dlists.  Also, for any object which is removed from the selected dlists,
 * or is otherwise not acceptable, create a declined_handle_t struct with
 * reason why not acceptable and add to the declined_objects dlist.
 */
int ORM_InitTask( task_context_t * context )
{
    int rc = EINVAL;

    LOGENTRY();


    if (context) {

        switch (context->action) {

        case EVMS_Task_Create:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

            rc = allocate_create_option_descriptors( context );
            if (rc == 0) {
                rc = get_acceptable_create_objects( context );
            }
            break;

        case EVMS_Task_Expand:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

            rc = allocate_expand_option_descriptors( context );
            if (rc == 0) {
                rc = get_acceptable_expand_objects( context );
            }
            break;

        case EVMS_Task_Shrink:

            context->min_selected_objects = 1;
            context->max_selected_objects = 1;

            rc = allocate_shrink_option_descriptors( context );
            if (rc == 0) {
                rc = get_acceptable_shrink_objects( context );
            }
            break;

        default:
               LOG_ERROR("context->action is unknown or unsupported\n");
               break;
        }
    }


    LOGEXITRC();
    return rc;
}


/*
 * Returns segment specific information
 */
int ORM_GetInfo( storage_object_t  * object, char *name,  extended_info_array_t * * info)
{
    int rc = EINVAL;
    extended_info_array_t   *Info;
    orm_private_data_t      *pdata = (orm_private_data_t *) object->private_data;


    LOGENTRY();

    // a measure of protection ...
    if ((info == NULL)||(pdata->signature != ORM_PDATA_SIGNATURE)) {
        LOGEXITRC();
        return rc;
    }

    rc    = ENOMEM;  // init to failed calloc
    *info = NULL;     // init to no info returned

    if ( object->object_type == EVMS_OBJECT ) {

        Info = ORM_EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( 3 * sizeof(extended_info_t) ) );
        if (Info) {

            Info->count = 2;

            SET_STRING_FIELD( Info->info[0].name, "Name" );
            SET_STRING_FIELD( Info->info[0].title, "Name" );
            SET_STRING_FIELD( Info->info[0].desc, "This is the name given to the storage object. It must be unique on the system.");
            Info->info[0].type               = EVMS_Type_String;
            Info->info[0].unit               = EVMS_Unit_None;
            SET_STRING_FIELD( Info->info[0].value.s, object->name );
            Info->info[0].collection_type    = EVMS_Collection_None;
            memset( &Info->info[0].group, 0, sizeof(group_info_t));

            SET_STRING_FIELD( Info->info[1].name, "Size" );
            SET_STRING_FIELD( Info->info[1].title, "Size" );
            SET_STRING_FIELD( Info->info[1].desc, "This is the size of the storage object after reserving space for metadata.");
            Info->info[1].type               = EVMS_Type_Unsigned_Int64;
            Info->info[1].unit               = EVMS_Unit_None;
            Info->info[1].value.ui64         = object->size;
            Info->info[1].collection_type    = EVMS_Collection_None;
            memset( &Info->info[1].group, 0, sizeof(group_info_t));

            *info = Info;

            rc = 0;
        }
        else {
            LOG_ERROR("unable to malloc memory for extended info array\n");
        }
    }

    LOGEXITRC();
    return rc;
}


/*
 *
 */
int ORM_GetPluginInfo( char * descriptor_name, extended_info_array_t * * info )
{
    int rc = EINVAL;
    extended_info_array_t   *Info;
    char                     version_string[64];
    char                     required_version_string[64];


    LOGENTRY();

    // a measure of protection ...
    if (info == NULL) {
        LOGEXITRC();
        return rc;
    }

    rc    = ENOMEM;  // init to failed calloc
    *info = NULL;     // init to no info returned

    Info = ORM_EngFncs->engine_alloc( sizeof( extended_info_array_t ) + ( 6 * sizeof( extended_info_t )) );
    if ( Info ) {

        Info->count = 5;

        sprintf( version_string, "%d.%d.%d",
                 MAJOR_VERSION,
                 MINOR_VERSION,
                 PATCH_LEVEL );

        sprintf( required_version_string, "%d.%d.%d",
                 ORM_PluginRecord.required_api_version.major,
                 ORM_PluginRecord.required_api_version.minor,
                 ORM_PluginRecord.required_api_version.patchlevel );

        SET_STRING_FIELD( Info->info[0].name, "Short Name" );
        SET_STRING_FIELD( Info->info[0].title, "Short Name" );
        SET_STRING_FIELD( Info->info[0].desc, "A short name given to this plugin." );
        Info->info[0].type               = EVMS_Type_String;
        Info->info[0].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[0].value.s, ORM_PluginRecord.short_name );
        Info->info[0].collection_type    = EVMS_Collection_None;
        memset( &Info->info[0].group, 0, sizeof( group_info_t ));

        SET_STRING_FIELD( Info->info[1].name, "Long Name" );
        SET_STRING_FIELD( Info->info[1].title, "Long Name" );
        SET_STRING_FIELD( Info->info[1].desc, "A long name given to this plugin." );
        Info->info[1].type               = EVMS_Type_String;
        Info->info[1].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[1].value.s, ORM_PluginRecord.long_name );
        Info->info[1].collection_type    = EVMS_Collection_None;
        memset( &Info->info[1].group, 0, sizeof( group_info_t ));

        SET_STRING_FIELD( Info->info[2].name, "Type" );
        SET_STRING_FIELD( Info->info[2].title, "Plugin Type" );
        SET_STRING_FIELD( Info->info[2].desc, "There are various types of plugins; each responsible for some kind of storage object." );
        Info->info[2].type               = EVMS_Type_String;
        Info->info[2].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[2].value.s, "Region Manager" );
        Info->info[2].collection_type    = EVMS_Collection_None;
        memset( &Info->info[2].group, 0, sizeof( group_info_t ));

        SET_STRING_FIELD( Info->info[3].name, "Version" );
        SET_STRING_FIELD( Info->info[3].title, "Plugin Version" );
        SET_STRING_FIELD( Info->info[3].desc, "This is the version number of the plugin." );
        Info->info[3].type               = EVMS_Type_String;
        Info->info[3].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[3].value.s, version_string );
        Info->info[3].collection_type    = EVMS_Collection_None;
        memset( &Info->info[3].group, 0, sizeof( group_info_t ));

        SET_STRING_FIELD( Info->info[4].name, "Required Version" );
        SET_STRING_FIELD( Info->info[4].title, "Required Engine Version" );
        SET_STRING_FIELD( Info->info[4].desc, "This is the version of the engine that the plugin requires. It will not run on older versions of the Engine." );
        Info->info[4].type               = EVMS_Type_String;
        Info->info[4].unit               = EVMS_Unit_None;
        SET_STRING_FIELD( Info->info[4].value.s, required_version_string );
        Info->info[4].collection_type    = EVMS_Collection_None;
        memset( &Info->info[4].group, 0, sizeof( group_info_t ));

        *info = Info;

        rc = 0;
    }


    LOGEXITRC();
    return rc;
}
