/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: plugin_cb.c
 */ 
 
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "widget.h"
#include "task.h"
#include "plugin_cb.h"
#include "thing.h"
#include "logging.h"
#include "help.h"

/*
 *
 *   void on_plugin_selection_button_clicked (GtkButton *, gpointer)
 *
 *   Description:
 *      This routine responds to plugin selection with creating the
 *      task given and creating the acceptable object selection dialog
 *      window to proceed with acceptable parameter and option selections.
 * 
 *   Entry:
 *      button    - address of the GtkButton widget
 *      user_data - contains code corresponding to task to initiate
 *
 *   Exit:
 *      The task context is created with the given plugin and task code.
 *      This window is hidden and another standard selection window is displayed.
 *
 */
void on_plugin_selection_button_clicked (GtkButton *button, gpointer user_data)
{
    GList          *window_list;
    GtkWidget      *clist;
    GtkWidget      *next_window;
    GtkWidget      *selection_window;
    task_action_t   action;
    plugin_handle_t plugin;
    plugin_handle_t prev_plugin;

    /*
     * Hide the selection window and either redisplay the existing task window
     * if it exists and the plugin selection did not change. If the task window
     * does not exist, create it. However, if the plugin selections changed and
     * the task window existed then destroy the current task window and create
     * a new one.
     */

    action           = GPOINTER_TO_UINT (user_data);
    selection_window = gtk_widget_get_toplevel (GTK_WIDGET (button));
    clist            = lookup_widget (GTK_WIDGET (button), "selection_window_clist");
    plugin           = GPOINTER_TO_UINT (get_single_select_current_row_data (GTK_CLIST (clist)));
    window_list      = get_window_list (selection_window);
    next_window      = gtk_object_get_data (GTK_OBJECT (selection_window), "next_window_id");
    prev_plugin      = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (selection_window), "previous_plugin"));

    if (window_list == NULL)
        window_list = g_list_append (window_list, selection_window);

    if (plugin != prev_plugin)
    {
        gint          rc;
        task_handle_t handle;

        rc = evms_create_task (plugin, action, &handle);

        if (rc != SUCCESS)
        {
            log_error ("%s: evms_create_task() returned error code %d.\n", __FUNCTION__, rc);

            display_results_window (rc, NULL, _("Unable to create the task context."), 
                                    NULL, FALSE, selection_window);
        }
        else
        {
            gint   count;
            gchar *next_button_text=NULL;
            
            /*
             * If we already created the next window for selecting
             * the acceptable objects (and maybe even other follow-on 
             * windows like for options), make sure we destroy them
             * all and clean up the list since our plugin selections
             * have apperently changed and so all the following panels
             * need to be recreated.
             */

            if (next_window != NULL)
                destroy_window_list (g_list_find (window_list, next_window));

            /*
             * If we have no options then set the Next button label to
             * indicate the actual task action rather than "Next".
             */
            
            if (evms_get_option_count (handle, &count) == SUCCESS && count == 0)
                next_button_text = get_task_action_string (handle);          
                
            next_window = create_standard_selection_window (_("Select Plugin Acceptable Objects"),
                                                            next_button_text,
                                                            acceptable_objects_help_text,
                                                            on_acceptable_objects_clist_realize,
                                                            on_acceptable_objects_button_clicked,
                                                            on_button_clicked_display_prev_window,
                                                            on_button_clicked_destroy_window_list,
                                                            on_acceptable_objects_clist_select_row,
                                                            on_acceptable_objects_clist_unselect_row,
                                                            GUINT_TO_POINTER (handle));

            window_list = g_list_append (window_list, next_window);

            set_window_list (next_window, window_list);
            set_window_list (selection_window, window_list);

            gtk_object_set_data (GTK_OBJECT (selection_window), "next_window_id" , next_window);
            gtk_object_set_data (GTK_OBJECT (selection_window), "previous_plugin", GUINT_TO_POINTER (plugin));

            /*
             * Connect the destroy signal to the next window. If the window is
             * destroyed, so is the task context.
             */

            gtk_signal_connect (GTK_OBJECT (next_window), "destroy", 
                                GTK_SIGNAL_FUNC (on_task_window_destroy), GUINT_TO_POINTER (handle));
                       
            if (next_button_text)
                g_free (next_button_text);
            
            gtk_widget_show (next_window);
            gtk_widget_hide (selection_window);            
        }
    }
    else
    {    
        gtk_widget_show (next_window);
        gtk_widget_hide (selection_window);
    }
}

/*
 *
 *   void on_plugin_selection_clist_realize (GtkCList *, evms_plugin_code_t,
 *                                           plugin_search_flags_t flags)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of plugins desired.
 * 
 *   Entry:
 *      clist - address of the selections GtkCList widget
 *      code  - the plugin code to enumerate and add to the list
 *      flags - plugin search flags
 *
 *   Exit:
 *      Selection list populated with plugins of the given type
 *
 */
void on_plugin_selection_clist_realize (GtkCList *clist, evms_plugin_code_t code,
                                        plugin_search_flags_t flags)
{
    gint            rc;
    handle_array_t *plugins;

    rc = evms_get_plugin_list (code, flags, &plugins);

    if (rc != SUCCESS)
    {
        log_error ("%s: evms_get_plugin_list() returned error code %d.\n", __FUNCTION__, rc);
    }
    else
    {
        gint     i;
        gboolean is_selected = (plugins->count == 1);

        /*
         * Since we will be using the second column for the plugin description
         * and hiding the unused third column, set the column to autoresize
         * based upon the length of the longest description so it does not
         * get initially cut off.
         */

        gtk_clist_set_column_auto_resize (clist, SL_SIZE_COLUMN, TRUE);
        
        for (i=0; i < plugins->count; i++)
             add_thing_to_selection_list (clist, plugins->handle[i], is_selected);

        evms_free (plugins);
    }

    /*
     * We won't need the standard three columns (pixmap, size, description) for
     * the plugin list so hide the last column.
     */

    gtk_clist_set_column_visibility (clist, SL_DESC_COLUMN, FALSE);
}
