/*
 *
 *   Copyright (c) International Business Machines  Corp., 2001, 2002
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or 
 *   (at your option) any later version.
 * 
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software 
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: setinfo.c
 */ 
 
#include <frontend.h>
#include <gtk/gtk.h>

#include "support.h"
#include "widget.h"
#include "task.h"
#include "setinfo.h"
#include "thing.h"
#include "readable.h"
#include "logging.h"
#include "help.h"

/*
 *
 *   void on_set_info_clist_realize (GtkWidget *, gpointer)
 *   
 *   Description:
 *      This routine populates the given GtkCList with the list
 *      of things that can have their properties changed through
 *      the EVMS_Task_Set_Info task.
 * 
 *   Entry:
 *      widget    - address of the selections GtkCList widget
 *      user_data - contains type of thing to set (volume, object, etc.)
 *
 *   Exit:
 *      Selection list populated with settable things
 *
 */
void on_set_info_clist_realize (GtkWidget *widget, gpointer user_data)
{
    gint            rc=0;
    GtkCList       *clist = GTK_CLIST (widget);
    object_type_t   type = GPOINTER_TO_UINT (user_data);
    handle_array_t *things;

    set_selection_window_clist_column_titles (clist, _("Size"), 
                                              make_object_type_readable_string (type), NULL);
    
    switch (type)
    {
        case VOLUME:
            rc = evms_get_volume_list (0, &things);
            break;

        case EVMS_OBJECT:
        case REGION:
        case SEGMENT:
        case DISK:
            rc = evms_get_object_list (type, DATA_TYPE, 0, 0, &things);
            break;

        case CONTAINER:
            rc = evms_get_container_list (0, &things);
            break;

        default:
            log_error ("%s: Unsupported selection type %d.\n", __FUNCTION__, type);
            rc = EINVAL;
            break;
    }

    if (rc != SUCCESS) 
    {
        log_error ("%s: evms_get_xxxx_list() returned error code %d.\n", __FUNCTION__, rc);
    }
    else
    {
        guint    i;
        gboolean is_selected = (things->count == 1);

        for (i=0; i < things->count; i++)
        {
            if (evms_can_set_info (things->handle[i]) == 0)
                add_thing_to_selection_list (clist, things->handle[i], is_selected);
        }

        if (clist->rows == 1)
            gtk_clist_select_row (clist, 0, 0);
        
        evms_free (things);
    }
}

/*
 *
 *   void on_set_info_button_clicked (GtkButton *, gpointer)
 *   
 *   Description:
 *      This routine initiates the EVMS_Task_Set_Info task
 *      and creates or redisplays the options window to
 *      allow changing the properties of the current selection.
 * 
 *   Entry:
 *      button    - the button that was clicked
 *      user_data - not used
 *
 *   Exit:
 *      Nothing is returned but hopefully the options notebook
 *      is displayed.
 *
 */
void on_set_info_button_clicked (GtkButton *button, gpointer user_data)
{
    GList          *window_list;
    GtkWidget      *clist;
    GtkWidget      *next_window;
    GtkWidget      *selection_window;
    object_handle_t object;
    object_handle_t prev_object;

    /*
     * Hide the selection window and either redisplay the existing options window
     * if it exists and the selection did not change. If the options window
     * does not exist, create it. However, if the selection changed and
     * the options window existed then destroy the current options window and create
     * a new one.
     */

    selection_window = gtk_widget_get_toplevel (GTK_WIDGET (button));
    clist            = lookup_widget (GTK_WIDGET (button), "selection_window_clist");
    object           = GPOINTER_TO_UINT (get_single_select_current_row_data (GTK_CLIST (clist)));
    window_list      = get_window_list (selection_window);
    next_window      = gtk_object_get_data (GTK_OBJECT (selection_window), "next_window_id");
    prev_object      = GPOINTER_TO_UINT (gtk_object_get_data (GTK_OBJECT (selection_window), "previous_object"));

    if (window_list == NULL)
        window_list = g_list_append (window_list, selection_window);

    if (object != prev_object)
    {
        gint          rc;
        task_handle_t handle;

        rc = evms_create_task (object, EVMS_Task_Set_Info, &handle);

        if (rc != SUCCESS)
        {
            log_error ("%s: evms_create_task() returned error code %d.\n", __FUNCTION__, rc);

            display_results_window (rc, NULL, _("Unable to create the task context."), 
                                    NULL, FALSE, selection_window);
        }
        else
        {            
            /*
             * If we already created the next window for changing
             * the object properties (through the options window),
             * make sure we destroy the window since our selection
             * has apperently changed and so all the following panel
             * needs to be recreated.
             */

            if (next_window != NULL)
                destroy_window_list (g_list_find (window_list, next_window));
            
            next_window = create_standard_task_window ("Modify Properties", 
                                                       _("Modify"),
                                                       NULL, 
                                                       on_button_clicked_display_prev_window,
                                                       on_button_clicked_destroy_window_list,
                                                       handle);

            window_list = g_list_append (window_list, next_window);

            set_window_list (next_window, window_list);
            set_window_list (selection_window, window_list);
                
            bind_options_callout_to_toplevel_widget (next_window, options_window_callout);

            gtk_object_set_data (GTK_OBJECT (selection_window), "next_window_id", next_window);
            gtk_object_set_data (GTK_OBJECT (selection_window), "previous_object", GUINT_TO_POINTER (object));

            /*
             * Connect the destroy signal to the next window. If the window is
             * destroyed, so is the task context.
             */

            gtk_signal_connect (GTK_OBJECT (next_window), "destroy", 
                                GTK_SIGNAL_FUNC (on_task_window_destroy), GUINT_TO_POINTER (handle));
                                            
            gtk_widget_show (next_window);
            gtk_widget_hide (selection_window);            
        }
    }
    else
    {
        gtk_widget_show (next_window);
        gtk_widget_hide (selection_window);
    }    
}

/*
 *
 *   void on_modify_thing_properties_menu_item_activate (GtkMenuItem *, gpointer)
 *
 *   Description:
 *      This routine will display the thing we intend to modify and
 *      allow the user to confirm the modification of its properties.
 * 
 *   Entry:
 *      menuitem  - the menuitem that initiated the action
 *      user_data - object handle
 *
 *   Exit:
 *      See description.
 *
 */
void on_modify_thing_properties_menu_item_activate (GtkMenuItem *menuitem, gpointer user_data)
{
    object_type_t type;

    if (evms_get_handle_object_type (GPOINTER_TO_UINT (user_data), &type) == SUCCESS)
    {
        gchar     *window_title;
        gchar     *help_text;
        GtkWidget *window;

        switch (type)
        {
            case VOLUME:
                window_title = _("Modify Logical Volume Properties");
                help_text = modify_volume_properties_help_text;
                break;

            case EVMS_OBJECT:
                window_title = _("Modify Feature Object Properties");
                help_text = modify_object_properties_help_text;
                break;

            case REGION:
                window_title = _("Modify Storage Region Properties");
                help_text = modify_region_properties_help_text;
                break;
            
            case CONTAINER:
                window_title = _("Modify Storage Container Properties");
                help_text = modify_container_properties_help_text;
                break;
            
            case SEGMENT:
                window_title = _("Modify Disk Segment Properties");
                help_text = modify_segment_properties_help_text;
                break;
            
            case DISK:
                window_title = _("Modify Logical Disk Properties");
                help_text = modify_disk_properties_help_text;
                break;
            
            default:
                window_title = _("Modify Properties");
                help_text = NULL;            
                break;
        }

        window = create_standard_selection_window (window_title,
                                                   NULL,
                                                   help_text,
                                                   add_thing_as_selected_list_item,
                                                   on_set_info_button_clicked,
                                                   NULL, NULL, NULL, NULL, user_data);

        gtk_widget_show (window);
    }
}
