/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libcsm.so
 *
 *   File: containers.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/utsname.h>
#include <uuid/uuid.h>

#include <plugin.h>

#include "csm_plugin.h"

int add_disk_to_container( storage_object_t    * ld,
			   storage_container_t * container )
{
        int rc;
        DISKSEG *seg=NULL, *dataseg=NULL;
        list_element_t iter,e;

        LOG_ENTRY();

        REQUIRE( isa_cluster_logical_disk(ld)==TRUE );
        REQUIRE( isa_cluster_container(container)==TRUE );
        REQUIRE( EngFncs->list_count(ld->parent_objects) > 0 );

        // find the csm data segment
        LIST_FOR_EACH( ld->parent_objects, iter, seg ){
                if ( seg->data_type == DATA_TYPE ) {
                        dataseg=seg;
                        break;
                }
        }

        e=EngFncs->insert_thing( container->objects_consumed,
                                 ld,
                                 INSERT_AFTER | EXCLUSIVE_INSERT,
                                 NULL );
        if (e!=NULL) {
                rc = 0;
        }
        else {
                rc = ENOMEM;
        }

        if (!rc) {

                // add data segment to the disk group produced object list.
                // there will NOT be a dataseg if we dont have access to the
                // container,i.e. not ours or deported storage.
                if (dataseg) {

                        e=EngFncs->insert_thing( container->objects_produced,
                                                 dataseg,
                                                 INSERT_AFTER | EXCLUSIVE_INSERT,
                                                 NULL );
                        if (e!=NULL) {
                                rc = 0;
                        }
                        else {
                                rc = ENOMEM;
                        }

                        if (!rc) {
                                dataseg->consuming_container = NULL;
                                dataseg->producing_container = container;

                                if (container->flags & SCFLAG_CLUSTER_PRIVATE) {
                                        dataseg->flags |= SOFLAG_CLUSTER_PRIVATE;
                                }
                                else if (container->flags & SCFLAG_CLUSTER_SHARED) {
                                        dataseg->flags |= SOFLAG_CLUSTER_SHARED;
                                }
                                else if (container->flags & SCFLAG_CLUSTER_DEPORTED) {
                                        dataseg->flags |= SOFLAG_CLUSTER_DEPORTED;
                                }

                                container->size  += dataseg->size;
                        }

                }

                if (!rc) {

                        ld->consuming_container  = container;

                        container->flags |= SCFLAG_DIRTY;     // discovery code should clear
                                                              // the DIRTY bit
                }
                else {
                        EngFncs->remove_thing( container->objects_consumed, ld );
                }

        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: csm_can_delete_container
 *
 *  If it is one of our csm containers then we can certainly
 *  delete it ... but only if we can unassign the CSM from
 *  each of the disks in the disk group (container). This is
 *  because we don't want the container being rediscovered
 *  during a rediscovery at some later time.
 *
 *  Naturally, check and make sure there are no storage objects
 *  consuming the csm data segs!
 */
int csm_can_delete_container( storage_container_t * container )
{
        LOGICALDISK *ld=NULL;
        DISKSEG *seg=NULL;
        int produced_objects=0;
        list_element_t iter;

        LOG_ENTRY();

        REQUIRE( csm_has_quorum == TRUE || csm_admin_mode == TRUE);
        REQUIRE( isa_cluster_container(container)==TRUE );

        LIST_FOR_EACH( container->objects_produced, iter, seg ){
                produced_objects += EngFncs->list_count(seg->parent_objects);
        }

        REQUIRE(produced_objects == 0);

        LIST_FOR_EACH( container->objects_consumed, iter, ld ) {
                REQUIRE( csm_can_unassign( ld ) == 0 );
        }

        LOG_EXIT_INT(0);
        return 0;
}


/*
 *  Function: csm_can_add_object
 *
 *  Our containers hold logical disks that produce CSM
 *  segments ... meaning they have CSM metadata.  So, we
 *  can only add the object if it is a csm logical disk.
 */
int csm_can_add_object( LOGICALDISK           * ld,
                        storage_container_t   * container )
{
        LOG_ENTRY();

        REQUIRE( csm_has_quorum == TRUE || csm_admin_mode == TRUE);
        REQUIRE( EngFncs->list_count(ld->parent_objects) == 0 );
	REQUIRE( ld->consuming_container == NULL );
        REQUIRE( isa_cluster_container(container) == TRUE);

        LOG_EXIT_INT(0);
        return 0;
}


/**
 * csm_can_expand_container
 *
 * Get the list of available disks and see which ones can be added to this
 * container. If any object can be added, create an expand-point to add to
 * this list.
 **/
int csm_can_expand_container(storage_container_t *container,
			     list_anchor_t expand_points)	/* of type expand_object_info_t */
{
	expand_object_info_t *expand_point;
	list_anchor_t objects = NULL;
	storage_object_t *object;
	sector_count_t limit = 0;
	list_element_t iter;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Checking if container %s can expand.\n", container->name);

        rc = EngFncs->get_object_list(DISK, DATA_TYPE, NULL,
				      NULL, TOPMOST, &objects);
	if (rc) {
		goto out;
	}

	prune_acceptable_disks(objects);
	LIST_FOR_EACH(objects, iter, object) {
		limit += object->size;
	}

	if (!limit) {
		LOG_DEBUG("No objects available to add to container %s.\n",
			  container->name);
		rc = ENOSPC;
		goto out;
	}

	expand_point = EngFncs->engine_alloc(sizeof(*expand_point));
	if (!expand_point) {
		rc = ENOMEM;
		goto out;
	}

	expand_point->container = container;
	expand_point->max_expand_size = limit;
	EngFncs->insert_thing(expand_points, expand_point, INSERT_AFTER, NULL);
	rc = 0;

out:
	EngFncs->destroy_list(objects);
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * csm_can_shrink_container
 *
 * If any one of the consumed objects can be removed, then the container can
 * shrink. Create a shrink_object_info_t and add it to the shrink_points list.
 **/
int csm_can_shrink_container(storage_container_t *container,
			     list_anchor_t shrink_points)	/* of type shrink_object_info_t */
{
	shrink_object_info_t *shrink_point;
	storage_object_t *object;
	sector_count_t limit = 0;
	list_element_t iter;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Checking if container %s can shrink.\n", container->name);

	LIST_FOR_EACH(container->objects_consumed, iter, object) {
		rc = csm_can_unassign(object);
		if (!rc) {
			limit += object->size;
		}
	}

	if (!limit) {
		LOG_DEBUG("No objects can be removed from container %s.\n",
			  container->name);
		rc = EBUSY;
		goto out;
	}

	shrink_point = EngFncs->engine_alloc(sizeof(*shrink_point));
	if (!shrink_point) {
		rc = ENOMEM;
		goto out;
	}

	shrink_point->container = container;
	shrink_point->max_shrink_size = limit;
	EngFncs->insert_thing(shrink_points, shrink_point, INSERT_AFTER, NULL);
	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * csm_can_expand_container_by
 *
 * Don't currently support expanding consumed disks.
 **/
int csm_can_expand_container_by(storage_container_t *container,
				storage_object_t *consumed_object,
				sector_count_t *delta_size)
{
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/**
 * csm_can_shrink_container_by
 *
 * Don't currently support shrinking consumed disks.
 **/
int csm_can_shrink_container_by(storage_container_t *container,
				storage_object_t *consumed_object,
				sector_count_t *delta_size)
{
	LOG_ENTRY();
	LOG_EXIT_INT(ENOSYS);
	return ENOSYS;
}

/*
 *  Function: csm_create_container_storage_object
 *
 *  If we can get the memory for the new storage container
 *  and if we can register the new name ... then we can
 *  create it.  This routine can be called from 3 different
 *  places ... (1) from the container create api (2) from
 *  the csm discover api (3) from the csm assign api.
 */
int csm_create_container_storage_object( char                 *name,
                                         storage_container_t **container,
                                         ece_nodeid_t         *nodeid,
                                         ece_clusterid_t      *clusterid,
                                         uint                  flags )
{
        storage_container_t *new_container;
        int  rc;
        container_private_data_t  *c_pdata=NULL;


        LOG_ENTRY();

        REQUIRE( name != NULL );
        REQUIRE( container != NULL );

        rc = allocate_csm_container( flags, &new_container );
        if (!rc) {

                c_pdata = (container_private_data_t *) new_container->private_data;

                memcpy(&c_pdata->clusterid, clusterid, sizeof(ece_clusterid_t) );

                memcpy(&c_pdata->nodeid, nodeid, sizeof(ece_nodeid_t) );

                new_container->disk_group = new_container;

//                sprintf( new_container->name, "%s", name );
                strcpy( new_container->name, name );

                rc = EngFncs->register_name(new_container->name);
                if (rc) {
                        free_csm_container(new_container);
                        new_container = NULL;
                }

        }
        else {
                LOG_ERROR("error, allocating container %s.\n", name);
                rc = ENOMEM;
        }

        *container = new_container;
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: csm_create_container
 *
 *  This api is just a wrapper around the lower level routine
 *  csm_create_container_storage_object.  The lower level routine
 *  is called when you know the create options, like during
 *  discovery.  So, the purpose of this routine is to collect
 *  evms task based options and make the call.
 */
int csm_create_container( list_anchor_t             objects,
                          option_array_t          * options,
                          storage_container_t    ** container )
{
        int rc=EINVAL;
        LOGICALDISK *ld;
        storage_container_t *new_container=NULL;
        list_element_t iter;

        LOG_ENTRY();

        REQUIRE(objects != NULL );
	REQUIRE(EngFncs->list_count != 0);
        REQUIRE(options != NULL );
        REQUIRE(container != NULL);
        REQUIRE(csm_has_quorum == TRUE || csm_admin_mode == TRUE);

        LIST_FOR_EACH(objects, iter, ld) {
                REQUIRE(ld->consuming_container == NULL);
                REQUIRE(EngFncs->list_count(ld->parent_objects) == 0);
        }

        LIST_FOR_EACH(objects, iter, ld) {
                rc = csm_assign( ld, options );

                if (rc) {
                        break;
                }
        }

	ld = EngFncs->first_thing(objects, NULL);
	new_container = ld->consuming_container;

        if (rc && new_container != NULL) {
                csm_delete_container( new_container, NULL );
        }

        *container = new_container;
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: csm_add_object
 *
 *  Our containers hold logical disks that produce CSM
 *  segments ... meaning they have CSM metadata.  So, we
 *  can only add the object if it is a csm logical disk.
 *  that is producing a csm data segment.
 */
int csm_add_object( LOGICALDISK         *ld,
                    storage_container_t *container,
                    option_array_t      *options )
{
        int rc;

        LOG_ENTRY();

        REQUIRE( EngFncs->list_count(ld->parent_objects) == 0 );
	REQUIRE( ld->consuming_container == NULL );
        REQUIRE( isa_cluster_container(container)==TRUE );

	rc = assign_cluster_segment_manager_to_disk( ld,
						     container );
	if (!rc) {
		// Tell other nodes in the cluster to rediscover everything.
		// We don't know how this disk appears on the other nodes,
		// so everything has to be rediscovered.
		EngFncs->rediscover_objects(NULL);

	}

        LOG_EXIT_INT(rc);
        return rc;
}


/**
 * csm_expand_container
 *
 * Expand the specified container by adding one or more objects, which are
 * specified in the input_objects list. Add each object using the specified
 * options.
 **/
int csm_expand_container(storage_container_t *container,
			 storage_object_t *consumed_object,
			 storage_object_t *expand_object,
			 list_anchor_t input_objects,
			 option_array_t *options)
{
	storage_object_t *object;
	list_element_t iter;
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Expanding container %s.\n", container->name);

	if (expand_object) {
		/* One of the consumed objects will be expanding.
		 * Not currently supported.
		 */
		rc = ENOSYS;
	} else {
		/* Add this list of new objects to the container. */
		LIST_FOR_EACH(input_objects, iter, object) {
			rc = csm_add_object(object, container, options);
			if (rc) {
				LOG_ERROR("Error adding object %s to container "
					  "%s. Aborting remaining expands.\n",
					  object->name, container->name);
				break;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * csm_shrink_container
 *
 * Shrink the specified container by removing one or more objects, which are
 * specified in the input_objects list. Remove each object using the specified
 * options.
 **/
int csm_shrink_container(storage_container_t *container,
			 storage_object_t *consumed_object,
			 storage_object_t *shrink_object,
			 list_anchor_t input_objects,
			 option_array_t *options)
{
	storage_object_t *object;
	list_element_t iter;
	int rc = 0;

	LOG_ENTRY();
	LOG_DEBUG("Shrinking container %s.\n", container->name);

	if (shrink_object) {
		/* One of the consumed objects will be shrinking.
		 * Not currently supported.
		 */
		rc = ENOSYS;
	} else {
		/* Remove this list of objects from the container. */
		LIST_FOR_EACH(input_objects, iter, object) {
			rc = csm_remove_object(object);
			if (rc) {
				LOG_ERROR("Error removing object %s from "
					  "container %s. Aborting remaining "
					  "shrinks.\n", object->name,
					  container->name);
				break;
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}

int csm_discard_container( storage_container_t * container )
{
	list_element_t iter1;
	list_element_t iter2;
	DISKSEG *seg;
	LOGICALDISK *ld;

        LOG_ENTRY();

	// The objects produced must not be in use by other objects.

	LIST_FOR_EACH(container->objects_produced, iter1, seg) {
		if (seg->data_type == DATA_TYPE) {
			REQUIRE( EngFncs->list_empty( seg->parent_objects ) == TRUE );
		}
	}

	// Remove the CSM from each disk consumed by the container.
	// The removeal of csm from the last disk will discard the container.

	LIST_FOR_EACH_SAFE(container->objects_consumed, iter1, iter2, ld) {
		remove_csm_from_disk( ld );
	}

        LOG_EXIT_INT(0);
        return 0;
}


/*
 *  Function: csm_remove_object
 *
 *  Our containers only hold logical disks that produce CSM
 *  segments ... meaning they have CSM metadata.  So, we
 *  can only remove the object if it is a csm logical disk.
 *  that is currently being consumed by one of our containers
 *  ... and ... we remove it by removing the segment managers
 *  metadata from the storage object.
 */
int csm_remove_object( LOGICALDISK *ld )
{
        int rc;

        LOG_ENTRY();

        REQUIRE( isa_cluster_logical_disk(ld)==TRUE );
        REQUIRE( isa_cluster_container(ld->consuming_container)==TRUE );

        rc = csm_unassign(ld);

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: csm_delete_container
 *
 *  Easy ... sanity check the container ... cant have any of
 *  its csm data segs being consumed by anyone and we must have
 *  admin privilage or quorum to do this.
 *
 *  If Ok then we can delete the container and release the
 *  the objects that are currently being consumed by it by
 *  telling the CSM seg mgr to unassign each of the consumed
 *  disk storage objects.
 */
int csm_delete_container( storage_container_t   * container,
                          list_anchor_t           objects_consumed )
{
        int rc=0;
        LOGICALDISK *ld;
        DISKSEG *seg;
        int produced_objects=0;
        list_element_t iter;

        LOG_ENTRY();

        REQUIRE( isa_cluster_container(container)==TRUE );
        REQUIRE( container->objects_produced != NULL );
        REQUIRE( container->objects_consumed != NULL );
        REQUIRE( csm_has_quorum == TRUE || csm_admin_mode == TRUE);

        LIST_FOR_EACH(container->objects_produced, iter, seg ){
                produced_objects += EngFncs->list_count(seg->parent_objects);
        }

        REQUIRE(produced_objects == 0);

        // For Each ... logical disk in the disk group ...
        // call the CSM unassign api to remove it.
        do {
                ld = EngFncs->first_thing(container->objects_consumed,NULL);
                if (ld!=NULL) {
                        rc = csm_unassign(ld);
                }

        } while (ld != NULL  &&  rc == 0);

        LOG_EXIT_INT(0);
        return 0;
}


/*
 *  Function:  csm_commit_container_changes
 *
 *  Really a NOP. Changes to a csm container are driven down
 *  to csm metadata segments which are marked dirty and
 *  committed to disk.  See ... the container set info
 *  code.
 */
int csm_commit_container_changes( storage_container_t   * container,
                                  unsigned int          commit_phase )
{
        LOG_ENTRY();

        REQUIRE( isa_cluster_container(container)==TRUE );

        if ( commit_phase > 0 ) {
                container->flags &= ~SCFLAG_DIRTY;
        }

        LOG_EXIT_INT(0);
        return 0;
}

/*
 *  Function:  csm_backup_container_metadata
 *
 *  CSM containers don't have metadata to backup. All metadata is backed-up
 *  on a per-segment basis. See csm_backup_metadata().
 */
int csm_backup_container_metadata( storage_container_t * container )
{
	LOG_ENTRY();
	LOG_EXIT_INT(0);
	return 0;
}

/*
 *  Function:  csm_get_container_info
 *
 *  Used to access container private info.
 */
int csm_get_container_info( storage_container_t     *container,
                            char                    * name,
                            extended_info_array_t  **info )
{
        int rc=ENOSYS;
        extended_info_array_t   *Info;
        container_private_data_t *c_pdata;
        const char *nodeid_string;

        LOG_ENTRY();

        REQUIRE(isa_cluster_container(container)==TRUE);

        c_pdata = (container_private_data_t *)container->private_data;

        Info = EngFncs->engine_alloc( sizeof(extended_info_array_t) + ( CONTAINER_GETINFO_COUNT * sizeof(extended_info_t) ) );

        if (Info) {

                Info->count = CONTAINER_GETINFO_COUNT;

                Info->info[CONTAINER_GETINFO_NAME_INDEX].name =
			EngFncs->engine_strdup( CONTAINER_GETINFO_NAME_STRING );
                Info->info[CONTAINER_GETINFO_NAME_INDEX].title = EngFncs->engine_strdup( _("Name") );
                Info->info[CONTAINER_GETINFO_NAME_INDEX].desc =
			EngFncs->engine_strdup( _("Enter the container name(e.g. node1-disk-group). Do not use any slashes in the name. Evms will prepend the csm namespace to the name(e.g. csm/node1-disk-group)."));
                Info->info[CONTAINER_GETINFO_NAME_INDEX].type               = EVMS_Type_String;
                Info->info[CONTAINER_GETINFO_NAME_INDEX].unit               = EVMS_Unit_None;
                Info->info[CONTAINER_GETINFO_NAME_INDEX].value.s = EngFncs->engine_strdup( container->name );
                Info->info[CONTAINER_GETINFO_NAME_INDEX].collection_type    = EVMS_Collection_None;
                memset(&Info->info[CONTAINER_GETINFO_NAME_INDEX].group, 0, sizeof(group_info_t));

		rc = EngFncs->nodeid_to_string( &c_pdata->nodeid, &nodeid_string );

                if (!rc) {
                        Info->info[CONTAINER_GETINFO_TYPE_INDEX].name =
				EngFncs->engine_strdup( CONTAINER_GETINFO_TYPE_STRING );
                        Info->info[CONTAINER_GETINFO_TYPE_INDEX].title = EngFncs->engine_strdup( _("Type") );
                        Info->info[CONTAINER_GETINFO_TYPE_INDEX].desc =
				EngFncs->engine_strdup( _("Enter the cluster storage type. The allowed values are: private, shared or deported."));
                        Info->info[CONTAINER_GETINFO_TYPE_INDEX].type = EVMS_Type_String;
                        Info->info[CONTAINER_GETINFO_TYPE_INDEX].unit = EVMS_Unit_None;

                        if (container->flags & SCFLAG_CLUSTER_PRIVATE) {
                                Info->info[CONTAINER_GETINFO_TYPE_INDEX].value.s = EngFncs->engine_strdup( _("private") );
                        }
                        else if (container->flags & SCFLAG_CLUSTER_SHARED) {
                                Info->info[CONTAINER_GETINFO_TYPE_INDEX].value.s = EngFncs->engine_strdup( _("shared") );
                        }
                        else if (container->flags & SCFLAG_CLUSTER_DEPORTED) {
                                Info->info[CONTAINER_GETINFO_TYPE_INDEX].value.s = EngFncs->engine_strdup( _("deported") );
                        }
                        else {
                                Info->info[CONTAINER_GETINFO_TYPE_INDEX].value.s = EngFncs->engine_strdup( _("unknown") );
                        }

                        Info->info[CONTAINER_GETINFO_TYPE_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[CONTAINER_GETINFO_TYPE_INDEX].group, 0, sizeof(group_info_t));

                        Info->info[CONTAINER_GETINFO_NODE_ID_INDEX].name =
				EngFncs->engine_strdup( CONTAINER_GETINFO_NODE_ID_STRING );
                        Info->info[CONTAINER_GETINFO_NODE_ID_INDEX].title = EngFncs->engine_strdup( _("Node") );
                        Info->info[CONTAINER_GETINFO_NODE_ID_INDEX].desc =
				EngFncs->engine_strdup( _("Enter the owner of this container (nodeid). Every container must have an owner and it must be one of the configured cluster nodes.") );
                        Info->info[CONTAINER_GETINFO_NODE_ID_INDEX].type               = EVMS_Type_String;
                        Info->info[CONTAINER_GETINFO_NODE_ID_INDEX].unit               = EVMS_Unit_None;
                        Info->info[CONTAINER_GETINFO_NODE_ID_INDEX].value.s = EngFncs->engine_strdup( nodeid_string );
                        Info->info[CONTAINER_GETINFO_NODE_ID_INDEX].collection_type    = EVMS_Collection_None;
                        memset( &Info->info[CONTAINER_GETINFO_NODE_ID_INDEX].group, 0, sizeof(group_info_t));
                }
                else {
                        EngFncs->engine_free( Info );
                        Info = NULL;
                }


        }
        else {
                rc = ENOMEM;
        }

        *info = Info;
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: csm_set_container_info
 *
 *  This api is used by the cluster segment manager to make changes
 *  to a disk group.  If you want to reassign a disk group to a
 *  different node or if you simply want to rename a disk group ...
 *  you would use modify container properties ... which would start
 *  a task that eventually calls this routine.  The changes are made
 *  by updating the metadata of each logical disk being consumed
 *  by the container and then marking these segments dirty so that
 *  the metadata will be updated on disk.
 */
int csm_set_container_info( storage_container_t * container,
                            option_array_t      * options )
{
        int rc=0;
        int i;
        container_private_data_t *c_pdata;
        container_private_data_t  saved_c_pdata;
        LOGICALDISK *ld;
        DISKSEG *seg;
        disk_private_data_t *disk_pdata;
        csm_header_t  *hdr1=NULL;
        csm_header_t  *hdr2=NULL;
        ece_nodeid_t  original_nodeid;
        char original_name[EVMS_NAME_SIZE+1];
        uint original_flags;
        char new_container_name[EVMS_NAME_SIZE+1];
        char storage_type[EVMS_NAME_SIZE];
        boolean container_was_accessible;
        list_element_t iter,iter2;
	list_anchor_t container_list;

        LOG_ENTRY();

        REQUIRE(container != NULL);
        REQUIRE(isa_cluster_container(container)==TRUE);
        REQUIRE(options != NULL);
        REQUIRE(csm_has_quorum == TRUE || csm_admin_mode == TRUE);

        container_was_accessible = isa_accessible_container( container );

        c_pdata = (container_private_data_t *) container->private_data;

        // save for possible restore
        memcpy(&saved_c_pdata, c_pdata, sizeof(container_private_data_t));
        memcpy(&original_nodeid, &c_pdata->nodeid, sizeof(ece_nodeid_t) );
        strncpy(original_name, container->name, EVMS_NAME_SIZE);
        original_flags = container->flags;
        new_container_name[0] = '\0';

        // get modify property options
        for (i = 0; i < options->count && rc==0; i++) {

                if (options->option[i].is_number_based) {

                        switch (options->option[i].number) {

                        case CONTAINER_SETINFO_NAME_INDEX:

				// Handle only a name change
				if (strcmp(options->option[i].value.s,container->name) != 0) {
					rc = EngFncs->validate_name(options->option[i].value.s);
					if (!rc) {
						strncpy( new_container_name,
							 options->option[i].value.s,
							 EVMS_NAME_SIZE);
					} else {
						LOG_ERROR("Validation of name \"%s\" failed with error code %d: %s\n",
							  options->option[i].value.s, rc, EngFncs->strerror(rc));
						LOG_EXIT_INT(rc);
						return rc;
					}
				}
                                break;

                        case CONTAINER_SETINFO_TYPE_INDEX:
                                strncpy(storage_type, options->option[i].value.s, EVMS_NAME_SIZE);
                                break;

                        case CONTAINER_SETINFO_NODE_ID_INDEX:

                                rc = EngFncs->string_to_nodeid( options->option[i].value.s,
								&c_pdata->nodeid );
				if (rc) {
					LOG_ERROR("Lookup of node name \"%s\" failed with error code %d: %s\n",
						  options->option[i].value.s, rc, EngFncs->strerror(rc));
					LOG_EXIT_INT(rc);
					return rc;
				}
                                break;

                        default:
				LOG_ERROR("%d is not a valid option index.\n", options->option[i].number);
				LOG_EXIT_INT(EINVAL);
                                return EINVAL;
                        }

                }
                else {
                        if (strcmp(options->option[i].name,CONTAINER_SETINFO_NAME_STRING) == 0) {

				// Handle only a name change
				if (strcmp(options->option[i].value.s,container->name) != 0) {
					rc = EngFncs->validate_name(options->option[i].value.s);
					if (!rc) {
						strncpy( new_container_name,
							 options->option[i].value.s,
							 EVMS_NAME_SIZE);
					} else {
						LOG_ERROR("Validation of name \"%s\" failed with error code %d: %s\n",
							  options->option[i].value.s, rc, EngFncs->strerror(rc));
						LOG_EXIT_INT(rc);
						return rc;
					}
				}

                        }
                        else if (strcmp(options->option[i].name,CONTAINER_SETINFO_TYPE_STRING) == 0) {
                                strncpy(storage_type, options->option[i].value.s, EVMS_NAME_SIZE );
                        }
                        else if (strcmp(options->option[i].name,CONTAINER_SETINFO_NODE_ID_STRING) == 0) {

                                rc = EngFncs->string_to_nodeid( options->option[i].value.s,
								&c_pdata->nodeid);
				if (rc) {
					LOG_ERROR("Lookup of node name \"%s\" failed with error code %d: %s\n",
						  options->option[i].value.s, rc, EngFncs->strerror(rc));
					LOG_EXIT_INT(rc);
					return rc;
				}
                        }
                        else {
				LOG_ERROR("%s is not a valid option name.\n", options->option[i].name);
				LOG_EXIT_INT(EINVAL);
                                return EINVAL;
                        }
                }
        }

	// Tell the other nodes in the cluster to rediscover.
	// the changes to the container.  If anything fails below, that's OK.
	// The other nodes will just rediscover the old stuff again.
	container_list = EngFncs->allocate_list();
	if ( container_list != NULL ) {
		list_element_t el;

		el = EngFncs->insert_thing( container_list, container, INSERT_AFTER, NULL );
		if ( el != NULL ) {
			EngFncs->rediscover_containers( container_list );
		}

		EngFncs->destroy_list( container_list );
	}

        // update container flags
        if (strcmp(storage_type,_("private"))==0) {
                container->flags &= ~SCFLAG_CLUSTER_SHARED;
                container->flags &= ~SCFLAG_CLUSTER_DEPORTED;
                container->flags |=  SCFLAG_CLUSTER_PRIVATE;
        }
        else if (strcmp(storage_type,_("shared"))==0) {
                container->flags &= ~SCFLAG_CLUSTER_PRIVATE;
                container->flags &= ~SCFLAG_CLUSTER_DEPORTED;
                container->flags |=  SCFLAG_CLUSTER_SHARED;
        }
        else if (strcmp(storage_type,_("deported"))==0) {
                container->flags &= ~SCFLAG_CLUSTER_PRIVATE;
                container->flags &= ~SCFLAG_CLUSTER_SHARED;
                container->flags |=  SCFLAG_CLUSTER_DEPORTED;
        }
        else {
                container->flags &= ~SCFLAG_CLUSTER_SHARED;
                container->flags &= ~SCFLAG_CLUSTER_DEPORTED;
                container->flags |=  SCFLAG_CLUSTER_PRIVATE;
        }

	if (new_container_name[0] != '\0') {
		rc = EngFncs->register_name(new_container_name);
		// The register should not fail since the name was
		// validated above.  But we'll check for an error
		// just in case.
		if (!rc) {
			EngFncs->unregister_name(container->name);
			memset(container->name, '\0', EVMS_NAME_SIZE+1);
			strcpy(container->name, new_container_name);
		} else {
			MESSAGE(_("Register new container name \"%s\" failed with error code %d: %s\n"),
				new_container_name, rc, EngFncs->strerror(rc));
			rc = 0;
		}
	}

        if (!rc) {
		// update metadata on each disk
                LIST_FOR_EACH( container->objects_consumed, iter, ld ){

                        disk_pdata = get_csm_disk_private_data(ld);

                        if (disk_pdata) {
                                hdr1 = ((seg_private_data_t *)disk_pdata->md1->private_data)->hdr;
                                hdr2 = ((seg_private_data_t *)disk_pdata->md2->private_data)->hdr;

                                strncpy(hdr1->container_name, container->name, EVMS_NAME_SIZE );
                                strncpy(hdr2->container_name, container->name, EVMS_NAME_SIZE );

                                memcpy(&hdr1->nodeid, &c_pdata->nodeid, sizeof(ece_nodeid_t));
                                memcpy(&hdr2->nodeid, &c_pdata->nodeid, sizeof(ece_nodeid_t));

                                if (container->flags & SCFLAG_CLUSTER_PRIVATE) {
                                        hdr1->flags &= ~CSM_CLUSTER_SHARED;
                                        hdr1->flags &= ~CSM_CLUSTER_DEPORTED;
                                        hdr1->flags |=  CSM_CLUSTER_PRIVATE;
                                }
                                else if (container->flags & SCFLAG_CLUSTER_SHARED) {
                                        hdr1->flags &= ~CSM_CLUSTER_PRIVATE;
                                        hdr1->flags &= ~CSM_CLUSTER_DEPORTED;
                                        hdr1->flags |=  CSM_CLUSTER_SHARED;
                                }
                                else {
                                        hdr1->flags &= ~CSM_CLUSTER_PRIVATE;
                                        hdr1->flags &= ~CSM_CLUSTER_SHARED;
                                        hdr1->flags |=  CSM_CLUSTER_DEPORTED;
                                }
                                hdr2->flags = hdr1->flags;

                                disk_pdata->md1->flags |= SOFLAG_DIRTY;
                                disk_pdata->md2->flags |= SOFLAG_DIRTY;
                        }
                        else {
				LOG_SERIOUS("Could not get CSM private data from disk %s.\n", ld->name);
                                rc = ENOLINK;
                                break;
                        }

                }
        }

        if (!rc) {
                container->flags |= SCFLAG_DIRTY;

                // Did we have access to the container?
		if ( container_was_accessible == TRUE ) {

			if (EngFncs->list_count(container->objects_produced) > 0) {
				EngFncs->discard( container->objects_produced );
                        }

		}

		// Do we now have access to the container?
		if ( isa_accessible_container(container) == TRUE ) {

			LIST_FOR_EACH(container->objects_consumed, iter2, ld) {

				disk_pdata = get_csm_disk_private_data(ld);

				if (disk_pdata) {
					hdr1 = ((seg_private_data_t *)disk_pdata->md1->private_data)->hdr;
					seg = create_csm_data_segment( ld,
								       container,
								       hdr1->start_useable,
								       hdr1->end_useable - hdr1->start_useable + 1 );

					if (seg != NULL) {
						insert_csm_segment_into_list( ld->parent_objects, seg );
						add_disk_to_container( ld, container );
						get_csm_segment_devmap_info( seg, container );
					}
				}
			}

			if (!EngFncs->list_empty(container->objects_produced)) {
				list_anchor_t discover_list = EngFncs->copy_list(container->objects_produced);

				if (discover_list != NULL) {
					EngFncs->discover(discover_list);

					EngFncs->destroy_list(discover_list);
				}
			}
		}
        }
        else {
                memcpy(c_pdata, &saved_c_pdata, sizeof(container_private_data_t));
                memcpy(&c_pdata->nodeid, &original_nodeid, sizeof(ece_nodeid_t) );
                strncpy(container->name, original_name, EVMS_NAME_SIZE);
                container->flags = original_flags;

		// Restore metadata
                LIST_FOR_EACH( container->objects_consumed, iter, ld ){

                        disk_pdata = get_csm_disk_private_data(ld);

                        if (disk_pdata) {
                                hdr1 = ((seg_private_data_t *)disk_pdata->md1->private_data)->hdr;
                                hdr2 = ((seg_private_data_t *)disk_pdata->md2->private_data)->hdr;

                                strncpy(hdr1->container_name, container->name, EVMS_NAME_SIZE );
                                strncpy(hdr2->container_name, container->name, EVMS_NAME_SIZE );

                                memcpy(&hdr1->nodeid, &c_pdata->nodeid, sizeof(ece_nodeid_t));
                                memcpy(&hdr2->nodeid, &c_pdata->nodeid, sizeof(ece_nodeid_t));

                                if (container->flags & SCFLAG_CLUSTER_PRIVATE) {
                                        hdr1->flags &= ~CSM_CLUSTER_SHARED;
                                        hdr1->flags &= ~CSM_CLUSTER_DEPORTED;
                                        hdr1->flags |=  CSM_CLUSTER_PRIVATE;
                                }
                                else if (container->flags & SCFLAG_CLUSTER_SHARED) {
                                        hdr1->flags &= ~CSM_CLUSTER_PRIVATE;
                                        hdr1->flags &= ~CSM_CLUSTER_DEPORTED;
                                        hdr1->flags |=  CSM_CLUSTER_SHARED;
                                }
                                else {
                                        hdr1->flags &= ~CSM_CLUSTER_PRIVATE;
                                        hdr1->flags &= ~CSM_CLUSTER_SHARED;
                                        hdr1->flags |=  CSM_CLUSTER_DEPORTED;
                                }
                                hdr2->flags = hdr1->flags;

                                disk_pdata->md1->flags |= SOFLAG_DIRTY;
                                disk_pdata->md2->flags |= SOFLAG_DIRTY;
                        }
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}


int csm_get_plugin_functions(storage_container_t     * container,
                             function_info_array_t * * actions)
{
        return ENOSYS;
}


int csm_plugin_function(storage_container_t * container,
                        task_action_t         action,
                        list_anchor_t               objects,
                        option_array_t      * options)
{
        return ENOSYS;
}


/*
 *  Function: csm_find_container
 *
 *  This is a helper routine that is used to locate and return
 *  a container storage object ... matching the specified name.
 */
int  csm_find_container( char *search_name, storage_container_t  **container )
{
        int rc;
        list_anchor_t  container_list;
        storage_container_t  *csm_container=NULL;
        list_element_t iter;

        LOG_ENTRY();

        REQUIRE(search_name != NULL);
        REQUIRE(container != NULL);

        LOG_DEBUG("Looking for csm container: %s\n", search_name );

        *container = NULL;

        rc = get_csm_container_list(&container_list);

        if (!rc) {

                LIST_FOR_EACH( container_list, iter, csm_container ) {
                        if (strncmp(csm_container->name,search_name,EVMS_NAME_SIZE)==0) {
                                *container = csm_container;
                                break;
                        }
                }

                EngFncs->destroy_list( container_list );
        }

        if (*container != NULL) {
                rc = 0;
        }
        else {
                rc = ENODEV;
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: allocate_csm_container
 *
 *  This is a helper routine that wrappers the engine allocate api
 *  so that csm private data can be setup before returning the
 *  new container storage object to the caller.  This routine is
 *  called with the flags for the object so that at create time
 *  the object can be marked NEW and DIRTY but at discovery time
 *  we can simply clear the flags.
 */
int allocate_csm_container( uint flags, storage_container_t **container )
{
        int rc;
        storage_container_t *new_container=NULL;
        container_private_data_t  *c_pdata=NULL;


        LOG_ENTRY();

        rc = EngFncs->allocate_container( NULL, &new_container );

        if (rc == 0) {

                new_container->plugin      = csm_plugin_record_ptr;
                new_container->flags       = flags;

                new_container->private_data = calloc(1, sizeof(container_private_data_t) );

                if (new_container->private_data) {
                        c_pdata = (container_private_data_t *)new_container->private_data;
                        c_pdata->signature    = CSM_CONTAINER_PDATA_SIGNATURE;
                }
                else {
                        LOG_ERROR("error, call to malloc private storage area failed\n");
                        EngFncs->free_container( new_container );
                        new_container = NULL;
                }
        }

        *container = new_container;
        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: free_csm_container
 *
 *  This is a helper routine that simply frees csm private data
 *  before freeing the container storage object itself.
 */
void free_csm_container( storage_container_t *container )
{
        LOG_ENTRY();

        if (container->private_data) free(container->private_data);

        EngFncs->free_container( container );

        LOG_EXIT_VOID();
}


/*
 *  Function: delete_all_csm_container_private_data
 *
 *  This is a helper routine that simply frees csm private data
 *  that is referenced by csm container objects.  It is called
 *  by the csm plug-in cleanup api.
 */
void delete_all_csm_container_private_data(void)
{
        list_anchor_t container_list;
        int rc;
        storage_container_t *container;
        list_element_t iter;

        LOG_ENTRY();

        rc = get_csm_container_list(&container_list);

        if (!rc) {

                LIST_FOR_EACH( container_list, iter, container ) {
                        if (container->private_data) free(container->private_data);
                }

                EngFncs->destroy_list( container_list );
        }

        LOG_EXIT_VOID();
}


/*
 *  Function:  get_csm_container_list
 *
 *  This function will return a list of csm containers if
 *  any exist.
 */
int get_csm_container_list( list_anchor_t *list )
{
        int rc;
        list_anchor_t  container_list;

        LOG_ENTRY();

        REQUIRE(list != NULL);

        container_list = EngFncs->allocate_list();

        REQUIRE(container_list != NULL);

        rc = EngFncs->get_container_list(  csm_plugin_record_ptr,
                                           NULL,
                                           0,
                                           &container_list);

        if ( rc || EngFncs->list_count(container_list) == 0 ) {
                EngFncs->destroy_list( container_list );
                container_list=NULL;
                rc = ENODATA;
        }

        *list = container_list;

        LOG_EXIT_INT(rc);
        return rc;
}


