/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdos.so
 *
 *   File: sn.c
 */

#include <stdlib.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>
#include <errno.h>

#include <plugin.h>

#include "defsegmgr.h"
#include "sn.h"




/*
 *  Needing to generate unique 32 bit serial numbers for OS/2 dlat tables, and not having
 *  a serial number generator handy, we'll just corrupt the use of the engine API
 *  register_name() by passing a contrived char string that has the serial number
 *  embedded within it.
 *
 *  This is the structure used to generate 32 bit serial numbers for drivelink objects
 *
 */
typedef struct _SN_ {
        char               prefix[4];            /* set to our PREFIX_STRING                                    */
        u_int32_t          serial_number;        /* the actual serial number                                    */
        char               suffix;               /* so we can NULL terminate the pseudo character string        */
} sn_t;


#define SN_PREFIX_STRING "SN__"              /* Prefix used when creating pseudo names                      */




/*
 *  This routine unregisters serial numbers with the engine
 */
int seg_unregister_serial_number( u_int32_t  serial_number )
{
        sn_t  sn = { SN_PREFIX_STRING, serial_number, 0x00};

        return( EngFncs->unregister_name( (char *) &sn ) );
}


/*
 *  This routine registers serial numbers with the engine
 */
int seg_register_serial_number( u_int32_t  serial_number )
{
        sn_t  sn = { SN_PREFIX_STRING, serial_number, 0x00};

        if (sn.serial_number==BAD_SERIAL_NUMBER) {
                return EINVAL;
        }

        return( EngFncs->register_name( (char *) &sn ) );
}


/*
 *  This routine generates a new serial number and registers it
 *  with the engine, returning the SN to the caller.
 */
u_int32_t   seg_gen_serial_number( u_int32_t  guess )
{
        int    rc=EINVAL;
        sn_t   sn = {SN_PREFIX_STRING, guess, 0x00};
        char  *cptr = (char *)&sn.serial_number;
        int    attempts = 0;
        struct timeval  tv;

        LOG_ENTRY();

        // try to validate serial number character string with engine
        while ((rc != 0)&&(attempts<25)) {

                // dont allow any zeroes in the serial number itself, which would
                // NULL terminate the char string too soon!
                if (cptr[0] == 0x00)  cptr[0] = 0x01;
                if (cptr[1] == 0x00)  cptr[1] = 0x02;
                if (cptr[2] == 0x00)  cptr[2] = 0x03;
                if (cptr[3] == 0x00)  cptr[3] = 0x04;

                rc = EngFncs->validate_name( (char *) &sn );

                // guess again
                if (rc != 0) {
                        rc = gettimeofday(&tv,NULL);
                        if (rc==0) {
                                sn.serial_number += (u_int32_t) tv.tv_usec;
                                rc = -1;  // keep looping
                        }
                        else {
                                sn.serial_number += (u_int32_t) &guess;
                        }
                }

                ++attempts;
        }

        if (rc) {
                sn.serial_number = BAD_SERIAL_NUMBER;
        }

        LOG_EXIT_INT(sn.serial_number);
        return sn.serial_number;
}



