/*
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM Region Manager
 * File: evms2/engine/plugins/lvm/lvm_groups.c
 *
 * Description: This file contains all functions related to the discovery,
 *              creation and management of volume groups in the LVM region
 *              manager.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "lvmregmgr.h"


/*** Volume Group Memory Allocation Functions ***/


/**
 * lvm_allocate_volume_group
 * @vg:		VG metadata to attach to the new group
 * @vg_name:	Name for the new group
 * @disk_group:	CSM disk-group (if any)
 *
 * Allocate memory for a new LVM volume group and all of its sub-fields.
 * Initialize the appropriate fields based on the vg parameter. At this
 * time, the group will not have any segments or regions, but will have a
 * container.
 *
 * If this function fails in any way, the vg argument is free'd.
 **/
lvm_volume_group_t * lvm_allocate_volume_group(vg_disk_t * vg,
					       char * vg_name,
					       storage_container_t * disk_group)
{
	lvm_volume_group_t * new_group;
	char container_name[EVMS_NAME_SIZE+1] = {0};
	int rc;

	LOG_ENTRY();

	lvm_translate_vg_name_to_container_name(vg_name,
						container_name, disk_group);

	/* The volume group itself. */
	new_group = EngFncs->engine_alloc(sizeof(lvm_volume_group_t));
	if (!new_group) {
		LOG_CRITICAL("Memory error creating container %s\n", vg_name);
		EngFncs->engine_free(vg);
		goto out;
	}

	new_group->vg = vg;

	/* A container to represent this group to EVMS. */
	rc = EngFncs->allocate_container(container_name,
					 &new_group->container);
	if (rc) {
		LOG_CRITICAL("Memory error creating container %s\n", vg_name);
		lvm_deallocate_volume_group(new_group);
		new_group = NULL;
		goto out;
	}

	/* Array of LVM lv_disk_t structures. This will hold all the LV
	 * metadata for this group.
	 */
	new_group->lv_array_disk = EngFncs->engine_alloc(EVMS_LVM_MAX_LV_METADATA_SIZE);
	if (!new_group->lv_array_disk) {
		LOG_CRITICAL("Memory error creating LV metadata for container %s\n",
			     vg_name);
		lvm_deallocate_volume_group(new_group);
		new_group = NULL;
		goto out;
	}


        /* For newly created groups, we are always going to make sure the LV
         * array is properly aligned on disk, so we can set lv_array to be
         * lv_array_disk. For discovered groups, this value will simply be
         * over-written when we read in the LV array from disk.
	 */
        new_group->lv_array = new_group->lv_array_disk;

	/* Initialize the remaining group fields. */
	new_group->container->plugin		= my_plugin_record;
	new_group->container->flags		= 0;
	new_group->container->size		= 0;
	new_group->container->private_data	= new_group;
	new_group->container->disk_group	= disk_group;
	new_group->freespace			= NULL;
	new_group->geometry.block_size		= 1024;
	new_group->geometry.bytes_per_sector	= 512;
	new_group->move_extents			= 0;
	new_group->pv_count			= 0;
	new_group->volume_count			= 0;
	new_group->flags			= 0;

	LOG_DETAILS("Created container %s\n", new_group->container->name);

out:
	LOG_EXIT_PTR(new_group);
	return new_group;
}

/**
 * lvm_deallocate_volume_group
 *
 * Delete all memory used by this volume group.
 **/
void lvm_deallocate_volume_group(lvm_volume_group_t * group)
{
	int i;

	LOG_ENTRY();
	LOG_DETAILS("Deleting container %s\n", group->container->name);

	/* Remove this group from the global list. */
	EngFncs->remove_thing(lvm_group_list, group);

	/* Delete the UUID list strings. */
	for (i = 1; i <= MAX_PV; i++) {
		if (group->uuid_list[i]) {
			EngFncs->engine_free(group->uuid_list[i]);
			group->uuid_list[i] = NULL;
		}
	}

	/* Delete the array of lv_disk_t structs. */
	if (group->lv_array_disk) {
		EngFncs->engine_free(group->lv_array_disk);
		group->lv_array_disk = NULL;
		group->lv_array = NULL;
	}

	/* Delete all logical volumes. */
	for (i = 1; i <= MAX_LV; i++) {
		if (group->volume_list[i]) {
			lvm_deallocate_logical_volume(group->volume_list[i]);
			group->volume_list[i] = NULL;
		}
	}

	/* Delete the freespace volume. */
	if (group->freespace) {
		lvm_deallocate_logical_volume(group->freespace);
		group->freespace = NULL;
	}

	/* Delete all physical volumes. */
	for (i = 1; i <= MAX_PV; i++) {
		if (group->pv_list[i]) {
			lvm_deallocate_physical_volume(group->pv_list[i]);
			group->pv_list[i] = NULL;
		}
	}

	/* Delete the EVMS container. */
	if (group->container) {
		EngFncs->free_container(group->container);
		group->container = NULL;
	}

	/* Delete the VG metadata. */
	if (group->vg) {
		EngFncs->engine_free(group->vg);
		group->vg = NULL;
	}

	EngFncs->engine_free(group);

	LOG_EXIT_VOID();
}


/*** Volume Group Discovery/Assembly Functions ***/


/**
 * lvm_find_group_for_uuid
 * @vg_uuid:	UUID of VG to search for.
 * @vg_name:	Text name of the VG that goes with the UUID.
 * @group:	Location to store the pointer to the group.
 *
 * Search for the specified UUID in the global list of existing groups. If
 * it is found, return a pointer to that group. If it is not found, the
 * return pointer will be NULL.
 **/
static void lvm_find_group_for_uuid(char * vg_uuid,
				    lvm_volume_group_t ** group)
{
	list_element_t itr;
	int rc;

	LOG_ENTRY();

	LIST_FOR_EACH(lvm_group_list, itr, *group) {
		rc = memcmp(vg_uuid, (*group)->vg->vg_uuid, UUID_LEN);
		if (!rc) {
			LOG_EXIT_VOID();
			return;
		}
	}

	*group = NULL;
	LOG_EXIT_VOID();
}

/**
 * lvm_find_group_for_pv
 * @segment:	Storage object to read the VG metadata from.
 * @pv:		PV metadata for this segment
 * @group:	Location to store a pointer to the appropriate group.
 *
 * This is a discover-time function. It reads the VG metadata info for the
 * specified segment, and locates the appropriate group that owns that
 * segment. If that group does not already exist, it is created and
 * initialized.
 **/
int lvm_find_group_for_pv(storage_object_t * segment,
			  pv_disk_t * pv,
			  lvm_volume_group_t ** group)
{
	vg_disk_t * vg;
	int rc = EINVAL;

	LOG_ENTRY();

	*group = NULL;

	/* Check for an unassigned PV. */
	if (pv->vg_name[0] == 0) {
		/* Unassigned PVs will no longer be tracked in EVMS. In
		 * addition, we will delete the PV sector for these nodes.
		 */
		LOG_DETAILS("PV %s is unassigned. Deleting PV sector.\n",
			    segment->name);
		lvm_erase_pv(segment);
		goto out;
	}

	/* Read the VG on-disk info for this PV. If this succeeds, it
	 * allocates a new VG metadata structure.
	 */
	rc = lvm_read_vg(segment, pv, &vg);
	if (rc) {
		goto out;
	}

	/* Use the UUID from the VG metadata to determine if this group
	 * has already been discovered and constructed.
	 */
	lvm_find_group_for_uuid(vg->vg_uuid, group);

	if (!*group) {
		/* Create a new group entry and add to the global list. */
		*group = lvm_allocate_volume_group(vg, pv->vg_name,
						   segment->disk_group);
		if (!*group) {
			rc = ENOMEM;
			goto out;
		}

		EngFncs->insert_thing(lvm_group_list, *group,
				      INSERT_AFTER, NULL);
	} else {
		EngFncs->engine_free(vg);
	}

	/* Read in the UUID list for this group, if it isn't present. */
	rc = lvm_read_uuid_list(segment, pv, *group);
	if (rc) {
		LOG_ERROR("Error reading UUID list for container %s.\n",
			  pv->vg_name);
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}


/**
 * lvm_increment_container_size
 *
 * In all regular groups, the size is incremented by the total number
 * of sectors available in the PEs of this PV.
 **/
static void lvm_increment_container_size(lvm_volume_group_t * group,
					 lvm_physical_volume_t * pv_entry)
{
	LOG_ENTRY();

 	group->container->size += pv_entry->pv->pe_total * pv_entry->pv->pe_size;

	LOG_EXIT_VOID();
}


/**
 * lvm_decrement_container_size
 *
 * The same as lvm_increment_container_size, but in reverse.
 **/
static void lvm_decrement_container_size(lvm_volume_group_t * group,
					 lvm_physical_volume_t * pv_entry)
{
	LOG_ENTRY();

	group->container->size -= pv_entry->pv->pe_total * pv_entry->pv->pe_size;

	LOG_EXIT_VOID();
}

/**
 * lvm_add_pv_to_group_list
 * @pv_entry:	PV to add to the group
 * @group:	Group to add the PV to
 *
 * Add the specified physical volume to the pv list for the specified group
 * and the segment list for the group's container.
 *
 * NOTE: add_pv_to_group_list is used during discovery to assemble volume
 * volume groups. Therefore, this function does not modify the VG info. To
 * do that, use the add_new_pv_to_group function.
 */
int lvm_add_pv_to_group_list(lvm_physical_volume_t * pv_entry,
			     lvm_volume_group_t * group)
{
	storage_object_t * segment = pv_entry->segment;
	int rc;

	LOG_ENTRY();

	/* Make sure the PV's UUID is listed. */
	rc = lvm_verify_pv_uuid(pv_entry, group);
	if (rc) {
		LOG_SERIOUS("PV %s does not belong in container %s\n",
			    segment->name, group->container->name);
		goto out;
	}

	/* Make sure a PV isn't already occupying this spot in the list. This
	 * really should never happen, because PV numbers are supposed to be
	 * unique within a group. Plus, we have just verified its location
	 * based on its UUID.
	 */
	if (group->pv_list[pv_entry->number]) {
		LOG_SERIOUS("PVs %s and %s have conflicting numbers: %ld\n",
			    segment->name,
			    group->pv_list[pv_entry->number]->segment->name,
			    pv_entry->number);
		rc = EINVAL;
		goto out;
	}

	/* Add the PV's segment to the group's container. */
	rc = lvm_append_segment_to_container(segment, group->container);
	if (rc) {
		goto out;
	}

	/* Add the PV to the group and update group information. */
	group->pv_list[pv_entry->number] = pv_entry;
	group->pv_count++;
	pv_entry->group = group;
	lvm_increment_container_size(group, pv_entry);
	group->geometry.block_size = max(group->geometry.block_size,
					 pv_entry->segment->geometry.block_size);
	group->geometry.bytes_per_sector = max(group->geometry.bytes_per_sector,
					       pv_entry->segment->geometry.bytes_per_sector);
	
	LOG_DETAILS("Added object %s to container %s\n",
		    segment->name, group->container->name);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_add_new_pv_to_group
 *
 * This function adds a PV to a group. In addition to placing the PV on
 * the group's lists, it also updates all necessary information in the
 * VG and the PV.
 **/
int lvm_add_new_pv_to_group(lvm_physical_volume_t * pv_entry,
			    lvm_volume_group_t * group)
{
	storage_object_t * segment = pv_entry->segment;
	int rc;

	LOG_ENTRY();

	/* Update the PV info. Needs to be done
	 * before adding it to the group.
	 */
	rc = lvm_update_pv_for_group(pv_entry, group);
	if (rc) {
		goto out;
	}

	/* Have to enter the PV's UUID in the group's list before adding
	 * the PV to the group.
	 */
	lvm_set_uuid_list_entry(group, pv_entry->number, pv_entry->pv->pv_uuid);

	/* Put the PV on the group's lists. */
	rc = lvm_add_pv_to_group_list(pv_entry, group);
	if (rc) {
		LOG_SERIOUS("Error adding object %s to container %s\n",
			    segment->name, group->container->name);
		goto out;
	}

	/* Update the VG info. */
	group->vg->pv_cur++;
	group->vg->pv_act++;
	group->vg->pe_total += pv_entry->pv->pe_total;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_consolidate_pvs
 *
 * When a PV is removed from a group, any PVs above that one in the
 * group's list need to be shifted down to fill in any gaps in the
 * list.
 **/
static void lvm_consolidate_pvs(lvm_volume_group_t * group)
{
	int i, j;

	LOG_ENTRY();

	/* For each PV in the group. */
	for (i = 1; i <= MAX_PV; i++) {
		if (!group->pv_list[i]) {
			continue;
		}

		/* Find the index of the lowest-numbered, unused list entry. */
		for (j = i - 1; j > 0 && ! group->pv_list[j]; j--) {
			;
		}
		j++;

		/* If a blank entry was found, move this PV to this new spot. */
		if (!group->pv_list[j]) {
			group->pv_list[j] = group->pv_list[i];
			group->pv_list[j]->number = j;
			group->pv_list[j]->pv->pv_number = j;
			lvm_set_uuid_list_entry(group, j,
						group->pv_list[i]->pv->pv_uuid);
			lvm_clear_uuid_list_entry(group, i);
			group->pv_list[i] = NULL;
		}
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_remove_pv_from_group
 *
 * This function is the reverse of lvm_add_pv_to_group_list. It takes the
 * specified PV out of its group, and updates all necessary information.
 *
 * IMPORTANT: This function does not currently perform a lot of checking
 * before removing the PV. Don't call this unless all volumes that use
 * this PV have been removed.
 *
 * NOTE: unlike add_pv_to_group_list, remove_pv_from_group is never used
 * during discover. Therefore, we can safely update the VG and PV
 * information during this function.
 **/
int lvm_remove_pv_from_group(lvm_physical_volume_t * pv_entry)
{
	lvm_volume_group_t * group = pv_entry->group;
	int rc = EINVAL;

	LOG_ENTRY();

	if (group->pv_list[pv_entry->number] != pv_entry) {
		LOG_ERROR("PV/VG inconsistency in PV %s, VG %s.\n",
			  pv_entry->segment->name, group->container->name);
		goto out;
	}

	/* Take the segment out of the container's list. */
	rc = lvm_remove_segment_from_container(pv_entry->segment);
	if (rc) {
		goto out;
	}

	/* Take the PV out of the group's list and update group information. */
	group->pv_list[pv_entry->number] = NULL;
	lvm_decrement_container_size(group, pv_entry);
	group->pv_count--;

	/* Update VG info. */
	lvm_clear_uuid_list_entry(group, pv_entry->number);
	lvm_consolidate_pvs(group);
	group->vg->pv_cur--;
	group->vg->pv_act--;
	group->vg->pe_total -= pv_entry->pv->pe_total;

	/* Erase all of the group-specific metadata. */
	lvm_erase_group_metadata(pv_entry);

	/* Update the PV info. */
	lvm_update_pv_for_no_group(pv_entry);

	LOG_DETAILS("Removed object %s from container %s\n",
		    pv_entry->segment->name, group->container->name);

out:
	LOG_EXIT_INT(rc);
	return rc;
}


/*** Volume Group Creation Functions ***/


/**
 * lvm_find_free_vg_number
 *
 * During creation of a new group, we need to find the next available
 * group number. This is slightly more complicated now that the groups
 * are kept in an unordered list.
 **/
static int lvm_find_free_vg_number(void)
{
	lvm_volume_group_t * group;
	char numbers[MAX_VG] = {FALSE};
	list_element_t itr;
	int i;

	LOG_ENTRY();

	/* Record the numbers currently in use. */
	LIST_FOR_EACH(lvm_group_list, itr, group) {
		numbers[group->vg->vg_number] = TRUE;
	}

	/* Find the first unused number. */
	for (i = 0; i < MAX_VG; i++) {
		if (!numbers[i]) {
			LOG_EXIT_INT(i);
			return i;
		}
	}

	LOG_ERROR("Maximum number of LVM containers exist (%d).\n", MAX_VG);
	LOG_EXIT_INT(-1);
	return -1;
}

/**
 * lvm_check_vg_name
 *
 * Run through all the existing volume groups and check if the specified
 * name is currently being used.
 **/
int lvm_check_vg_name(char * vg_name)
{
	lvm_volume_group_t * group;
	char this_vg_name[NAME_LEN] = {0};
	list_element_t itr;
	int rc = EINVAL;

	LOG_ENTRY();

	if (vg_name[0] == 0) {
		LOG_ERROR("Must specify a name for the new container.\n");
		goto out;
	}

	LIST_FOR_EACH(lvm_group_list, itr, group) {
		lvm_translate_container_name_to_vg_name(group, this_vg_name);
		if (! strncmp(vg_name, this_vg_name, NAME_LEN)) {
			MESSAGE(_("%s is already in use as a container name\n"),
				vg_name);
			rc = EEXIST;
			goto out;
		}
	}
	rc = 0;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_check_pe_size
 *
 * During volume group creation, the user-specified PE size must be a power
 * of 2 between 8k and 16GB. If it is in range, but not a power of 2, round
 * down until a suitable value is found. If it is out of range, reset to
 * the nearest limit.
 **/
int lvm_check_pe_size(u_int32_t * pe_size)
{
	unsigned long mask = 1;
	int rc = 0;

	LOG_ENTRY();

	if (*pe_size < LVM_MIN_PE_SIZE) {
		LOG_WARNING("PE size %d below lower limit.\n", *pe_size);
		LOG_WARNING("Resetting PE size to %ld.\n", LVM_MIN_PE_SIZE);
		*pe_size = LVM_MIN_PE_SIZE;
		rc = -1;
	} else if (*pe_size > LVM_MAX_PE_SIZE) {
		LOG_WARNING("PE size %d above upper limit.\n", *pe_size);
		LOG_WARNING("Resetting PE size to %ld.\n", LVM_MAX_PE_SIZE);
		*pe_size = LVM_MAX_PE_SIZE;
		rc = -1;
	} else if ((*pe_size & (*pe_size - 1)) != 0) {
		LOG_WARNING("PE size %d not a power of 2.\n", *pe_size);
		while ((*pe_size & (*pe_size - 1)) != 0) {
			*pe_size =  *pe_size & ~mask;
			mask = mask << 1;
		}
		LOG_WARNING("Rounding PE size down to %d.\n", *pe_size);
		rc = -1;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_initialize_new_vg
 *
 * During creation of a new volume group, we need to fill in the VG disk
 * structure with appropriate values, based on the user options.
 */
vg_disk_t * lvm_initialize_new_vg(u_int32_t pe_size)
{
	vg_disk_t * vg;
	int rc;

	LOG_ENTRY();

	vg = EngFncs->engine_alloc(sizeof(vg_disk_t));
	if (!vg) {
		LOG_CRITICAL("Memory error creating VG metadata\n");
		goto out;
	}

	/* Get a new UUID for this group. */
	memset(vg->vg_uuid, 0, NAME_LEN);
	rc = lvm_create_uuid(vg->vg_uuid);
	if (rc) {
		EngFncs->engine_free(vg);
		vg = NULL;
		goto out;
	}

	/* Get a new number for this group. */
	vg->vg_number = lvm_find_free_vg_number();
	if (vg->vg_number < 0) {
		EngFncs->engine_free(vg);
		vg = NULL;
		goto out;
	}

	/* Fill in remaining fields. */
	vg->vg_access	= VG_READ | VG_WRITE;
	vg->vg_status	= VG_ACTIVE | VG_EXTENDABLE;
	vg->lv_max	= MAX_LV;
	vg->lv_cur	= 0;
	vg->lv_open	= 0;
	vg->pv_max	= MAX_PV;
	vg->pv_cur	= 0;
	vg->pv_act	= 0;
	vg->dummy	= 0;
	vg->vgda	= 0;
	vg->pe_size	= pe_size;
	vg->pe_total	= 0;
	vg->pe_allocated= 0;	
	vg->pvg_total	= 0;

out:
	LOG_EXIT_PTR(vg);
	return vg;
}


/*** Volume Group Membership Functions ***/


/**
 * lvm_check_segment_for_group
 *
 * Examine a single segment to determine if it is currently owned by
 * the specified LVM volume group. If the specified group is NULL, it
 * checks that the segment is not owned by LVM at all.
 **/
int lvm_check_segment_for_group(storage_object_t * segment,
				lvm_volume_group_t * group)
{
	int rc = 0;

	LOG_ENTRY();

	if (group) {
		if (segment->consuming_container != group->container) {
			LOG_WARNING("Object %s is not in container %s\n",
				    segment->name, group->container->name);
			rc = EINVAL;
			goto out;
		}
	} else if (segment->consuming_container &&
		   segment->consuming_container->plugin == my_plugin_record) {
		LOG_WARNING("Object %s is owned by LVM\n", segment->name);
		rc = EINVAL;
		goto out;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_check_segment_for_group_inclusion
 *
 * This function examines a single segment to determine if it is
 * currently available to be added to an LVM volume group. This can be
 * easily determined by seeing if the segment has any parent objects.
 **/
int lvm_check_segment_for_group_inclusion(storage_object_t * segment,
					  lvm_volume_group_t * group)
{
	lvm_logical_volume_t * volume;
	int rc = 0;

	LOG_ENTRY();

       	/* Is this segment already in a container? */
	if (segment->consuming_container) {
		LOG_WARNING("Object %s is in a non-LVM container %s\n",
			    segment->name, segment->consuming_container->name);
		rc = EINVAL;
	}

	/* Not in a container. Does the segment have any parent objects? */
	else if (! EngFncs->list_empty(segment->parent_objects)) {
		LOG_WARNING("Object %s has parent objects.\n", segment->name);
		rc = EINVAL;
	}

	/* No parents. Is this segment a compatibility volume? */
	else if (segment->volume) {
		LOG_WARNING("Object %s is volume %s. Must revert the volume first.\n",
			    segment->name, segment->volume->name);
		rc = EINVAL;
	}

	/* Not a compatibility volume. Is this an LVM region
	 * from the same group we are trying to add to?
	 */
	else if (segment->plugin == my_plugin_record) {
		volume = segment->private_data;
		if (volume && volume->group == group) {
			LOG_WARNING("Object %s is an LVM region from container %s.\n",
				    segment->name, group->container->name);
			LOG_WARNING("Cannot add %s as a new PV to %s.\n",
				    segment->name, group->container->name);
			rc = EINVAL;
		}
	}

	/* The segment must be a minimum size to allow for enough space for
	 * the LVM metadata.
	 */
	else if (segment->size < LVM_MIN_PV_SIZE) {
		LOG_WARNING("Object %s is only %"PRIu64" sectors.\n",
			    segment->name, segment->size);
		LOG_WARNING("Minimum size for LVM PVs is %lu sectors.\n",
			    LVM_MIN_PV_SIZE);
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_transfer_segment_to_group
 *
 * This function does the work of moving an existing physical volume
 * from its current group to a new group during group creation. If the
 * segment is not in a group to start with, a new physical volume is
 * created for it.
 **/
int lvm_transfer_segment_to_group(storage_object_t * segment,
				  lvm_volume_group_t * group)
{
	lvm_physical_volume_t * pv_entry;
	int rc;

	LOG_ENTRY();

	/* Get the physical volume for this segment. If it
	 * isn't a PV yet, create a new one.
	 */
	pv_entry = lvm_get_pv_for_segment(segment);
	if (!pv_entry) {
		pv_entry = lvm_create_pv_from_segment(segment);
		if (!pv_entry) {
			rc = EINVAL;
			goto out;
		}
	} else {
		/* Remove the physical volume from its current group. */
		rc = lvm_remove_pv_from_group(pv_entry);
		if (rc) {
			goto out;
		}
	}

	/* Add the physical volume to the new group. */
	rc = lvm_add_new_pv_to_group(pv_entry, group);
	if (rc) {
		LOG_SERIOUS("Error adding object %s to container %s\n",
			    segment->name, group->container->name);
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_fix_group_after_pv_removal
 *
 * If PVs were missing during discovery, the user has the choice of
 * deleting those PVs permanently from the group. After that has been
 * done, the VG metadata must be fixed up as well to reflect the
 * actual number of PVs and PEs in the group.
 **/
void lvm_fix_group_after_pv_removal(lvm_volume_group_t * group)
{
	int i;

	LOG_ENTRY();

	/* Update the count of PVs. */
	group->vg->pv_cur = group->pv_count;
	group->vg->pv_act = group->pv_count;

	/* The pe_total and pe_allocated fields must be recalculated
	 * based on the remaining PVs in the group.
	 */
	group->vg->pe_total = 0;
	group->vg->pe_allocated = 0;
	for (i = 1; i <= MAX_PV; i++) {
		if (group->pv_list[i]) {
			group->vg->pe_total += group->pv_list[i]->pv->pe_total;
			group->vg->pe_allocated += group->pv_list[i]->pv->pe_allocated;
		}
	}

	/* Since the pe_total field changed, we need to
	 * rebuild the freespace for this container.
	 */
	lvm_update_freespace_volume(group);

	/* The group has changed. Mark it dirty. */
	group->container->flags |= SCFLAG_DIRTY;

	LOG_EXIT_VOID();
}


/*** Volume Group Expand/Shrink Functions ***/


/**
 * lvm_can_add_object
 *
 * Can the specified object be added to the specified container.
 **/
int lvm_can_add_object(storage_object_t * object,
		       storage_container_t * container)
{
	lvm_volume_group_t * group = container->private_data;
	u_int32_t pe_size;
	int rc;

	LOG_ENTRY();

	/* Check that the segment can be added to a group. */
	rc = lvm_check_segment_for_group_inclusion(object, group);
	if (rc) {
		goto out;
	}

	/* Make sure the group has space for a new PV. */
	if (group->vg->pv_cur >= MAX_PV) {
		LOG_WARNING("Container %s already has maximum number of PVs (%d).\n",
			    container->name, MAX_PV);
		rc = EINVAL;
		goto out;
	}

	/* Check that the segment is big enough to add to this group. */
	pe_size = group->vg->pe_size;
	rc = lvm_check_segment_for_pe_size(object, &pe_size);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_can_remove_object
 *
 * Can the specified object be removed from its container.
 **/
int lvm_can_remove_object(storage_object_t * object)
{
	storage_container_t * container = object->consuming_container;
	lvm_volume_group_t * group;
	lvm_physical_volume_t * pv_entry;
	int rc = 0;

	LOG_ENTRY();

	/* Check that the segment is part of an LVM container. */
	if (!container) {
		LOG_WARNING("Object %s is not in a container.\n", object->name);
		rc = EINVAL;
		goto out;
	}
	if (container->plugin != my_plugin_record) {
		LOG_WARNING("Object %s is in non-LVM container %s.\n",
			    object->name, container->name);
		rc = EINVAL;
		goto out;
	}
	group = container->private_data;
	
	/* Make sure this PV isn't the last one in the group. To take the last
	 * PV out, use the lvm_destroy_container function. We don't want empty
	 * groups hanging around.
	 */
	if (group->pv_count == 1) {
		LOG_WARNING("Cannot remove the last object from container %s\n",
			    container->name);
		LOG_WARNING("Use delete_container to remove this container.\n");
		rc = EINVAL;
		goto out;
	}

	/* Get the PV for this segment. */
	pv_entry = lvm_get_pv_for_segment(object);
	if (!pv_entry) {
		LOG_ERROR("Could not find PV entry for object %s\n",
			  object->name);
		rc = EINVAL;
		goto out;
	}

	/* Make sure the PV is not in use by any volumes. */
	if (pv_entry->pv->lv_cur) {
		LOG_WARNING("Object %s is in use by %d regions\n",
			    object->name, pv_entry->pv->lv_cur);
		rc = EINVAL;
		goto out;
	}

	/* Make sure no extents are being moved to this PV. */
	if (pv_entry->move_extents > 0) {
		LOG_WARNING("Object %s is scheduled to have %d extents moved to it\n",
			    object->name, pv_entry->move_extents);
		rc = EINVAL;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_can_shrink_pv
 *
 * The specified PV can be shrunk if it has available extents at the end of
 * the object. If the PV is shrinkable, set the shrink_limit to the amount
 * of space that could be freed.
 **/
int lvm_can_shrink_pv(lvm_physical_volume_t *pv_entry, u_int64_t *shrink_limit)
{
	lvm_volume_group_t *group = pv_entry->group;
	u_int32_t i, unused_extents = 0;
	int rc = EBUSY;

	LOG_ENTRY();
	LOG_DEBUG("Checking if PV %s can be shrunk.\n",
		  pv_entry->segment->name);

	/* Work backwards from the end of the PE-map
	 * and see how many extents are unused.
	 */
	for (i = pv_entry->pv->pe_total; i > 0; i--) {
		if (lvm_pe_is_available(pv_entry->pe_map + (i-1))) {
			unused_extents++;
		} else {
			break;
		}
	}

	if (unused_extents) {
		if (unused_extents == pv_entry->pv->pe_total) {
			/* Always need at least one extent per PV. */
			unused_extents--;
		}
		LOG_DEBUG("PV %s can shrink by %u extents.\n",
			  pv_entry->segment->name, unused_extents);
		*shrink_limit = unused_extents * group->vg->pe_size;
		rc = 0;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * check_extra_extents_count
 *
 * Verify that there is sufficient space in the metadata PE table to hold the
 * specified number of additional extents. If not enough space is available,
 * reduce new_pes appropriately.
 **/
void check_extra_extents_count(lvm_physical_volume_t *pv_entry,
			       u_int32_t *new_pes)
{
	u_int64_t pe_start, pe_map_start, pe_map_size;
	u_int32_t total_pes, available_pes;

	LOG_ENTRY();

	pe_map_start = bytes_to_sectors(pv_entry->pv->pe_on_disk.base);
	pe_start = lvm_get_pe_start(pv_entry);
	pe_map_size = pe_start - pe_map_start;

	total_pes = (pe_map_size << EVMS_VSECTOR_SIZE_SHIFT) /
		    sizeof(pe_disk_t);
	available_pes = total_pes - pv_entry->pv->pe_total;

	if (*new_pes > available_pes) {
		LOG_WARNING("Requested expanding PV %s by %u PEs, but only "
			    "have metadata space available for %u new PEs.\n",
			    pv_entry->segment->name, *new_pes, available_pes);
		*new_pes = available_pes;
	}

	LOG_EXIT_VOID();
}

/**
 * lvm_expand_pv_in_container
 *
 * Tell the specified PV in this container to expand using the input-objects
 * and options. After it expands, update the PV's internal data to reflect
 * the increased space.
 **/
int lvm_expand_pv_in_container(storage_container_t *container,
			       storage_object_t *consumed_object,
			       storage_object_t *expand_object,
			       list_anchor_t input_objects,
			       option_array_t *options)
{
	lvm_volume_group_t *group = container->private_data;
	u_int32_t new_extents, old_pe_count, new_pe_count;
	lvm_physical_extent_t *old_pe_map, *new_pe_map;
	u_int64_t old_size, new_size, delta_size;
	lvm_physical_volume_t *pv_entry;
	u_int32_t i;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Expanding object %s in container %s.\n",
		  consumed_object->name, container->name);

	/* Make sure the consumed-object is really a PV. */
	if (consumed_object->consuming_container != container) {
		LOG_ERROR("Attempt to expand object %s which isn't a PV in "
			  "container %s.\n", consumed_object->name,
			  container->name);
		rc = EINVAL;
		goto out;
	}

	pv_entry = lvm_get_pv_for_segment(consumed_object);
	if (!pv_entry) {
		rc = EINVAL;
		goto out;
	}

	/* Tell the PV object to expand. Save the old size. */
	old_size = consumed_object->size;
	rc = consumed_object->plugin->functions.plugin->expand(consumed_object,
							       expand_object,
							       input_objects,
							       options);
	if (rc) {
		LOG_ERROR("Error expanding object %s in container %s.\n",
			  consumed_object->name, container->name);
		goto out;
	}

	/* Calculate the number of new extents after the object expands. */
	new_size = consumed_object->size;
	delta_size = new_size - old_size;
	new_extents = delta_size / group->vg->pe_size;

	/* Check that there's enough metadata space for the number
	 * of new extents. It will round down if necessary.
	 */
	check_extra_extents_count(pv_entry, &new_extents);

	/* Save the old PE map and allocate a new PE map. */
	old_pe_map = pv_entry->pe_map;
	old_pe_count = pv_entry->pv->pe_total;
	new_pe_count = old_pe_count + new_extents;
	pv_entry->pv->pe_total = new_pe_count;
	pv_entry->pv->pv_size = new_size;
	pv_entry->pe_map = NULL;

	rc = lvm_allocate_pe_map(pv_entry);
	if (rc) {
		pv_entry->pv->pe_total = old_pe_count;
		pv_entry->pv->pv_size = old_size;
		pv_entry->pe_map = old_pe_map;
		goto out;
	}
	new_pe_map = pv_entry->pe_map;

	/* Update the group information. */
	group->vg->pe_total += new_extents;
	container->size += new_extents * group->vg->pe_size;

	/* Copy the contents of the old PE map to the new PE map.
	 * For each old PE, point that PE's LE at the new PE.
	 */
	for (i = 0; i < old_pe_count; i++) {
		new_pe_map[i].pe = old_pe_map[i].pe;
		new_pe_map[i].le = old_pe_map[i].le;
		new_pe_map[i].new_le = old_pe_map[i].new_le;

		if (old_pe_map[i].le &&
		    old_pe_map[i].le->pe == &(old_pe_map[i])) {
			old_pe_map[i].le->pe = &(new_pe_map[i]);
		}
		if (old_pe_map[i].new_le &&
		    old_pe_map[i].new_le->new_pe == &(old_pe_map[i])) {
			old_pe_map[i].new_le->new_pe = &(new_pe_map[i]);
		}
	}

	/* The freespace volume needs to be adjusted. */
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_SERIOUS("Error updating freespace for container %s\n",
			    container->name);
		goto out;
	}

	EngFncs->engine_free(old_pe_map);
	container->flags |= SCFLAG_DIRTY;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * lvm_shrink_pv_in_container
 *
 * Tell the specified PV in this container to shrink using the input-objects
 * and options. After it shrinks, update the PV's internal data to reflect
 * the decreased space.
 **/
int lvm_shrink_pv_in_container(storage_container_t *container,
			       storage_object_t *consumed_object,
			       storage_object_t *shrink_object,
			       list_anchor_t input_objects,
			       option_array_t *options)
{
	u_int64_t shrink_limit, old_size, new_size, delta_size;
	u_int32_t removed_extents, old_pe_count, new_pe_count;
	lvm_volume_group_t *group = container->private_data;
	lvm_physical_extent_t *old_pe_map, *new_pe_map;
	lvm_physical_volume_t *pv_entry;
	u_int32_t i;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Shrinking object %s in container %s.\n",
		  consumed_object->name, container->name);

	/* Make sure the consumed-object is really a PV. */
	if (consumed_object->consuming_container != container) {
		LOG_ERROR("Attempt to shrink object %s which isn't a PV in "
			  "container %s.\n", consumed_object->name,
			  container->name);
		rc = EINVAL;
		goto out;
	}

	pv_entry = lvm_get_pv_for_segment(consumed_object);
	if (!pv_entry) {
		rc = EINVAL;
		goto out;
	}

	/* Check that we can allow this PV to shrink. */
	rc = lvm_can_shrink_pv(pv_entry, &shrink_limit);
	if (rc) {
		LOG_ERROR("Cannot shrink object %s.\n", consumed_object->name);
		goto out;
	}

	/* Tell the PV object to shrink. Save the old size. */
	old_size = consumed_object->size;
	rc = consumed_object->plugin->functions.plugin->shrink(consumed_object,
							       shrink_object,
							       input_objects,
							       options);
	if (rc) {
		LOG_ERROR("Error shrinking object %s in container %s.\n",
			  consumed_object->name, container->name);
		goto out;
	}

	/* Calculate the number of extents to remove after the object shrunk. */
	new_size = consumed_object->size;
	delta_size = old_size - new_size;
	removed_extents = (delta_size / group->vg->pe_size) + 
			  ((delta_size % group->vg->pe_size) ? 1 : 0);

	/* Save the old PE map and allocate a new PE map. */
	old_pe_map = pv_entry->pe_map;
	old_pe_count = pv_entry->pv->pe_total;
	new_pe_count = old_pe_count - removed_extents;
	pv_entry->pv->pe_total = new_pe_count;
	pv_entry->pv->pv_size = new_size;
	pv_entry->pe_map = NULL;

	rc = lvm_allocate_pe_map(pv_entry);
	if (rc) {
		pv_entry->pv->pe_total = old_pe_count;
		pv_entry->pv->pv_size = old_size;
		pv_entry->pe_map = old_pe_map;
		goto out;
	}
	new_pe_map = pv_entry->pe_map;

	/* Update the group information. */
	group->vg->pe_total -= removed_extents;
	container->size -= removed_extents * group->vg->pe_size;

	/* Copy the contents of the old PE map to the new PE map.
	 * For each old PE, point that PE's LE at the new PE.
	 */
	for (i = 0; i < new_pe_count; i++) {
		new_pe_map[i].pe = old_pe_map[i].pe;
		new_pe_map[i].le = old_pe_map[i].le;
		new_pe_map[i].new_le = old_pe_map[i].new_le;

		if (old_pe_map[i].le &&
		    old_pe_map[i].le->pe == &(old_pe_map[i])) {
			old_pe_map[i].le->pe = &(new_pe_map[i]);
		}
		if (old_pe_map[i].new_le &&
		    old_pe_map[i].new_le->new_pe == &(old_pe_map[i])) {
			old_pe_map[i].new_le->new_pe = &(new_pe_map[i]);
		}
	}

	/* The freespace volume needs to be adjusted. */
	rc = lvm_update_freespace_volume(group);
	if (rc) {
		LOG_SERIOUS("Error updating freespace for container %s\n",
			    container->name);
		goto out;
	}


	/* Update the container's size. */
	EngFncs->engine_free(old_pe_map);
	container->flags |= SCFLAG_DIRTY;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

