/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: LVM2 Plugin
 * File: evms2/engine/plugins/lvm2/vgda.c
 *
 * Routines for managing the text VG descriptor.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <time.h>
#include <sys/utsname.h>
#include <plugin.h>
#include "lvm2.h"

#define NEXT_CHAR(p) (((*(p) == '\\') && (*((p)+1) != '\0')) ? (p)+=2 : (p)++)
#define IS_WHITE_SPACE(c) (((c) == ' ') || ((c) == '\t') || \
			   ((c) == '\n') || ((c) == '\r'))
#define IS_KEY_VALUE_DELIMITER(c) (((c) == '=') || ((c) == ':'))

#define parse_error_msg()  do { LOG_ERROR("Parse error!\n"); } while (0);

static key_value_t *parse_value(char **pp, char *key);

static char *skip_white_space(char *p, char *extra_chars)
{
	char c = *p;

	LOG_ENTRY();

	while (c &&
	       (IS_WHITE_SPACE(c) ||
		c == '#' ||
	        (extra_chars &&
		 strchr(extra_chars, c) != NULL))) {
		/* Comments are white space. */
		if (c == '#') {
			while (c != '\n') {
				NEXT_CHAR(p);
				c = *p;
			}
		}

		NEXT_CHAR(p);
		c = *p;
	}

	LOG_EXIT_PTR(p);
	return p;
}

/* Convert a two digit ASCII hex string to an integer. */
static int hex_byte(char *p)
{
	int i, value = 0;

	LOG_ENTRY();

	for (i = 0; i < 2; i++) {
		value <<= 8;
		if ((*p >= '0') && (*p <= '9')) {
			value += *p - '0';
		} else if ((*p >= 'a') && (*p <= 'f')) {
			value += (*p - 'a') + 10;
		} else if ((*p >= 'A') && (*p <= 'F')) {
			value += (*p - 'A') + 10;
		} else {
			parse_error_msg();
			value = 0;
			break;
		}
	}

	LOG_EXIT_INT(value);
	return value;
}

/* Convert a three digit ASCII octal string to an integer. */
static int oct_byte(char *p)
{
	int i, value = 0;

	LOG_ENTRY();

	for (i = 0; i < 3; i++) {
		value <<= 8;
		if ((*p >= '0') && (*p <= '7')) {
			value += *p - '0';
		} else {
			parse_error_msg();
			value = 0;
			break;
		}
	}

	LOG_EXIT_INT(value);
	return value;
}

static void compress_escapes(char *string)
{
	LOG_ENTRY();

	while (*string) {
		if (*string == '\\') {
			char *p, *p2;

			p = string;
			p2 = p + 2;
			/* Handle special escape characters. */
			switch (*(p+1)) {
			case 'n':
				*p = '\n';
				break;
			case 'r':
				*p = '\r';
				break;
			case 'b':
				*p = '\b';
				break;
			case 't':
				*p = '\t';
				break;
			case 'f':
				*p = '\f';
				break;
			case 'x':
				*p = hex_byte(p+1);
				p2 = p + 4;
				break;
			case '0':
			case '1':
			case '2':
			case '3':
			case '4':
			case '5':
			case '6':
			case '7':
			case '8':
			case '9':
				*p = oct_byte(p+1);
				p2 = p + 4;
				break;
			default:
				*p = *(p+1);
			}

			for (p++; *p2; p++, p2++) {
				*p = *p2;
			}
		}

		string++;
	}

	LOG_EXIT_VOID();
}

static char *find_string_end(char *p, char *delimiters)
{
	char quote;

	LOG_ENTRY();

	while (*p) {
		if (IS_WHITE_SPACE(*p)) {
			break;
		}

		if (delimiters) {
			char *p2 = delimiters;
			while (*p2 && (*p != *p2)) {
				p2++;
			}

			if (*p2) {
				/* Found a delimiter. */
				break;
			}
		}

		switch (*p) {
		case '\'':
		case '\"':
			quote = *p;
			p++;
			while (*p && *p != quote) {
				/* Skip over escaped characters. */
				if (*p == '\\') {
					/* Except null. */
					if (*(p+1) != '\0') {
						p++;
					}
				}
				p++;
			}

			if (*p == quote) {
				/* Skip over the closing '\''. */
				p++;
			} else {
				parse_error_msg();
			}
			break;

		case '\\':
			/* Skip over escaped character. */
			if (*p == '\\') {
				/* Except nul. */
				if (*(p+1) != '\0') {
					p++;
				}
			}
			p++;
			break;

		default:
			p++;
		}
	}

	LOG_EXIT_PTR(p);
	return p;
}

static char *get_string(char *p)
{
	char *p2 = find_string_end(p, NULL);
	int str_len = p2 - p;
	char *string = EngFncs->engine_alloc(str_len + 1);
	char quote;
	boolean finished;

	LOG_ENTRY();

	memcpy(string, p, str_len);
	string[str_len] = '\0';

	/* Strip off any outside quote marks. */
	p = string;
	finished = FALSE;
	while (*p && !finished) {
		switch (*p) {
		case '\'':
		case '\"':
			quote = *p;
			/* Remove the opening quote by
			 * copying the string over it.
			 */
			for (p2 = p; *p2; p2++) {
				*p2 = *(p2+1);
			}

			/* Search for the closing quote. */
			while (*p && *p != quote) {
				/* NEXT_CHAR() skips over escaped
				 * characters properly.
				 */
				NEXT_CHAR(p);
			}

			/* Remove the closing quote by
			 * copying the string over it.
			 */
			for (p2 = p; *p2; p2++) {
				*p2 = *(p2+1);
			}
			break;

		default:
			finished = TRUE;
		}
	}

	compress_escapes(string);

	LOG_EXIT_PTR(string);
	return string;
}

static char *parse_key(char **pp)
{
	char *p = *pp;
	char *key = NULL;

	LOG_ENTRY();

	p = skip_white_space(p, NULL);
	if (*p) {
		key = p;

		/* Skip over the key. */
		p = find_string_end(p, "=:");

		/* Terminate the key. */
		if (*p) {
			*p = '\0';
			p++;
		}

		compress_escapes(key);
	}

	*pp = p;
	LOG_EXIT_PTR(key);
	return key;
}

static key_value_t *parse_section(char **pp, char *key)
{
	char *p = *pp;
	key_value_t *kv, *kv_child, *next = NULL;

	LOG_ENTRY();

	kv = EngFncs->engine_alloc(sizeof(key_value_t));
	kv->key = EngFncs->engine_strdup(key);
	kv->type = SECTION;

	/* Skip over the '{'.*/
	p++;
	p = skip_white_space(p, NULL);

	while (*p && *p != '}') {
		key = parse_key(&p);
		if (key) {
			if (*p) {
				p = skip_white_space(p, "=:");
				if (*p) {
					kv_child = parse_value(&p, key);
					if (kv_child) {
						/* Add this "child" node at the
						 * end of the "section" list.
						 */
						if (!kv->value.section) {
							kv->value.section =
								next = kv_child;
						} else {
							next->next = kv_child;
							next = kv_child;
						}
					}
				} else {
					parse_error_msg();
				}
			} else {
				parse_error_msg();
			}
		}

		p = skip_white_space(p, NULL);
	}

	if (*p) {
		/* Skip over the '}'.*/
		p++;
	} else {
		parse_error_msg();
	}

	*pp = p;
	LOG_EXIT_PTR(kv);
	return kv;
}

static key_value_t *parse_array(char **pp, char *key)
{
	int array_count = 8;
	char *p = *pp;
	char *p2;
	char c;
	key_value_t *kv;

	LOG_ENTRY();

	kv = EngFncs->engine_alloc(sizeof(*kv));
	kv->key = EngFncs->engine_strdup(key);
	kv->type = ARRAY;
	kv->value.array = EngFncs->engine_alloc(sizeof(*(kv->value.array)));
	kv->value.array->strings = EngFncs->engine_alloc(sizeof(char *) * array_count);

	/* Skip over the '['.*/
	p++;
	p = skip_white_space(p, NULL);
	while (*p && *p != ']') {

		/* An array entry can itself contain brackets. */
		int bracket_count = 0;

		p2 = find_string_end(p, "[],");
		while (*p2 == '[' || bracket_count != 0) {
			switch (*p2) {
			case '[':
				bracket_count ++;
				p2++;
				break;

			case ']':
				bracket_count--;
				p2++;
				break;

			default:
				/* Hit a ',' or white space. We are
				 * at the end of the entry.
				 */
				bracket_count = 0;
			}

			if (bracket_count != 0) {
				p2 = find_string_end(p2, "[],");
			}
		}

		c = *p2;
		*p2 = '\0';

		if (kv->value.array->count >= array_count) {
			/* Need more space in the array. */
			array_count += 8;
			kv->value.array->strings =
				EngFncs->engine_realloc(kv->value.array->strings,
							(sizeof(char *) * array_count));
		}

		kv->value.array->strings[kv->value.array->count] = get_string(p);

		if (kv->value.array->strings[kv->value.array->count]) {
			kv->value.array->count++;
		} else {
			LOG_CRITICAL("Error making a copy of string %s.\n", p);
			break;
		}

		*p2 = c;
		p = skip_white_space(p2, ",");
	}

	if (kv && kv->value.array->count < array_count) {
		/* NULL terminate the string array for safety. */
		kv->value.array->strings[kv->value.array->count] = NULL;
	}

	if (*p == ']') {
		/* Skip over the ']'.*/
		p++;
	} else {
		parse_error_msg();
	}

	*pp = p;
	LOG_EXIT_PTR(kv);
	return kv;
}

static key_value_t *parse_string(char **pp, char *key)
{
	key_value_t *kv = NULL;

	LOG_ENTRY();

	kv = EngFncs->engine_alloc(sizeof(*kv));
	kv->key = EngFncs->engine_strdup(key);
	kv->type = VALUE;
	kv->value.string = get_string(*pp);

	*pp = find_string_end(*pp, NULL);

	LOG_EXIT_PTR(kv);
	return kv;
}

static key_value_t *parse_value(char **pp, char *key)
{
	key_value_t *kv = NULL;

	LOG_ENTRY();

	/* *(*p)) is guaranteed to point to a non-null character. */
	switch (*(*pp)) {
	case '{':
		kv = parse_section(pp, key);
		break;

	case '[':
		kv = parse_array(pp, key);
		break;

	case '}':
	case ']':
		parse_error_msg();
		break;

	default:
		kv = parse_string(pp, key);
	}

	LOG_EXIT_PTR(kv);
	return kv;
}

/**
 * parse_vg_metadata
 *
 * Transform the text buffer from the VG metadata into a tree of key-value
 * nodes representing the VG information. This code is basically the same
 * as the engine code that parses the EVMS config file.
 **/
key_value_t *parse_vg_metadata(char *buffer)
{
	key_value_t *kv, *next = NULL, *tree = NULL;
	char *key;

	LOG_ENTRY();

	while (*buffer) {
		key = parse_key(&buffer);
		if (key) {
			if (*buffer) {
				buffer = skip_white_space(buffer, "=:");
				if (*buffer) {
					kv = parse_value(&buffer, key);
					if (kv) {
						/* Add this node to the end
						 * of the list.
						 */
						if (!tree) {
							tree = next = kv;
						} else {
							next->next = kv;
							next = kv;
						}
					}
				} else {
					parse_error_msg();
				}
			} else {
				parse_error_msg();
			}
		}
	}

	LOG_EXIT_PTR(tree);
	return tree;
}

/**
 * find_key
 *
 * Search the specified key-value list for a node with the specified key.
 **/
key_value_t *find_key(key_value_t *list, char *key)
{
	key_value_t *node = list;

	LOG_ENTRY();

	while (node) {
		if (!strcmp(key, node->key)) {
			break;
		}
		node = node->next;
	}

	LOG_EXIT_PTR(node);
	return node;
}

/**
 * delete_vgda_tree
 *
 * Delete all nodes from the specified key-value tree.
 **/
void delete_vgda_tree(key_value_t *tree)
{
	key_value_t *node, *next;

	for (node = tree; node; node = next) {
		next = node->next;
		EngFncs->engine_free(node->key);
		switch (node->type) {
		case VALUE:
			EngFncs->engine_free(node->value.string);
			break;
		case ARRAY:
			EngFncs->engine_free(node->value.array->strings);
			EngFncs->engine_free(node->value.array);
			break;
		case SECTION:
			delete_vgda_tree(get_section(node));
			break;
		default:
			break;
		}
		EngFncs->engine_free(node);
	}
}

/**
 * realloc_vgda_buffer
 *
 * Allocate a buffer for the text-VGDA. If the buffer already exists,
 * allocate a larger one and copy the existing buffer to the new one.
 **/
static int realloc_vgda_buffer(char **buffer, unsigned long *size)
{
	char *buf;
	unsigned long sz;
	int rc = 0;

	LOG_ENTRY();

	/* Start with one sector, then keep doubling the buffer size. */
	sz = (*size) ? *size * 2 : EVMS_VSECTOR_SIZE;

	buf = EngFncs->engine_alloc(sz);
	if (!buf) {
		LOG_ERROR("Error allocating %lu byte buffer for VGDA.\n", sz);
		rc = ENOMEM;
		goto out;
	}

	/* Copy the contents from the old buffer to the new. */
	if (*buffer) {
		memcpy(buf, *buffer, *size);
		EngFncs->engine_free(*buffer);
	}

	*buffer = buf;
	*size = sz;

	LOG_DETAILS("Allocated %lu byte buffer for VGDA.\n", sz);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * write_vgda_string
 * @string:	String to append to the buffer.
 * @buffer:	Buffer to write string into.
 * @offset:	Offset in buffer to write the string.
 * @size:	Size of buffer.
 *
 * Write a string into the VGDA buffer. If the string is going to overwrite
 * the end of the buffer, reallocate a larger buffer first.
 **/
int write_vgda_string(char *string,
		      char **buffer,
		      unsigned long *offset,
		      unsigned long *size)
{
	unsigned long string_size;
	int rc = 0;

	LOG_ENTRY();

	string_size = strlen(string);
	if (*offset + string_size >= *size) {
		rc = realloc_vgda_buffer(buffer, size);
		if (rc) {
			goto out;
		}
	}

	sprintf(*buffer + *offset, string);
	*offset += string_size;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * vgda_add_pv
 *
 * Add the information about this PV to the VGDA buffer.
 **/
static int vgda_add_pv(storage_object_t *object,
		       char **buffer,
		       unsigned long *offset,
		       unsigned long *size)
{
	pv_data_t *pv_data = object->consuming_private_data;
	char pv_name[25];
	char pv_uuid[LVM2_UUID_LEN+1];
	char buf[256];
	int rc = 0;

	LOG_ENTRY();

	/* Section starts with the "name" of the PV. */
	snprintf(pv_name, 25, "pv%u {\n", pv_data->pv_index);
	WRITE_STRING(pv_name, buffer, offset, size, rc);
	
	/* UUID */
	format_uuid(pv_data->uuid, pv_uuid);
	snprintf(buf, 256, "id = \"%s\"\n", pv_uuid);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* Name of object. */
	snprintf(buf, 256, "device = \"/dev/%s\"\n", object->name);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* Status flags. */
	WRITE_STRING("status = [", buffer, offset, size, rc);
	rc = write_flags(pv_data->flags, PV_FLAGS, buffer, offset, size);
	if (rc) {
		goto out;
	}
	WRITE_STRING("]\n", buffer, offset, size, rc);

	/* PE start. */
	snprintf(buf, 256, "pe_start = %"PRIu64"\n", pv_data->pe_start);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* PE count. */
	snprintf(buf, 256, "pe_count = %"PRIu64"\n", pv_data->pe_count);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* FIXME: Anything else??? */

	/* End of this PV section. */
	WRITE_STRING("}\n", buffer, offset, size, rc);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * vgda_add_pvs
 *
 * Add the PV information for this container to the VGDA buffer.
 **/
static int vgda_add_pvs(storage_container_t *container,
			char **buffer,
			unsigned long *offset,
			unsigned long *size)
{
	storage_object_t *object;
	list_element_t iter;
	int rc = 0;

	LOG_ENTRY();

	WRITE_STRING("physical_volumes {\n", buffer, offset, size, rc);

	LIST_FOR_EACH(container->objects_consumed, iter, object) {
		rc = vgda_add_pv(object, buffer, offset, size);
		if (rc) {
			goto out;
		}
	}

	WRITE_STRING("}\n", buffer, offset, size, rc);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * vgda_add_region_mapping
 *
 * Add the information about this region mapping to the VGDA buffer. 
 **/
static int vgda_add_region_mapping(region_mapping_t *r_map,
				   unsigned int index,
				   char **buffer,
				   unsigned long *offset,
				   unsigned long *size)
{
	u_int64_t i;
	char buf[256];
	int rc = 0;

	LOG_ENTRY();

	/* Section begins with mapping "name". */
	snprintf(buf, 256, "segment%u {\n", index);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* Start extent. */
	snprintf(buf, 256, "start_extent = %"PRIu64"\n", r_map->start_le);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* Extent count. */
	snprintf(buf, 256, "extent_count = %"PRIu64"\n", r_map->le_count);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* Type.
	 * FIXME: Allow other mappings besides "striped".
	 */
	snprintf(buf, 256, "type = \"striped\"\n");
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* Stripe count. */
	snprintf(buf, 256, "stripe_count = %"PRIu64"\n", r_map->stripe_count);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* Stripe size. */
	if (r_map->stripe_count > 1) {
		snprintf(buf, 256, "stripe_size = %"PRIu64"\n", r_map->stripe_size);
		WRITE_STRING(buf, buffer, offset, size, rc);
	}

	/* FIXME: Anything else??? */

	/* Stripe descriptors. */
	WRITE_STRING("stripes = [\n", buffer, offset, size, rc);

	for (i = 0; i < r_map->stripe_count; i++) {
		/* PV "name". */
		snprintf(buf, 256, "\"pv%u\",",
			 r_map->le_maps[i].map[0].pe->pv_data->pv_index);
		WRITE_STRING(buf, buffer, offset, size, rc);

		/* Starting PE. */
		snprintf(buf, 256, "%"PRIu64,
			 r_map->le_maps[i].map[0].pe->number);
		WRITE_STRING(buf, buffer, offset, size, rc);

		if (i < r_map->stripe_count - 1) {
			WRITE_STRING(",\n", buffer, offset, size, rc);
		} else {
			WRITE_STRING("\n", buffer, offset, size, rc);
		}
	}
	
	WRITE_STRING("]\n", buffer, offset, size, rc);

	/* Close this mapping section. */
	WRITE_STRING("}\n", buffer, offset, size, rc);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * vgda_add_region
 *
 * Add the information about this region to the VGDA buffer.
 **/
static int vgda_add_region(storage_object_t *region,
			   char **buffer,
			   unsigned long *offset,
			   unsigned long *size)
{
	region_data_t *r_data = region->private_data;
	region_mapping_t *r_map;
	list_element_t iter;
	char lv_name[EVMS_NAME_SIZE];
	char lv_uuid[LVM2_UUID_LEN+1];
	char buf[256];
	int i = 1, rc = 0;

	LOG_ENTRY();

	/* The section begins with the region name. */
	region_name_to_lv_name(region->name, lv_name,
			       region->producing_container->name);
	WRITE_STRING(lv_name, buffer, offset, size, rc);
	WRITE_STRING(" {\n", buffer, offset, size, rc);

	/* UUID */
	format_uuid(r_data->uuid, lv_uuid);
	snprintf(buf, 256, "id = \"%s\"\n", lv_uuid);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* Status flags. */
	WRITE_STRING("status = [", buffer, offset, size, rc);
	rc = write_flags(r_data->flags, LV_FLAGS, buffer, offset, size);
	if (rc) {
		goto out;
	}
	WRITE_STRING("]\n", buffer, offset, size, rc);

	/* Segment count */
	snprintf(buf, 256, "segment_count = %u\n",
		 EngFncs->list_count(r_data->mappings));
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* FIXME: Anything else??? */

	/* Segment descriptors. */
	LIST_FOR_EACH(r_data->mappings, iter, r_map) {
		rc = vgda_add_region_mapping(r_map, i++, buffer, offset, size);
		if (rc) {
			goto out;
		}
	}

	/* Close this region section. */
	WRITE_STRING("}\n", buffer, offset, size, rc);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * vgda_add_regions
 *
 * Add the region information for this container to the VGDA buffer.
 **/
static int vgda_add_regions(storage_container_t *container,
			    char **buffer,
			    unsigned long *offset,
			    unsigned long *size)
{
	storage_object_t *region;
	list_element_t iter;
	int rc = 0;

	LOG_ENTRY();

	WRITE_STRING("logical_volumes {\n", buffer, offset, size, rc);

	LIST_FOR_EACH(container->objects_produced, iter, region) {
		if (region->data_type != DATA_TYPE) {
			continue;
		}

		rc = vgda_add_region(region, buffer, offset, size);
		if (rc) {
			goto out;
		}
	}

	WRITE_STRING("}\n", buffer, offset, size, rc);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * vgda_add_container
 *
 * Add the container information to the VGDA buffer.
 **/
static int vgda_add_container_info(storage_container_t *container,
				   char **buffer,
				   unsigned long *offset,
				   unsigned long *size)
{
	container_data_t *c_data = container->private_data;
	char vg_uuid[LVM2_UUID_LEN+10];
	char buf[256];
	int rc;

	LOG_ENTRY();

	/* Section begins with the container name. */
	container_name_to_vg_name(container->name, buf);
	WRITE_STRING(buf, buffer, offset, size, rc);
	WRITE_STRING(" {\n", buffer, offset, size, rc);

	/* Container parameters. */

	/* UUID */
	format_uuid(c_data->uuid, vg_uuid);
	snprintf(buf, 256, "id = \"%s\"\n", vg_uuid);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* Sequence number. */
	snprintf(buf, 256, "seqno = %lu\n", c_data->sequence);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* Status flags. */
	WRITE_STRING("status = [", buffer, offset, size, rc);
	rc = write_flags(c_data->flags, VG_FLAGS, buffer, offset, size);
	if (rc) {
		goto out;
	}
	WRITE_STRING("]\n", buffer, offset, size, rc);

	/* PE size. */
	snprintf(buf, 256, "extent_size = %"PRIu64"\n", c_data->pe_size);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* Max LVs and PVs. */
	snprintf(buf, 256, "max_lv = %lu\n", c_data->max_lvs);
	WRITE_STRING(buf, buffer, offset, size, rc);

	snprintf(buf, 256, "max_pv = %lu\n", c_data->max_pvs);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* FIXME: Anything else??? */

	/* Physical volume info. */
	rc = vgda_add_pvs(container, buffer, offset, size);
	if (rc) {
		goto out;
	}

	/* Logical volume info. */
	if (EngFncs->list_count(container->objects_produced) > 1) {
		rc = vgda_add_regions(container, buffer, offset, size);
		if (rc) {
			goto out;
		}
	}

	/* Close the container section. */
	WRITE_STRING("}\n", buffer, offset, size, rc);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * vgda_add_comments
 *
 * Add comments to the VGDA buffer that follow the container information.
 **/
static int vgda_add_comments(char **buffer,
			     unsigned long *offset,
			     unsigned long *size)
{
	time_t now = time(NULL);
	struct utsname name;
	char buf[256];
	int rc = 0;

	LOG_ENTRY();

	rc = uname(&name);
	if (rc) {
		goto out;
	}

	WRITE_STRING("# Generated by EVMS: ", buffer, offset, size, rc);

	snprintf(buf, 256, "%s\n", ctime(&now));
	WRITE_STRING(buf, buffer, offset, size, rc);

	WRITE_STRING("contents = \"Text Format Volume Group\"\n",
		     buffer, offset, size, rc);

	snprintf(buf, 256, "version = %u\n", LVM2_FMTT_VERSION);
	WRITE_STRING(buf, buffer, offset, size, rc);

	/* FIXME: Is anything supposed to go in the description? */
	WRITE_STRING("description = \"\"\n", buffer, offset, size, rc);

	snprintf(buf, 256, "creation_host = \"%s\"\t# %s %s %s %s %s\n",
		 name.nodename, name.sysname, name.nodename,
		 name.release, name.version, name.machine);
	WRITE_STRING(buf, buffer, offset, size, rc);

	snprintf(buf, 256, "creation_time = %lu\t# %s\n", now, ctime(&now));
	WRITE_STRING(buf, buffer, offset, size, rc);

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * prepare_vgda_buffer
 *
 * Create a buffer containing the text VGDA that can be written to disk.
 **/
int prepare_vgda_buffer(storage_container_t *container, char **buffer)
{
	unsigned long size = 0, offset = 0;
	char *buf = NULL;
	int rc;

	LOG_ENTRY();
	LOG_DEBUG("Preparing VGDA buffer for container %s.\n", container->name);

	rc = vgda_add_container_info(container, &buf, &offset, &size);
	if (rc) {
		goto out;
	}

	rc = vgda_add_comments(&buf, &offset, &size);
	if (rc) {
		goto out;
	}

	*buffer = buf;

out:
	if (rc) {
		EngFncs->engine_free(buf);
	}
	LOG_EXIT_INT(rc);
	return rc;
}

