/*
 *
 *   (C) Copyright IBM Corp. 2002, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * 	Module: rsct_ece.c
 */




#include "rsct_internal.h"
#include "rsct_mem_info.h"
#include "eceinternal.h"

#include <pthread.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/select.h>
#include <ha_gs.h>
#include <ece.h>
#include <plugin.h>
#include <assert.h>
#include <sys/types.h>
#include <sys/wait.h>



/*******************macros*******************/
#define LOCAL_NAME                   "RSCT plugin"
#define GROUP_NAME                   "RSCT_plugin_group"
#define MAX_MSG_COUNT                 100000

/*******************gloable variables**********************/
GSList*                  gl_cb_list;
ha_gs_token_t            gl_provider_token;
ha_gs_token_t            gl_subscriber_token;
engine_functions_t *     gl_ece_engine_funcs;

pthread_mutex_t          gl_ack_lock;
pthread_cond_t           gl_ack_cond;
unsigned                 gl_ack_arrived;

pthread_mutex_t          gl_send_lock;
pthread_mutex_t          gl_send_func_lock;

pthread_mutex_t          gl_ack_thread_lock;
pthread_cond_t           gl_ack_thread_cond;
unsigned                 gl_ack_thread_activate;

pthread_mutex_t          gl_rsct_thread_lock;
pthread_cond_t           gl_rsct_thread_cond;
unsigned                 gl_rsct_thread_activate;

unsigned                 gl_ack_thread_dispatch_once;

pthread_mutex_t          gl_event_queue_lock;


pthread_t                gl_rsct_thread;
pthread_t                gl_appcb_thread;

int                      gl_proposal_id=0;
pthread_mutex_t          gl_proposal_id_lock;

static unsigned long     gl_max_rsct_msg_size ;


static unsigned long     gl_msg_count=0;
pthread_mutex_t          gl_msg_count_lock;

ha_gs_descriptor_t       gl_descriptor;

ece_msg_t*               gl_ecemsg=NULL;


/*******************extern global variables****************/
extern int gl_transid;


/*******************internal static functions***************/


static void
unregister_all()
{
	
	LOG_ENTRY();
	
	g_slist_free(gl_cb_list);
	
	LOG_EXIT_VOID();
	return;
}


static void *
appcb(void *arg)
{
	LOG_ENTRY();
	
	pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS,NULL);

	while(1){
		pthread_testcancel();
		if(RSCT_appcb_length() >0)
			RSCT_appcb_run_remove_first();
	}
	
	LOG_EXIT_PTR(NULL);
	return(NULL);
}

static int get_msg_count()
{
	int msg_count;
	
	LOG_ENTRY();
	
	pthread_mutex_lock(&gl_msg_count_lock);
	msg_count=gl_msg_count++;
	pthread_mutex_unlock(&gl_msg_count_lock);
	
	LOG_EXIT_INT(msg_count);
	return(msg_count);
	
}

//only called by the act and rsct  thread
static void
RSCT_non_blocking_dispatch()
{

	ha_gs_rc_t gs_rc;
	
	LOG_ENTRY();
	gs_rc=ha_gs_dispatch(HA_GS_NON_BLOCKING);
	if(gs_rc !=HA_GS_OK){
		LOG_ERROR("ERROR in RSCT_non_blocking_dispath()\n");
	}
	
	LOG_EXIT_VOID();
	return;
}

static int
get_local_instance_number()
{
	
	int ret;
	
	LOG_ENTRY();
	
	if(gl_ece_engine_funcs->get_engine_mode() ==ENGINE_DAEMON)
		ret=MASTER_INSTANCE_NUM;
	else
		ret=SLAVE_INSTANCE_NUM;
	
	LOG_EXIT_INT(ret);
	return(ret);
}

static void *
rsct_dispatch(void *arg)
{
	
	fd_set my_fd;
	int rc;
	struct timeval tv;
	
	LOG_ENTRY();
	
	pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS,NULL);
	LOG_DEBUG("RSCT starts\n");
	
	while(1){
		pthread_testcancel();
		FD_ZERO(&my_fd);
		FD_SET(0, &my_fd);
		FD_SET(gl_descriptor, &my_fd);
		tv.tv_sec=1;
		tv.tv_usec=0;
		rc = select(gl_descriptor + 1, &my_fd, NULL, NULL, &tv);
		if(rc < 0) {
			LOG_EXIT_PTR(NULL);
			return(NULL);
		}
		if(gl_descriptor && FD_ISSET(gl_descriptor, &my_fd)) {
			
			LOG_DEBUG("there are data in coming\n");
			RSCT_non_blocking_dispatch();
		}
	}
	LOG_EXIT_PTR(NULL);
	return(NULL);
}

static gint
cb_compare (gconstpointer a,
	    gconstpointer b)
{
	
	const registercb_t* rcb_a=a;
	const ece_cb_t      cb_b=b;
	int ret;
	
	LOG_ENTRY();
	
	if(rcb_a->reg_callback== cb_b)
		ret = 0;
	else
		ret = -1;
	
	LOG_EXIT_INT(ret);
	return(ret);
	
}

static int
frag_send_msg(ece_msg_t* ecemsg)
{
	
	int gs_rc;
	unsigned long seqno=0;
	unsigned long flen;
	unsigned long tlen;
	int i;
	rsct_msg_t* rsctmsg;
	ha_gs_proposal_info_t    proposal_info;
	int temp;
	int message_size;
	char* sendBuf;
	int msg_count;
	int total_frag_num=0;
	int local_node_number;
	int sleep_time=1000;
	
	LOG_ENTRY();
	
	
	msg_count=get_msg_count();
	ha_gs_get_node_number(&local_node_number);
	
	tlen=ecemsg->size;
	flen=gl_max_rsct_msg_size - sizeof(rsct_msg_t);
	
	temp= tlen <= flen? tlen:flen;
	
	message_size =sizeof(rsct_msg_t)+temp;
	sendBuf=(char*)malloc(message_size );
	memset(sendBuf,0,message_size);
	
	rsctmsg=(rsct_msg_t*)sendBuf;
	memcpy(&rsctmsg->ecemsg,ecemsg,sizeof(ece_msg_t));
	
	LOG_DEBUG("sendout msg_count=%d to node %s,msgsize=%d\n",
		  msg_count,
		  &ecemsg->node,
		  ecemsg->size);
	
	if(tlen==0)
		total_frag_num=1;
	else
		total_frag_num=(tlen-1)/flen +1;
	
	
	LOG_DEBUG("tlen=%d, flen=%d,total_frag_num=%d\n",
		  tlen,
		  flen,
		  total_frag_num);
	
	for(i=0;i < total_frag_num;i++){
		int length;
		
		
		rsctmsg->ecemsg.msg=sendBuf + sizeof(rsct_msg_t);
		
		if(i == (tlen-1)/flen)
			length=(tlen-1)%flen +1;
		else
			length=flen;
		
		if(tlen ==0)length=0;
		if(length!=0)
			memcpy(rsctmsg->ecemsg.msg, ecemsg->msg + flen*i, length );
		rsctmsg->type=RSCT_MSG;
		rsctmsg->seqno=seqno++;
		rsctmsg->flen=flen;
		rsctmsg->tlen=tlen;
		rsctmsg->data_length=length;
		rsctmsg->send_node_number=local_node_number;
		
		rsctmsg->msg_count=msg_count;
		do{
			
			//give some rest to rsct between the interval
			usleep(sleep_time);
			
			proposal_info.gs_message_request.gs_num_phases  = HA_GS_1_PHASE;
			proposal_info.gs_message_request.gs_time_limit  = TIME_LIMIT;
			proposal_info.gs_message_request.gs_message.gs_length  =
				length+sizeof(rsct_msg_t);
			proposal_info.gs_message_request.gs_message.gs_message = (char*)sendBuf;
			
			pthread_mutex_lock(&gl_send_lock);	
			
			gs_rc = ha_gs_send_message(gl_provider_token,
						   &proposal_info);
			
			LOG_DEBUG("frag_msg send out with gs_rc=%d, seqno=%d\n",
				  gs_rc,
				  rsctmsg->seqno);
			//release the lock for send
			pthread_mutex_unlock(&gl_send_lock);
			
			if(gs_rc != HA_GS_OK){
				if(gs_rc == HA_GS_COLLIDE) {
					sleep_time +=sleep_time;
					LOG_ERROR("warning:msg sending collide:"
						  " msg_count=%d,seqno=%d,length=%d\n",
						  rsctmsg->msg_count,
						  rsctmsg->seqno,
						  proposal_info.gs_message_request.
						  gs_message.gs_length);
				}
				else {
					LOG_ERROR("ERROR:"
						     "ha_gs_send_message() failed rc=%d\n", gs_rc);
					free(sendBuf);
					RETURN(EINVAL);
				}
			}
		}while(gs_rc !=HA_GS_OK);
	}

	LOG_DEBUG("frag_send success\n");
	free(sendBuf);
	LOG_EXIT_INT(0);
	return(0);
	
}


/*******************RSCT plugin interface functions **********/

static void
RSCT_cleanup_evms_plugin(void)
{
	
	ha_gs_rc_t gs_rc;
	ha_gs_proposal_info_t    proposal_info;
	
	LOG_ENTRY();
	
	
	usleep(100000);
	//kill two child threads
	if(gl_rsct_thread)
		pthread_cancel(gl_rsct_thread);
	if(gl_appcb_thread)
		pthread_cancel(gl_appcb_thread);
	
	waitpid(gl_rsct_thread,NULL,WUNTRACED);
	waitpid(gl_appcb_thread,NULL,WUNTRACED);
	
	proposal_info.gs_leave_request.gs_num_phases = HA_GS_1_PHASE;
	proposal_info.gs_leave_request.gs_time_limit = TIME_LIMIT;
	
	gs_rc = ha_gs_leave( gl_provider_token,
			     &proposal_info);
	if(gs_rc != HA_GS_OK) {
		LOG_ERROR("ERROR: leaving the group failed, rc=%d\n",
			     gs_rc);
	}
	
	//unregister all registered callback functions
	unregister_all();
	
	RSCT_mem_cleanup();
	
	RSCT_INTERNAL_cb_cleanup();
	
	
	if(gl_cb_list != NULL)
		g_slist_free(gl_cb_list)                    ;
	
	ha_gs_quit();
	
	LOG_EXIT_VOID();
	return;
}

static int
RSCT_setup_evms_plugin(engine_functions_t *func)
{
	
	ha_gs_rc_t               gs_rc;
	ha_gs_responsiveness_t   responsiveness;
	ha_gs_proposal_info_t    proposal_info;
	ha_gs_group_attributes_t group_attributes;
	
	ha_gs_proposal_info_t    *subscriber_proposal_info;
	
	int local_node_number;
	int local_instance_number;
	ece_nodeid_t local_nodeid;
	
	ha_gs_limits limits;

	EngFncs = func;
	LOG_ENTRY();
	
	
	if(!func || !func->get_engine_mode) {
		LOG_ERROR("Error: Invalid Input (Arg #1)\n");
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}

	gl_ece_engine_funcs=func;
	
	//initialize mutex, lock and others for ack
	pthread_mutex_init(&gl_ack_lock,NULL);
	pthread_cond_init(&gl_ack_cond,NULL);
	gl_ack_arrived=0;
	
	//initialize mutex, lock and others for ack thread
	pthread_mutex_init(&gl_ack_thread_lock,NULL);
	pthread_cond_init(&gl_ack_thread_cond,NULL);
	gl_ack_thread_activate=0;
	
	
	//initialize mutex, lock and others for main thread
	pthread_mutex_init(&gl_rsct_thread_lock,NULL);
	pthread_cond_init(&gl_rsct_thread_cond,NULL);
	gl_rsct_thread_activate=1;

	//initialize lock for event queue
	pthread_mutex_init(&gl_event_queue_lock,NULL);
	
	//initialize gl_ack_thread_dispatch_once
	gl_ack_thread_dispatch_once=0;
	
	//initizalize gl_proposal_id_lock
	pthread_mutex_init(&gl_proposal_id_lock,NULL);
	
	//initialize gl_msg_count_lock
	pthread_mutex_init(&gl_msg_count_lock,NULL);

	//initialize gl_send_lock and gl_send_func_lock
	pthread_mutex_init(&gl_send_lock,NULL);
	pthread_mutex_init(&gl_send_func_lock,NULL);
	
	
	/*initialize the group service*/
	responsiveness.gs_responsiveness_type     = HA_GS_NO_RESPONSIVENESS;
	responsiveness.gs_responsiveness_interval = 0;
	responsiveness.gs_responsiveness_response_time_limit = 0;
	responsiveness.gs_counter_location        = NULL;
	responsiveness.gs_counter_length          = 0;
	
	gs_rc = ha_gs_init(&gl_descriptor,
			   HA_GS_SOCKET_NO_SIGNAL | HA_GS_ENABLE_ADAPTER_INFO,
			   &responsiveness,
			   NULL,
			   NULL,
			   RSCT_INTERNAL_delayed_error_cb,
			   (ha_gs_query_cb_t*)0);
	
	
	
	if(gs_rc != HA_GS_OK) {
		LOG_ERROR("*** ha_gs_init failed rc=%d ***\n", gs_rc);
		LOG_EXIT_INT(ENODEV);
		return(ENODEV);
	}
	
	
	/* get the max msg size for rsct*/
	
	gs_rc=ha_gs_get_limits(&limits);
	assert(gs_rc == HA_GS_OK && "ha_gs_get_limits() failed\n");
	
	gl_max_rsct_msg_size =limits.max_provider_message_length;
	
	
	/*initialize membership information*/
	RSCT_mem_init();

	/* initiazlize internal callback functions*/
	RSCT_INTERNAL_cb_init();
	
	
	/*init applcation call back */
	RSCT_appcb_init();
	
	/*join as a provider*/
	
	proposal_info.gs_join_request.gs_group_attributes    = &group_attributes;
	if(func->get_engine_mode() ==ENGINE_DAEMON)
		proposal_info.gs_join_request.gs_provider_instance   = MASTER_INSTANCE_NUM;
	else
		proposal_info.gs_join_request.gs_provider_instance   = SLAVE_INSTANCE_NUM;
	
	proposal_info.gs_join_request.gs_provider_local_name = LOCAL_NAME;
	proposal_info.gs_join_request.gs_n_phase_protocol_callback
		= RSCT_INTERNAL_n_phase_cb;
	proposal_info.gs_join_request.gs_protocol_approved_callback
		= RSCT_INTERNAL_approved_cb;
	proposal_info.gs_join_request.gs_protocol_rejected_callback
		= RSCT_INTERNAL_rejected_cb ;
	proposal_info.gs_join_request.gs_announcement_callback
		= RSCT_INTERNAL_announcement_cb;
	proposal_info.gs_join_request.gs_merge_callback      = NULL;

	group_attributes.gs_version                      = 1;
	group_attributes.gs_sizeof_group_attributes
		= sizeof(ha_gs_group_attributes_t);
	group_attributes.gs_client_version               = 1;
	group_attributes.gs_batch_control
		= HA_GS_NO_BATCHING | HA_GS_DEACTIVATE_ON_FAILURE;
	group_attributes.gs_num_phases                   = HA_GS_1_PHASE;
	group_attributes.gs_source_reflection_num_phases = HA_GS_1_PHASE;
	group_attributes.gs_group_default_vote           = HA_GS_VOTE_APPROVE;
	group_attributes.gs_merge_control                = HA_GS_DISSOLVE_MERGE;
	group_attributes.gs_time_limit                   = JOIN_FAILURE_TIME_LIMIT;
	group_attributes.gs_source_reflection_time_limit = 0;
	group_attributes.gs_group_name                   = GROUP_NAME;
	group_attributes.gs_source_group_name            = NULL;

	gs_rc = ha_gs_join(&gl_provider_token,
			   &proposal_info);
	if(gs_rc != HA_GS_OK) {
		LOG_ERROR("ha_gs_join failed rc=%d \n", gs_rc);
		LOG_EXIT_INT(ENODEV);
		return(ENODEV);
	}
	
	//subscribe node membership group
	
	subscriber_proposal_info =
		(ha_gs_proposal_info_t*)malloc(sizeof(ha_gs_proposal_info_t));
	
	subscriber_proposal_info->gs_subscribe_request.gs_subscription_control
		= HA_GS_SUBSCRIBE_ALL_MEMBERSHIP;
	subscriber_proposal_info->gs_subscribe_request.gs_subscription_group
		= HA_GS_HOST_MEMBERSHIP_GROUP;
	//= HA_GS_ALL_ADAPTER_MEMBERSHIP_GROUP;
	subscriber_proposal_info->gs_subscribe_request.gs_subscription_callback
		= RSCT_INTERNAL_subscriber_cb;
	
	gs_rc = ha_gs_subscribe(&gl_subscriber_token,
				subscriber_proposal_info);
	if(gs_rc != HA_GS_OK) {
		free(subscriber_proposal_info);
		LOG_ERROR("ha_gs_subscribe failed rc=%d\n", gs_rc);
		RETURN(ENODEV);
	}
	
	free(subscriber_proposal_info);
	
	//put self into the membership table
	if(func->get_engine_mode() ==ENGINE_DAEMON)
		local_instance_number = MASTER_INSTANCE_NUM;
	else
		local_instance_number = SLAVE_INSTANCE_NUM;
	ha_gs_get_node_number(&local_node_number);
	RSCT_mem_get_mynode(&local_nodeid);
	RSCT_plugin_add(local_node_number,
			local_instance_number,
			&local_nodeid);
	
	//spawn a thread to handle callbacks
	if(pthread_create(&gl_rsct_thread, NULL, rsct_dispatch, NULL)) {
		LOG_ERROR("main rsct thread:Thread_create() error!");
		RSCT_cleanup_evms_plugin();
		RETURN(EAGAIN);
	}
	
	
	//spawn a thread to handle membership and message callback for
	//application
	if(pthread_create(&gl_appcb_thread, NULL, appcb, NULL)) {
		LOG_ERROR("appcb thread: Thread_create() error!\n");
		RSCT_cleanup_evms_plugin();
		RETURN(EAGAIN);
	}
	
	LOG_DEBUG("gl_rsct_thread=%x, gl_appcb_thread=%x\n",
		     gl_rsct_thread,
		     gl_appcb_thread);
	
	
	usleep(100000);//give sometime for initializtion
	
	LOG_EXIT_INT(0);
	return(0);
	
}



static int
RSCT_register_callback(ece_callback_type_t type,
		       ece_cb_t cb)
{
	
	registercb_t*  rcb;
	int num_entries;
	ece_event_t* event;
	ece_event_t* leave_event;
	int size;
	int real_size;
	int temp;
	
	LOG_ENTRY();
	usleep(100000);
	
	rcb=(registercb_t*)malloc(sizeof(registercb_t));
	rcb->reg_type=type;
	rcb->reg_callback=cb;
	gl_cb_list= g_slist_append (gl_cb_list,rcb );
	
	
	do{
		num_entries= RSCT_mem_get_num_config_nodes();
		size= sizeof(ece_event_t) +sizeof(ece_nodeid_t)*num_entries;

		event=(ece_event_t*) malloc(size);
		event->num_entries=num_entries;
		
		temp=RSCT_node_get_membership(event);
		
	} while(temp!=0);
	
	
	leave_event=(ece_event_t*)malloc(sizeof(ece_event_t));
	
	real_size = sizeof(ece_event_t)
		+ sizeof(ece_nodeid_t)*event->num_entries;
	
	//get quorum info
	event->quorum_flag= RSCT_mem_has_quorum();
	leave_event->quorum_flag= RSCT_mem_has_quorum();
	
	if(type == DELTAS){
		//call it twice: one for join with all members
		//               the other for leave with none
		
		event->type =DELTA_JOIN;
		RSCT_appcb_add(CALLBACK_MEMBERSHIP,rcb, real_size,event);
		
		leave_event->num_entries=0;
		leave_event->type =DELTA_LEAVE;
		RSCT_appcb_add(CALLBACK_MEMBERSHIP,rcb, sizeof(ece_event_t),leave_event);
		
	}else if (type == FULL_MEMBERSHIP ){
		//call it once with all members
		event->type =MEMBERSHIP;
		RSCT_appcb_add(CALLBACK_MEMBERSHIP,rcb, real_size,event);
		
	}else{
		fprintf(stderr,"Wrong type in registering callback functions\n");
		gl_cb_list= g_slist_remove (gl_cb_list,rcb );
		free(rcb);
		free(event);
		LOG_EXIT_INT(0);
		return(0);
	}
	
	
	//deliver the first callback
	LOG_EXIT_INT(0);
	return(0);
}

static int
RSCT_unregister_callback(ece_cb_t cb)
{
	
	registercb_t*  rcb;
	GSList* temp;
	
	LOG_ENTRY();
	
	temp=g_slist_find_custom(gl_cb_list,cb,cb_compare);
	if(temp == NULL ||g_slist_length(temp)==0){
		LOG_ERROR("ERROR: RSCT_unregister_callback()--function not found.\n");
	}
	else if (g_slist_length(temp) >1 ){
		LOG_ERROR("ERROR: RSCT_unregister_callback()--more than one function found.\n");
	}
	
	rcb=g_slist_nth_data(temp,0);
	
	//this will free the memory allocated in RSCT_register_callback()
	gl_cb_list = g_slist_remove(gl_cb_list,(gpointer)rcb);
	LOG_EXIT_INT(0);
	return(0);
}



static int
RSCT_send_msg(ece_msg_t *ecemsg)
{
	
	ece_nodeid_t      all_nodes=(ece_nodeid_t)ECE_ALL_NODES;
	boolean           isBroadcast;
	
	int               local_node_number;
	ece_nodeid_t*      local_nodeid;
	int               local_instance_number;
	
	int                out_instance_number=0;
	ece_nodeid_t*      out_nodeid;

	LOG_ENTRY();
	
	ha_gs_get_node_number(&local_node_number);
	local_nodeid=RSCT_mem_map_table_lookup(local_node_number);
	
	assert(local_nodeid &&"local_nodeid not found in table?!");
	
	local_instance_number=get_local_instance_number();
	
	if(strncmp((char*)&ecemsg->node,
		   (char*)&all_nodes,
		   sizeof(ece_nodeid_t) )  == 0)
		isBroadcast=TRUE;
	else
		isBroadcast=FALSE;
	
	if(!isBroadcast){
		if(local_instance_number==MASTER_INSTANCE_NUM)
			out_instance_number=SLAVE_INSTANCE_NUM;
		else
			out_instance_number=MASTER_INSTANCE_NUM;
		out_nodeid=&ecemsg->node;
		
		if(! RSCT_plugin_is_present(out_nodeid,out_instance_number)){
			usleep(100000);
		}
		
		if(! RSCT_plugin_is_present(out_nodeid,out_instance_number)){
			LOG_ERROR("Error:the destation node %s %s is not running\n",
				     (char*)out_nodeid,
				     out_instance_number==MASTER_INSTANCE_NUM?"master":"slave");
			RSCT_mem_print();
			LOG_EXIT_INT(EAGAIN);
			return(EAGAIN);
		}
	}
	
	pthread_mutex_lock(&gl_send_func_lock);
	
	frag_send_msg( ecemsg);
	
	all_nodes=(ece_nodeid_t)ECE_ALL_NODES;
	//wait until the ACK is arrived
	if( strncmp( ((char*)&(ecemsg->node)),
		     (char*)(&all_nodes),
		     sizeof(ece_nodeid_t))!=0){
		
		if(pthread_self() == gl_rsct_thread){
			//activate the ack thread
			LOG_DEBUG("activate the ACK thread\n");
			if(pthread_self() == gl_rsct_thread){
				pthread_mutex_lock(&gl_ack_thread_lock);
				gl_ack_thread_activate=1;
				pthread_cond_signal(&gl_ack_thread_cond);
				pthread_mutex_unlock(&gl_ack_thread_lock);
			}
		}
		
		
		//wait for ACK
		LOG_DEBUG("wait for ACK from %s %s plugin......\n",
			     &ecemsg->node,
			     out_instance_number==MASTER_INSTANCE_NUM?"master":"slave");
		
		pthread_mutex_lock(&gl_ack_lock);
		gl_ecemsg = ecemsg;
		
		if( RSCT_plugin_is_present(&ecemsg->node,out_instance_number))
			while (gl_ack_arrived == 0){
				pthread_cond_wait(&gl_ack_cond, &gl_ack_lock);
				gl_ack_arrived=1;
			}
		
		gl_ack_arrived=0;
		
		gl_ecemsg = NULL;
		
		pthread_mutex_unlock(&gl_ack_lock);
		LOG_DEBUG("get the ACK!\n");
		
	}
	pthread_mutex_unlock(&gl_send_func_lock);
	
	LOG_DEBUG("RSCT_send_msg done\n");
	LOG_EXIT_INT(0);
	return(0);
}



static int
RSCT_get_clusterid(ece_clusterid_t *clusterid)
{	
	const char* cid=RSCT_mem_get_clusterid();
	
	LOG_ENTRY();
	
	memset(clusterid, 0, sizeof(ece_clusterid_t));
	strcpy((char *)clusterid, cid);
	
	LOG_EXIT_INT(0);
	return(0);
}

static int
RSCT_get_my_nodeid(ece_nodeid_t *nodeid)
{	
	
	LOG_ENTRY();
	RSCT_mem_get_mynode(nodeid);
	LOG_EXIT_INT(0);
	return(0);
}

static int
RSCT_get_num_config_nodes(uint* num)
{
	int count;
	
	LOG_ENTRY();
	
	count=RSCT_mem_get_num_config_nodes();
	if(count >=0){
		(*num)=count;
		RETURN(0);
	}
	
	LOG_ERROR("Error: Not initialized\n");
	LOG_EXIT_INT(-1);
	return(-1);
}

static int
RSCT_get_all_nodes(unsigned int *num_entries,
		   ece_nodeid_t *nodes)
{
	int count;
	
	LOG_ENTRY();
	if(!nodes || !num_entries) {
		LOG_ERROR("Error: Invalid Input\n");
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}
	
	count = RSCT_mem_get_num_config_nodes();
	
	if( count > *num_entries){
		
		*num_entries=count;
		LOG_EXIT_INT(ENOSPC);
		return(ENOSPC);
	}
	
	RSCT_mem_get_all_nodes(nodes);
	
	LOG_EXIT_INT(0);
	return(0);
}

static int
RSCT_get_membership(ece_event_t* event)
{
	
	int ret;
	
	LOG_ENTRY();
	ret=RSCT_node_get_membership(event);
	
	LOG_EXIT_INT(ret);
	return(ret);
}

static int
RSCT_nodeid_to_string(const ece_nodeid_t *nodeid,
		      char *str,
		      uint *str_len)
{
	
	int len;
	LOG_ENTRY();
	if(!nodeid) {
		LOG_ERROR("Error: Invalid Input (Arg #1)\n");	
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}
	if((len = strlen((char *)nodeid)) >= *str_len){
		*str_len = len+1;
		LOG_EXIT_INT(ENOSPC);
		return(ENOSPC);
	}
	
	if(!str) {
		LOG_ERROR("Error: Invalid Input (Arg #2)\n");
		LOG_EXIT_INT(EINVAL);		
		return(EINVAL);
	}
	strncpy(str, (char *)nodeid, *str_len);
	str[len]='\0';
	
	LOG_EXIT_INT(0);
	return(0);
}

static int
RSCT_string_to_nodeid(const char* str,
		      ece_nodeid_t* nodeid)
{
	
	LOG_ENTRY();
	
	if(!str || !nodeid || strlen(str) >= sizeof(ece_nodeid_t)){
		LOG_ERROR("Error: Invalid Input\n");
		LOG_EXIT_INT(EINVAL);
		return(EINVAL);
	}
	bzero(nodeid,sizeof(ece_nodeid_t));
	strcpy((char *)nodeid, str);
	
	LOG_EXIT_INT(0);
	return(0);
}

static int
RSCT_get_plugin_info(char  * info_name,
		     extended_info_array_t** info_array){
	
	
	extended_info_array_t   * info = NULL;
	char                    buffer[51] = {0};
	int                     i = 0;
	
	LOG_ENTRY();
	
	/* Parameter check */
	if (!info_array) {
		RETURN(EFAULT);
	}
	
	if (!info_name) {
		if (!(info = ENGINE_ALLOC(sizeof(extended_info_array_t) +
					  sizeof(extended_info_t)*6))) {
			LOG_ERROR("Error allocating memory for info array\n");
			LOG_EXIT_INT(ENOMEM);
			return(ENOMEM);
		}
		
		info->info[i].name = EngFncs->engine_strdup("ShortName");
		info->info[i].title = EngFncs->engine_strdup(_("Short Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A short name given to this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(ece_plugin_record.short_name);
		i++;
		
		info->info[i].name = EngFncs->engine_strdup("LongName");
		info->info[i].title = EngFncs->engine_strdup(_("Long Name"));
		info->info[i].desc = EngFncs->engine_strdup(_("A longer, more descriptive name for this plug-in"));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(ece_plugin_record.long_name);
		i++;
		
		info->info[i].name = EngFncs->engine_strdup("Type");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Type"));
		info->info[i].desc =
			EngFncs->engine_strdup("There are various types of plug-ins, each responsible for some kind of storage object or logical volume."));
		info->info[i].type = EVMS_Type_String;
		info->info[i].value.s = EngFncs->engine_strdup(_("Cluster Manager"));
		i++;
		
		info->info[i].name = EngFncs->engine_strdup("Version");
		info->info[i].title = EngFncs->engine_strdup(_("Plug-in Version"));
		info->info[i].desc =
			EngFncs->engine_strdup(_("This is the version number of the plug-in."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d", MAJOR_VERSION,
			 MINOR_VERSION, PATCH_LEVEL);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;
		
		info->info[i].name = EngFncs->engine_strdup("Required_Engine_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Engine Services Version"));
		info->info[i].desc =
			EngFncs->engine_strdup(_("This is the version of the Engine services that this plug-in requires.  "
						 "It will not run on older versions of the Engine services."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d",
			 ece_plugin_record.required_engine_api_version.major,
			 ece_plugin_record.required_engine_api_version.minor,
			 ece_plugin_record.required_engine_api_version.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;
		
		info->info[i].name = EngFncs->engine_strdup("Required_Cluster_Version");
		info->info[i].title = EngFncs->engine_strdup(_("Required Cluster API Version"));
		info->info[i].desc =
			EngFncs->engine_strdup(_("This is the version of the Engine's cluster plug-in API that this plug-in requires.  "
						 "It will not run on older versions of the Engine's cluster plug-in API."));
		info->info[i].type = EVMS_Type_String;
		snprintf(buffer, 50, "%d.%d.%d",
			 ece_plugin_record.required_plugin_api_version.cluster.major,
			 ece_plugin_record.required_plugin_api_version.cluster.minor,
			 ece_plugin_record.required_plugin_api_version.cluster.patchlevel);
		info->info[i].value.s = EngFncs->engine_strdup(buffer);
		i++;
	} else {
		LOG_ERROR("No support for extra plugin information about \"%s\"\n", info_name);
		RETURN(EINVAL);
	}
	
	info->count = i;
	*info_array = info;
	
	LOG_EXIT_INT(0);
	return(0);
	
}

static int
RSCT_get_plugin_functions(ece_nodeid_t * nodeid,
			  function_info_array_t** actions)
{
	/* not sure what to fill here TOBEDONE*/
	LOG_ENTRY();
	LOG_ERROR("Error: Not supported Yet\n");
	
	LOG_EXIT_INT(ENOSYS);
	return(ENOSYS);
}


static int
RSCT_plugin_function(ece_nodeid_t * nodeid,
		     task_action_t  action,
		     list_anchor_t  objects,
		     option_array_t * options)
{
	
	/* not sure what to fill here TOBEDONE*/
	LOG_ENTRY();
	LOG_ERROR("Error: Not supported Yet\n");
	
	LOG_EXIT_INT(ENOSYS);
	return(ENOSYS);
}

static cluster_functions_t ft={
  setup_evms_plugin:          RSCT_setup_evms_plugin,
  cleanup_evms_plugin:        RSCT_cleanup_evms_plugin,
  register_callback:          RSCT_register_callback,
  unregister_callback:        RSCT_unregister_callback,
  send_msg:                   RSCT_send_msg,
  get_clusterid:              RSCT_get_clusterid,
  get_my_nodeid:              RSCT_get_my_nodeid,
  get_num_config_nodes:       RSCT_get_num_config_nodes,
  get_all_nodes:              RSCT_get_all_nodes,
  get_membership:             RSCT_get_membership,
  nodeid_to_string:           RSCT_nodeid_to_string,
  string_to_nodeid:           RSCT_string_to_nodeid,
  get_plugin_info:            RSCT_get_plugin_info,
  get_plugin_functions:       RSCT_get_plugin_functions,
  plugin_function:            RSCT_plugin_function
};


plugin_record_t ece_plugin_record = {
	
	id:			SetPluginID(EVMS_OEM_IBM,
					    EVMS_CLUSTER_MANAGER_INTERFACE_MODULE,
					    2),
	
	version:		{major:      MAJOR_VERSION,
				 minor:      MINOR_VERSION,
				 patchlevel: PATCH_LEVEL},
	
	required_engine_api_version: {major:      15,
				      minor:      0,
				      patchlevel: 0},
	
	required_plugin_api_version: {cluster: {major:      1,
						minor:      0,
						patchlevel: 0} },
	
	short_name:		"RSCT",
	long_name:		"Cluster Manager RSCT",
	oem_name:		"NCSA",
	
	functions:	{cluster:   &ft},
	
	container_functions:   NULL
};

plugin_record_t * evms_plugin_records[] = {&ece_plugin_record, NULL};

