# (C) Copyright IBM Corp. 2004
#
# This program is free software;  you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY;  without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
# the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program;  if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
#
# Common subroutines for calling dmsetup and getting Device-Mapper information.

package Evms::DM;

use strict;
use warnings;

use Evms::Log;

BEGIN {
	use Exporter();
	our (@ISA, @EXPORT);
        @ISA = qw(Exporter);
        @EXPORT = qw(&dm_create
		     &dm_remove
		     &dm_remove_all
		     &dm_suspend
		     &dm_resume
		     &dm_load
		     &dm_clear
		     &dm_rename
		     &dm_ls
		     &dm_info
		     &dm_deps
		     &dm_status
		     &dm_table
		     &dm_wait
		     &dm_targets
		     &dm_version
		     &dm_device_exists
		     &dm_get_device_number
		     &dm_get_event_number
		    );
}

# dm_create
sub dm_create($)
{
	my $name = $_[0];
	my $rc;

	`dmsetup create "$name" 2>&1`;
	$rc = $? >> 8;

	return $rc;
}

# dm_remove
sub dm_remove($)
{
	my $name = $_[0];
	my $rc;

	`dmsetup remove "$name" 2>&1`;
	$rc = $? >> 8;

	return $rc;
}

# dm_remove_all()
sub dm_remove_all
{
	`dmsetup remove_all 2>&1`;
}

# dm_suspend
sub dm_suspend($)
{
	my $name = $_[0];
	my $rc;

	`dmsetup suspend "$name" 2>&1`;
	$rc = $? >> 8;

	return $rc;
}

# dm_resume
sub dm_resume($)
{
	my $name = $_[0];
	my $rc;

	`dmsetup resume "$name" 2>&1`;
	$rc = $? >> 8;

	return $rc;
}

# dm_load
sub dm_load($ $)
{
	my $name = $_[0];
	my $table = $_[1];
	my $rc;

	`dmsetup load "$name" $table 2>&1`;
	$rc = $? >> 8;

	return $rc;

}

# dm_clear
sub dm_clear($)
{
	my $name = $_[0];
	my $rc;

	`dmsetup clear "$name" 2>&1`;
	$rc = $? >> 8;

	return $rc;
}

# dm_rename
sub dm_rename($ $)
{
	my $name = $_[0];
	my $new_name = $_[1];
	my $rc;

	`dmsetup rename "$name" "$new_name" 2>&1`;
	$rc = $? >> 8;

	return $rc;
}

# dm_ls
sub dm_ls()
{
	my @output;

	@output = `dmsetup ls 2>&1`;
	if ($? >> 8) {
		return;
	}

	if ($output[0] =~ /No devices found/) {
		return;
	}

	return @output;
}

# dm_info
sub dm_info($)
{
	my $name = $_[0];
	my @output;

	@output = `dmsetup info "$name" 2>&1`;
	if ($? >> 8) {
		return;
	}

	if ($output[0] =~ /No devices found/ ||
	    $output[0] =~ /Device does not exist/) {
		return;
	}

	return @output;
}

# dm_deps
sub dm_deps($)
{
	my $name = $_[0];
	my @output;

	@output = `dmsetup deps "$name" 2>&1`;
	if ($? >> 8) {
		return;
	}

	if ($output[0] =~ /No devices found/) {
		return;
	}

	return @output;
}

# dm_status
sub dm_status($)
{
	my $name = $_[0];
	my @output;

	@output = `dmsetup status "$name" 2>&1`;
	if ($? >> 8) {
		return;
	}

	if ($output[0] =~ /No devices found/) {
		return;
	}

	return @output;
}

# dm_table
sub dm_table($)
{
	my $name = $_[0];
	my @output;

	@output = `dmsetup table "$name" 2>&1`;
	if ($? >> 8) {
		return;
	}

	if ($output[0] =~ /No devices found/) {
		return;
	}

	return @output;
}

# dm_wait
sub dm_wait($)
{
	my $name = $_[0];
	my $rc;

	`dmsetup wait "$name" 2>&1`;
	$rc = $? >> 8;

	return $rc;
}

# dm_targets
sub dm_targets()
{
	my @output;

	@output = `dmsetup targets 2>&1`;
	if ($? >> 8) {
		return;
	}

	return @output;
}

# dm_version
sub dm_version()
{
	my @output;

	@output = `dmsetup version 2>&1`;
	if ($? >> 8) {
		return;
	}

	return @output;
}

# dm_device_exists
# Returns 0 if device exists, 1 if device doesn't exist.
sub dm_device_exists($)
{
	my $name = $_[0];
	my @info;

	@info = dm_info($name);

	return (@info) ? 0 : 1;
}

# dm_get_device_number
sub dm_get_device_number($)
{
	my $name = $_[0];
	my @info;
	my $line;
	my $major = 0;
	my $minor = 0;

	@info = dm_info($name);
	if (@info) {
		($line) = grep(/Major, minor/, @info);
		if ($line =~ /Major, minor:\s+(\d+),\s+(\d+)/) {
			$major = $1;
			$minor = $2;
		}
	}

	return ($major, $minor);
}

# dm_get_event_number
sub dm_get_event_number
{
	my $name = $_[0];
	my @info;
	my $line;
	my $event = -1;

	@info = dm_info($name);
	if (@info) {
		($line) = grep(/Event number/, @info);
		if ($line =~ /Event number:\s+(\d+)/) {
			$event = $1;
		}
	}

	return $event;
}

1;

