/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdrivelink.so
 *
 *   File: dl_helpers.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "dl_common.h"


/*
 *  Function: dl_get_drivelink_list
 *
 *  Called to build a list of drivelink storage objects.
 */
int dl_get_drivelink_list( list_anchor_t *list )
{
        int rc;
        list_anchor_t tlist;
        list_element_t iter1, iter2;
        storage_object_t * obj;

        LOG_ENTRY();

        tlist = EngFncs->allocate_list();

        REQUIRE(tlist != NULL);

        rc = EngFncs->get_object_list( 0,
                                              DATA_TYPE,
                                              dl_plugin_record,
                                              NULL,
                                              0,
                                              &tlist );
        if (rc==0) {
                LIST_FOR_EACH_SAFE(tlist, iter1, iter2, obj) {
                        if ( dl_isa_missing_child(obj)==TRUE ) {
                                LOG_DEBUG("pruning %s from drivelink list\n", obj->name);
                                EngFncs->delete_element(iter1);
                        }
                }
                *list = tlist;
        }
        else {
                EngFncs->destroy_list(tlist);
        }

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: dl_isa_parent
 *
 *  Called by a drivelink to see if the specified object is
 *  a top level parent object.  Meaning that it is found at
 *  a higher level in the feature stack.
 */
boolean dl_isa_parent( storage_object_t *parent,
                       storage_object_t *object )
{
        boolean rc = FALSE;
        storage_object_t *obj;
        list_element_t iter;

        LOG_ENTRY();

        LIST_FOR_EACH(object->parent_objects, iter, obj) {
                if (parent == obj) {
                        rc = TRUE;
                        break;
                }
                if (dl_isa_parent(parent,obj)==TRUE) {
                        rc = TRUE;
                        break;
                }
        }

        LOG_EXIT_BOOL(rc);
        return rc;
}


/*
 *  Function: dl_setup_geometry
 *
 *  Called to provide a geometry for the drivelink storage object.
 *  Child objects may differ in geometry and hard sector sizes. Still,
 *  a drivelink aggregate must provide basic geometry info in the
 *  drivelink storage object.  So, this routine will setup this
 *  information as best as it can.
 *
 *  The routine will actually walk the child objects and see if it
 *  can produce meaningful CHS values. However, the only fields of
 *  real importance are the hard sector size and block sizes.
 *  These fields are important to FSIM plugins.
 *
 *  Called by:  discovery code, create code, expand and shrink code
 */
int dl_setup_geometry( storage_object_t  *drivelink )
{
        storage_object_t *child;
        list_element_t iter;
        boolean got_initial_CHS_values = FALSE;
        boolean got_cylinder_size_consensus = TRUE;

        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(drivelink)==TRUE);

        // clear any possible old values ... we may have just finished
        // resizing the drivelink and have values from old drivelink object.
        memset( &drivelink->geometry, 0, sizeof(geometry_t) );

        // walk child objects
        LIST_FOR_EACH(drivelink->child_objects, iter, child) {

                if ( !dl_isa_missing_child(child) ) {

                        if ( got_initial_CHS_values == FALSE ) {

                                if ( (child->geometry.heads != 0) &&
                                     (child->geometry.sectors_per_track != 0)) {

                                        drivelink->geometry.heads             = child->geometry.heads;
                                        drivelink->geometry.sectors_per_track = child->geometry.sectors_per_track;

                                        got_initial_CHS_values = TRUE;
                                }

                        }
                        else if ( got_cylinder_size_consensus == TRUE ) {

                                if ( (child->geometry.heads != drivelink->geometry.heads ) ||
                                     (child->geometry.bytes_per_sector != drivelink->geometry.bytes_per_sector)) {
                                        got_cylinder_size_consensus = FALSE;
                                }

                        }

                        if (child->geometry.bytes_per_sector > drivelink->geometry.bytes_per_sector) {
                                drivelink->geometry.bytes_per_sector = child->geometry.bytes_per_sector;
                        }

                        if (child->geometry.block_size > drivelink->geometry.block_size) {
                                drivelink->geometry.block_size = child->geometry.block_size;
                        }

                }
        }


        // if we were unable to get default CHS values then contrive
        // fake ones using very very common values.
        if ( ( got_initial_CHS_values == FALSE ) ||
             ( got_cylinder_size_consensus == FALSE )) {
                drivelink->geometry.heads = 255;
                drivelink->geometry.sectors_per_track = 63;
        }

        // same with the hard sector size
        if ( drivelink->geometry.bytes_per_sector == 0 ) {
                LOG_DEBUG("faking hard sector size\n");
                drivelink->geometry.bytes_per_sector = 512;
        }

        // as well as the block size
        if ( drivelink->geometry.block_size == 0 ) {
                LOG_DEBUG("faking block size\n");
                drivelink->geometry.block_size = 1024;
        }

        // finally, calculate a cylinder count for this aggregate
        drivelink->geometry.cylinders = drivelink->size /
                                        (drivelink->geometry.heads*drivelink->geometry.sectors_per_track);

        LOG_DEBUG(" Drivelink Geometry ...\n");
        LOG_DEBUG("                cylinders: %"PRIu64"\n", drivelink->geometry.cylinders );
        LOG_DEBUG("                    heads: %d\n", drivelink->geometry.heads);
        LOG_DEBUG("                  sectors: %d\n", drivelink->geometry.sectors_per_track);
        LOG_DEBUG("      sector size (bytes): %d\n", drivelink->geometry.bytes_per_sector);
        LOG_DEBUG("       block size (bytes): %"PRIu64"\n", drivelink->geometry.block_size);

        LOG_EXIT_INT(0);
        return 0;
}


/*
 *  Function: dl_get_parent
 *
 *  Called to get the drivelink parent object.
 */
storage_object_t * dl_get_parent( storage_object_t  *object )
{
        storage_object_t *parent=NULL;

        LOG_ENTRY();

        if (object) {
                parent = EngFncs->first_thing(object->parent_objects, NULL);
                if (dl_isa_drivelink(parent) == FALSE ) {
                        parent = NULL;
                }
        }

        LOG_EXIT_PTR(parent);
        return parent;
}



/*
 *  Function:  dl_get_last_child
 *
 *  Called to find the last child storage object in a drivelink and return it.
 *
 *  Returns: pointer to last child storage object if successful
 *           NULL pointer otherwise.
 */
storage_object_t *  dl_get_last_child( storage_object_t *object )
{
        storage_object_t          *last_child=NULL;
        drivelink_private_data_t  *pdata;

        LOG_ENTRY();

        if ( dl_isa_drivelink(object) == TRUE ) {
                pdata =  (drivelink_private_data_t *) object->private_data;
                last_child = pdata->drive_link[pdata->drive_link_count-1].object;
        }

        LOG_EXIT_PTR(last_child);
        return last_child;
}


/*
 *  Function:  dl_isa_complete_aggregate
 *
 *  Called to test if the specified storage object is a completed
 *  drive link. Meaning ... we have found all the child storage
 *  objects.
 *
 *  If the link count says there are N children in the drivelink
 *  but when walking the ordering table we discover a missing
 *  child ... no storage object for a link in the ordering table ...
 *  then the drivelink is incomplete.
 *
 *  Returns: TRUE only if we have all the child objects needed.
 *
 */
boolean dl_isa_complete_aggregate( storage_object_t *drivelink )
{
        int     i;
        drivelink_private_data_t *pdata;

        if (dl_isa_drivelink(drivelink)==TRUE) {
                pdata = (drivelink_private_data_t *)drivelink->private_data;
                for (i=0; i<pdata->drive_link_count; i++) {
                        if ( pdata->drive_link[i].object == NULL ||
                                dl_isa_missing_child(pdata->drive_link[i].object)==TRUE) {
                                return FALSE;
                        }
                }
                return TRUE;
        }
        return FALSE;
}



/*
 *  Function: dl_build_feature_header
 *
 *  Called to build a drive link feature header for the specified
 *  child object link.
 *
 *
 *
 *  A picture is worth a thousand words
 *  ----------------------------------------------------------------------------------------------------+
 *           |  2nd copy of metadata  | 1st copy of metadata | 2nd feature header  | 1st feature header |
 *  ---------------------------------------------------------------------------------------------------+
 *                                                            LSN=n-1               LSN=n
 *
 *
 *  Returns: feature header in the supplied buffer if RC=0
 *
 */
int dl_build_feature_header( storage_object_t      *drivelink,
                             drive_link_t          *child,
                             evms_feature_header_t *fh    )
{
        int rc=0;
        drivelink_private_data_t *pdata;
        u_int32_t  feature_header_sequence_number=0;
        u_int32_t  feature_header_data1_start_lsn=0;
        u_int32_t  feature_header_data1_size     =0;
        u_int32_t  feature_header_data2_start_lsn=0;
        u_int32_t  feature_header_data2_size     =0;

        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(drivelink) == TRUE);
        REQUIRE(child != NULL);
        REQUIRE(fh != NULL);

        LOG_DEBUG("building feature header for object %s\n", child->object->name );

        pdata = (drivelink_private_data_t *)drivelink->private_data;

        // provide an incremented feature header sequence number
        feature_header_sequence_number   = pdata->sequence_number;

        // 1st copy of metadata is found immediately before feature header sectors
        feature_header_data1_start_lsn =  child->object->size -
                                          (FEATURE_HEADER_SECTOR_COUNT*2) -
                                          DRIVELINK_METADATA_SECTOR_COUNT;

        feature_header_data1_size      =  DRIVELINK_METADATA_SECTOR_COUNT;

        // 2nd copy of metadata is found immediately before 1st copy of metadata
        feature_header_data2_start_lsn = feature_header_data1_start_lsn - DRIVELINK_METADATA_SECTOR_COUNT;
        feature_header_data2_size      = DRIVELINK_METADATA_SECTOR_COUNT;

        fh->signature                  = EVMS_FEATURE_HEADER_SIGNATURE;
        fh->feature_id                 = EVMS_DRIVE_LINK_PLUGIN_ID;
        fh->sequence_number            = feature_header_sequence_number;
        fh->feature_data1_size         = feature_header_data1_size;
        fh->feature_data2_size         = feature_header_data2_size;
        fh->feature_data1_start_lsn    = feature_header_data1_start_lsn;
        fh->feature_data2_start_lsn    = feature_header_data2_start_lsn;

        strncpy( fh->object_name, pdata->parent_object_name,EVMS_VOLUME_NAME_SIZE );

        child->object->flags |= SOFLAG_FEATURE_HEADER_DIRTY;

        LOG_EXIT_INT(rc);
        return rc;
}


/*
 *  Function: dl_free_object
 *
 *  Frees up a drive link object that I allocated using the
 *  malloc_drive_link_object() routine.
 *
 *  Returns: nothing
 *
 */
void dl_free_drivelink_object( storage_object_t  *object )
{
        if (object) {
                drivelink_private_data_t * pdata = (drivelink_private_data_t *)object->private_data;
                if (pdata) {
                        EngFncs->unregister_name( object->name);
                        dl_unregister_serial_number(pdata->parent_serial_number);
                        free(object->private_data);
                        object->private_data = NULL;
                }
                EngFncs->free_evms_object(object);
        }
}


/*
 *
 *  Function: dl_malloc_object
 *
 *  Routine will allocate memory for a new parent drive link storage object
 *  and initialize the new object for the caller.  The only parm is
 *  the parent's drive link serial number.
 *
 *  Returns: address of new storage object if successful, NULL otherwise
 *
 */
storage_object_t * dl_malloc_drivelink_object( void )
{
        int                 rc;
        storage_object_t   *object = NULL;
        drivelink_private_data_t *pdata;

        rc = EngFncs->allocate_evms_object(NULL, &object);
        if ( rc == 0 ) {

                pdata = (drivelink_private_data_t *) calloc(1,sizeof(drivelink_private_data_t));

                if ( pdata == NULL ) {
                        EngFncs->free_evms_object( object );
                        object = NULL;
                }
                else {
                        pdata->signature     = EVMS_DRIVELINK_SIGNATURE;
                        object->plugin       = dl_plugin_record;
                        object->private_data = pdata;
                }
        }

        return object;
}


/*
 *  Function: dl_build_ordered_child_object_list
 *
 *  Called to build a child_objects list from the drivelink
 *  ordering table.
 *
 *  Returns 0 if successful and the child_objects
 *  list should contain all the objects from the drive link
 *  in the proper order.
 */
int dl_build_ordered_child_object_list( storage_object_t  *drivelink, list_anchor_t *target_list )
{
        int rc, i;
        list_anchor_t objects;
        list_element_t iter;
        drivelink_private_data_t *pdata;

        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(drivelink) == TRUE);
        REQUIRE(target_list != NULL);

        pdata   = (drivelink_private_data_t *) drivelink->private_data;
        objects = *target_list;

        REQUIRE(objects != NULL);

        EngFncs->delete_all_elements(objects);

        for(i=0,rc=0; (i<pdata->drive_link_count)&&(rc==0); i++) {
                if ( pdata->drive_link[i].object != NULL ) {
                        iter = EngFncs->insert_thing(objects, pdata->drive_link[i].object,
                                                     INSERT_AFTER, NULL);
                        if (!iter) {
                                rc = ENOMEM;
                        }
                }
        }

        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Function: dl_table_fixup
 *
 *  Called to walk a drivelink table and build missing
 *  child objects when a child link is missing from the
 *  table. On exit, the drivelink is complete ... though it
 *  may have some missing child objects.
 */
int dl_table_fixup(storage_object_t * drivelink)
{
        int i,rc;
        drivelink_private_data_t *pdata;

        LOG_ENTRY();
        REQUIRE(dl_isa_drivelink(drivelink)==TRUE);
        pdata = (drivelink_private_data_t *)drivelink->private_data;
        for (i=0,rc=0; i<pdata->drive_link_count && rc==0; i++) {
                if (pdata->drive_link[i].object == NULL) {
                        rc = dl_build_missing_child( drivelink, i );
                }
        }
        if (rc==0) {
                rc = dl_build_ordered_child_object_list( drivelink, &drivelink->child_objects );
        }
        LOG_EXIT_INT(rc);
        return rc;
}

/*
 *  Function: dl_build_linear_mapping
 *
 *  Called to walk a drivelink and create a mapping between the
 *  logical sector numbers of the drivelink and the useable areas
 *  of the child objects.
 */
int dl_build_linear_mapping( storage_object_t *drivelink )
{
        int i, rc=0;
        lsn_t lsn=0;
        drivelink_private_data_t *pdata;

        LOG_ENTRY();

        REQUIRE(dl_isa_drivelink(drivelink) == TRUE);

        pdata = (drivelink_private_data_t *)drivelink->private_data;

        drivelink->size = 0;

        for (i=0; i<pdata->drive_link_count; i++) {

                drivelink->size += pdata->ordering_table[i].child_vsize;

                pdata->drive_link[i].start_lsn = lsn;
                pdata->drive_link[i].end_lsn   = lsn + pdata->ordering_table[i].child_vsize - 1;

                lsn = pdata->drive_link[i].end_lsn + 1;
        }

        LOG_DEBUG("Drivelink (%s) Size= %"PRIu64"  Linear Mapping ....\n", drivelink->name, drivelink->size);
        for (i=0; i<pdata->drive_link_count; i++) {
                if (pdata->drive_link[i].object == NULL) {
                        LOG_DEBUG("\tChild: n/a\n");
                }
                else {
                        LOG_DEBUG("\tChild: %s\n",  pdata->drive_link[i].object->name);
                }
                LOG_DEBUG( "\t\tstart_lsn= %"PRIu64"   end_lsn= %"PRIu64"   size= %"PRIu64"\n",
                           pdata->drive_link[i].start_lsn,
                           pdata->drive_link[i].end_lsn,
                           pdata->drive_link[i].sector_count );
        }

        LOG_EXIT_INT(rc);
        return rc;
}
