/*
 *
 *   (C) Copyright IBM Corp. 2001, 2003
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 *   Module: libdrivelink.so
 *
 *   File: dl_helpers.h
 */
#ifndef _DL_HELPERS_HEADER_
#define _DL_HELPERS_HEADER_  1

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>

#include <plugin.h>

#include "dl_common.h"

int 			dl_get_drivelink_list( list_anchor_t *list );

boolean 		dl_isa_parent( storage_object_t *parent,storage_object_t *object );

int 			dl_setup_geometry( storage_object_t  *dlink );

storage_object_t * 	dl_get_parent( storage_object_t  *object );

storage_object_t *  	dl_get_last_child( storage_object_t *object );

boolean 		dl_isa_complete_aggregate( storage_object_t *parent );

int 			dl_build_feature_header( storage_object_t      *object,         
				       		 drive_link_t          *link,           
				       		 evms_feature_header_t *fh    );

void 			dl_free_drivelink_object( storage_object_t  *object );

storage_object_t * 	dl_malloc_drivelink_object( void );

int 			dl_build_ordered_child_object_list( storage_object_t  *dlink,
							    list_anchor_t     *target_list );

int 			dl_table_fixup(storage_object_t * drivelink);

int 			dl_build_linear_mapping( storage_object_t *drivelink );


/*
 * Tests if the storage object is a valid drivelink object
 */
static inline boolean dl_isa_drivelink( storage_object_t *object )
{

	if ( object != NULL &&
	     object->plugin == dl_plugin_record &&
	     object->private_data != NULL ) {

                if ( ((drivelink_private_data_t *)object->private_data)->signature==EVMS_DRIVELINK_SIGNATURE) {
			return TRUE;
                }

	}

	return FALSE;
}

/* 
 * Determines if the child serial number is a duplicate by seeing if it
 * already appears in the drivelink ordering table. Child serial numbers
 * are only unique within their drivelink.
 */
static inline boolean  dl_isa_dup_child_sn( storage_object_t *drivelink, u_int32_t sn )        
{
	int i;
	drivelink_private_data_t *pdata = (drivelink_private_data_t *)drivelink->private_data;      
	dot_entry_t *table;

	table = &pdata->ordering_table[0];

	for (i=0; i<pdata->drive_link_count; i++) {
		if ( ((dot_entry_t *)table+i)->child_serial_number == sn ) {
			return TRUE;
		}
	}
	return FALSE;
}

/* 
 * Returns the index into the drivelink ordering table for the specified child
 * link object.  A child object is specified by its serial number.
 */
static inline int dl_get_child_index_by_sn( storage_object_t *drivelink, u_int32_t sn )        
{
	int i;
	drivelink_private_data_t *pdata = (drivelink_private_data_t *)drivelink->private_data;      
	dot_entry_t *table;
	
	table = &pdata->ordering_table[0];

	for (i=0; i<pdata->drive_link_count; i++) {
		if ( ((dot_entry_t *)table+i)->child_serial_number == sn ) {
			return i;
		}
	}
	return BAD_DOT_INDEX;
}

/*
 * Converts drivelink metadata from DISK format to CPU format
 */
static inline void dl_disk_metadata_to_cpu( drivelink_metadata_t *metadata )
{
	int i;

	metadata->signature              = DISK_TO_CPU32(metadata->signature);
	metadata->crc                    = DISK_TO_CPU32(metadata->crc);
	metadata->version.major          = DISK_TO_CPU32(metadata->version.major);
	metadata->version.minor          = DISK_TO_CPU32(metadata->version.minor);
	metadata->version.patchlevel     = DISK_TO_CPU32(metadata->version.patchlevel);
	metadata->flags                  = DISK_TO_CPU32(metadata->flags);
	metadata->sequence_number        = DISK_TO_CPU64(metadata->sequence_number);
	metadata->child_serial_number    = DISK_TO_CPU64(metadata->child_serial_number);
	metadata->parent_serial_number   = DISK_TO_CPU64(metadata->parent_serial_number);
	metadata->child_count            = DISK_TO_CPU64(metadata->child_count);

	for (i=0; i<EVMS_DRIVELINK_MAX_ENTRIES; i++) {
		metadata->ordering_table[i].child_serial_number = DISK_TO_CPU64(metadata->ordering_table[i].child_serial_number);
		metadata->ordering_table[i].child_vsize         = DISK_TO_CPU64(metadata->ordering_table[i].child_vsize );
	}

}

/*
 * Converts drivelink metadata from CPU format to DISK format
 */
static inline void dl_cpu_metadata_to_disk( drivelink_metadata_t *metadata )
{
	int i;

	metadata->signature              = CPU_TO_DISK32(metadata->signature);
	metadata->crc                    = CPU_TO_DISK32(metadata->crc);
	metadata->version.major          = CPU_TO_DISK32(metadata->version.major);
	metadata->version.minor          = CPU_TO_DISK32(metadata->version.minor);
	metadata->version.patchlevel     = CPU_TO_DISK32(metadata->version.patchlevel);
	metadata->flags                  = CPU_TO_DISK32(metadata->flags);
	metadata->sequence_number        = CPU_TO_DISK64(metadata->sequence_number);
	metadata->child_serial_number    = CPU_TO_DISK64(metadata->child_serial_number);
	metadata->parent_serial_number   = CPU_TO_DISK64(metadata->parent_serial_number);
	metadata->child_count            = CPU_TO_DISK64(metadata->child_count);

	for (i=0; i<EVMS_DRIVELINK_MAX_ENTRIES; i++) {
		metadata->ordering_table[i].child_serial_number = CPU_TO_DISK64(metadata->ordering_table[i].child_serial_number);
		metadata->ordering_table[i].child_vsize         = CPU_TO_DISK64(metadata->ordering_table[i].child_vsize );
	}

}

/*
 * Debug routine to display a drivelink feature header 
 */ 
static inline int dl_display_feature_header(evms_feature_header_t  *fh)
{
	if (fh!=NULL) {
		LOG_DEBUG("Feature Header ...\n");
		LOG_DEBUG("\t       FH      flags: 0x%X\n", fh->flags );
		LOG_DEBUG("\t       FH    feat id: 0x%X\n", fh->feature_id );
		LOG_DEBUG("\t       FH      major: 0x%X\n", fh->version.major );
		LOG_DEBUG("\t       FH      minor: 0x%X\n", fh->version.minor );
		LOG_DEBUG("\t       FH      patch: 0x%X\n", fh->version.patchlevel );
		LOG_DEBUG("\t       FH   sequence: %"PRIu64"\n", fh->sequence_number );
		LOG_DEBUG("\t       FH    padding: %"PRIu64"\n", fh->alignment_padding );
		LOG_DEBUG("\t       FH       lsn1: %"PRIu64"\n", fh->feature_data1_start_lsn );
		LOG_DEBUG("\t       FH data size1: %"PRIu64"\n", fh->feature_data1_size );
		LOG_DEBUG("\t       FH       lsn2: %"PRIu64"\n", fh->feature_data2_start_lsn );
		LOG_DEBUG("\t       FH data size2: %"PRIu64"\n", fh->feature_data2_size );
		LOG_DEBUG("\t       FH  obj depth: %d\n",   fh->object_depth);
		LOG_DEBUG("\t       FH   obj name: %s\n",   fh->object_name);
		return 0;
	}
	return EINVAL;
}
#endif
