/*
 *   (C) Copyright IBM Corp. 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * Module: Error Plugin
 * File: evms2/engine/plugins/error/options.c
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <plugin.h>
#include "error.h"

/**
 * generate_error_name
 **/
void generate_error_name(char *input_name, char *error_name)
{
	LOG_ENTRY();
	snprintf(error_name, EVMS_NAME_SIZE, "%s/%s", ERROR_NAME, input_name);
	LOG_EXIT_VOID();
}

/**
 * init_create_task
 **/
int init_create_task(task_context_t *context)
{
	option_desc_array_t *od = context->option_descriptors;
	int i, rc = 0;

	LOG_ENTRY();

	/* Initialize the "Name" option. Allocate space for the string value
	 * now so we can just strcpy during set_option().
	 */
	i = ERROR_OPTION_NAME_IDX;
	od->option[i].name = EngFncs->engine_strdup(ERROR_OPTION_NAME_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Name for the new error object."));
	od->option[i].type = EVMS_Type_String;
	od->option[i].min_len = 1;
	od->option[i].max_len = EVMS_NAME_SIZE;
	od->option[i].flags = EVMS_OPTION_FLAGS_NO_INITIAL_VALUE;
	od->option[i].value.s = EngFncs->engine_alloc(EVMS_NAME_SIZE + 1);
	if (!od->option[i].value.s) {
		rc = ENOMEM;
		goto out;
	}

	/* Initialize the "Size" option. */
	i = ERROR_OPTION_SIZE_IDX;
	od->option[i].name = EngFncs->engine_strdup(ERROR_OPTION_SIZE_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Size of the new error object."));
	od->option[i].type = EVMS_Type_Unsigned_Int64;
	od->option[i].unit = EVMS_Unit_Sectors;
	od->option[i].flags = EVMS_OPTION_FLAGS_AUTOMATIC |
			      EVMS_OPTION_FLAGS_NOT_REQUIRED;
	od->option[i].value.ui64 = (u_int64_t) -1;

	/* Initialize the "Type" option. */
	i = ERROR_OPTION_TYPE_IDX;
	od->option[i].name = EngFncs->engine_strdup(ERROR_OPTION_TYPE_STR);
	od->option[i].title = EngFncs->engine_strdup(_("Type of the new error object."));
	od->option[i].type = EVMS_Type_String;
	od->option[i].min_len = 1;
	od->option[i].max_len = EVMS_NAME_SIZE;
	od->option[i].flags = EVMS_OPTION_FLAGS_AUTOMATIC |
			      EVMS_OPTION_FLAGS_NOT_REQUIRED;
	od->option[i].constraint_type = EVMS_Collection_List;
	od->option[i].constraint.list = EngFncs->engine_alloc(sizeof(value_list_t) +
							      sizeof(value_t) * 4);
	od->option[i].value.s = EngFncs->engine_alloc(EVMS_NAME_SIZE + 1);
	if (!od->option[i].constraint.list || !od->option[i].value.s) {
		rc = ENOMEM;
		goto out;
	}
	od->option[i].constraint.list->value[0].s = EngFncs->engine_strdup("Disk");
	od->option[i].constraint.list->value[1].s = EngFncs->engine_strdup("Segment");
	od->option[i].constraint.list->value[2].s = EngFncs->engine_strdup("Region");
	od->option[i].constraint.list->value[3].s = EngFncs->engine_strdup("Feature_Object");
	od->option[i].constraint.list->count = 4;
	strncpy(od->option[i].value.s, "Disk", EVMS_NAME_SIZE);

	od->count = ERROR_OPTION_COUNT;

out:
	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * set_create_option
 **/
int set_create_option(task_context_t *context,
		      u_int32_t index,
		      value_t *value,
		      task_effect_t *effect)
{
	option_desc_array_t *od = context->option_descriptors;
	char error_name[EVMS_NAME_SIZE+1];
	int i, rc = EINVAL;

	LOG_ENTRY();

	switch (index) {
	case ERROR_OPTION_NAME_IDX:
		generate_error_name(value->s, error_name);
		rc = EngFncs->register_name(error_name);
		if (!rc) {
			EngFncs->unregister_name(error_name);
			strncpy(od->option[index].value.s,
				value->s, EVMS_NAME_SIZE);
		}
		break;

	case ERROR_OPTION_SIZE_IDX:
		od->option[index].value.ui64 = value->ui64;
		rc = 0;
		break;

	case ERROR_OPTION_TYPE_IDX:
		for (i = 0; i < od->option[index].constraint.list->count; i++) {
			if (! strcmp(od->option[index].constraint.list->value[i].s, value->s)) {
				strncpy(od->option[index].value.s,
					value->s, EVMS_NAME_SIZE);
				rc = 0;
			}
		}
		break;

	default:
		break;
	}

	LOG_EXIT_INT(rc);
	return rc;
}

/**
 * get_object_type
 **/
static int get_object_type(char *type_string)
{
	int type = 0;

	LOG_ENTRY();

	if (! strcasecmp(type_string, "Disk")) {
		type = DISK;
	} else if (! strcasecmp(type_string, "Segment")) {
		type = SEGMENT;
	} else if (! strcasecmp(type_string, "Region")) {
		type = REGION;
	} else if (! strcasecmp(type_string, "Feature_Object")) {
		type = EVMS_OBJECT;
	}

	LOG_EXIT_INT(type);
	return type;
}

/**
 * parse_options
 **/
void parse_options(option_array_t *options,
		   char **input_name,
		   u_int64_t *size,
		   object_type_t *type)
{
	u_int i;

	LOG_ENTRY();

	/* Default values. */
	*input_name = NULL;
	*size = (u_int64_t) -1;
	*type = DISK;

	for (i = 0; i < options->count; i++) {
		/* If only the name-based index is specified, get the number. */
		if (!options->option[i].is_number_based) {
			if (!strcmp(options->option[i].name,
				    ERROR_OPTION_NAME_STR)) {
				options->option[i].number = ERROR_OPTION_NAME_IDX;

			} else if (!strcmp(options->option[i].name,
					   ERROR_OPTION_SIZE_STR)) {
				options->option[i].number = ERROR_OPTION_SIZE_IDX;

			} else if (!strcmp(options->option[i].name,
					   ERROR_OPTION_TYPE_STR)) {
				options->option[i].number = ERROR_OPTION_TYPE_IDX;

			} else {
				continue;
			}
		}

		/* Use the number-based index to record the option. */
		switch (options->option[i].number) {
		case ERROR_OPTION_NAME_IDX:
			*input_name = options->option[i].value.s;
			LOG_DEBUG("Name option: %s\n", *input_name);
			break;

		case ERROR_OPTION_SIZE_IDX:
			*size = options->option[i].value.ui64;
			LOG_DEBUG("Size option: %"PRIu64"\n", *size);
			break;

		case ERROR_OPTION_TYPE_IDX:
			*type = get_object_type(options->option[i].value.s);
			LOG_DEBUG("Type option: %s\n", options->option[i].value.s);
			break;

		default:
			break;
		}
	}

	LOG_EXIT_VOID();
}

/**
 * verify_options
 **/
int verify_options(char *input_name, object_type_t type)
{
	char error_name[EVMS_NAME_SIZE+1];
	int rc;

	LOG_ENTRY();

	if (!input_name) {
		LOG_ERROR("No name specified for new object.\n");
		rc = EINVAL;
		goto out;
	}

	/* Check that the specified name isn't in use already. */
	generate_error_name(input_name, error_name);
	rc = EngFncs->register_name(error_name);
	if (rc) {
		LOG_ERROR("Name \"%s\" already in use.\n", error_name);
		goto out;
	}
	EngFncs->unregister_name(error_name);

	/* Make sure the type is valid for a storage-object. */
	if (!(type == DISK   || type == SEGMENT ||
	      type == REGION || type == EVMS_OBJECT)) {
		LOG_ERROR("Type %u isn't a valid object type.\n", type);
		rc = EINVAL;
		goto out;
	}

out:
	LOG_EXIT_INT(rc);
	return rc;
}

