/*
 *   (C) Copyright IBM Corp. 2001, 2004
 *
 *   This program is free software;  you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY;  without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See
 *   the GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program;  if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * evms_metadata_restore
 *
 * Command-line utility to restore the EVMS metadata saved in a file.
 */

#define _GNU_SOURCE     /* Used to pick up definition of lseek64 to avoid */
			/* compiler warning messages. */
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <dirent.h>
#include <fcntl.h>
#include <getopt.h>
#include <libgen.h>
#include <frontend.h>
#include <locale.h>
#include <langinfo.h>
#include <limits.h>
#include <time.h>
#include <enginestructs.h>

#define REQUIRED_ENGINE_API_MAJOR 10
#define REQUIRED_ENGINE_API_MINOR 1
#define REQUIRED_ENGINE_API_PATCH 0

/* Engine APIs that are not published in appAPI.h */
int evms_engine_read(object_handle_t handle,
		     lsn_t           lsn,
		     sector_count_t  length,
		     void          * buffer);

int evms_engine_write(object_handle_t handle,
		      lsn_t           lsn,
		      sector_count_t  length,
		      void          * buffer);

int evms_mark_for_rediscover(char * name);

int evms_rediscover(void);


static char * prog_name;
static boolean restore_all = FALSE;
static debug_level_t debug_level = -1;
static char * log_file = NULL;
static boolean list = FALSE;
static char * date_string = NULL;
static char * metadata_file = NULL;
static boolean recursive = FALSE;
static boolean single = FALSE;
static boolean parents = FALSE;
static boolean quiet   = FALSE;
static boolean verbose = FALSE;
static char date_time_format[24] = {0};
static char date_time_format_help[32] = {0};
static const char * default_metadata_dir = DEFAULT_METADATA_DIR;
#define STANDARD_DATE_TIME_FORMAT	"%Y/%m/%d%n%H:%M:%S"
#define STANDARD_DATE_TIME_FORMAT_HELP	"yyyy/mm/md[ HH:MM:SS]"
/* Some countries use '.' for a date separator. */
#define STANDARD_DATE_TIME_FORMAT_ALT	"%Y.%m.%d%n%H:%M:%S"


typedef struct metadata_info_s {
	struct metadata_info_s * next;
	char                   * parent_name;
	char                   * child_name;
	sector_count_t           offset;
	sector_count_t           length;
	int			 flags;
	void                   * metadata;
} metadata_info_t;

#define MI_VALID_ON_DISK	(1<<0)

typedef struct tree_node_s {
	char                   * name;
	struct tree_node_s     * parent;
	struct tree_node_s     * sibling;
	struct tree_node_s     * child;
	struct metadata_info_s * info;
} tree_node_t;

char msg_buf[10240];

#define LOG_CRITICAL(msg, args...)	do { sprintf(msg_buf, "%s: %s", __FUNCTION__, msg); \
					     evms_write_log_entry(CRITICAL,	prog_name, msg_buf , ## args);} while (0)
#define LOG_SERIOUS(msg, args...)	do { sprintf(msg_buf, "%s: %s", __FUNCTION__, msg); \
					     evms_write_log_entry(SERIOUS,	prog_name, msg_buf , ## args);} while (0)
#define LOG_ERROR(msg, args...)		do { sprintf(msg_buf, "%s: %s", __FUNCTION__, msg); \
					     evms_write_log_entry(ERROR,	prog_name, msg_buf , ## args);} while (0)
#define LOG_WARNING(msg, args...)	do { sprintf(msg_buf, "%s: %s", __FUNCTION__, msg); \
					     evms_write_log_entry(WARNING,	prog_name, msg_buf , ## args);} while (0)
#define LOG_DEFAULT(msg, args...)	do { sprintf(msg_buf, "%s: %s", __FUNCTION__, msg); \
					     evms_write_log_entry(DEFAULT,	prog_name, msg_buf , ## args);} while (0)
#define LOG_DETAILS(msg, args...)	do { sprintf(msg_buf, "%s: %s", __FUNCTION__, msg); \
					     evms_write_log_entry(DETAILS,	prog_name, msg_buf , ## args);} while (0)
#define LOG_DEBUG(msg, args...)		do { sprintf(msg_buf, "%s: %s", __FUNCTION__, msg); \
					     evms_write_log_entry(DEBUG,	prog_name, msg_buf , ## args);} while (0)
#define LOG_EXTRA(msg, args...)		do { sprintf(msg_buf, "%s: %s", __FUNCTION__, msg); \
					     evms_write_log_entry(EXTRA,	prog_name, msg_buf , ## args);} while (0)
#define LOG_ENTRY_EXIT(msg, args...)	do { sprintf(msg_buf, "%s: %s", __FUNCTION__, msg); \
					     evms_write_log_entry(ENTRY_EXIT,	prog_name, msg_buf , ## args);} while (0)
#define LOG_EVERYTHING(msg, args...)	do { sprintf(msg_buf, "%s: %s", __FUNCTION__, msg); \
					     evms_write_log_entry(EVERYTHING,	prog_name, msg_buf , ## args);} while (0)

/* Macros to log entry and exit from subroutines. */
#define LOG_ENTRY()			LOG_ENTRY_EXIT("Enter.\n")
#define LOG_EXIT_INT(x)			LOG_ENTRY_EXIT("Exit.  Return value = %d\n", x)
#define LOG_EXIT_U64(x)			LOG_ENTRY_EXIT("Exit.  Return value = %"PRIu64"\n", x)
#define LOG_EXIT_PTR(x)			LOG_ENTRY_EXIT("Exit.  Return pointer = %p\n", x)
#define LOG_EXIT_BOOL(x)		LOG_ENTRY_EXIT("Exit.  Return is %s\n", x ? "TRUE" : "FALSE")
#define LOG_EXIT_VOID()			LOG_ENTRY_EXIT("Exit.\n")

#define MSG(msg, args...)         do { if (!quiet) { printf(msg , ## args); } LOG_DEFAULT(msg , ## args); } while (0)
#define VERBOSE_MSG(msg, args...) do { if (verbose) { printf(msg , ## args); } LOG_DEBUG(msg , ## args); } while (0)
#define ERROR_MSG(msg, args...)   do { fprintf(stderr, msg , ## args); LOG_CRITICAL(msg , ## args); } while (0)


static boolean is_numeric(char * str) {
	while (*str != '\0') {
		if ((*str < '0') || (*str > '9')) {
			return FALSE;
		}
		str++;
	}
	return TRUE;
}


static void build_date_time_format(void) {
	strcpy(date_time_format, nl_langinfo(D_FMT));
	strcat(date_time_format, "%n");
	strcat(date_time_format, nl_langinfo(T_FMT));
}


static void build_date_time_format_help(void) {

	char * format;
	char * pch;

	format = nl_langinfo(D_FMT);

	for (pch = format; *pch != '\0'; pch++) {
		if (*pch == '%') {
			switch (*(pch + 1)) {
				case 'd':
				case 'e':
					strcat(date_time_format_help, "dd");
					break;

				case 'D':
					strcat(date_time_format_help, "mm/dd/yy");
					break;

				case 'm':
					strcat(date_time_format_help, "mm");
					break;

				case 'y':
					strcat(date_time_format_help, "yy");
					break;

				case 'Y':
					strcat(date_time_format_help, "yyyy");
					break;
			}
			/* Skip over '%'. */
			pch++;

		} else {
			int len = strlen(date_time_format_help);

			date_time_format_help[len] = *pch;
			date_time_format_help[len+1] = '\0';
		}
	}

	strcat(date_time_format_help, "[ ");

	format = nl_langinfo(T_FMT);

	for (pch = format; *pch != '\0'; pch++) {
		if (*pch == '%') {
			switch (*(pch + 1)) {
				case 'H':
				case 'I':
					strcat(date_time_format_help, "HH");
					break;

				case 'M':
					strcat(date_time_format_help, "MM");
					break;

				case 'n':
				case 't':
					strcat(date_time_format_help, " ");
					break;

				case 'p':
					if (nl_langinfo(T_FMT_AMPM) == NULL) {
						break;
					}
					if (nl_langinfo(T_FMT_AMPM)[0] != '\0') {
						strcat(date_time_format_help, "(");
						strcat(date_time_format_help, nl_langinfo(AM_STR));
						strcat(date_time_format_help, "|");
						strcat(date_time_format_help, nl_langinfo(PM_STR));
						strcat(date_time_format_help, ")");
					}
					break;

				case 'R':
					strcat(date_time_format_help, "HH:MM");
					break;

				case 'r':
					strcat(date_time_format_help, "HH:MM:SS");

					if (nl_langinfo(T_FMT_AMPM) == NULL) {
						break;
					}
					if (nl_langinfo(T_FMT_AMPM)[0] != '\0') {
						strcat(date_time_format_help, " (");
						strcat(date_time_format_help, nl_langinfo(AM_STR));
						strcat(date_time_format_help, "|");
						strcat(date_time_format_help, nl_langinfo(PM_STR));
						strcat(date_time_format_help, ")");
					}
					break;

				case 'S':
					strcat(date_time_format_help, "SS");
					break;

				case 'T':
					strcat(date_time_format_help, "HH:MM:SS");
					break;

			}
			/* Skip over '%'. */
			pch++;

		} else {
			int len = strlen(date_time_format_help);

			date_time_format_help[len] = *pch;
			date_time_format_help[len+1] = '\0';
		}
	}

	strcat(date_time_format_help, "]");
}


static void show_help(void) {

	build_date_time_format_help();

	     /*---------------------------75 columns--------------------------------------*/
	MSG(_("\nUsage: %s [options] [thing_name ...]\n"), prog_name);
	MSG(_("Restore backed up metadata.\n\n"

	      "Options:\n"
	      "  [-a, --all]\n"
	      "        Restore the metadata for all objects, containers, and volumes.\n"
	      "        You must use -a if you do not give any thing names.\n"
	      "  [-d debug_level, --debug-level debug_level]\n"
	      "        debug_level = [0-9] |\n"
	      "                      [critical | serious | error | warning | default |\n"
	      "                       details | entry_exit | debug | extra | everything]\n"
	      "  [-D date/time, --date date/time]\n"
	      "        Use backup file that is the most recent to the date/time.\n"
	      "        The date/time is in the format \"%s\"\n"
	      "        or in the format \"%s\"\n"
	      "  [-f metadata_file_name, --metadata-file metadata_file_name]\n"
	      "        If neither -D nor -f are not used, the most recent file in the \n"
	      "        default directory will be used.\n"
	      "        The default directory is %s.\n"
	      "  [-l log_file_name, --log-file log_file_name]\n"
	      "        EVMS log file name\n"
	      "  [-L, --list]\n"
	      "        Print the contents of the metadata file to standard out.\n"
	      "        -L cannot be used with -r nor with -s.\n"
	      "  [-p, --parents]\n"
	      "        Restore the parent(s) built from the things rather than the\n"
              "        metadata to build the things.\n"
	      "  [-r, --recursive]\n"
	      "        Restore the metadata to build the things as well as restoring any\n"
	      "        metadata to build the children of the things.\n"
	      "        If -p is used, restore all the parents built from the list of\n"
	      "        things and then recursively restore all the parents build from those\n"
	      "        parents.\n"
	      "        -r cannot be used with -s nor with -L.\n"
	      "        -r is the default behavior.\n"
	      "  [-s, --single]\n"
	      "        Restore only the metadata needed to build the things from their\n"
	      "        child objects.  Do not restore the metadata to build the child\n"
	      "        objects.\n"
	      "        If -p us used, restore any parents built from the things.\n"
	      "        -s cannot be used with -r nor with -L.\n"
	      "  [-q, --quiet]\n"
	      "        Quiet mode\n"
	      "  [-v, --verbose]\n"
	      "        Verbose mode\n"
	      "  [-h | -? | --help]\n"
	      "        Display this help\n\n"),
	    date_time_format_help,
	    STANDARD_DATE_TIME_FORMAT_HELP,
	    default_metadata_dir);
}


static int parse_options(int      argc,
			 char * * argv) {

	int c, rc = 0;
	char * short_opts = "ad:D:f:l:Lprsqvh?";
	struct option long_opts[] = { { "all",           no_argument,       NULL, 'a'},
				      { "debug-level",   required_argument, NULL, 'd'},
				      { "date",          required_argument, NULL, 'D'},
				      { "log-file",      required_argument, NULL, 'l'},
				      { "list",          no_argument,       NULL, 'L'},
				      { "metadata-file", required_argument, NULL, 'f'},
				      { "parents",       no_argument,       NULL, 'p'},
				      { "recursive",     no_argument,       NULL, 'r'},
				      { "single",        no_argument,       NULL, 's'},
				      { "quiet",         no_argument,       NULL, 'q'},
				      { "verbose",       no_argument,       NULL, 'v'},
				      { "help",          no_argument,       NULL, 'h'},
				      { NULL,            0,                 NULL,   0} };

	while ((c = getopt_long(argc, argv, short_opts,
				long_opts, NULL)) != EOF) {
		switch (c) {
			case 'a':
				restore_all = TRUE;
				break;

			case 'd':
				if (is_numeric(optarg)) {
					int level = atoi(optarg);

					if ((level < CRITICAL) ||
					    (level > EVERYTHING)) {
						ERROR_MSG(_("%s is not a valid debug level.\n"),
							  optarg);

						/* Display the help. */
						rc = EINVAL;
					}
				} else if (strcasecmp(optarg, "critical") == 0) {
					debug_level = CRITICAL;
				} else if (strcasecmp(optarg, "serious") == 0) {
					debug_level = SERIOUS;
				} else if (strcasecmp(optarg, "error") == 0) {
					debug_level = ERROR;
				} else if (strcasecmp(optarg, "warning") == 0) {
					debug_level = WARNING;
				} else if (strcasecmp(optarg, "default") == 0) {
					debug_level = DEFAULT;
				} else if (strcasecmp(optarg, "details") == 0) {
					debug_level = DETAILS;
				} else if (strcasecmp(optarg, "entry_exit") == 0) {
					debug_level = ENTRY_EXIT;
				} else if (strcasecmp(optarg, "debug") == 0) {
					debug_level = DEBUG;
				} else if (strcasecmp(optarg, "extra") == 0) {
					debug_level = EXTRA;
				} else if (strcasecmp(optarg, "everything") == 0) {
					debug_level = EVERYTHING;
				} else {
					ERROR_MSG(_("%s is not a valid debug level.\n"),
						  optarg);

					/* Display the help. */
					rc = EINVAL;
				}

				break;

			case 'D':
				if (metadata_file == NULL) {
					date_string = strdup(optarg);

				} else {
					ERROR_MSG(_("You cannot specify both a metadata file and a metadata backup date.\n"));
					rc = EINVAL;
				}
				break;

			case 'f':
				if (date_string == NULL) {
					metadata_file = strdup(optarg);

				} else {
					ERROR_MSG(_("You cannot specify both a metadata backup date and a metadata file.\n"));
					rc = EINVAL;
				}
				break;

			case 'l':
				log_file = strdup(optarg);
				break;

			case 'L':
				list = TRUE;
				if (single) {
					(_("You cannot specify -L (list) and -s (single) at the same time.\n"));

					/* Display the help. */
					rc = EINVAL;
				}
				if (recursive) {
					ERROR_MSG(_("You cannot specify -L (list) and -r (recursive) at the same time.\n"));

					/* Display the help. */
					rc = EINVAL;
				}
				break;

			case 'p':
				parents = TRUE;
				break;

			case 'r':
				recursive = TRUE;
				if (single) {
					ERROR_MSG(_("You cannot specify -r (recursive) and -s (single) at the same time.\n"));

					/* Display the help. */
					rc = EINVAL;
				}
				if (list) {
					ERROR_MSG(_("You cannot specify -r (recursive) and -L (list) at the same time.\n"));

					/* Display the help. */
					rc = EINVAL;
				}
				break;

			case 's':
				single = TRUE;
				if (recursive) {
					ERROR_MSG(_("You cannot specify -s (single) and -r (recursive) at the same time.\n"));

					/* Display the help. */
					rc = EINVAL;
				}
				if (list) {
					ERROR_MSG(_("You cannot specify -s (single) and -L (list) at the same time.\n"));

					/* Display the help. */
					rc = EINVAL;
				}
				break;

			case 'q':
				quiet = TRUE;
				if (verbose) {
					ERROR_MSG(_("You cannot specify -v (verbose) and -q (quiet) at the same time.  "
						    "-q will override -v.\n"));
					verbose = FALSE;
				}
				break;

			case 'v':
				verbose = TRUE;
				if (quiet) {
					ERROR_MSG(_("You cannot specify -q (quiet) and -v (verbose) at the same time.  "
						    "-v will override -q.\n"));
					quiet = FALSE;
				}
				break;

			case 'h':
			case '?':
				/* Display the help. */
				rc = EINVAL;
				break;

			default:
				ERROR_MSG(_("%s -- unrecognized option \"%c\"\n\n"),
					  prog_name, c);

				/* Display the help. */
				rc = EINVAL;
				break;
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static char * find_newest_backup_file(const char * dir_name,
				      struct tm  * time) {

	DIR * dir;
	struct dirent * de;
	char * file_name = NULL;
	char * full_file_name = NULL;
	char oldest_date[20];

	if (time != NULL) {
		sprintf(oldest_date, "%04u-%02u-%02u-%02u.%02u.%02u",
			time->tm_year + 1900,
			time->tm_mon + 1,
			time->tm_mday,
			time->tm_hour,
			time->tm_min,
			time->tm_sec);
	}

	dir = opendir(dir_name);

	if (dir != NULL) {

		for (de = readdir(dir); de != NULL; de = readdir(dir)) {
			if (strncmp(de->d_name,
				    EVMS_METADATA_FILE_NAME_PREFIX,
				    EVMS_METADATA_FILE_NAME_PREFIX_LEN) != 0) {
				continue;
			}

			if (time != NULL) {
				if (strcmp(de->d_name + EVMS_METADATA_FILE_NAME_PREFIX_LEN,
					   oldest_date) > 0) {
					continue;
				}
			}

			if (file_name == NULL) {
				file_name = strdup(de->d_name);

			} else {
				if (strcmp(de->d_name + EVMS_METADATA_FILE_NAME_PREFIX_LEN,
					   file_name + EVMS_METADATA_FILE_NAME_PREFIX_LEN) > 0) {
					free(file_name);
					file_name = strdup(de->d_name);
				}
			}
		}

		closedir(dir);
	}

	if (file_name != NULL) {
		full_file_name = malloc(strlen(dir_name) + strlen(file_name) + 2);
		if (full_file_name) {
			strcpy(full_file_name, dir_name);
			strcat(full_file_name, file_name);
			MSG(_("Using metadata backup file %s.\n"), full_file_name);
		}
	}

	LOG_EXIT_PTR(full_file_name);
	return full_file_name;
}


static boolean validate_date(struct tm * date) {

	boolean result = TRUE;

	if ((date->tm_sec < 0) || (date->tm_sec > 60)) {
		VERBOSE_MSG(_("Seconds value of %d is out of range.\n"), date->tm_sec);
		result = FALSE;
	}
	if ((date->tm_min < 0) || (date->tm_min > 59)) {
		VERBOSE_MSG(_("Minues value of %d is out of range.\n"), date->tm_min);
		result = FALSE;
	}
	if ((date->tm_hour < 0) || (date->tm_hour > 23)) {
		VERBOSE_MSG(_("Hour value of %d is out of range.\n"), date->tm_hour);
		result = FALSE;
	}
	if ((date->tm_mday < 1) || (date->tm_mday > 31)) {
		VERBOSE_MSG(_("Date value of %d is out of range.\n"), date->tm_mday);
		result = FALSE;
	}
	if ((date->tm_mon < 0) || (date->tm_mon > 11)) {
		VERBOSE_MSG(_("Month value of %d is out of range.\n"), date->tm_mon);
		result = FALSE;
	}
	if (date->tm_year < 70) {
		VERBOSE_MSG(_("Year value of %d (%d) is out of range.\n"), date->tm_year, date->tm_year + 1900);
		result = FALSE;
	}

	LOG_EXIT_BOOL(result);
	return result;
}


static inline boolean got_valid_date(struct tm * date, char * remainder) {

	/*
	 * remainder will be NULL if the date string didn't have a time.
	 * It's OK not to specify a time.
	 */
	if (remainder == NULL) {
		if (validate_date(date)) {
			return TRUE;
		} else {
			return FALSE;
		}
	} else {
		/*
		 * If there is any remainder it must be an empty string.
		 * If it's not empty, that means there are left over
		 * characters in the date string that did not get parsed.
		 */
		if (*remainder != '\0') {
			VERBOSE_MSG(_("Extra characters given for the date/time: \"%s\"\n"),
				    remainder);
			return FALSE;
		} else {
			if ((*remainder == '\0') && validate_date(date)) {
				return TRUE;
			} else {
				return FALSE;
			}
		}
	}
}


static int get_file_by_date(const char * date_str,
			    const char * dir,
			    char     * * p_file_name) {

	int rc = 0;
	char * file_name;

	LOG_ENTRY();

	if (date_str != NULL) {

		struct tm date;
		char * remainder;

		/*
		 * Set the time part to 29:59:60 in case the user specified only
		 * a date and not a date and time.  If only a date was given,
		 * then we want the oldest backup file on that date, not the
		 * oldest backup file before that date.
		 */
		memset(&date, 0, sizeof(date));

		date.tm_hour = 23;
		date.tm_min  = 59;
		date.tm_sec  = 60;	/* There are leap seconds. */

		build_date_time_format();
		remainder = strptime(date_str, date_time_format, &date);
		
		if (!got_valid_date(&date, remainder)) {
			/* Try the "standard" format. */
			memset(&date, 0, sizeof(date));

			date.tm_hour = 23;
			date.tm_min  = 59;
			date.tm_sec  = 60;

			remainder = strptime(date_str, STANDARD_DATE_TIME_FORMAT, &date);

			if (!got_valid_date(&date, remainder)) {
				/* Try the alternate "standard" format. */
				memset(&date, 0, sizeof(date));

				date.tm_hour = 23;
				date.tm_min  = 59;
				date.tm_sec  = 60;

				remainder = strptime(date_str, STANDARD_DATE_TIME_FORMAT_ALT, &date);

				if (!got_valid_date(&date, remainder)) {
					build_date_time_format_help();
					ERROR_MSG(_("Error parsing the date/time \"%s\".  "
						    "The format for the date/time is either \"%s\" or \"%s\".\n"),
						  date_str,
						  date_time_format_help,
						  STANDARD_DATE_TIME_FORMAT_HELP);
					LOG_EXIT_INT(EINVAL);
					return EINVAL;;
				}
			}
		}

		file_name = find_newest_backup_file(default_metadata_dir, &date);

		if (file_name == NULL) {
			ERROR_MSG(_("No metadata backup files before %s found in directory %s.\n"),
				  date_str, default_metadata_dir);
			rc = ENOENT;
		}

	} else {
		/* Use the newest backup in the directory. */
		file_name = find_newest_backup_file(default_metadata_dir, NULL);

		if (file_name == NULL) {
			ERROR_MSG(_("No metadata backup files found in directory %s.\n"),
				  default_metadata_dir);
			rc = ENOENT;
		}
	}

	*p_file_name = file_name;

	LOG_EXIT_INT(rc);
	return rc;
}


static char db_file_name[PATH_MAX];

static int open_db(char * file_name) {

	int fd = 0;

	LOG_ENTRY();

	/*
	 * If the file name contains some form of path, relative or absolute,
	 * use the file name as-is.
	 */
	if (strchr(file_name, '/') != NULL) {
		fd = open(file_name, O_RDONLY);
		if (fd < 0) {
			ERROR_MSG(_("Error opening file %s.  Error code was %d: %s\n"),
				  file_name, errno, strerror(errno));
			LOG_EXIT_INT(-errno);
			return -errno;
		}

	} else {
		/*
		 * It's just a file name.  Try to open the file in the
		 * current directory.  If that fails, try to open the file
		 * in the default directory.
		 */
		fd = open(file_name, O_RDONLY);
		if (fd < 0) {
			if (errno == ENOENT) {
				strcpy(db_file_name, default_metadata_dir);
				strcat(db_file_name, file_name);

				fd = open(db_file_name, O_RDONLY);
				if (fd < 0) {
					ERROR_MSG(_("Error opening file %s.  Error code was %d: %s\n"),
						  file_name, errno, strerror(errno));
					LOG_EXIT_INT(-errno);
					return -errno;
				}

			} else {
				ERROR_MSG(_("Error opening file %s.  Error code was %d: %s\n"),
					  file_name, errno, strerror(errno));
				LOG_EXIT_INT(-errno);
				return -errno;
			}
		}
	}

	LOG_EXIT_INT(fd);
	return fd;
}


static int verify_db_header(metadata_db_entry_t * db_entry) {

	u_int32_t save_CRC;
	u_int32_t new_CRC;

	LOG_ENTRY();

	if (strncmp(db_entry->signature, EVMS_METADATA_DB_SIGNATURE, sizeof(db_entry->signature)) != 0) {
		ERROR_MSG(_("Entry does not have the correct signature in the header.\n"));
		LOG_EXIT_INT(EINVAL);
		return EINVAL;
	}

	save_CRC = db_entry->header_crc;
	db_entry->header_crc = 0;

	new_CRC	= evms_calculate_crc(EVMS_INITIAL_CRC, db_entry, sizeof(*db_entry));
	db_entry->header_crc = save_CRC;

	if (DISK_TO_CPU32(save_CRC) != new_CRC) {
		ERROR_MSG(_("Entry's CRC does not match.\n"));
		LOG_EXIT_INT(EINVAL);
                return EINVAL;
	}

	LOG_EXIT_INT(0);
	return 0;
}


static int next_db_entry_header(int                   fd,
				metadata_db_entry_t * db_entry) {

	int rc = 0;
	int bytes_read;

	LOG_ENTRY();

	bytes_read = read(fd, db_entry, sizeof(metadata_db_entry_t));

	if (bytes_read < 0) {
		ERROR_MSG(_("Read of metadata file failed.  Error code %d: %s\n"), errno, strerror(errno));
		rc = errno;

	} else if (bytes_read == 0) {
		LOG_DEBUG("No bytes read.  Assuming end of file.\n");
		rc = ENODATA;

	} else {
		rc = verify_db_header(db_entry);

		if (rc == 0) {
			db_entry->offset = DISK_TO_CPU64(db_entry->offset);
			db_entry->length = DISK_TO_CPU64(db_entry->length);
			db_entry->metadata_crc = DISK_TO_CPU32(db_entry->metadata_crc);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int verify_metadata(metadata_db_entry_t * db_entry,
			   void                * metadata) {

	int rc = 0;
	u_int32_t new_CRC;

	LOG_ENTRY();

	new_CRC = evms_calculate_crc(EVMS_INITIAL_CRC, metadata,
				     (u_int32_t) db_entry->length << EVMS_VSECTOR_SIZE_SHIFT);

	if (new_CRC != db_entry->metadata_crc) {
		ERROR_MSG(_("CRC for the metadata does not match.\n"));
		rc = EINVAL;
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int get_metadata(int                   fd,
			metadata_db_entry_t * db_entry,
			void              * * p_metadata) {
	int rc = 0;
	void * metadata = NULL;
	u_int64_t metadata_byte_length;
	ssize_t bytes_read;

	LOG_ENTRY();

	metadata_byte_length = db_entry->length << EVMS_VSECTOR_SIZE_SHIFT;

	metadata = malloc(metadata_byte_length);
	if (metadata != NULL) {
		bytes_read = read(fd, metadata, metadata_byte_length);

		if (bytes_read < 0) {
			ERROR_MSG(_("Read of the metadata file failed.  Error code %d: %s\n"),
				  errno, strerror(errno));
			rc = errno;

		} else if (bytes_read < metadata_byte_length) {
			LOG_DEBUG("Not enough bytes read from the metadata file.  "
				  "Needed %d. Only got %d.\n", metadata_byte_length, bytes_read);
			rc = ENODATA;

		} else {
			rc = verify_metadata(db_entry, metadata);
		}

		if (rc != 0) {
			free(metadata);
			metadata = NULL;
		}

	} else {
		ERROR_MSG(_("Failed to allocate memory for a buffer to read in metadata.\n"));
		rc = ENOMEM;
	}

	*p_metadata = metadata;

	LOG_EXIT_INT(rc);
	return rc;
}


static int make_metadata_info(int                   fd,
			      metadata_db_entry_t * db_entry,
			      metadata_info_t   * * p_metadata_info) {

	metadata_info_t * metadata_info;
	int rc = 0;

	LOG_ENTRY();

	metadata_info = calloc(1, sizeof(metadata_info_t));

	if (metadata_info != NULL) {
		
		metadata_info->parent_name = strdup(db_entry->parent);
		metadata_info->child_name = strdup(db_entry->child);
		metadata_info->offset = db_entry->offset;
		metadata_info->length = db_entry->length;

		if (metadata_info->length != 0) {
			rc = get_metadata(fd, db_entry, &metadata_info->metadata);

			if (rc != 0) {
				free(metadata_info);
				metadata_info = NULL;
			}
		}

	} else {
		ERROR_MSG(_("Failed to allocate memory for a metadata tree entry.\n"));
		rc = ENOMEM;
	}

	*p_metadata_info = metadata_info;

	LOG_EXIT_PTR(rc);
	return rc;
}


static int insert_metadata_info(metadata_info_t * metadata_info,
				tree_node_t   * * tree) {

	int rc = 0;
	tree_node_t * * prev;
	tree_node_t * curr;

	LOG_ENTRY();

	for (prev = tree, curr = *tree;
	    (curr != NULL) &&
	    (strcmp(metadata_info->parent_name, curr->name) != 0);
	    prev = &(curr->sibling), curr = curr->sibling);

	if (curr == NULL) {

		/* Didn't find a matching child/parent.  Make one. */
		curr = calloc(1, sizeof(tree_node_t));

		if (curr == NULL) {
			ERROR_MSG(_("Failed to allocate memory for a parent tree node.\n"));
			rc = ENOMEM;
			goto error_out;
		}

		curr->name = strdup(metadata_info->parent_name);

		*prev = curr;
	}

	/* Find a child node that matches the metadata_info's child. */
	for (prev = &(curr->child), curr = curr->child;
	    (curr != NULL) &&
	    (strcmp(metadata_info->child_name, curr->name) != 0);
	    prev = &(curr->sibling), curr = curr->sibling);

	if (curr == NULL) {

		/* Didn't find a matching child/parent.  Make one. */
		curr = calloc(1, sizeof(tree_node_t));

		if (curr == NULL) {
			ERROR_MSG(_("Failed to allocate memory for a child/parent tree node.\n"));
			rc = ENOMEM;
			goto error_out;
		}

		curr->name = strdup(metadata_info->child_name);
		
		*prev = curr;
	}

	/* Add this info to the list of info for the child/parent. */
	metadata_info->next = curr->info;
	curr->info = metadata_info;

error_out:
	LOG_EXIT_INT(rc);
	return rc;
}


static void free_metadata_info(metadata_info_t * info) {

	LOG_ENTRY();

	if (info->next != NULL) {
		free_metadata_info(info->next);
	}

	free(info->parent_name);
	free(info->child_name);

	if (info->metadata != NULL) {
		free(info->metadata);
	}

	free(info);

	LOG_EXIT_VOID();
	return;
}


static void free_tree(tree_node_t * tree) {

	LOG_ENTRY();

	if (tree->parent != NULL) {
		free_tree(tree->parent);
	} else if (tree->child != NULL) {
		free_tree(tree->child);
	} else if (tree->sibling != NULL) {
		free_tree(tree->sibling);
	}

	if (tree->info != NULL) {
		free_metadata_info(tree->info);
	}

	free(tree->name);

	free(tree);

	LOG_EXIT_VOID();
	return;
}


static int get_tree(int             fd,
		    char        * * things,
		    boolean         get_parents,
		    tree_node_t * * p_tree) {

	int rc = 0;
	int i;
	int entry_number = 0;
	metadata_db_entry_t db_entry;
	tree_node_t * tree = NULL;
	metadata_info_t * metadata_info = NULL;

	LOG_ENTRY();

	memset(&db_entry, 0, sizeof(db_entry));

	lseek64(fd, 0, SEEK_SET);

	rc = next_db_entry_header(fd, &db_entry);
	while (rc == 0) {
		metadata_info = NULL;

		if (get_parents) {
			for (i = 0; things[i] != NULL; i++) {
				if (strcmp(db_entry.child, things[i]) == 0) {
					rc = make_metadata_info(fd, &db_entry, &metadata_info);
					break;
				}
			}
		} else {
			for (i = 0; things[i] != NULL; i++) {
				if (strcmp(db_entry.parent, things[i]) == 0) {
					rc = make_metadata_info(fd, &db_entry, &metadata_info);
					break;
				}
			}
		}

		/* metadata_info != NULL when rc == 0 */
		if (metadata_info != NULL) {
			rc = insert_metadata_info(metadata_info, &tree);

		} else {
			/*
			 * Move the current file position over the db_entry's
			 * metadata.
			 */
			if (db_entry.length != 0) {
				lseek64(fd, db_entry.length << EVMS_VSECTOR_SIZE_SHIFT, SEEK_CUR);
			}
		}

		if (rc == 0) {
			entry_number++;
			rc = next_db_entry_header(fd, &db_entry);
		}
	}

	/*
	 * next_db_entry_header() returns ENODATA at the end of the file.
	 * It's not a bad error code.
	 */
	if (rc == ENODATA) {
		rc = 0;

	} else {
		ERROR_MSG(_("Error encountered processing entry %d in the metadata backup file.\n"), entry_number);
		if (tree != NULL) {
			free_tree(tree);
			tree = NULL;
		}
	}

	*p_tree = tree;

	LOG_EXIT_INT(rc);
	return rc;
}


static int get_child_things(tree_node_t * tree,
			    char    * * * p_child_things) {

	int rc = 0;
	tree_node_t * parent;
	tree_node_t * child;
	char * * child_things = NULL;
	int num_child_things = 0;

	#define BLOCK_ALLOC 8

	LOG_ENTRY();

	child_things = calloc(1, BLOCK_ALLOC * sizeof(char *));
	if (child_things == NULL) {
		ERROR_MSG(_("Failed to allocate memory for a child things array.\n"));
		rc = ENOMEM;
	}

	for (parent = tree;
	     (rc == 0) && (parent != NULL);
	     parent = parent->sibling) {

		for (child = parent->child;
		     child != NULL;
		     child = child->sibling) {

			if (child->name == NULL) {
				continue;
			}
			if (strlen(child->name) == 0) {
				continue;
			}

			/*
			 * Make sure there is room for the new entry plus the
			 * terminating NULL entry.
			 */
			if ((num_child_things + 1) % BLOCK_ALLOC == 0) {
				child_things = realloc(child_things, (num_child_things + 1 + BLOCK_ALLOC) * sizeof(char *));

				if (child_things == NULL) {
					ERROR_MSG(_("Failed to allocate memory for a child things array.\n"));
					rc = ENOMEM;
					break;
				}
			}

			child_things[num_child_things] = strdup(child->name);
			num_child_things++;
		}
	}

	if (child_things != NULL) {
		child_things[num_child_things] = NULL;
	}

	*p_child_things = child_things;

	LOG_EXIT_INT(rc);
	return rc;
}


static int get_parent_things(tree_node_t * tree,
			     char    * * * p_parent_things) {

	int rc = 0;
	tree_node_t * parent;
	char * * parent_things = NULL;
	int num_parent_things = 0;

	#define BLOCK_ALLOC 8

	LOG_ENTRY();

	parent_things = calloc(1, BLOCK_ALLOC * sizeof(char *));
	if (parent_things == NULL) {
		ERROR_MSG(_("Failed to allocate memory for a parent things array.\n"));
		rc = ENOMEM;
	}

	for (parent = tree;
	     (rc == 0) && (parent != NULL);
	     parent = parent->sibling) {

		if (parent->name == NULL) {
			continue;
		}
		if (strlen(parent->name) == 0) {
			continue;
		}

		/*
		 * Make sure there is room for the new entry plus the
		 * terminating NULL entry.
		 */
		if ((num_parent_things + 1) % BLOCK_ALLOC == 0) {
			parent_things = realloc(parent_things, (num_parent_things + 1 + BLOCK_ALLOC) * sizeof(char *));

			if (parent_things == NULL) {
				ERROR_MSG(_("Failed to allocate memory for a parent things array.\n"));
				rc = ENOMEM;
				break;
			}
		}

		parent_things[num_parent_things] = strdup(parent->name);
		num_parent_things++;
	}

	if (parent_things != NULL) {
		parent_things[num_parent_things] = NULL;
	}

	*p_parent_things = parent_things;

	LOG_EXIT_INT(rc);
	return rc;
}


/*
 * Check if all the metadata for the list of objects is on the objects.
 *
 * "list" points to a list of nodes linked by their "sibling" pointers.
 * Each node's "info" pointer points to a metadata_info_t that contains the
 * definition of a chunk of metadata.  Chunks of metadata on the same object
 * are linked by their node.sibling pointers.
 */
static int validate_metadata(tree_node_t * list) {

	int rc = 0;
	tree_node_t * curr;
	metadata_info_t * metadata_info;
	u_int64_t metadata_byte_length;
	void * buffer;

	LOG_ENTRY();

	for (curr = list;
	    (rc == 0) && (curr != NULL);
	    curr = curr->sibling) {

		for (metadata_info = curr->info;
		    (rc == 0) & (metadata_info != NULL);
		    metadata_info = metadata_info->next) {

			object_handle_t obj_handle;

			LOG_DEBUG("Validate metadata for parent %s on child %s.\n",
				  metadata_info->parent_name, metadata_info->child_name);

			if (metadata_info->flags & MI_VALID_ON_DISK) {
				LOG_DEBUG("The metadata have already been validated.\n");
				continue;
			}

			metadata_byte_length = metadata_info->length << EVMS_VSECTOR_SIZE_SHIFT;

			if (metadata_byte_length == 0) {
				LOG_DEBUG("There are no metadata to validate.\n");
				continue;
			}

			rc = evms_get_object_handle_for_name(DISK | SEGMENT | REGION | EVMS_OBJECT,
							     metadata_info->child_name,
							     &obj_handle);

			if (rc != 0) {
				LOG_DEBUG("Error looking up a handle for name %s.  "
					  "Error code from evms_get_object_handle_for_name() was %d %s.\n",
					  metadata_info->child_name, rc, evms_strerror(rc));
				break;
			}

			buffer = malloc(metadata_byte_length);
			if (buffer == NULL) {
				ERROR_MSG(_("Failed to allocate a buffer for reading in metadata.\n"));
				rc = ENOMEM;
				break;
			}

			rc = evms_engine_read(obj_handle,
					      metadata_info->offset,
					      metadata_info->length,
					      buffer);

			if (rc != 0) {
				LOG_DEBUG("evms_engine_read() of file %s failed with error %d: %s\n",
					  metadata_info->child_name, rc, strerror(rc));

			} else if (memcmp(metadata_info->metadata,
					  buffer,
					  metadata_byte_length) != 0) {

				LOG_DEBUG("Metadata for %s on child %s at offset %"PRIu64" of length %"PRIu64" does not compare with the metadata on %s.\n",
					  metadata_info->parent_name,
					  metadata_info->child_name,
					  metadata_info->offset,
					  metadata_byte_length,
					  metadata_info->child_name);
				rc = ENODATA;

			} else {
				/* Mark this one as validated. */
				metadata_info->flags |= MI_VALID_ON_DISK;
			}

			free(buffer);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static boolean has_stop_data(object_handle_t  obj_handle,
			     sector_count_t * size) {

	boolean result = FALSE;
	int rc;
	evms_feature_header_t * stop_data;
	handle_object_info_t * info;

	LOG_ENTRY();

	*size = 0;

	stop_data = malloc(sizeof(evms_feature_header_t));
	if (stop_data == NULL) {
		ERROR_MSG(_("Failed to allocate memory for a stop data buffer.\n"));
		LOG_EXIT_BOOL(result);
		return result;
	}

	rc = evms_get_info(obj_handle, &info);
	if (rc == 0) {
		*size = info->info.disk.size;

		rc = evms_engine_read(obj_handle,
				      info->info.disk.size - 1,
				      1,
				      stop_data);
		if (rc == 0) {
			if (stop_data->signature == DISK_TO_CPU32(EVMS_FEATURE_HEADER_SIGNATURE)) {
				u_int32_t disk_crc = DISK_TO_CPU32(stop_data->crc);
				u_int32_t new_crc;

				stop_data->crc = 0;
				new_crc = evms_calculate_crc(EVMS_INITIAL_CRC,
							     stop_data,
							     sizeof(*stop_data));

				if (disk_crc == new_crc) {
					if (stop_data->flags & EVMS_VOLUME_DATA_STOP) {
						result = TRUE;
					}
				}
			}

		} else {
			LOG_DEBUG("evms_engine_read(%d, %"PRIu64", %"PRIu64", %p) failed with error code %d: %s\n",
				  obj_handle,
				  info->info.disk.size - 1,
				  1,
				  stop_data,
				  rc, evms_strerror(rc));
		}

		if (!result) {

			/* Examine the backup sector. */
			rc = evms_engine_read(obj_handle,
					      info->info.disk.size - 2,
					      1,
					      stop_data);
			if (rc == 0) {

				if (stop_data->signature == DISK_TO_CPU32(EVMS_FEATURE_HEADER_SIGNATURE)) {
					u_int32_t disk_crc = DISK_TO_CPU32(stop_data->crc);
					u_int32_t new_crc;

					stop_data->crc = 0;
					new_crc = evms_calculate_crc(EVMS_INITIAL_CRC,
								     stop_data,
								     sizeof(stop_data));

					if (disk_crc == new_crc) {
						if (stop_data->flags & EVMS_VOLUME_DATA_STOP) {
							result = TRUE;
						}
					}
				}

			} else {
				LOG_DEBUG("evms_engine_read(%d, %"PRIu64", %"PRIu64", %p) failed with error code %d: %s\n",
					  obj_handle,
					  info->info.disk.size - 2,
					  1,
					  stop_data,
					  rc, evms_strerror(rc));
			}
		}

		evms_free(info);
	}

	free(stop_data);

	LOG_EXIT_BOOL(result);
	return result;
}


/*
 * Write out all the metadata in the metadata_info_t list that is headed by
 * node->child.
 */
static int write_metadata(tree_node_t * node) {

	int rc = 0;
	metadata_info_t * metadata_info;
	char offset_string[24];
	char length_string[24];

	LOG_ENTRY();	

	for (metadata_info = node->info;
	    (rc == 0) && (metadata_info != NULL);
	    metadata_info = metadata_info->next){

		object_handle_t obj_handle;

		LOG_DEBUG("Write metadata for parent %s on child %s.\n",
			  metadata_info->parent_name, metadata_info->child_name);

		if ((metadata_info->length == 0) &&
		    (memcmp(metadata_info->parent_name, EVMS_DEV_NODE_PATH, EVMS_DEV_NODE_PATH_LEN) != 0)) {

			/* Nothing to write and it's not a volume. */
			LOG_DEBUG("There are no metadata to write.\n");
			continue;
		}

		rc = evms_get_object_handle_for_name(DISK | SEGMENT | REGION | EVMS_OBJECT,
						     metadata_info->child_name,
						     &obj_handle);
		if (rc != 0) {
			ERROR_MSG(_("Error looking up a handle for name %s.  "
				    "Error code from evms_get_object_handle_for_name() was %d: %s\n"),
				  metadata_info->child_name, rc, evms_strerror(rc));
			break;
		}

		if (metadata_info->length != 0) {
			rc = evms_engine_write(obj_handle,
					       metadata_info->offset,
					       metadata_info->length,
					       metadata_info->metadata);
			if (rc != 0) {
				sprintf(offset_string, "%"PRIu64, metadata_info->offset);
				sprintf(length_string, "%"PRIu64, metadata_info->length);
				ERROR_MSG(_("Failed to write metadata of length %s to object %s at offset %s.  "
					    "Error code was %d: %s\n"),
					  metadata_info->child_name,
					  length_string,
					  offset_string,
					  rc, evms_strerror(rc));
			}

		} else {
			/*
			 * Check if it is a compatibility volume.  (If the name
			 * of the parent after "/dev/evms/" is the same as the
			 * child name, it's a compatibility volume.)
			 * Compatibility volumes have no metadata.  But the
			 * volume's object may have stop data left on it.  Check
			 * for stop data and erase it if found.
			 */
			sector_count_t size;
			evms_feature_header_t * buffer;

			if ((strcmp(metadata_info->parent_name + EVMS_DEV_NODE_PATH_LEN,
				    metadata_info->child_name) == 0) &&
			    has_stop_data(obj_handle, &size)) {

				LOG_DEBUG("Erase the stop data from %s.\n", metadata_info->child_name);

				buffer = calloc(1, sizeof(*buffer));
				if (buffer != NULL) {

					rc = evms_engine_write(obj_handle,
							       size - EVMS_FEATURE_HEADER_SECTORS,
							       EVMS_FEATURE_HEADER_SECTORS,
							       buffer);

					if (rc == 0) {
						rc = evms_engine_write(obj_handle,
								       size - (EVMS_FEATURE_HEADER_SECTORS * 2),
								       EVMS_FEATURE_HEADER_SECTORS,
								       buffer);
						if (rc != 0) {
							sprintf(offset_string, "%"PRIu64, size - (EVMS_FEATURE_HEADER_SECTORS * 2));
							sprintf(length_string, "%"PRIu64, (u_int64_t) EVMS_FEATURE_HEADER_SECTORS);
							ERROR_MSG(_("Failed to write metadata of length %s to object %s at offset %s.  "
								    "Error code was %d: %s\n"),
								  metadata_info->child_name,
								  length_string,
								  offset_string,
								  rc, evms_strerror(rc));
						}

					} else {
						sprintf(offset_string, "%"PRIu64, size - EVMS_FEATURE_HEADER_SECTORS);
						sprintf(length_string, "%"PRIu64, (u_int64_t) EVMS_FEATURE_HEADER_SECTORS);
						ERROR_MSG(_("Failed to write metadata of length %s to object %s at offset %s.  "
							    "Error code was %d: %s\n"),
							  metadata_info->child_name,
							  length_string,
							  offset_string,
							  rc, evms_strerror(rc));
					}
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int mark_children_for_rediscover(tree_node_t * parent) {

	int rc = 0;
	tree_node_t * child;
	object_handle_t handle;
	handle_object_info_t * info;

	LOG_ENTRY();

	LOG_DEBUG("Mark children of \"%s\" for rediscovery.\n", parent->name);

	for (child = parent->child;
	     (rc == 0) && (child != NULL);
	     child = child->sibling) {

		LOG_DEBUG("Attempt to mark %s for rediscovery.\n", child->name);

		rc = evms_get_object_handle_for_name(DISK | SEGMENT | REGION |
						     CONTAINER | EVMS_OBJECT | VOLUME,
						     child->name,
						     &handle);
		if (rc != 0) {
			if (rc == ENOENT) {
				LOG_DEBUG("Child object %s does not exist.\n", child->name);
				rc = 0;

			} else {
				ERROR_MSG(_("Failed to get the handle for object %s.  Error code was %d: %s\n"),
					    child->name, rc, evms_strerror(rc));
				break;
			}

		} else {
			rc = evms_get_info(handle, &info);
			if (rc == 0) {
				switch (info->type) {
					case DISK:
					case SEGMENT:
					case REGION:
					case EVMS_OBJECT:
						rc = evms_mark_for_rediscover(child->name);
						if (rc != 0) {
							ERROR_MSG(_("Failed mark object %s for rediscovery.  Error code was %d: %s\n"),
								    child->name, rc, evms_strerror(rc));
						}
						break;

					case CONTAINER:
						LOG_DEBUG("Don't need to mark containers for rediscovery.\n");
						break;

					case VOLUME:
						LOG_DEBUG("Can't mark volumes for rediscovery.\n");
						break;

					default:
						LOG_SERIOUS("Aaak! I don't know how to mark for rediscovery a thing of type %d (%#x).\n",
							    info->type, info->type);
				}

				evms_free(info);

			} else {
				ERROR_MSG(_("Failed to get the info for object %s.  Error code was %d: %s\n"),
					    child->name, rc, evms_strerror(rc));
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int restore_children(int      fd,
			    char * * things,
			    boolean  recursive) {

	int rc = 0;
	int i;
	tree_node_t * tree;
	tree_node_t * parent;
	tree_node_t * child;
	object_handle_t handle;

	LOG_ENTRY();

	rc = get_tree(fd, things, FALSE, &tree);

	if (recursive) {
		char * * child_things = NULL;

		rc = get_child_things(tree, &child_things);
		if (rc != 0) {
			LOG_EXIT_INT(rc);
			return rc;
		}

		if (child_things[0] != NULL) {
			rc = restore_children(fd, child_things, TRUE);
		}

		free(child_things);

		if (rc != 0) {
			LOG_EXIT_INT(rc);
			return rc;
		}
	}

	for (i = 0; things[i] != NULL; i++) {
		tree_node_t *curr;

		for (curr = tree;
		     (curr != NULL) && (strcmp(things[i], curr->name) != 0);
		     curr = curr->sibling);

		if (curr == NULL) {
			MSG(_("No backup information found for %s\n"), things[i]);
		}
	}

	if (tree == NULL) {
		LOG_EXIT_INT(ENOENT);
		return ENOENT;
        }


	for (parent = tree; (rc == 0) && (parent != NULL); parent = parent->sibling) {

		if (parent->child == NULL) {
			continue;
		}

		if (strlen(parent->child->name) == 0) {
			continue;
		}

		rc = evms_get_object_handle_for_name(DISK | SEGMENT | REGION | EVMS_OBJECT |
						     CONTAINER | VOLUME,
						     parent->name,
						     &handle);
		if ((rc != 0) && (rc != ENOENT)) {
			ERROR_MSG(_("Failed to get the handle for %s.  Error code was %d: %s\n"),
				  parent->name, rc, evms_strerror(rc));
			break;
		}

		if (rc == 0) {
			/*
			 * The thing exists.
			 * Check to see if its metadata are layed out correctly
			 * on its children.
			 */
			LOG_DEBUG("Parent %s exists.\n", parent->name);

			rc = validate_metadata(parent->child);

			if (rc != 0) {
				/*
				 * The parent and its siblings will need to be
				 * rediscovered.
				 */
				evms_mark_for_rediscover(parent->name);

				/*
				 * Set the return code to indicate that the
				 * parent object does not exist so that the code
				 * that follows will write the metadata on the
				 * children.
				 */
				rc = ENOENT;
			}
		}

		if (rc == ENOENT) {
			rc = 0;

			for (child = parent->child;
			     (rc == 0) && (child != NULL);
			     child = child->sibling) {
				rc = write_metadata(child);
			}

			if (rc == 0) {
				rc = mark_children_for_rediscover(parent);
				if (rc == 0) {
					rc = evms_rediscover();
					if (rc != 0) {
						ERROR_MSG(_("Failed to rediscover new metadata.  "
                                                            "Error code was %d: %s\n"),
							    rc, evms_strerror(rc));
					}
				}
			}
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int restore_parents(int      fd,
			   char * * things,
			   boolean  recursive) {

	int rc = 0;
	int i;
	tree_node_t * tree;
	tree_node_t * child;
	tree_node_t * parent;
	object_handle_t handle;

	LOG_ENTRY();

	rc = get_tree(fd, things, TRUE, &tree);

	for (i = 0; things[i] != NULL; i++) {
		tree_node_t *curr;

		for (curr = tree;
		     (curr != NULL) && (strcmp(things[i], curr->child->name) != 0);
		     curr = curr->sibling);

		if (curr == NULL) {
			MSG(_("No backup information found for %s\n"), things[i]);
		}
	}

	if (tree == NULL) {
		LOG_EXIT_INT(ENOENT);
		return ENOENT;
        }

	for (parent = tree; (rc == 0) && (parent != NULL); parent = parent->sibling) {

		if (parent->child == NULL) {
			continue;
		}

		if (strlen(parent->child->name) == 0) {
			continue;
		}

		rc = evms_get_object_handle_for_name(DISK | SEGMENT | REGION | EVMS_OBJECT |
						     CONTAINER | VOLUME,
						     parent->name,
						     &handle);
		if ((rc != 0) && (rc != ENOENT)) {
			ERROR_MSG(_("Failed to get the handle for %s.  Error code was %d: %s\n"),
				  parent->name, rc, evms_strerror(rc));
			break;
		}

		if (rc == 0) {
			/*
			 * The thing exists.
			 * Check to see if its metadata are layed out correctly
			 * on its children.
			 */
			 rc = validate_metadata(parent->child);

			 if (rc != 0) {
				 /*
				  * The parent and its siblings will need to be
				  * rediscovered.
				  */
				 evms_mark_for_rediscover(parent->name);

				 /*
				  * Set the return code to indicate that the
				  * parent object does not exist so that the
				  * code that follows will write the metadata on
				  * the children.
				  */
				 rc = ENOENT;
			 }
		}

		if (rc == ENOENT) {
			rc = 0;

			for (child = parent->child;
			     (rc == 0) && (child != NULL);
			     child = child->sibling) {
				rc = write_metadata(child);
			}

			if (rc == 0) {
				rc = mark_children_for_rediscover(parent);
				if (rc == 0) {
					rc = evms_rediscover();
					if (rc != 0) {
						ERROR_MSG(_("Failed to rediscover new metadata.  "
                                                            "Error code was %d: %s\n"),
							    rc, evms_strerror(rc));
					}
				}
			}
		}
	}

	if ((rc == 0) && recursive) {

		char * * parent_things = NULL;

		rc = get_parent_things(tree, &parent_things);
		if (rc == 0) {
			if (parent_things[0] != NULL) {
				rc = restore_parents(fd, parent_things, TRUE);
			}

			free(parent_things);
		}
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static void print_db_entry_header(metadata_db_entry_t * db_entry,
				  int                   number) {

	char offset_string[24];
	char length_string[24];

	LOG_ENTRY();

	/*
	 * The "offset" and "length" in the db_entry have been converted to
	 * bytes for ease in doing file I/O.  Print them as the original sector
	 * values.
	 */
	sprintf(offset_string, "%"PRIu64, db_entry->offset);
	sprintf(length_string, "%"PRIu64, db_entry->length);

	printf(_("Entry #:  %d\n"
		 "Parent:   %s\n"
		 "Child:    %s\n"
		 "Offset:   %s\n"
		 "Length:   %s\n\n"),
               number,
	       (strlen(db_entry->parent) != 0) ? db_entry->parent : _("(none)"),
	       (strlen(db_entry->child)  != 0) ? db_entry->child  : _("(none)"),
	       offset_string,
	       length_string);

	LOG_EXIT_VOID();
	return;
}


static void print_metadata(metadata_db_entry_t * db_entry,
			   void                * metadata) {

	int i;
	int count = 0;
	u_int8_t * p1 = metadata;
	u_int8_t * p2 = metadata;
	u_int64_t metadata_byte_length;

	LOG_ENTRY();

	metadata_byte_length = db_entry->length << EVMS_VSECTOR_SIZE_SHIFT;

	/*
	 * We know that the metadata are in sector chunks.  So we can dump 16
	 * byte lines without having to worry about fractional rows or buffer
	 * overruns.
         */
	printf(_("Metadata:\n"));
	while (count < metadata_byte_length) {
		printf("%08x  ", count);
		for (i = 0; i < 8; i++) {
			printf("%02x ", *p1++);
		}
		printf(" ");
		for (i = 0; i < 8; i++) {
			printf("%02x ", *p1++);
		}
		printf(" |");
		for (i = 0; i < 16; i++) {
			if (((*p2 & 0x7f) < 0x20) ||
			    ((*p2 & 0x7f) == 0x7f)) {
				printf(".");
			} else {
				printf("%c", *p2);
			}
			p2++;
		}
		printf("|\n");
		count += 16;
	}

	printf("\n");

	LOG_EXIT_VOID();
	return;
}


static int do_list(int fd) {

	int rc = 0;
	int entry_number = 0;
	metadata_db_entry_t db_entry;
	void * metadata;

	LOG_ENTRY();

	rc = next_db_entry_header(fd, &db_entry);
	while (rc == 0) {

		print_db_entry_header(&db_entry, entry_number);

		if (db_entry.length != 0) {
			rc = get_metadata(fd, &db_entry, &metadata);

			if (rc == 0) {
				print_metadata(&db_entry, metadata);

				free(metadata);
			}
		}

		if (rc == 0) {
			rc = next_db_entry_header(fd, &db_entry);
		}

		entry_number ++;
	}

	/*
	 * next_db_entry() returns ENODATA at the end of the file.  It's not a
	 * bad error code.
	 */
	if (rc == ENODATA) {
		rc = 0;

	} else {
		ERROR_MSG(_("Error encountered processing entry %d in the metadata backup file."
			    "Error code was %d: %s\n"),
			  entry_number, rc, evms_strerror(rc));
	}

	LOG_EXIT_INT(rc);
	return rc;
}


static int message_callback(char   * message_text,
			    int    * answer,
			    char * * choices)
{
	LOG_ENTRY();

	MSG("%s", message_text);

	if (choices != NULL) {
		MSG(_("%s: Responding with default selection \"%s\".\n"),
		       prog_name, choices[*answer]);
	}

	LOG_EXIT_INT(0);
	return 0;
}


static void status_callback(char * message_text)
{
	LOG_ENTRY();

	VERBOSE_MSG("%s", message_text);

	LOG_EXIT_VOID();
}


static ui_callbacks_t callbacks = {
	user_message: message_callback,
	user_communication: NULL,
	progress: NULL,
	status: status_callback
};


int main(int argc, char * * argv)
{
	int rc;
	evms_version_t engine_api_version;
	char * all_things[] = {"", NULL};
	char * * things = NULL;
	int fd;
	char * temp_str;

	setlocale(LC_MESSAGES, "");
	setlocale(LC_TIME, "");
	bindtextdomain(PACKAGE, LOCALEDIR);
	textdomain(PACKAGE);

	prog_name = basename(argv[0]);

	/*
	 * Get the default metadata directory.  Add on a terminating '/' if it
	 * doesn't have one.
	 */
	evms_get_config_string("engine.metadata_backup_dir", &default_metadata_dir);
	if (default_metadata_dir[strlen(default_metadata_dir) - 1] != '/') {
		temp_str = malloc(strlen(default_metadata_dir) + 2);
		strcpy(temp_str, default_metadata_dir);
		strcat(temp_str, "/");
		default_metadata_dir = temp_str;
	}

	rc = parse_options(argc, argv);
	if (rc) {
		show_help();
		return rc;
	}

	/* Default is recursive if no action is specified. */
	if (!recursive && ! single && !list) {
		recursive = TRUE;
	}

	if (optind < argc) {
		things = &argv[optind];

	} else {
		if (restore_all) {
			things = all_things;

		} else {
			if (!list) {
				ERROR_MSG(_("You must use the -a (--all) switch if you do not explicitly specify the things to restore.\n"));
				show_help();
				return EINVAL;
			}
		}
	}
	
        if (metadata_file == NULL) {
		rc = get_file_by_date(date_string, default_metadata_dir, &metadata_file);
		if (rc != 0) {
			return rc;
		}
	}

	fd = open_db(metadata_file);
	if (fd < 0) {
		rc = -fd;
		return rc;
	}

	if (recursive || single) {
		evms_get_api_version(&engine_api_version);

		if ((engine_api_version.major != REQUIRED_ENGINE_API_MAJOR) ||
		    (engine_api_version.major == REQUIRED_ENGINE_API_MAJOR &&
		     engine_api_version.minor <  REQUIRED_ENGINE_API_MINOR)) {
			fprintf(stderr, _("EVMS API version mismatch.\n"));
			fprintf(stderr, _("\tEVMS provides %d.%d.%d\n"),
				engine_api_version.major, engine_api_version.minor, engine_api_version.patchlevel);
			fprintf(stderr, _("\t%s requires %d.%d.%d\n"),
				prog_name, REQUIRED_ENGINE_API_MAJOR,
				REQUIRED_ENGINE_API_MINOR, REQUIRED_ENGINE_API_PATCH);
			return 1;
		}

		VERBOSE_MSG(_("Opening the EVMS Engine...\n"));
		rc = evms_open_engine(NULL, ENGINE_READWRITE,
				      &callbacks, debug_level, log_file);
		if (rc) {
			ERROR_MSG(_("evms_open_engine() returned error code %d: %s\n"),
				  rc, evms_strerror(rc));
			return rc;
		}
		VERBOSE_MSG(_("The EVMS Engine is open.\n"));
	}

	if (single) {
		if (things != NULL) {
			VERBOSE_MSG(_("Restoring metadata...\n"));

			if (parents) {
				rc = restore_parents(fd, things, FALSE);
			} else {
				rc = restore_children(fd, things, FALSE);
			}

			if (rc == 0) {
				MSG(_("Metadata restored successfully.\n"));
			}

		} else {
			ERROR_MSG(_("You must specify at least one thing for a single restore.\n"));
			rc = EINVAL;
		}

	} else if (recursive) {
		VERBOSE_MSG(_("Restoring metadata...\n"));

		if (parents) {
			rc = restore_parents(fd, things, TRUE);
		} else {
			rc = restore_children(fd, things, TRUE);
		}

		if (rc == 0) {
			MSG(_("Metadata restored successfully.\n"));
		}

	} else if (list) {
		do_list(fd);
	}

	/* If we didn't open the engine, the close will fail.  No harm done. */
	evms_close_engine();

	return rc;
}

