/*************************************************************
*  This file is part of the Surface Evolver source code.     *
*  Programmer:  Ken Brakke, brakke@susqu.edu                 *
*************************************************************/

/**********************************************************************
*
*  File: sqcurve2.c
*
*  Purpose: Does calculations needed for including square curvature
*              in energy. Linear model only.
*              Named quantity methods.
*/

#include "include.h"

/* auxiliary structures needed to hold accumulated data */
/* per vertex structure */
struct v_curve { REAL area;     /* area around vertex */
                 REAL a;    /* allocated area */
                 REAL force[MAXCOORD];  /* force on vertex */
                 REAL f;        /* scalar force */
                 REAL h;        /* scalar curvature */
                 REAL star_force[MAXCOORD];  /* star area gradient */
                 REAL deriv2[MAXCOORD][MAXCOORD];  /* self second deriv */
                 REAL normal[MAXCOORD]; /* for effective area */
                 REAL norm;        /* square length of normal */
                 REAL fpgradf[MAXCOORD]; /* for projection gradient part */
                 REAL vol;  /* allocated volume of vertex */
                 int sign;    /* positive if H outward */
                 REAL term;  /* for squared integrands */
               };
static struct v_curve *v_curve;

/* per edge structure */
struct e_curve { REAL deriv[2][MAXCOORD]; /* dA_head/dv_tail */
                 REAL aderiv[2][MAXCOORD]; /* extra alloc area derivs */
                 REAL deriv2[MAXCOORD][MAXCOORD];
                 REAL volderiv[2][MAXCOORD];
                    /* [0] is deriv of head vol wrt tail */
               }; 
static struct e_curve *e_curve;

static int H0_flag; /* set to use (H - H_0)^2 */
static REAL h0_value;  /* value of H_0 */
static REAL sq_mean_mix; /* proportion for mix_sq_mean_curvature */
static REAL selfsim_coeff; /* self similarity */

long sqcurve_init_timestamp; /* when last initialized */
long sqcurve_grad_init_timestamp; /* when last initialized */

/* local prototypes */
void sqcurve_energy_precalc ARGS((vertex_id *,REAL (*)[MAXCOORD],WRAPTYPE*));
void sqcurve_grad_precalc ARGS((vertex_id *,edge_id *,REAL (*)[MAXCOORD],
    WRAPTYPE*));

/***********************************************************************
*
*  Function: sqcurve_method_init()
*
*  Purpose: Initializes data structures for square curvature.
*/

void sqcurve_method_init(mode,mi)
int mode; /* METHOD_VALUE or METHOD_GRADIENT */
struct method_instance *mi;
{ int k,n;
  facet_id f_id;
  struct gen_quant_method *gm;
  MAT2D(x,MAXCOORD,MAXCOORD);
  vertex_id vv_id;
  WRAPTYPE wraps[FACET_VERTS];

  if ( web.modeltype != LINEAR )
     kb_error(1758,"sq_mean_curvature method method only for LINEAR model.\n",RECOVERABLE);

 
  if ( everything_quantities_flag )
     GEN_QUANTS[sq_mean_curv_quantity_num].modulus = 
        globals[square_curvature_param].value.real;

  if ( sq_mean_curvature_mi < 0 )
  { /* see what method indices correspond to what methods */
     for ( n=0,gm = basic_gen_methods ; gm->name[0] != ' ' ; gm++,n++ )
     { if ( stricmp(gm->name,"sq_mean_curvature") == 0 ) 
            sq_mean_curvature_mi = n;
        if ( stricmp(gm->name,"eff_area_sq_mean_curvature") == 0 ) 
            eff_area_sq_mean_curvature_mi = n;
        if ( stricmp(gm->name,"normal_sq_mean_curvature") == 0 ) 
            normal_sq_mean_curvature_mi = n;
        if ( stricmp(gm->name,"mix_sq_mean_curvature") == 0 ) 
            mix_sq_mean_curvature_mi = n;
     }
  }

  if ( mi->gen_method == eff_area_sq_mean_curvature_mi )
      if ( SDIM != 3 )
          kb_error(1759,"eff_area_sq_mean_curvature method only for 3D space.\n",
                RECOVERABLE);
  if ( mi->gen_method == normal_sq_mean_curvature_mi )
      if ( SDIM != 3 )
          kb_error(1760,"normal_sq_mean_curvature method only for 3D space.\n",
                RECOVERABLE);
  if ( mi->gen_method == normal_sq_mean_curvature_mi )
      if ( SDIM != 3 )
          kb_error(1761,"mix_sq_mean_curvature method only for 3D space.\n",
                RECOVERABLE);

  /* get mix coefficient */
  if ( mi->gen_method == normal_sq_mean_curvature_mi )
  { k = lookup_global("sq_mean_mix");
     if ( k >= 0 ) sq_mean_mix = globals[k].value.real;
     else sq_mean_mix = 0.0;
  }

  /* see if using (H - H_0)^2 adjustment */
  H0_flag = 0;
  h0_value = 0;
  k = lookup_global("h_zero");
  if ( k >= 0 ) 
   { H0_flag = 1;
     h0_value = globals[k].value.real;
   }
  if ( self_similar_flag )
  { int param = lookup_global(SELFSIM_NAME);
    if ( param < 0 ) /* missing, so add */
        { param = add_global(SELFSIM_NAME);
          globals[param].value.real = 1.0;  /* default */
          globals[param].flags |=  ORDINARY_PARAM | SURFACE_PARAMETER;
        }
     selfsim_coeff = globals[param].value.real/6; /* area and volume factor */
     if ( H0_flag == 0 ) { H0_flag = 1; h0_value = 0.0; }
  }
  if ( vedge_timestamp < top_timestamp )
     { make_vedge_lists();
        vedge_timestamp = top_timestamp;
     }

  /* necessary precalculations, if anything changed */
  if ( ( mode == METHOD_VALUE )
     && ( sqcurve_init_timestamp < global_timestamp ) ) 
  {
     if ( v_curve ) myfree((char*)v_curve);
     v_curve = (struct v_curve *)mycalloc(web.skel[VERTEX].max_ord+1,
                         sizeof(struct v_curve));

     FOR_ALL_FACETS(f_id)
     { /* get side vectors */
        int i,j;
        vertex_id v_id[FACET_VERTS];
        facetedge_id fe_id;
        REAL side[FACET_EDGES][MAXCOORD];

        generate_facet_fe_init();
        for ( i = 0 ; i < FACET_EDGES ; i++ )
        {
          generate_facet_fe(f_id,&fe_id);
          v_id[i] = get_fe_tailv(fe_id);
        }
        get_facet_verts(f_id,x,wraps);  /* in tail order */
        for ( i = 0 ; i < FACET_EDGES ; i++ )
        { int ii = (i+1)%FACET_EDGES;
          for ( j = 0 ; j < SDIM ; j++ )
             side[i][j] = x[ii][j] - x[i][j];
        }
        sqcurve_energy_precalc(v_id,side,wraps);
     }

     sqcurve_init_timestamp = global_timestamp; 
  }

  if ( ( mode == METHOD_GRADIENT )
     && ( sqcurve_grad_init_timestamp < global_timestamp )  ) 
  {
     MAT2D(a,MAXCOORD,MAXCOORD);
     REAL af[MAXCOORD], gaf[MAXCOORD];
     MAT2D(grad,MAXCOORD,MAXCOORD);
     MAT2D(adft,MAXCOORD,MAXCOORD);
     MAT3D(seconds,MAXCOORD,MAXCOORD,MAXCOORD);

     /* see if anything changed from last initialization */

     if ( v_curve ) myfree((char*)v_curve);
     v_curve = (struct v_curve *)mycalloc(web.skel[VERTEX].max_ord+1,
                         sizeof(struct v_curve));
     if ( e_curve ) myfree((char*)e_curve);
     e_curve = (struct e_curve *)mycalloc(web.skel[EDGE].max_ord+1,
                         sizeof(struct e_curve));
      

     FOR_ALL_FACETS(f_id)
     { /* get side vectors */
        int i,j;
        vertex_id v_id[FACET_VERTS];
        edge_id e_id[FACET_VERTS];
        facetedge_id fe_id;
        REAL side[FACET_EDGES][MAXCOORD];

        generate_facet_fe_init();
        for ( i = 0 ; i < FACET_EDGES ; i++ )
        {
          generate_facet_fe(f_id,&fe_id);
          v_id[i] = get_fe_tailv(fe_id);
          e_id[i] = get_fe_edge(fe_id);
        }
        get_facet_verts(f_id,x,wraps);  /* in tail order */
        for ( i = 0 ; i < FACET_EDGES ; i++ )
        { int ii = (i+1)%FACET_EDGES;
          for ( j = 0 ; j < SDIM ; j++ )
             side[i][j] = x[ii][j] - x[i][j];
        }
        sqcurve_grad_precalc(v_id,e_id,side,wraps);
     }

    FOR_ALL_VERTICES(vv_id) 
    { struct v_curve *vc = v_curve + ordinal(vv_id);
      if ( !boundary_curvature_flag ) vc->a = vc->area; 
      if ( (get_vattr(vv_id) & CONSTRAINT) && !(sqcurve_ignore_constr) )
         {  conmap_t * conmap = get_v_constraint_map(vv_id);
             int i,j,oncount = 0;
             struct constraint *con[MAXCONPER];
             REAL perp[MAXCOORD];
             REAL dummy;

             for ( j = 1 ; j <= (int)conmap[0] ; j++ )
                  if ( conmap[j] & CON_HIT_BIT )
                        con[oncount++] = get_constraint(conmap[j]);

             /* stuff for gradient of projection operator */
             for ( j = 0 ; j < oncount ; j++ )
                 eval_second(con[j]->formula,get_coord(vv_id),SDIM,&dummy,
                      grad[j],seconds[j],vv_id);

             /* construct matrix A */
             for ( i = 0 ; i < oncount ; i++ )
                for ( j = 0 ; j < oncount ; j++ )
                  a[i][j] = SDIM_dot(grad[i],grad[j]);

             /* invert */
             mat_inv(a,oncount);

             mat_mult(a,grad,adft,oncount,oncount,SDIM);
             matvec_mul(adft,vc->force,af,oncount,SDIM);
             vec_mat_mul(af,grad,gaf,oncount,SDIM);
             for ( k = 0 ; k < SDIM ; k++ )
             { REAL sum = 0.0;
               for ( i = 0 ; i < oncount ; i++ )
                 for ( j = 0 ; j < SDIM ; j++ )
                 { sum += af[i]*seconds[i][k][j]*gaf[j];
                sum += vc->force[j]*seconds[i][k][j]*af[i];
              }
              vc->fpgradf[k] = -2*sum;
             } 
  
             constr_proj(TANGPROJ,oncount,con,get_coord(vv_id),
                                 vc->force,perp,NULL,NO_DETECT,NULLID);
             for ( j = 0 ; j < SDIM ; j++ )
                vc->force[j] -= perp[j];

             constr_proj(TANGPROJ,oncount,con,get_coord(vv_id),
                                 vc->normal,perp,NULL,NO_DETECT,NULLID);
             for ( j = 0 ; j < SDIM ; j++ )
                  vc->normal[j] -= perp[j];
         }

        if ( H0_flag ) 
         { 
            vc->norm = SDIM_dot(vc->normal,vc->normal);
            vc->f = SDIM_dot(vc->force,vc->normal);
            vc->h = vc->f/vc->norm*3;
            vc->term = vc->h - h0_value;
         }
     }
     sqcurve_grad_init_timestamp = global_timestamp; 
  }
}

/********************************************************************
*
*  Function: sqcurve_energy_precalc()
*
*  Purpose:  Does square curvature energy calculation for a facet.
*
*/

void sqcurve_energy_precalc(v_id,side,wraps)
vertex_id *v_id;  /* vertex list for facet */
REAL (*side)[MAXCOORD];  /* side vectors */
WRAPTYPE *wraps; /* in case of symmetry group */
{ 
  REAL t1t1,t1t2,t2t2;
  REAL det;
  struct v_curve *vc[FACET_VERTS];
  int i,j;
  REAL area;

  t1t1 = SDIM_dot(side[0],side[0]);
  t1t2 = SDIM_dot(side[0],side[1]);
  t2t2 = SDIM_dot(side[1],side[1]);

  det = t1t1*t2t2 - t1t2*t1t2;

  area = sqrt(det)/2;
  for ( i = 0 ; i < FACET_VERTS ; i++ )
     { vc[i] = v_curve + ordinal(v_id[i]);
        vc[i]->area += area;
     }
  if ( boundary_curvature_flag ) /* apportion area differently */
     { int fixcount = 0;
        for ( i = 0 ; i < FACET_VERTS ; i++ )
          if ( get_vattr(v_id[i]) & (BOUNDARY|FIXED) ) fixcount++;
        for ( i = 0 ; i < FACET_VERTS ; i++ )
        { vc[i] = v_curve+ordinal(v_id[i]);
          if ( !(get_vattr(v_id[i]) & (BOUNDARY|FIXED)) )
             vc[i]->a += 3*area/(3-fixcount);
        }
     }

  if ( area > 0.0 )
    if ( web.symmetry_flag )
    { REAL f[FACET_VERTS][MAXCOORD];
      REAL wforce[MAXCOORD];  /* unwrapped forces */
      for ( i = 0 ; i < SDIM ; i++ )
      { f[0][i] = -(t2t2*side[0][i]-t1t2*side[1][i])/4/area;
         f[1][i] = (t2t2*side[0][i]-t1t2*side[1][i])/4/area;
         f[2][i] = (t1t1*side[1][i]-t1t2*side[0][i])/4/area;
         f[1][i] -= (t1t1*side[1][i]-t1t2*side[0][i])/4/area;
      }
      for ( i = 0 ; i < FACET_VERTS ; i++ )  /* vertex loop */
      { (*sym_form_pullback)(get_coord(v_id[i]),wforce,f[i],wraps[i]);
         for ( j = 0 ; j < SDIM ; j++ )
            vc[i]->force[j] += wforce[j];
      }
    }
    else for ( i = 0 ; i < SDIM ; i++ )
    { vc[0]->force[i] -= (t2t2*side[0][i]-t1t2*side[1][i])/4/area;
      vc[1]->force[i] += (t2t2*side[0][i]-t1t2*side[1][i])/4/area;
      vc[2]->force[i] += (t1t1*side[1][i]-t1t2*side[0][i])/4/area;
      vc[1]->force[i] -= (t1t1*side[1][i]-t1t2*side[0][i])/4/area;
    }

  /* accumulate normal vector at each vertex;  should pullback */
    { REAL normal[MAXCOORD];
      cross_prod(side[0],side[1],normal);
      for ( i = 0 ; i < FACET_VERTS ; i ++ )
      { if ( web.symmetry_flag && wraps[i] )
         { REAL wnorm[MAXCOORD];
            (*sym_form_pullback)(get_coord(v_id[i]),wnorm,normal,wraps[i]);
            for ( j = 0 ; j < SDIM ; j++ )
              vc[i]->normal[j] += wnorm[j];
         }
         else
            for ( j = 0 ; j < SDIM ; j++ )
              vc[i]->normal[j] += normal[j];
      }
    }
}


/************************************************************************
*
*  Function: sqcurve_grad_precalc()
*
*  Purpose:  Does square curvature grad calculation for a facet.
*
*/

void sqcurve_grad_precalc(v_id,e_id,side,wraps)
vertex_id *v_id; /* vertex list of facet */
edge_id *e_id;     /* edge list */
REAL (*side)[MAXCOORD];  /* side vectors */
WRAPTYPE *wraps;
{ 
  REAL det;
  struct v_curve *vc[FACET_VERTS];
  int i,j,k;
  REAL force[FACET_VERTS][MAXCOORD];
  REAL tt[FACET_VERTS][FACET_VERTS]; /* side dot products */
  REAL area;
  struct e_curve *ec[FACET_EDGES];
  int fixcount=0;

  for ( j = 0 ; j < FACET_VERTS ; j++ )
     for ( k = 0 ; k <= j ; k++ )
        tt[j][k] = tt[k][j] = SDIM_dot(side[j],side[k]);

  det = tt[0][0]*tt[1][1] - tt[0][1]*tt[0][1];

  area = sqrt(det)/2;
  for ( i = 0 ; i < FACET_VERTS ; i++ )
     { vc[i] = v_curve + ordinal(v_id[i]);
       vc[i]->area += area;
     }
  if ( boundary_curvature_flag ) /* apportion area differently */
     { fixcount = 0;
       for ( i = 0 ; i < FACET_VERTS ; i++ )
          if ( get_vattr(v_id[i]) & (BOUNDARY|FIXED) ) fixcount++;
       for ( i = 0 ; i < FACET_VERTS ; i++ )
          if ( !(get_vattr(v_id[i]) & (BOUNDARY|FIXED)) )
             vc[i]->a += 3*area/(3-fixcount);
     }
  for ( i = 0 ; i < FACET_EDGES ; i++ )
     ec[i] = e_curve + ordinal(e_id[i]);

  memset((char*)force,0,sizeof(force));
  for ( j = 0 ; j < FACET_VERTS ; j++ )
     { int i1 = (j+1)%FACET_VERTS;
        int i2 = (j+2)%FACET_VERTS;
        for ( i = 0 ; i < SDIM ; i++ )
          force[j][i] = (tt[i1][i1]*side[i2][i] - tt[i1][i2]*side[i1][i])
                                    /4/area;
     }

  /* first and second derivatives at vertices; should add pullback */
  for ( i = 0 ; i < FACET_VERTS ; i++ )
     { int ii = (i+1)%FACET_VERTS; /* opposite side from vertex i */
        for ( j = 0 ; j < SDIM ; j++ )
          vc[i]->force[j] += force[i][j];
        if ( boundary_curvature_flag && (fixcount != 3) )
         for ( j = 0 ; j < SDIM ; j++ )
          vc[i]->star_force[j] += 3*force[i][j]/(3-fixcount);
        for ( j = 0 ; j < SDIM ; j++ )
          { vc[i]->deriv2[j][j] +=  tt[ii][ii]/4/area;
             for ( k = 0 ; k < SDIM ; k++ )
                vc[i]->deriv2[j][k] -= (force[i][j]*force[i][k]+0.25*side[ii][j]
                                                *side[ii][k])/area;
          }
     }

  /* now first and second derivatives on edges; should add pullback */
  for ( i = 0 ; i < FACET_EDGES ; i++ )
  { 
    int i1 =  (i+1)%FACET_EDGES;
    int i2 =  (i+2)%FACET_EDGES;
    for ( j = 0 ; j < SDIM ; j++ )
    { ec[i]->deriv2[j][j] += tt[i1][i2]/4/area;
      for ( k = 0 ; k < SDIM ; k++ )
        if ( inverted(e_id[i]) )
           ec[i]->deriv2[k][j] += (-side[i1][j]*side[i2][k]/2
                                + side[i2][j]*side[i1][k]/4
                                -force[i1][j]*force[i][k])/area;
         else
           ec[i]->deriv2[j][k] += (-side[i1][j]*side[i2][k]/2
                                   + side[i2][j]*side[i1][k]/4
                                   -force[i1][j]*force[i][k])/area;
      if ( boundary_curvature_flag 
                     && (get_vattr(v_id[i2]) & (FIXED|BOUNDARY)) )
      { if ( inverted(e_id[i]) )
        { ec[i]->aderiv[0][j] += 0.5*force[i1][j];
          ec[i]->aderiv[1][j] += 0.5*force[i][j];
        }
        else
        { ec[i]->aderiv[0][j] += 0.5*force[i][j];
          ec[i]->aderiv[1][j] += 0.5*force[i1][j];
        }
      }

      if ( inverted(e_id[i]) )
      { ec[i]->deriv[0][j] += force[i1][j];
        ec[i]->deriv[1][j] += force[i][j];
      }
      else
      { ec[i]->deriv[0][j] += force[i][j];
        ec[i]->deriv[1][j] += force[i1][j];
      }
    }
    if ( self_similar_flag )
    { REAL cp[MAXCOORD];
      int ii;
      cross_prod(get_coord(get_edge_tailv(e_id[i])),
                       get_coord(get_edge_headv(e_id[i])), cp);
      ii =  inverted(e_id[i1]) ? 0 : 1;
      for ( j = 0 ; j < SDIM ; j++ ) ec[i1]->volderiv[ii][j] += cp[j];
      ii =  inverted(e_id[i2]) ? 1 : 0;
      for ( j = 0 ; j < SDIM ; j++ ) ec[i2]->volderiv[ii][j] += cp[j];
    }
  } 
  
  /* accumulate normal vector at each vertex */
    { REAL normal[MAXCOORD];
      cross_prod(side[0],side[1],normal);
      for ( i = 0 ; i < 3 ; i ++ )
         for ( j = 0 ; j < SDIM ; j++ )
            vc[i]->normal[j] += normal[j];
    }
}


/*************************************************************************
*
*  function: sqcurve_method_value()
*
*  purpose: calculate squared mean curvature of given vertex
*              from precalculated data.
*
*/

REAL sqcurve_method_value(v_info)
struct qinfo *v_info;
{
     vertex_id v_id = v_info->v[0];
      REAL h,venergy;
      ATTR attr = get_vattr(v_id);
      int ordv = ordinal(v_id);
      struct v_curve *vc;
      REAL denom,f;
      REAL area; /* curvature normalization area */

     /* kludge check here, needed for info_only individual attributes */
     if ( sqcurve_init_timestamp < global_timestamp ) 
              sqcurve_method_init(METHOD_VALUE,METH_INST+v_info->method);

      vc = v_curve + ordv;
      if ((attr & BOUNDARY) && !(METH_INST[v_info->method].flags & IGNORE_CONSTR))
          return 0.0;
      if ((attr & FIXED) && !(METH_INST[v_info->method].flags & IGNORE_FIXED))
          return 0.0;
      if ( vc->area == 0.0 ) return 0.0;
      if ( !boundary_curvature_flag )
         { vc->a = vc->area; area = vc->area/3; } 
      else { area = vc->area/3; }
      if ( (attr & CONSTRAINT) && !(METH_INST[v_info->method].flags & IGNORE_CONSTR) )
         {
             conmap_t * conmap = get_v_constraint_map(v_id);
             int j,oncount = 0;
             struct constraint *con[MAXCONPER];
             REAL perp[MAXCOORD];

             for ( j = 1 ; j <= (int)conmap[0] ; j++ )
                  if ( conmap[j] & CON_HIT_BIT )
                        con[oncount++] = get_constraint(conmap[j]);

             constr_proj(TANGPROJ,oncount,con,get_coord(v_id),
                                 vc->force,perp,NULL,NO_DETECT,NULLID);
             for ( j = 0 ; j < SDIM ; j++ )
                vc->force[j] -= perp[j];

             constr_proj(TANGPROJ,oncount,con,get_coord(v_id),
                                 vc->normal,perp,NULL,NO_DETECT,NULLID);
             for ( j = 0 ; j < SDIM ; j++ )
                  vc->normal[j] -= perp[j];
         }
      if ( METH_INST[v_info->method].gen_method == normal_sq_mean_curvature_mi )
        { f = SDIM_dot(vc->force,vc->force);
          denom = SDIM_dot(vc->force,vc->normal);
          if ( denom == 0.0 )  h = 0.0;
          else h = 3*f/denom;  /* mean, and  normal was twice area */
          if ( H0_flag ) h -= h0_value;
          venergy = h*h;
        }
      else if ( METH_INST[v_info->method].gen_method==mix_sq_mean_curvature_mi )
      {
          /* normal_sq_mean part */
          f = SDIM_dot(vc->force,vc->force);
          denom = SDIM_dot(vc->force,vc->normal);
          if ( denom == 0.0 )  h = 0.0;
          else h = 3*f/denom;  /* mean, and  normal was twice area */

            /* add other part */
            vc->h = h = h*sq_mean_mix + SDIM_dot(vc->force,vc->normal)/
                          SDIM_dot(vc->normal,vc->normal)*3*(1-sq_mean_mix);

            h -= h0_value;
          venergy = h*h;
      }
      else if ( H0_flag )
         { REAL term,sim;
            vc->h = h = SDIM_dot(vc->force,vc->normal)/
                          SDIM_dot(vc->normal,vc->normal)*3;
                            /* since vc->normal = 6*volgrad */
            term = h - h0_value;
            if ( self_similar_flag )
            { vc->vol = SDIM_dot(get_coord(v_id),vc->normal);
              sim = selfsim_coeff*vc->vol;
              term -= sim;
            }
            venergy = term*term;
         }
      else if ( METH_INST[v_info->method].gen_method == eff_area_sq_mean_curvature_mi )
        { f = SDIM_dot(vc->force,vc->force);
          denom = SDIM_dot(vc->normal,vc->normal);
          if ( denom == 0.0 ) venergy = 0.0;
          else
             venergy = 9*f/denom;  /* 9 = 36/4 */
        }
      else /* plain squared curvature */
         venergy = SDIM_dot(vc->force,vc->force)/area/area/4;

      return venergy*vc->a/3;
}

/*************************************************************************
*
*  function: sqcurve_method_grad()
*
*  purpose:  Convert square curvature data into forces for a vertex.
*
*/

REAL sqcurve_method_grad(v_info)                
struct qinfo *v_info;
{
  vertex_id v_id = v_info->v[0];
  edge_id e_id,ee_id,eee_id;
  int i,j;
  REAL e,e1,e2,denom;
  REAL f[MAXCOORD];
  REAL fudge1,fudge2,fudge3; /* combinations of values */
  REAL fudge11,fudge12,fudge13; /* combinations of values */
  REAL fudge21,fudge22,fudge23; /* combinations of values */
  REAL h; /* curvature */
  REAL area; /* curvature normalization area */
  REAL a;     /* integral area allocation */
  struct v_curve *vc = v_curve + ordinal(v_id);
  ATTR attr = get_vattr(v_id);
  REAL  ad[MAXCOORD];
  REAL venergy = 0.0;
  REAL *grad = v_info->grad[0]; 

  if ( div_normal_curvature_flag ) 
      kb_error(1761,"Force not implemented yet for div_normal_curvature.\n",
         RECOVERABLE );

  for ( i = 0 ; i < SDIM ; i++ ) grad[i] = 0.0;

  if ((attr & BOUNDARY) && !(METH_INST[v_info->method].flags & IGNORE_CONSTR))
         return 0.0;
  if ((attr & FIXED) && !(METH_INST[v_info->method].flags & IGNORE_FIXED))
         return 0.0;
  if ( vc->area == 0.0 ) return 0.0;
  if ( !boundary_curvature_flag ) {  area = a = vc->a/3; } 
  else { a = vc->a/3; area = vc->area/3; }
  for ( i = 0 ; i < SDIM  ;i++ ) /* alloc area deriv */
      if ( boundary_curvature_flag ) ad[i] = vc->star_force[i]/3;
      else ad[i] = vc->force[i]/3;

  /* vertex self-second derivatives */
  if ( METH_INST[v_info->method].gen_method == normal_sq_mean_curvature_mi )
  { e = SDIM_dot(vc->force,vc->force);
    denom = SDIM_dot(vc->force,vc->normal);
    if ( denom != 0.0 )
    {
       h = 3*e/denom;
       if ( H0_flag ) h -= h0_value;
       venergy = h*h;
       fudge1 = 4*h/denom*vc->area;
       fudge2 = fudge1*e/denom/2;
       fudge3 = h*h/3;
       for ( i = 0 ; i < SDIM ; i++ )
       f[i] = fudge1*SDIM_dot(vc->force,vc->deriv2[i])
                   - fudge2*SDIM_dot(vc->normal,vc->deriv2[i])
                   + fudge3*(boundary_curvature_flag ? vc->star_force[i] :
                        vc->force[i]);
    }
    else 
       for ( i = 0 ; i < SDIM ; i++ ) f[i] =0.0;
  }
  else if ( METH_INST[v_info->method].gen_method==mix_sq_mean_curvature_mi )
  {
    /* normal_sq_mean part */
    e = SDIM_dot(vc->force,vc->force);
    denom = SDIM_dot(vc->force,vc->normal);
    if ( denom != 0.0 )
    {
       h = 3*e/denom;
       if ( H0_flag ) h -= h0_value;

       fudge1 = 4/denom;
       fudge2 = fudge1*e/denom/2;

       fudge3 = h*h/3;
       for ( i = 0 ; i < SDIM ; i++ )
       f[i] = fudge1*SDIM_dot(vc->force,vc->deriv2[i])
                   - fudge2*SDIM_dot(vc->normal,vc->deriv2[i]);

    }
    else 
       for ( i = 0 ; i < SDIM ; i++ ) f[i] =0.0;

      /* add other part */
    {
      REAL net;
      REAL fd[MAXCOORD];
      REAL aread[MAXCOORD];
      for ( i = 0 ; i < SDIM  ;i++ )
        fd[i] = SDIM_dot(vc->normal,vc->deriv2[i])/vc->norm*3;
      for ( i = 0 ; i < SDIM  ;i++ )
        { aread[i] = vc->force[i]/3;
           if ( boundary_curvature_flag ) ad[i] = vc->star_force[i]/3;
           else ad[i] = aread[i];
        }
      h = (h*sq_mean_mix+(1-sq_mean_mix)*vc->term);  /* make combination */
      for ( i = 0  ; i < SDIM ; i++ )
        {
           net = 2*h*(f[i]*sq_mean_mix+(1-sq_mean_mix)*fd[i])*a + h*h*ad[i];
           grad[i] += net;
        }
      }
    venergy = h*h;
  }
  else if ( METH_INST[v_info->method].gen_method == eff_area_sq_mean_curvature_mi )
  { e = SDIM_dot(vc->force,vc->force);
    denom = SDIM_dot(vc->normal,vc->normal);
    if ( denom != 0.0 )
    { for ( i = 0 ; i < SDIM ; i++ )
        f[i] = 18*SDIM_dot(vc->force,vc->deriv2[i])/denom*a
          + 9*e/denom*ad[i];
       venergy = 9*e/denom;  /* 9 = 36/4 */
    }
    else
       for ( i = 0 ; i < SDIM ; i++ ) f[i] =0.0;
  }
  else /* squared curvature */
    { e = SDIM_dot(vc->force,vc->force)/vc->area*3.0/4;
       for ( i = 0 ; i < SDIM ; i++ )
    { f[i] = (2*SDIM_dot(vc->force,vc->deriv2[i])
            - 4/3.0*e*vc->force[i])/vc->area*3.0/4;
       f[i] += vc->fpgradf[i]/vc->area*3.0/4;
    }
       venergy = e*3/vc->area/4;
    }

  if ( H0_flag  && 
    !(METH_INST[v_info->method].gen_method == normal_sq_mean_curvature_mi) )
    {
      REAL net,sim;
      REAL fd[MAXCOORD],simd[MAXCOORD];
      REAL aread[MAXCOORD];
      for ( i = 0 ; i < SDIM  ;i++ )
        fd[i] = SDIM_dot(vc->normal,vc->deriv2[i])/vc->norm*3;
      for ( i = 0 ; i < SDIM  ;i++ )
        { aread[i] = vc->force[i]/3;
           if ( boundary_curvature_flag ) ad[i] = vc->star_force[i]/3;
           else ad[i] = aread[i];
        }
      if ( self_similar_flag )
      { vc->vol = SDIM_dot(get_coord(v_id),vc->normal);
        sim = selfsim_coeff*vc->vol/area;
        vc->term -= sim/area;
        for ( i = 0 ; i < SDIM  ;i++ )
           { simd[i] = selfsim_coeff*vc->normal[i];
              grad[i] += 2*vc->term* (- simd[i])*a;
           }
      }
      for ( i = 0  ; i < SDIM ; i++ )
        {
           net = 2*vc->term*fd[i]*a + vc->term*vc->term*ad[i];
           grad[i] += net;
        }
    }
  else
    for ( i = 0 ; i < SDIM ; i++ )
       grad[i] += f[i];
    

  ee_id = get_vertex_edge(v_id);
  eee_id = ee_id;
  if ( valid_id(eee_id) ) do
     {
  vertex_id headv;
  vertex_id tailv;
  struct e_curve *ec;
  struct v_curve *vc1;
  struct v_curve *vc2;
  REAL s[MAXCOORD],cross1[MAXCOORD],cross2[MAXCOORD];
  REAL denom1,denom2;
  REAL s0[MAXCOORD];
  REAL wa[MAXCOORD],wb[MAXCOORD],w[MAXCOORD],aw[MAXCOORD],vw[MAXCOORD];
  facetedge_id fe;
  facet_id f_id;

  if ( inverted(eee_id) ) e_id = inverse_id(eee_id);
  else e_id = eee_id;
  headv = get_edge_headv(e_id);
  tailv = get_edge_tailv(e_id);
  ec = e_curve + ordinal(e_id);
  vc1 = v_curve + ordinal(tailv);
  vc2 = v_curve + ordinal(headv);
  
  if ( (vc1->area == 0.0) || (vc2->area == 0.0) ) continue;
  if ( METH_INST[v_info->method].gen_method == normal_sq_mean_curvature_mi )
   { facetedge_id fe_a;
      facetedge_id fe_b;
      REAL sa[MAXCOORD],sb[MAXCOORD];

      fe = get_edge_fe(e_id);
      fe_a = get_prev_edge(fe);
      fe_b = get_prev_edge(get_next_facet(fe));
      get_edge_side(e_id,s0);

      get_edge_side(get_fe_edge(fe_a),sa);
      cross_prod(vc1->force,sa,cross1);
      if ( (assume_oriented_flag && inverted(get_fe_facet(fe_a)))
         || (!assume_oriented_flag && triple_prod(vc1->normal,sa,s0) < 0.0 ) )
        for ( i = 0 ; i < SDIM ; i++ ) cross1[i] = -cross1[i];
      if ( fe_b != fe_a ) /* check for single edge on constraint */
        { get_edge_side(get_fe_edge(fe_b),sb);
           cross_prod(vc1->force,sb,s);
           if ( (assume_oriented_flag && inverted(get_fe_facet(fe_b)))
         || (!assume_oriented_flag && triple_prod(vc1->normal,sb,s0) < 0.0 ) )
              for ( i = 0 ; i < SDIM ; i++ ) cross1[i] -= s[i];
           else  for ( i = 0 ; i < SDIM ; i++ ) cross1[i] += s[i];
        }

      e1 = SDIM_dot(vc1->force,vc1->force);
      denom1 = SDIM_dot(vc1->force,vc1->normal);
      if ( denom1 != 0.0 )
      { h = 3*e1/denom1;
        if ( H0_flag ) h -= h0_value;
        fudge11 = 4*h/denom1*vc1->area;
        fudge12 = fudge11*e1/denom1/2;
        fudge13 = h*h/3;
      }
      else  fudge11 = fudge12 = fudge13 = 0.0; 

      e2 = SDIM_dot(vc2->force,vc2->force);
      denom2 = SDIM_dot(vc2->force,vc2->normal);
      if ( denom2 != 0.0 )
      { h = 3*e2/denom2;
        if ( H0_flag ) h -= h0_value;
        fudge21 = 4*h/denom2*vc2->area;
        fudge22 = fudge21*e2/denom2/2;
        fudge23 = h*h/3;
      }
      else  fudge21 = fudge22 = fudge23 = 0.0; 

      cross_prod(vc2->force,sa,cross2);
      if ( (assume_oriented_flag && !inverted(get_fe_facet(fe_a)))
    || (!assume_oriented_flag && triple_prod(vc2->normal,sa,s0) > 0.0 ) )
        for ( i = 0 ; i < SDIM ; i++ ) cross2[i] = -cross2[i];
      if ( fe_b != fe_a ) /* check for single edge on constraint */
        { cross_prod(vc2->force,sb,s);
           if ( (assume_oriented_flag && !inverted(get_fe_facet(fe_b)))
    || (!assume_oriented_flag && triple_prod(vc2->normal,sb,s0) > 0.0 ) )
              for ( i = 0 ; i < SDIM ; i++ ) cross2[i] -= s[i];
           else  for ( i = 0 ; i < SDIM ; i++ ) cross2[i] += s[i];
        }
   }
  else if ( METH_INST[v_info->method].gen_method == eff_area_sq_mean_curvature_mi )
   { 
      facetedge_id fe_a;
      facetedge_id fe_b;
      REAL sa[MAXCOORD],sb[MAXCOORD];

      fe = get_edge_fe(e_id);
      fe_a = get_prev_edge(fe);
      fe_b = get_prev_edge(get_next_facet(fe));
      get_edge_side(e_id,s0);

      get_edge_side(get_fe_edge(fe_a),sa);
      cross_prod(vc1->normal,sa,cross1);
      if ( (assume_oriented_flag && inverted(get_fe_facet(fe_a)))
           || (!assume_oriented_flag &&  SDIM_dot(cross1,s0) < 0.0 ) )
        for ( i = 0 ; i < SDIM ; i++ ) cross1[i] = -cross1[i];
      if ( fe_b != fe_a ) /* check for single edge on constraint */
        { get_edge_side(get_fe_edge(fe_b),sb);
           cross_prod(vc1->normal,sb,s);
           if ( (assume_oriented_flag && inverted(get_fe_facet(fe_b)))
             || (!assume_oriented_flag && SDIM_dot(s,s0) < 0.0 ) )
              for ( i = 0 ; i < SDIM ; i++ ) cross1[i] -= s[i];
           else  for ( i = 0 ; i < SDIM ; i++ ) cross1[i] += s[i];
        }

      e1 = SDIM_dot(vc1->force,vc1->force);
      denom1 = SDIM_dot(vc1->normal,vc1->normal);
      e2 = SDIM_dot(vc2->force,vc2->force);
      denom2 = SDIM_dot(vc2->normal,vc2->normal);

      cross_prod(vc2->normal,sa,cross2);
      if ( (assume_oriented_flag && !inverted(get_fe_facet(fe_a)))
            || (!assume_oriented_flag &&  SDIM_dot(cross2,s0) > 0.0 ) )
        for ( i = 0 ; i < SDIM ; i++ ) cross2[i] = -cross2[i];
      if ( fe_b != fe_a ) /* check for single edge on constraint */
        { cross_prod(vc2->normal,sb,s);
           if ( (assume_oriented_flag && !inverted(get_fe_facet(fe_b)))
            || (!assume_oriented_flag &&  SDIM_dot(s,s0) > 0.0 ) )
              for ( i = 0 ; i < SDIM ; i++ ) cross2[i] -= s[i];
           else  for ( i = 0 ; i < SDIM ; i++ ) cross2[i] += s[i];
        }
   }
  else  /* squared curvature */
   { e1 = SDIM_dot(vc1->force,vc1->force)/vc1->area*3.0/4;
      e2 = SDIM_dot(vc2->force,vc2->force)/vc2->area*3.0/4;
   }

  if ( inverted(eee_id) )
  { int tattr = get_vattr(tailv);
    if ( (!(tattr & FIXED) || 
                  (METH_INST[v_info->method].flags & IGNORE_FIXED))
   && (!(tattr & BOUNDARY) || 
                  (METH_INST[v_info->method].flags & IGNORE_CONSTR))
    )
    { /* force on head due to curvature at tail */
       if ( METH_INST[v_info->method].gen_method == normal_sq_mean_curvature_mi )
        for ( i = 0 ; i < SDIM ; i++ )
          { f[i] = fudge11*SDIM_dot(vc1->force,ec->deriv2[i])
               - fudge12*(cross1[i]+SDIM_dot(vc1->normal,ec->deriv2[i]))
               + fudge13*ec->deriv[1][i] ;
          }
       else if (METH_INST[v_info->method].gen_method==eff_area_sq_mean_curvature_mi )
          for ( i = 0 ; i < SDIM ; i++ )
          { if ( denom1 != 0.0 )
                f[i] = 6*SDIM_dot(vc1->force,ec->deriv2[i])/denom1*vc1->area
                  + 3*e1/denom1*ec->deriv[1][i] 
                  - 6*e1/denom1/denom1*cross1[i]*vc1->area;
            else f[i] = 0.0;
          }
       else  /* squared curvature */
        for ( i = 0 ; i < SDIM ; i++ )
          f[i] = (- 4/3.0*e1*ec->deriv[1][i]
          + 2*SDIM_dot(vc1->force,ec->deriv2[i]))/vc1->area*3.0/4;
       if ( H0_flag &&
          !(METH_INST[v_info->method].gen_method == normal_sq_mean_curvature_mi) )
        { REAL fd[MAXCOORD],net;
          fe = get_edge_fe(e_id);
          get_edge_side(get_fe_edge(get_next_edge(fe)),wa);
          get_edge_side(get_fe_edge(get_next_edge(get_next_facet(fe))),wb);
          f_id = get_fe_facet(fe);
          if ( inverted(f_id) )
            for ( i = 0 ; i < SDIM ; i++ ) w[i] = wb[i] - wa[i];
          else for ( i = 0 ; i < SDIM ; i++ ) w[i] = wa[i] - wb[i];
          cross_prod(w,vc1->force,aw);
          cross_prod(w,vc1->normal,vw);
          for ( i = 0 ; i < SDIM ; i++ )
          { fd[i] = (SDIM_dot(vc1->normal,ec->deriv2[i])/vc1->norm
              + aw[i]/vc1->norm - vc1->f*2*vw[i]/vc1->norm/vc1->norm)*3;
            ad[i] = ec->deriv[1][i]/3;
          }
          area = vc1->area/3;
          a = vc1->a/3;
          for ( i = 0 ; i < SDIM ; i++ )
          {
            net = 2*vc1->term*fd[i]*a 
                    + vc1->term*vc1->term*(ad[i]+ec->aderiv[1][i]/3);
            if ( self_similar_flag )
              net += 2*vc1->term*(-selfsim_coeff*ec->volderiv[1][i])*a;
            grad[i] += net;
          }
        }
       else
        for ( i = 0 ; i < SDIM ; i++ )
          grad[i] += f[i];
       }
    }
    else /* not inverted */
    if ( (!(get_vattr(headv) & FIXED) || 
                  (METH_INST[v_info->method].flags & IGNORE_FIXED))
   && (!(get_vattr(headv) & BOUNDARY) || 
                  (METH_INST[v_info->method].flags & IGNORE_CONSTR))
    )
    { /* force on tail due to curvature at head */
       if ( METH_INST[v_info->method].gen_method == normal_sq_mean_curvature_mi )
        for ( i = 0 ; i < SDIM ; i++ )
          { f[i] = fudge23*ec->deriv[0][i] - fudge22*cross2[i];
            for ( j = 0 ; j < SDIM ; j++ )
                   f[i] += fudge21*vc2->force[j]*ec->deriv2[j][i]
                            - fudge22*vc2->normal[j]*ec->deriv2[j][i];
          }
       else if ( METH_INST[v_info->method].gen_method == eff_area_sq_mean_curvature_mi )
        for ( i = 0 ; i < SDIM ; i++ )
          { if ( denom2 != 0.0 )
              { f[i] =    3*e2/denom2*ec->deriv[0][i] 
                            - 6*e2/denom2/denom2*cross2[i]*vc2->area;
                for ( j = 0 ; j < SDIM ; j++ )
                   f[i] += 6*vc2->force[j]*ec->deriv2[j][i]/denom2*vc2->area;
              }
            else f[i] = 0.0;
          }
       else  /* squared curvature */
        for ( i = 0 ; i < SDIM ; i++ )
        { f[i] = -4/3.0*e2*ec->deriv[0][i]/vc2->area*3.0/4;
           for ( j = 0 ; j < SDIM ; j++ )
               f[i] += 2*vc2->force[j]*ec->deriv2[j][i]/vc2->area*3.0/4;
        }

       if ( H0_flag &&
          !(METH_INST[v_info->method].gen_method == normal_sq_mean_curvature_mi) )
        { REAL fd[MAXCOORD],net;
          fe = get_edge_fe(e_id);
          get_edge_side(get_fe_edge(get_next_edge(fe)),wa);
          get_edge_side(get_fe_edge(get_next_edge(get_next_facet(fe))),wb);
          f_id = get_fe_facet(fe);
          if ( inverted(f_id) )
            for ( i = 0 ; i < SDIM ; i++ ) w[i] = wb[i] - wa[i];
          else for ( i = 0 ; i < SDIM ; i++ ) w[i] = wa[i] - wb[i];
          cross_prod(vc2->force,w,aw);
          cross_prod(vc2->normal,w,vw);
          for ( i = 0 ; i < SDIM ; i++ )
          { for ( j = 0, fd[i] = 0.0 ; j < SDIM ; j++ )
               fd[i] += vc2->normal[j]*ec->deriv2[j][i];
            fd[i] = (fd[i]/vc2->norm
                        + aw[i]/vc2->norm - vc2->f*2*vw[i]/vc2->norm/vc2->norm)*3;
            ad[i] = ec->deriv[0][i]/3;
          }
          area = vc2->area/3;
          a = vc2->a/3;
          for ( i = 0 ; i < SDIM ; i++ )
          {
            net = 2*vc2->term*fd[i]*a 
                    + vc2->term*vc2->term*(ad[i]+ec->aderiv[0][i]/3);
            if ( self_similar_flag )
              net += 2*vc2->term*(-selfsim_coeff*ec->volderiv[0][i])*a;
            grad[i] += net;
          }
        }
       else
        for ( i = 0 ; i < SDIM ; i++ )
          grad[i] += f[i];
    }
  eee_id = get_next_tail_edge(eee_id);
} while ( !equal_id(eee_id,ee_id) );
    return venergy*vc->a/3;
}  /* end sqcurve_method_grad */


/*************************************************************************
              square mean curvature for string

              With optional power in variable sqcurve_string_power 

**************************************************************************/
/************************************************************************
*
*  function: sqcurve_string_init()
*
*  purpose:  get power, if any.
*/
void sqcurve_string_init(mode,mi)
int mode;
struct method_instance *mi;
{ 
}
/************************************************************************
*
*  function: sqcurve_string_value()
*
*  purpose:  Calculate square curvature energy for string model
*                Works locally vertex by vertex. Assumes two edges per vertex.
*
*/

REAL sqcurve_string_value(v_info)
struct qinfo *v_info;
{
  REAL s1,s2,s1s2;  /* edge lengths */
  REAL *side1 = v_info->sides[0][0],*side2 = v_info->sides[0][1];
  REAL energy;
  REAL hh,a1,a2;
  REAL power;

  if ( v_info->vcount < 3 ) return 0.;

  if ( METH_INST[v_info->method].flags & METH_PARAMETER_1 )
     power = METH_INST[v_info->method].parameter_1;
  else power = 2.0;
  s1 = sqrt(SDIM_dot(side1,side1));
  s2 = sqrt(SDIM_dot(side2,side2));
  s1s2 = SDIM_dot(side1,side2);

  a1 = 1 + s1s2/s1/s2;
  a2 = (s1 + s2)/2;
  if ( (a1 <= 0.0) || (a2 == 0.0) ) return 0.0;

  hh = 2*a1/a2/a2;
  energy = a2*pow(hh,power/2);

  return energy;
}

/************************************************************************
*
*  function: sqcurve_string_grad()
*
*  purpose:  Calculate square curvature force for string model
*                Works locally vertex by vertex. 
*
*/

REAL sqcurve_string_grad(v_info)
struct qinfo *v_info;
{
  REAL s1,s2,s1s2;  /* edge lengths */
  REAL *side1 = v_info->sides[0][0],*side2 = v_info->sides[0][1];
  int  i,k;
  REAL hh,a1,a2,energy;
  REAL a1g[2][MAXCOORD],a2g[2][MAXCOORD];
  REAL s[2];
  REAL *side[2];
  REAL term; /* common factor */
  REAL power;

  if ( v_info->vcount < 3 ) return 0.0;

  if ( METH_INST[v_info->method].flags & METH_PARAMETER_1 )
     power = METH_INST[v_info->method].parameter_1;
  else power = 2.0;

  s1 = sqrt(SDIM_dot(side1,side1));
  s2 = sqrt(SDIM_dot(side2,side2));
  s1s2 = SDIM_dot(side1,side2);

  a1 = 2 + 2*s1s2/s1/s2;
  a2 = (s1 + s2)/2;
  if (a2 == 0.0) return 0.0;

  hh = a1/a2/a2;
  energy = a2*pow(hh,power/2);

  side[0] = side1; side[1] = side2; s[0] = s1; s[1] = s2;
  for ( k = 0 ; k < 2 ; k++ )
  { REAL coeff = 2*s1s2/s[k]/s[k]/s[k]/s[1-k];
     for ( i = 0 ; i < SDIM ; i++ )
        { a1g[k][i] = 2*side[1-k][i]/s1/s2 - coeff*side[k][i];
          a2g[k][i] = 0.5*side[k][i]/s[k];
        }
  }
  for ( i = 0 ; i < SDIM ; i++ ) v_info->grad[0][i] = 0.0;
  term = a2*pow(hh,power/2-1); 
  for ( k = 0 ; k < 2 ; k++ )
     for ( i = 0 ; i < SDIM ; i++ )
     { REAL f;
        f = a2g[k][i]*energy/a2 + term*power/2
              *(a1g[k][i]/a2/a2 - 2*a1/a2/a2/a2*a2g[k][i]);
        v_info->grad[k+1][i] = f; 
        v_info->grad[0][i] += -f; 
     }
  return energy;

}


/************************************************************************
*
*  function: sqcurve_string_hess()
*
*  purpose:  Calculate square curvature hessian for string model
*                Works locally vertex by vertex. 
*
*/

REAL sqcurve_string_hess(v_info)
struct qinfo *v_info;
{
  REAL s1,s2,s1s2;  /* edge lengths */
  REAL *side1 = v_info->sides[0][0],*side2 = v_info->sides[0][1];
  REAL *side[2];
  int  i,k,kk,ii;
  REAL a1,a2,energy;
  REAL a1g[2][MAXCOORD],a2g[2][MAXCOORD];
  REAL a1h[2][2][MAXCOORD][MAXCOORD],a2h[2][2][MAXCOORD][MAXCOORD];
  REAL s[2];
  REAL ****h = v_info->hess;
  REAL term; /* common factor */

  if ( v_info->vcount < 3 ) return 0.0;

  if ( METH_INST[v_info->method].flags & METH_PARAMETER_1 )
     kb_error(1762,"Hessian not implemented for sqcurve_string_power != 2.\n",
        RECOVERABLE);

  s1 = sqrt(SDIM_dot(side1,side1));
  s2 = sqrt(SDIM_dot(side2,side2));
  s1s2 = SDIM_dot(side1,side2);

  a1 = 1 + s1s2/s1/s2;
  a2 = (s1 + s2)/2;
  if ( a2 == 0.0 ) return 0.0;

  energy = 2*a1/a2;

  side[0] = side1; side[1] = side2; s[0] = s1; s[1] = s2;
  for ( k = 0 ; k < 2 ; k++ )
  { term = s1s2/s[k]/s[k]/s[k]/s[1-k];
     for ( i = 0 ; i < SDIM ; i++ )
        { a1g[k][i] = side[1-k][i]/s1/s2 - term*side[k][i];
          a2g[k][i] = 0.5*side[k][i]/s[k];
        }
  }
  for ( i = 0 ; i < SDIM ; i++ ) v_info->grad[0][i] = 0.0;
  term = a1/a2/a2;
  for ( k = 0 ; k < 2 ; k++ )
     for ( i = 0 ; i < SDIM ; i++ )
     { REAL f = 2*(a1g[k][i]/a2 - term*a2g[k][i]);
        v_info->grad[k+1][i] = f; 
        v_info->grad[0][i] += -f; 
     }

  /* hessian */
  memset((char*)a1h,0,sizeof(a1h));
  memset((char*)a2h,0,sizeof(a2h));
  for ( k = 0 ; k < 2 ; k++ )
  { REAL terma = 3*s1s2/s[k]/s[k]/s[k]/s[k]/s[k]/s[1-k];
     REAL termb = s1s2/s[k]/s[k]/s[k]/s[1-k]/s[1-k]/s[1-k];
     REAL termc = 1/s[k]/s[1-k]/s[1-k]/s[1-k];
     term = 1/s[k]/s[k]/s[k]/s[1-k];
     for ( i = 0 ; i < SDIM ; i++ )
     { a1h[k][1-k][i][i] += 1/s1/s2;
        for ( ii = 0 ; ii < SDIM ; ii++ )
        { a1h[k][k][i][ii] += -side[1-k][i]*term*side[k][ii];
          a1h[k][1-k][i][ii] += 
              -side[1-k][i]*termc*side[1-k][ii];
        }
        for ( ii = 0 ; ii < SDIM ; ii++ )
        { a1h[k][k][i][ii] += -side[1-k][ii]*term*side[k][i];
          a1h[k][1-k][i][ii] += -side[k][ii]*term*side[k][i];
          a1h[k][k][i][ii] += terma*side[k][i]*side[k][ii];
          a1h[k][1-k][i][ii] += termb*side[k][i]*side[1-k][ii];
        }
        a1h[k][k][i][i] += - s1s2*term;
        a2h[k][k][i][i] += 0.5/s[k];
        for ( ii = 0 ; ii < SDIM ; ii++ )
          a2h[k][k][i][ii] += -0.5*side[k][i]/s[k]/s[k]/s[k]*side[k][ii];

     }
  }
  for ( k = 0 ; k < 2 ; k++ )
    for ( kk = 0 ; kk < 2 ; kk++ )
     for ( i = 0 ; i < SDIM ; i++ )
      for ( ii = 0 ; ii < SDIM ; ii++ )
      { REAL f = 2*(a1h[k][kk][i][ii]/a2
                         - a1g[k][i]*a2g[kk][ii]/a2/a2
                         - a1g[kk][ii]*a2g[k][i]/a2/a2
                         - a1*a2h[k][kk][i][ii]/a2/a2
                         + 2*a1*a2g[k][i]*a2g[kk][ii]/a2/a2/a2);
         h[k+1][kk+1][i][ii] += f;
         h[k+1][0][i][ii] += -f;
         h[0][kk+1][i][ii] += -f;
         h[0][0][i][ii] += f;
      }
  return energy;
}
