/*
    ext2prepare.c -- prepares an ext2 fs for an online resize
    Copyright (C) 1999, 2000 Andreas Dilger <adilger@turbolinux.com>
    Copyright (C) 1999 Lennert Buytenhek <buytenh@gnu.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
  This program is to be used in cooperation with Andreas Dilger's
  online resize patch for the linux kernel.
*/

static const char _ext2prepare_c[] = "$Id: ext2prepare.c,v 1.14 2001/03/15 20:35:09 adilger Exp $";

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <getopt.h>
#include <sys/stat.h>
#include "ext2.h"

int	 need_res = 0;

#define EXT2_IND_UNCHANGED	1
#define EXT2_IND_CHANGED	2

void usage(FILE *outfile, char *progname)
{
	fprintf(outfile,
		"usage: %s [-d] [-q] [-v] [-V] device max_new_size[bkmgt]\n"
		"\t-d, --debug    : turn debug info on\n"
		"\t-f, --force    : force prepare without checking fs state\n"
		"\t-q, --quiet    : be quiet (print only errors)\n"
		"\t-v, --verbose  : be verbose\n"
		"\t-V, --version  : print version and exit\n"
		"\tnew_size is in ext2 blocks (1k, 2k, or 4k) (default),\n"
		"\tdisk Blocks (512), Kilo-, Mega-, Giga-, or Terabytes\n",
		progname);
	exit(1);
}

void parse_args(int *argc, char **argv[], int *flags)
{
	static struct option long_options[] = {
		{"debug", 0, 0, 'd'},
		{"force", 0, 0, 'f'},
		{"quiet", 0, 0, 'q'},
		{"verbose", 0, 0, 'v'},
		{"version", 0, 0, 'V'},
		{NULL, 0, 0, 0}};
	int c;
	int dummy;

	while ((c = getopt_long(*argc, *argv, "dfqvV", long_options,
				&dummy)) != -1) {
		switch (c) {
		case 'd': *flags |= FL_DEBUG; *flags &= ~FL_QUIET; break;
		case 'f': *flags |= FL_FORCE; break;
		case 'q': *flags |= FL_QUIET;
			  *flags &= ~(FL_VERBOSE | FL_DEBUG); break;
		case 'v': *flags |= FL_VERBOSE; *flags &= ~FL_QUIET; break;
		case 'V': *flags |= FL_VERSION; *flags &= ~FL_QUIET; return;
		}
	}

	*argc -= optind;
	*argv += optind;
}

/* Calculate the maximum number of blocks that we need to reserve in Bond */
blk_t calc_blocks(struct ext2_fs *fs)
{
	blk_t need_res;

	/* For a non-sparse filesystem, the maximum number of blocks to
	 * reserve happens when we have half of the filesystem full.
	 * If we are past half full, then we need fewer reserved blocks.
	 */
	if (!(fs->flags & FL_SPARSE))
		need_res =(fs->newgdblocks-max(fs->gdblocks,fs->newgdblocks/2))*
			  (fs->newgroups -max(fs->numgroups, fs->newgroups /2));

	/* For a sparse filesystem, the maximum number of blocks reserved at
	 * one time is not a simple function of the filesystem size.  Insead
	 * of actually working out the exact maximum, we have pre-calculated
	 * that the maximum number of groups with GDT blocks in a sparse ext2
	 * filesystem is 26, (2k -> 26, 4k -> 25, 8k -> 23, but will typically
	 * be much fewer), so we reserve indirect blocks for this many blocks.
	 * Since this calculation only affects the number of indirect blocks
	 * we pre-allocate, we do not need to get it exactly right.
	 *
	 * For a filesystem with a 32-bit block numbers (all current ext2
	 * filesystems), we are limited by the number of blocks in the fs:
	 *
	 * 1k blocksize => 4294967296 blocks / 8192 blocks/group = 524288 grp
	 *              => 524288 groups / 32 groups/block = 16384 GDT blocks!!
	 *                 (limit of 262106 groups by 8188 GDT blocks/group)
	 * 2k blocksize => 4294967296 blocks / 16384 blocks/group = 262144 grp
	 *              => 262144 groups / 64 groups/block = 4096 GDT blocks
	 * 4k blocksize => 4294967296 blocks / 32768 blocks/group = 131072 grp
	 *              => 131072 groups / 128 groups/block = 1024 GDT blocks
	 * 8k blocksize => 4294967296 blocks / 65536 blocks/group = 65536 grp
	 *              => 65536 groups / 256 groups/block = 256 GDT blocks
	 */
	else switch (fs->blocksize) {
	case 1024: need_res = (fs->newgdblocks - fs->gdblocks) * 26; break;
	case 2048: need_res = (fs->newgdblocks - fs->gdblocks) * 26; break;
	case 4096: need_res = (fs->newgdblocks - fs->gdblocks) * 25; break;
	case 8192: need_res = (fs->newgdblocks - fs->gdblocks) * 23; break;
	}

	if (fs->flags & FL_DEBUG)
		printf("calc_blocks: newgroups = %d, "
		       "newgdblocks = %d, oldgdblocks = %d, need_res = %d\n",
		       fs->newgroups, fs->newgdblocks, fs->gdblocks, need_res);

	return need_res;
}


blk_t getfreezeroedblock(struct ext2_fs *fs)
{
	int i;
	blk_t offset;

	for (i = 0, offset = fs->sb.s_first_data_block; i < fs->numgroups;
	     i++, offset += fs->sb.s_blocks_per_group) {
		blk_t j;

		/* Don't allocate inode blocks in range where we may need
		 * to move GDT blocks later */
		for (j = fs->itoffset + fs->inodeblocks + fs->newgdblocks;
		     j < fs->sb.s_blocks_per_group; j++) {
			if (!ext2_get_block_state(fs, offset + j)) {
				struct ext2_buffer_head *bh;

				ext2_set_block_state(fs, offset + j, 1, 1);
				bh = ext2_bcreate(fs, offset + j);
				bh->dirty = 1;
				ext2_brelse(bh, 0);

				return offset + j;
			}
		}
	}

	return 0;
}

/* allocate an indirect block, if necessary */
int check_inode_ind(struct ext2_fs *fs, blk_t *block, blk_t *blocks, int decr)
{
	if (decr)
		need_res -= fs->u32perblock;

	if (*block == 0) {
		*block = getfreezeroedblock(fs);
		if (*block == 0)
			return 0;
		*blocks += fs->blocksize >> 9;
		if (fs->flags & FL_DEBUG) {
			printf("check_inode_ind: add indirect 0x%06x ", *block);
			if (decr)
				printf("(%d remain)",
				       (need_res + fs->u32perblock - 1) /
				       fs->u32perblock);
			else
				printf("(double indirect)");
			printf("\n");
		}
		return EXT2_IND_CHANGED;
	}

	return EXT2_IND_UNCHANGED;
}

int check_inode_dind(struct ext2_fs *fs, blk_t *block, blk_t *blocks)
{
	struct ext2_buffer_head	*bh;
	__u32			*udata;
	int			 change = 0;
	int			 res;
	int			 i;

	if ((res = check_inode_ind(fs, block, blocks, 0)) == 0)
		return 0;
	change |= res;
	bh = ext2_bread(fs, *block);
	udata = (__u32 *)bh->data;
	for (i = 0; i < fs->u32perblock && need_res > 0; i++, udata++) {
		if ((res = check_inode_ind(fs, udata, blocks, 1)) == 0)
			return 0;
		change |= res;
	}
	bh->dirty = change & EXT2_IND_CHANGED ? 1: 0 ;
	ext2_brelse(bh, 0);

	return change;
}

int make_inode_ind(struct ext2_fs *fs, struct ext2_inode *inode)
{
	need_res -= EXT2_NDIR_BLOCKS;
	if (need_res > 0 &&
	    check_inode_ind(fs, inode->i_block + EXT2_IND_BLOCK,
			    &inode->i_blocks, 1) == 0) {
		fprintf(stderr, "%s: can't reserve indirect block\n", fs->prog);
		return 0;
	}

	if (need_res > 0 &&
	    check_inode_dind(fs, inode->i_block + EXT2_DIND_BLOCK,
			    &inode->i_blocks) == 0) {
		fprintf(stderr, "%s: can't reserve dindirect blocks\n",
			fs->prog);
		return 0;
	}

	if (need_res > 0) {
		fprintf(stderr, "%s: too many reserved blocks\n", fs->prog);
		return 0;
	}
	return 1;
}

int create_resize_inode(struct ext2_fs *fs)
{
	struct ext2_inode	tmp;
	struct ext2_inode	*inode = &tmp;
	int			diff;
	int			i;
	blk_t			block;

	if (fs->flags & FL_DEBUG)
		printf(__FUNCTION__ "\n");

	ext2_read_inode(fs, EXT2_RESIZE_INO, inode);
	if (inode->i_mode == 0 || inode->i_links_count == 0) {
		if (fs->flags & FL_VERBOSE)
			printf("creating resize inode (%03d)\n",
				EXT2_RESIZE_INO);
		memset(inode, 0, sizeof(struct ext2_inode));
		inode->i_mode = S_IFREG | S_IRUSR | S_IWUSR;
		inode->i_ctime = time(NULL);
		inode->i_links_count = 1;
		inode->i_flags = EXT2_NODUMP_FL;
	}

	if (!make_inode_ind(fs, inode))
		return 0;

	diff = fs->newgdblocks - fs->gdblocks;
	/* we store the number of reserved blocks in the inode version field */
	inode->i_version = diff;

	for (i = 0, block = fs->sb.s_first_data_block + fs->gdblocks + 1;
	     i < fs->numgroups;
	     i++, block += fs->sb.s_blocks_per_group) {
		int j;

		if (!ext2_bg_has_super(fs, i))
			continue;

		for (j = 0; j < diff; j++) {
			if (ext2_block_iterate(fs, inode, block + j,
					       EXT2_ACTION_FIND) < 0 &&
			    ext2_block_iterate(fs, inode, block + j,
					       EXT2_ACTION_ADD) < 0) {
				fprintf(stderr,
					"%s: error reserving block 0x%06x\n",
					fs->prog, block + j);
				return 0;
			}
		}
	}

	inode->i_atime = inode->i_mtime = time(NULL);
	ext2_write_inode(fs, EXT2_RESIZE_INO, inode);

	return 1;
}


int ext2_prepare_fs(struct ext2_fs *fs)
{
	int status;

	if (fs->flags & FL_VERBOSE)
		printf("prepare for %u blocks\n", fs->newblocks);

	if (fs->newblocks <= fs->sb.s_blocks_count)
		return 0;

	if ((fs->relocator_pool =
	     (unsigned char *)malloc(ext2_relocator_pool_size << 10)) == NULL) {
		fprintf(stderr, "%s: error allocating relocator pool!\n",
			fs->prog);
		return 0;
	}
	fs->relocator_pool_end = fs->relocator_pool +
				 (ext2_relocator_pool_size << 10);

	status = ext2_block_relocate(fs);

	if (status)
		status = ext2_metadata_push(fs);

	if (status)
		status = create_resize_inode(fs);

	free(fs->relocator_pool);
	fs->relocator_pool = NULL;
	fs->relocator_pool_end = NULL;

	return status;
}

int main(int argc, char *argv[])
{
	struct ext2_dev_handle	*handle;
	struct ext2_fs		*fs;
	char			*dev;
	char			*progname;
	blk_t			 maxres;
	blk_t			 resize = 0;
	char			 mod = '\0';
	int			 flags = FL_SAFE;

	progname = strrchr(argv[0], '/') ? strrchr(argv[0], '/') + 1 : argv[0];
	parse_args(&argc, &argv, &flags);

	if (!(flags & FL_QUIET))
		ext2_print_version(stdout, progname);
	if (flags & FL_VERSION)
		return 0;

	if (argc != 2)
		usage(stderr, progname);

	dev = argv[0];
	handle = ext2_make_dev_handle_from_file(dev, progname);
	if (handle == NULL) {
		fprintf(stderr, "%s: can't open %s\n", progname, dev);
		return 1;
	}

	sscanf(argv[1], "%i%c", &resize, &mod);

	switch (mod) {	/* Order of these options is important!! */
		case 't':
		case 'T': resize <<= 10; /* no break */
		case 'g':
		case 'G': resize <<= 10; /* no break */
		case 'm':
		case 'M': resize <<= 10; /* no break */
		case 'k':
		case 'K': /* Need to open fs to find block size */
		case  0 : break;
		case 'b':
		case 'B': resize >>= 1; break;
		default: usage(stderr, progname);
	}

	if (mod)
		flags |= FL_KB_BLOCKS;
	if ((fs = ext2_open(handle, resize, flags | FL_PREPARE)) == NULL) {
		fprintf(stderr, "%s: can't open %s\n", progname, dev);
		return 1;
	}

	/* For non-sparse 1k filesystems over 128GB, over one-quarter of
	 * the filesystem would be used by group descriptor blocks!
	 */
	if (fs->blocksize == 1024 && fs->newblocks > 1<<27 &&
	    !(fs->flags & FL_SPARSE)) {
		fprintf(stderr,
			"%s: won't prepare 1k block filesystem over 128GB.\n"
			"\tToo much space would be wasted.\n", progname);
		return 1;
	} else if ((fs->blocksize == 4096 && fs->newblocks > 1<<30) ||
		   fs->newblocks > 1<<31) {
		fprintf(stderr,
			"%s: can't resize filesystem over 2TB.\n", progname);
		return 1;
	}

	need_res = calc_blocks(fs);

	/* Maximum number of blocks we can reserve depends on block size,
	 * and the fact we do not use triple-indirect blocks.
	 */
	maxres = (1 + fs->u32perblock) * fs->u32perblock + EXT2_NDIR_BLOCKS;
	if (need_res > maxres) {
		/* Round down to the next multiple of the group size */
		fs->newblocks &= ~(fs->sb.s_blocks_per_group - 1);
		need_res = calc_blocks(fs);
		while (need_res > maxres) {
			fs->newblocks -= fs->sb.s_blocks_per_group;
			need_res = calc_blocks(fs);
		}
		fprintf(stderr,
		       "%s: can't prepare for filesystem over %d blocks\n"
		       "\tuntil the current filesystem grows larger.\n",
		       progname, fs->newblocks);
		return 1;
	}

	if (!ext2_prepare_fs(fs))
		return 2;

	if (!(flags & FL_QUIET))
		printf("%s: prepared for maximum size of %d blocks\n",
		       fs->prog, fs->newgdblocks *
		       (fs->blocksize / sizeof(struct ext2_group_desc)) *
		       fs->sb.s_blocks_per_group);
	ext2_close(fs);

	return 0;
}
