/*
    ext2online.c -- resizes a mounted ext2 fs
    Copyright (C) 1999, 2000 Andreas Dilger <adilger@turbolinux.com>
    Copyright (C) 1999 Lennert Buytenhek <buytenh@gnu.org>

    Parts of pathname canonicalization:
    Copyright (C) 1993 Rick Sladkey <jrs@world.std.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/*
  This program is to be used in cooperation with Andreas Dilger's
  ext2-online resize patch for the linux kernel 2.0, 2.2, or 2.3.
*/

static const char _ext2online_c[] = "$Id: ext2online.c,v 1.24 2004/09/30 14:12:01 sct Exp $";

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <mntent.h>
#include <sys/mount.h> /* mount.h may define these next two, but if not... */
#if !defined(MS_MGC_VAL) || !defined(MS_REMOUNT)
#include <linux/fs.h>
#endif
#include <signal.h>
#include <errno.h>
#include <stdarg.h>
#include <limits.h>		/* for PATH_MAX */
#ifndef PATH_MAX
#define PATH_MAX 8192
#endif
#include "ext2.h"

#define MAX_READLINKS 32
#define streq(s, t)	(strcmp ((s), (t)) == 0)

FILE *completion_output;

static char * canonicalize (const char *path);

static void usage(FILE *outfile, char *progname)
{
	fprintf(outfile,
		"usage: %s [-C fd] [-dfqvV] device [new_size[bkmgt]]\n"
		"\t-C, --completion : print completion information\n"
		"\t-d, --debug      : turn debug info on\n"
		"\t-f, --force      : skip safety checks\n"
		"\t-q, --quiet      : be quiet (print only errors)\n"
		"\t-v, --verbose    : be verbose\n"
		"\t-V, --version    : print version and exit\n"
		"\tfd is the file descriptor to output completion data to,\n"
		"\tnew_size is in ext2 blocks (1k, 2k, or 4k) (default),\n"
		"\tdisk Blocks (512 byte), Kilo-, Mega-, Giga-, or Terabytes\n",
		progname);
	exit(1);
}

static void parse_args(int *argc, char **argv[], int *flags)
{
	static struct option long_options[] = {
		{"completion", 1, 0, 'C'},
		{"debug", 0, 0, 'd'},
		{"force", 0, 0, 'f'},
		{"quiet", 0, 0, 'q'},
		{"verbose", 0, 0, 'v'},
		{"version", 0, 0, 'V'},
		{NULL, 0, 0, 0}};
	int c;
	int dummy;

	while ((c = getopt_long(*argc, *argv, "C:dfqvV", long_options,
				&dummy)) != -1) {
		switch (c) {
		case 'C': *flags |= FL_COMPLETION;
			  *flags &= ~(FL_QUIET|FL_DEBUG|FL_VERBOSE);
			  completion_output = fdopen(atoi(optarg), "w");
			  if(completion_output == NULL) {
				  perror(optarg);
				  usage(stderr, *argv[0]);
			  }
			  break;
		case 'd': *flags |= FL_DEBUG;
			  *flags &= ~(FL_QUIET|FL_COMPLETION);
			  break;
		case 'f': *flags |= FL_FORCE; break;
		case 'q': *flags |= FL_QUIET;
			  *flags &= ~(FL_COMPLETION | FL_DEBUG | FL_VERBOSE);
			  break;
		case 'v': *flags |= FL_VERBOSE;
			  *flags &= ~(FL_QUIET|FL_COMPLETION); break;
		case 'V': *flags |= FL_VERSION; *flags &= ~FL_QUIET; return;
		default: usage(stderr, *argv[0]);
		}
	}

	*argc -= optind;
	*argv += optind;
}

/* Find the mountpoint or device from the string in *dev */
static void find_mount(char *filename, char **dev, char **dir, char **type,
		       char **opts)
{
	FILE		*tabfile;
	struct mntent	*ent;

	tabfile = setmntent(filename, "r");

	while  ((ent = getmntent(tabfile)) != NULL) {
		if (strcmp(*dev, ent->mnt_fsname) == 0) {
			*dir = (char *)malloc(strlen(ent->mnt_dir) + 1);
			if (*dir == NULL)
				*dir = ent->mnt_dir;
			else
				strcpy(*dir, ent->mnt_dir);
			break;
		} else if (strcmp(*dev, ent->mnt_dir) == 0) {
			*dir = *dev;
			*dev = (char *)malloc(strlen(ent->mnt_fsname) + 1);
			if (*dev == NULL)
				*dev = ent->mnt_dir;
			else
				strcpy(*dev, ent->mnt_fsname);
			break;
		}
	}
	if (*dir != NULL && ent != NULL) {
		*type = (char *)malloc(strlen(ent->mnt_type) + 1);
		if (*type == NULL)
			*type = ent->mnt_type;
		else
			strcpy(*type, ent->mnt_type);
		/* Leave room at end for ",resize=<NEW_SIZE>:RSV" parameter */
		*opts = (char *)malloc(strlen(ent->mnt_opts) + 25);
		if (*opts == NULL) {
			fprintf(stderr,
				"%s: no memory for mount options - "
				"resize aborted\n", *dev);
			*dir = NULL;
		} else {
			char *index, *index2;

			if (hasmntopt(ent, MNTOPT_RO)) {
				fprintf(stderr,
					"%s: read-only mount, resize aborted\n",
					*dev);
				*dir = NULL;
			}

again:
			/* Don't copy options that "mount" parses for us */
			if ((index = hasmntopt(ent, "noatime")) != NULL ||
			    (index = hasmntopt(ent, "nodev")) != NULL ||
			    (index = hasmntopt(ent, "noexec")) != NULL ||
			    (index = hasmntopt(ent, "nosuid")) != NULL ||
			    (index = hasmntopt(ent, "rw")) != NULL ||
			    (index = hasmntopt(ent, "sync")) != NULL ||
			    (index = hasmntopt(ent, "loop")) != NULL) {
				if ((index2 = strchr(index, ',')) == NULL)
					index[0] = '\0';
				else
					while ((*index++ = *++index2) != '\0');
				goto again;
			}

			if (ent->mnt_opts[0] == '\0')
				*opts[0] = '\0';
			else
				sprintf(*opts, "%s,", ent->mnt_opts);
		}
	}
	endmntent(tabfile);
}


static int ext2_check_one_iterate(struct ext2_fs *fs)
{
	unsigned int three = 1, five = 5, seven = 7;
	unsigned int group;

	for (group = 0; group < fs->numgroups;
	     group = ext2_list_backups(fs, &three, &five, &seven)) {
		blk_t gdb = fs->sb.s_first_data_block + 1 +
				group * fs->sb.s_blocks_per_group;
		blk_t block;

		for (block = fs->gdblocks; block < fs->newgdblocks; block++) {
			if (fs->flags & FL_DEBUG)
				printf("checking for group block %d in Bond\n",
				       gdb + block);

			if (ext2_block_iterate(fs, &fs->resize, gdb + block,
					       EXT2_ACTION_FIND) < 0) {
				fprintf(stderr,
					"%s: group %d, block %u not reserved\n",
					fs->prog, group, gdb + block);
				return 0;
			}
		}
	}

	return 1;
}

static int ext2_check_one_rsv(struct ext2_fs *fs)
{
	struct ext2_buffer_head *dindir_bh;
	int apb = fs->blocksize / sizeof(__u32);
	__u32 *dindir_buf;
	blk_t gdb_num, pri_blk;
	int ret = 1;

	dindir_bh = ext2_bread(fs, fs->resize.i_block[EXT2_DIND_BLOCK]);

	dindir_buf = (__u32 *)(dindir_bh->data);
	for (gdb_num = fs->gdblocks,
	     pri_blk = fs->sb.s_first_data_block + 1 + fs->gdblocks;
	     gdb_num < fs->newgdblocks; gdb_num++, pri_blk++) {
		struct ext2_buffer_head *pri_bh;
		unsigned int three = 1, five = 5, seven = 7;
		unsigned int group;
		int last = 0;

		if (dindir_buf[gdb_num % apb] != pri_blk) {
			fprintf(stderr, "found %d not %d at %d[%d]\n",
				dindir_buf[gdb_num % apb], pri_blk,
				fs->resize.i_block[EXT2_DIND_BLOCK],
				gdb_num % apb);
			ret = 0;
			goto exit_dind;
		}

		pri_bh = ext2_bread(fs, pri_blk);

		while ((group = ext2_list_backups(fs, &three, &five, &seven)) <
		       fs->numgroups) {
			__u32 *pri_buf = (__u32 *)(pri_bh->data);
			blk_t bku_blk;

			bku_blk = pri_blk + group * fs->sb.s_blocks_per_group;

			if (fs->flags & FL_DEBUG)
				printf("checking for group block %d in Bond\n",
				       bku_blk);
			if (pri_buf[last] != bku_blk) {
				fprintf(stderr, "found %d not %d at %d[%d]\n",
					pri_buf[last], bku_blk, pri_blk, last);
				ext2_brelse(pri_bh, 0);
				ret = 0;
				goto exit_dind;
			}
			++last;
		}
		ext2_brelse(pri_bh, 0);
	}
exit_dind:
	ext2_brelse(dindir_bh, 0);
	return ret;
}

/* Check that all of the needed group descriptor blocks are reserved */
static int ext2_check_newgdblocks(struct ext2_fs *fs)
{
	if (fs->sb.s_feature_compat & EXT2_FEATURE_COMPAT_RESIZE_INODE)
		return ext2_check_one_rsv(fs);

	return ext2_check_one_iterate(fs);
}

/* Create group descriptors, bitmaps, inodes for each new group */
static blk_t ext2_make_group(struct ext2_fs *fs, int group, blk_t bpg)
{
	struct ext2_buffer_head	*bh;
	struct ext2_group_desc	*gdp;
	blk_t			 start;
	blk_t			 new_bb, new_ib, new_itend;
	int			 max_inodes;
	int			 new_resgd;
	int			 has_sb;
	int			 i;

	new_resgd = fs->resgdblocks + fs->gdblocks - fs->newgdblocks;
	if (fs->flags & FL_VERBOSE) {
		printf("creating group %3d with %5u blocks "
		       "(rsvd = %2u, newgd = %2u)\r",
		       group, bpg, new_resgd, fs->newgdblocks);
		fflush(stdout);
	}

	new_itend = fs->itoffset + fs->inodeblocks;
	if (bpg < new_itend + 50) {
		/* This would only happen on the last group */
		printf("\n%s: unable to use %u blocks at end of device\n",
			fs->prog, bpg);
		return 0;
	}

	start = fs->sb.s_first_data_block + group * fs->sb.s_blocks_per_group;

	if (fs->flags & FL_DEBUG)
		printf("\nusing itoffset of %d\n", fs->itoffset);

	has_sb = ext2_bg_has_super(fs, group);

	if (fs->stride) {
		blk_t new_bmap = new_itend +
			(fs->stride * group % (bpg - new_itend));
		new_bb = new_bmap >= bpg ? new_itend : new_bmap;
	} else
		new_bb = has_sb ? fs->itoffset - 2 : 0;
	new_ib = (new_bb + 1 >= bpg) ? new_itend : new_bb + 1;

	/* Create the primary copy of the group descriptor table */
	bh = ext2_bread(fs, fs->sb.s_first_data_block + 1 +
			group / (fs->blocksize/sizeof(struct ext2_group_desc)));
	if(bh == NULL)
	{
		printf("\n%s: unable to read block %ld\n", fs->prog,
		       (long) fs->sb.s_first_data_block + 1 +
		       group / (fs->blocksize/sizeof(struct ext2_group_desc)));
		return 0;
	}
	gdp = (struct ext2_group_desc *)bh->data +
	      (group % (fs->blocksize/sizeof(struct ext2_group_desc)));

	gdp->bg_block_bitmap = start + new_bb;
	gdp->bg_inode_bitmap = start + new_ib;
	gdp->bg_inode_table  = start + fs->itoffset;
	gdp->bg_free_blocks_count = bpg - fs->inodeblocks - 2 -
		(has_sb ? 1 + fs->gdblocks + fs->resgdblocks: 0);
	gdp->bg_free_inodes_count = fs->sb.s_inodes_per_group;
	gdp->bg_used_dirs_count = 0;
	gdp->bg_pad = 0;
	gdp->bg_reserved[0] = 0;
	gdp->bg_reserved[1] = 0;
	gdp->bg_reserved[2] = 0;

	/* Update the in-memory copy of the group descriptor as well */
	memcpy(&fs->gd[group], gdp, sizeof(struct ext2_group_desc));

	if (fs->flags & FL_DEBUG) {
		printf("new block bitmap is at 0x%04x\n",
		       fs->gd[group].bg_block_bitmap);
		printf("new inode bitmap is at 0x%04x\n",
		       fs->gd[group].bg_inode_bitmap);
		printf("new inode table is at 0x%04x-0x%04x\n",
		       fs->gd[group].bg_inode_table,
		       fs->gd[group].bg_inode_table + fs->inodeblocks - 1);
		printf("new group has %d free blocks\n",
		       fs->gd[group].bg_free_blocks_count);
		printf("new group has %d free inodes (%d blocks)\n",
		       fs->gd[group].bg_free_inodes_count, fs->inodeblocks);
	}

	if (fs->flags & FL_IOCTL) {
		blk_t ret;
		struct ext2_new_group_input input;

		input.group = group;
		input.block_bitmap = fs->gd[group].bg_block_bitmap;
		input.inode_bitmap = fs->gd[group].bg_inode_bitmap;
		input.inode_table = fs->gd[group].bg_inode_table;
		input.blocks_count = bpg;
		input.reserved_blocks = bpg * fs->r_frac / 100;

		ret = ext2_ioctl(fs, EXT2_IOC_GROUP_ADD, &input) ? bpg : 0;

		ext2_brelse(bh, 0);

		return ret;
	}

	/* Write only this specific group to disk, so we don't corrupt
	 * other group descriptors within this block.
	 */
	fs->devhandle->ops->direct_write(fs->devhandle->cookie, gdp,
		(fs->sb.s_first_data_block + 1) * fs->blocksize +
		group * sizeof(struct ext2_group_desc),
		sizeof(struct ext2_group_desc));
	ext2_brelse(bh, 0);

	/* Clear metadata blocks to show all blocks/inodes unused */
	ext2_zero_blocks(fs, fs->gd[group].bg_block_bitmap, 1);
	ext2_zero_blocks(fs, fs->gd[group].bg_inode_bitmap, 1);
	ext2_zero_blocks(fs, fs->gd[group].bg_inode_table, fs->inodeblocks);

	/* Mark in-use metadata blocks */
	bh = ext2_bread(fs, fs->gd[group].bg_block_bitmap);
	bh->dirty = 1;
	set_bit(bh->data, new_bb);
	set_bit(bh->data, new_ib);
	for (i = fs->itoffset; i < new_itend; i++)
		set_bit(bh->data, i);

	/* Mark metadata blocks used in block bitmap */
	if (has_sb) {
		if (fs->flags & FL_DEBUG) {
			printf("mark superblock 0x%04x used\n", start);
			printf("mark group desc. 0x%04x-0x%04x used\n",
				start + 1, start + fs->newgdblocks);
			if (new_resgd)
				printf("mark reserved descriptors "
					"0x%04x-0x%04x used\n",
					start + fs->newgdblocks + 1,
					start + fs->gdblocks + fs->resgdblocks);
		}

		set_bit(bh->data, 0);		/* superblock */
		for (i = 1; i <= fs->newgdblocks; i++)
			set_bit(bh->data, i);	/* group descriptor blocks */
		for (; i <= fs->gdblocks + fs->resgdblocks; i++) {
			set_bit(bh->data, i);	/* reserved GDT blocks */
			if (ext2_block_iterate(fs, &fs->resize, start + i,
					       EXT2_ACTION_FIND) < 0 &&
			    ext2_block_iterate(fs, &fs->resize, start + i,
					       EXT2_ACTION_ADD) < 0)
				fprintf(stderr,
					"%s: error reserving block 0x%06x\n",
					fs->prog, start + i);
		}
	}

	/* Mark unavailable blocks at end of group used */
	if (fs->flags & FL_DEBUG && bpg != fs->blocksize * 8)
		printf("mark end blocks 0x%04x-0x%04x used\n",
			start + bpg, start + fs->blocksize * 8 - 1);

	for (i = bpg; i < fs->blocksize * 8; i++)
		set_bit(bh->data, i);
	ext2_brelse(bh, 0);

	/* Mark unavailable inodes at end of table used */
	max_inodes = fs->blocksize * 8;
	if (max_inodes != fs->sb.s_inodes_per_group) {
		if (fs->flags & FL_DEBUG)
			printf("mark %d unavailable end inodes used\n\n",
			       max_inodes - fs->sb.s_inodes_per_group);

		bh = ext2_bread(fs, new_ib + start);
		bh->dirty = 1;
		for (i = fs->sb.s_inodes_per_group; i < max_inodes; i++)
			set_bit(bh->data, i);
		ext2_brelse(bh, 0);
	}

	return bpg;
}

/*
 * Extend the last existing group and create new groups if necessary to
 * handle the new filesystem size.  If we are using the ioctl interface
 * then this is doing all of the work.
 */
static blk_t ext2_online_primary(struct ext2_fs *fs)
{
	int	group;
	blk_t	size;
	int	first = 1;
	int	bscount = 0;
	blk_t	count = 0;
	blk_t	ra = 0;

	if (fs->newblocks == fs->sb.s_blocks_count)
		return fs->newblocks;

	if (fs->flags & FL_DEBUG) {
		printf("\n%d old groups, %d blocks\n", fs->numgroups,
			fs->gdblocks);
		printf("%d new groups, %d blocks\n", fs->newgroups,
			fs->newgdblocks);
	}

	/* We need to use more GDT blocks - see if they are available */
	if (fs->newgdblocks > fs->gdblocks && !ext2_check_newgdblocks(fs))
		return 0;

	/* Fill last existing group */
	size = min(fs->numgroups * fs->sb.s_blocks_per_group +
			fs->sb.s_first_data_block, fs->newblocks);

	if (size > fs->sb.s_blocks_count && fs->flags & FL_DEBUG)
		printf("Filling last group to %u blocks\n", size);

	/*
	 * We call this regardless of whether we want to resize or not, so
	 * that we can determine whether the filesystem supports the ioctl
	 * interface.  It will return success if we try to resize to the
	 * current size.  Older code will return ENOTTY for unknown ioctls.
	 */
	if (ext2_ioctl(fs, EXT2_IOC_GROUP_EXTEND, &size))
		fs->flags |= FL_IOCTL;
	else if (errno != ENOTTY)
		return 0;

	ra = fs->newblocks - size;

	/* Add new groups */
	for (group = fs->numgroups; group < fs->newgroups; group++) {
		blk_t blocks, added;

		blocks = min(fs->sb.s_blocks_per_group, fs->newblocks - size);
		added = ext2_make_group(fs, group, blocks);
		if (!added)
			return 0;

		size += added;
		count += added;
		if (fs->flags & FL_COMPLETION) {
			if(first) {
				first = 0;
				fprintf(completion_output, "Added ");
			} else {
				int i;
				for(i = 0; i < bscount; i++)
					putc('\b', completion_output);
			}
			bscount = fprintf(completion_output, "%u/%u... ",
					  count, ra);
			fflush(completion_output);
		}
	}

	if(size && fs->flags & FL_COMPLETION) {
		fprintf(completion_output, "\n");
	}

	return size;
}

/*
 * Copy primary superblock, group descriptors to all "sparse" groups after
 * we are sure that the resize has worked
 */
static void ext2_online_finish(struct ext2_fs *fs)
{
	int			 group;
	blk_t			 block;
	blk_t			 gdb;

	/* Remove blocks as required from Bond inode */
	for (group = 0, gdb = fs->sb.s_first_data_block + 1;
	     group < fs->numgroups;
	     group++, gdb += fs->sb.s_blocks_per_group)
		if (ext2_bg_has_super(fs, group))
			for (block = fs->gdblocks;
			     block < fs->newgdblocks;
			     block++) {
				if (fs->flags & FL_DEBUG)
					printf("Checking for group block %d "
					       "in Bond\n", gdb);
				ext2_block_iterate(fs, &fs->resize, gdb + block,
						   EXT2_ACTION_DELETE);
			}

	/* Save the new number of reserved GDT blocks in the resize inode */
	fs->resize.i_generation = fs->resgdblocks + fs->gdblocks -
		fs->newgdblocks;

	/* Re-read kernel-updated superblock from disk, and fix up the rest
	 * of the fs structure so that the backup superblock copies are OK.
	 */
	fs->devhandle->ops->direct_read(fs->devhandle->cookie, &fs->sb,
					1024, 1024);
	fs->gdblocks = fs->newgdblocks;
	fs->numgroups = fs->newgroups;
	fs->metadirty = EXT2_META_BACKUP;
}

int main(int argc, char *argv[])
{
	char			*dev, *type, *opts, *dir = NULL;
	char			*progname;
	struct ext2_dev_handle	*handle;
	struct ext2_fs		*fs;
	blk_t			 resize = 0, realsize = 0;
	char			 mod = '\0';
	int			 flags = FL_SAFE;

	progname = strrchr(argv[0], '/') ? strrchr(argv[0], '/') + 1 : argv[0];
	parse_args(&argc, &argv, &flags);

	if (!(flags & FL_QUIET))
		ext2_print_version(stdout, progname);
	if (flags & FL_VERSION)
		return 0;

	if (argc < 1 || argc > 2)
		usage(stderr, progname);

	dev = canonicalize(argv[0]);

	find_mount(MOUNTED, &dev, &dir, &type, &opts);
	if (dir == NULL) {
		fprintf(stderr, "%s: can't find %s, is it mounted?\n",
			progname, dev);
		usage(stderr, progname);
	}

	handle = ext2_make_dev_handle_from_file(dev, dir, progname);
	if (handle == NULL)
		return 1;

	/* See if the user has specified a filesystem size */
	if (argc == 2) {
		sscanf(argv[1], "%i%c", &resize, &mod);

		switch (mod) {	/* Order of these options is important!! */
			case 't':
			case 'T': resize <<= 10; /* no break */
			case 'g':
			case 'G': resize <<= 10; /* no break */
			case 'm':
			case 'M': resize <<= 10; /* no break */
			case 'k': /* Need to open fs to find block size */
			case 'K': break;
			case  0 : break;
			case 'b':
			case 'B': resize >>= 1; break;
			default: usage(stderr, progname);
		}
	}

	if (mod)
		flags |= FL_KB_BLOCKS;
	if ((fs = ext2_open(handle, resize, flags | FL_ONLINE)) == NULL) {
		fprintf(stderr, "%s: can't open %s\n", progname, dev);
		return 1;
	}

	if (fs->newblocks == fs->sb.s_blocks_count) {
		fprintf(stderr, "%s: new size is same as current (%u)\n",
			fs->prog, fs->newblocks);
		return 0;
	}

	if (fs->newblocks < fs->sb.s_blocks_count) {
		/* Not currently, but it would be possible if the groups at
		 * the end of the FS are not in use at all (no inode, blocks)
		 * We could even do a system() call to lvreduce after the
		 * mount() call returned, when we can get a status code back...
		 */
		fprintf(stderr, "%s: %s has %u blocks cannot shrink to %u\n",
			fs->prog, dev, fs->sb.s_blocks_count, fs->newblocks);
		return 2;
	}

	ext2_read_inode(fs, EXT2_RESIZE_INO, &fs->resize);
	fs->resize.i_mtime = 0;
	if ((realsize = ext2_online_primary(fs)) == 0) {
		fprintf(stderr, "\n%s: unable to resize %s\n", progname, dev);
		return 3;
	} else if (realsize == fs->sb.s_blocks_count) {
		fprintf(stderr, "%s: new size is same as current (%u)\n",
			progname, realsize);
		return 0;
	}

	if (!(fs->flags & FL_IOCTL)) {
		fs->newblocks = realsize;
		printf("\n%s: resizing to %u blocks\n",
		       progname, fs->newblocks);

		if (fs->resize.i_mtime) {
			ext2_write_inode(fs, EXT2_RESIZE_INO, &fs->resize);
			fs->resize.i_mtime = 0;
		}
		ext2_sync(fs);

		/* Add the resize option to the end of the other mount options */
		sprintf(opts + strlen(opts), "resize=%u", fs->newblocks);
		if (fs->resgdblocks + fs->gdblocks - fs->newgdblocks)
			sprintf(opts + strlen(opts), ":%u",
				fs->resgdblocks +fs->gdblocks -fs->newgdblocks);

		if (fs->flags & FL_DEBUG) {
			printf("Calling mount() for %s(%s) with %s\n",
			       dev, dir, opts);
#if 0
			printf("Press ENTER to continue:\n");
			mod = getchar();
#endif
		}

		/* Ignore basic interrupts at this point, so we have a chance
		 * to clean up.  Before now, if the resize was interrupted,
		 * it didn't matter, as we haven't made any real changes to
		 * the filesystem.
		 */
		{
			struct sigaction act;
			act.sa_handler = SIG_IGN;
			act.sa_flags = SA_RESTART;

			sigaction(SIGHUP, &act, NULL);
			sigaction(SIGINT, &act, NULL);
			sigaction(SIGQUIT, &act, NULL);
		}

		if (mount(dev, dir, type, MS_MGC_VAL | MS_REMOUNT, opts) < 0) {
			int err = errno;
			fprintf(stderr, "%s: resize failed while in kernel\n",
				progname);
			perror(progname);
			if (err == EINVAL || err == ENOTTY)
				fprintf(stderr, "%s: does the kernel support "
					"online resizing?\n", progname);

			return 4;
		}

		/* Clean up the rest of the filesystem and the Bond inode */
		ext2_online_finish(fs);
		if (fs->resize.i_mtime)
			ext2_write_inode(fs, EXT2_RESIZE_INO, &fs->resize);
	}
	ext2_close(fs);

	return 0;
}


/* Fatal error.  Print message and exit.  */
void
die (int err, const char *fmt, ...) {
     va_list args;

     va_start (args, fmt);
     vfprintf (stderr, fmt, args);
     fprintf (stderr, "\n");
     va_end (args);

     exit (err);
}


void *
xmalloc (size_t size) {
     void *t;

     if (size == 0)
          return NULL;

     t = malloc (size);
     if (t == NULL)
          die (2, "not enough memory");

     return t;
}

char *
xstrdup (const char *s) {
     char *t;

     if (s == NULL)
          return NULL;

     t = strdup (s);

     if (t == NULL)
          die (2, "not enough memory");

     return t;
}

/* The following canonicalisation code is taken directly from
 * util-linux's mount code. */

/* this leaks some memory - unimportant for resize */
char *
myrealpath(const char *path, char *resolved_path, int maxreslth) {
	int readlinks = 0;
	char *npath;
	char link_path[PATH_MAX+1];
	int n;
	char *buf;
	int m;

	npath = resolved_path;

	/* If it's a relative pathname use getcwd for starters. */
	if (*path != '/') {
		if (!getcwd(npath, maxreslth-2))
			return NULL;
		npath += strlen(npath);
		if (npath[-1] != '/')
			*npath++ = '/';
	} else {
		*npath++ = '/';
		path++;
	}

	/* Expand each slash-separated pathname component. */
	while (*path != '\0') {
		/* Ignore stray "/" */
		if (*path == '/') {
			path++;
			continue;
		}
		if (*path == '.' && (path[1] == '\0' || path[1] == '/')) {
			/* Ignore "." */
			path++;
			continue;
		}
		if (*path == '.' && path[1] == '.' &&
		    (path[2] == '\0' || path[2] == '/')) {
			/* Backup for ".." */
			path += 2;
			while (npath > resolved_path+1 &&
			       (--npath)[-1] != '/')
				;
			continue;
		}
		/* Safely copy the next pathname component. */
		while (*path != '\0' && *path != '/') {
			if (npath-resolved_path > maxreslth-2) {
				errno = ENAMETOOLONG;
				return NULL;
			}
			*npath++ = *path++;
		}

		/* Protect against infinite loops. */
		if (readlinks++ > MAX_READLINKS) {
			errno = ELOOP;
			return NULL;
		}

		/* See if last pathname component is a symlink. */
		*npath = '\0';
		n = readlink(resolved_path, link_path, PATH_MAX);
		if (n < 0) {
			/* EINVAL means the file exists but isn't a symlink. */
			if (errno != EINVAL)
				return NULL;
		} else {
			/* Note: readlink doesn't add the null byte. */
			link_path[n] = '\0';
			if (*link_path == '/')
				/* Start over for an absolute symlink. */
				npath = resolved_path;
			else
				/* Otherwise back up over this component. */
				while (*(--npath) != '/')
					;

			/* Insert symlink contents into path. */
			m = strlen(path);
			buf = xmalloc(m + n + 1);
			memcpy(buf, link_path, n);
			memcpy(buf + n, path, m + 1);
			path = buf;
		}
		*npath++ = '/';
	}
	/* Delete trailing slash but don't whomp a lone slash. */
	if (npath != resolved_path+1 && npath[-1] == '/')
		npath--;
	/* Make sure it's null terminated. */
	*npath = '\0';
	return resolved_path;
}


/* Make a canonical pathname from PATH.  Returns a freshly malloced string.
   It is up the *caller* to ensure that the PATH is sensible.  i.e.
   canonicalize ("/dev/fd0/.") returns "/dev/fd0" even though ``/dev/fd0/.''
   is not a legal pathname for ``/dev/fd0''.  Anything we cannot parse
   we return unmodified.   */
static char * canonicalize (const char *path) 
{
     char *canonical;

     if (path == NULL)
	  return NULL;

     if (streq(path, "none") || streq(path, "proc") || streq(path, "devpts"))
	  return xstrdup(path);

     canonical = xmalloc (PATH_MAX+2);
  
     if (myrealpath (path, canonical, PATH_MAX+1))
	  return canonical;

     free(canonical);
     return xstrdup(path);
}
