/*
 *	HT Editor
 *	symmath.cc
 *
 *	Copyright (C) 2001 Stefan Weyergraf (stefan@weyergraf.de)
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License version 2 as
 *	published by the Free Software Foundation.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <string.h>

#include "tools.h"
#include "symmath.h"

#define ATOM_SYM_INT		MAGICD("SMA\x00")
#define ATOM_SYM_INT_SYMBOL	MAGICD("SMA\x01")
#define ATOM_SYM_INT_CONST	MAGICD("SMA\x02")

#define ATOM_SYM_BOOL		MAGICD("SMA\x10")
#define ATOM_SYM_BOOL_SYMBOL	MAGICD("SMA\x11")
#define ATOM_SYM_BOOL_INTCMP	MAGICD("SMA\x12")

/* C operator precedence (for output) */

b_op copp_mul[] = {
	b_mul,
	b_div,
	b_mod,
	b_invalid
};

b_op copp_add[] = {
	b_add,
	b_sub,
	b_invalid
};

b_op copp_bin[] = {
	b_and,
	b_or,
	b_xor,
	b_invalid
};

b_op *c_op_prec[] = {
	copp_bin,			/* lowest */
	copp_add,
	copp_mul,			/* biggest */
	NULL
};

UINT get_op_prec(b_op bop, b_op **op_prec)
{
	UINT k = 0;
	while (*op_prec) {
		b_op *l = *op_prec;
		while (*l != b_invalid) {
			if (*l == bop) return k;
			l++;
		}
		op_prec++;
		k++;
	}
	return 0;
}

/*
 *	CLASS sym_int_token
 */

sym_int_token::sym_int_token()
{
}

bool sym_int_token::compare_eq(sym_int_token *t)
{
	return false;
}

bool sym_int_token::evaluate(UINT *i)
{
	return false;
}

int sym_int_token::nstrfy(char *buf, int n)
{
	return 0;
}

/*
 *	CLASS sym_int_symbol
 */

sym_int_symbol::sym_int_symbol(char *n)
{
	name = strdup(n);
}

sym_int_symbol::~sym_int_symbol()
{
	free(name);
}

bool sym_int_symbol::compare_eq(sym_int_token *t)
{
	sym_int_symbol *s = (sym_int_symbol*)t;
	return (strcmp(name, s->name) == 0);
}

object *sym_int_symbol::duplicate()
{
	sym_int_symbol *p = new sym_int_symbol(name);
	return p;
}

bool sym_int_symbol::evaluate(UINT *i)
{
	return false;
}

int sym_int_symbol::nstrfy(char *buf, int n)
{
/* FIXME: use n */
	return sprintf(buf, "%s", name);
}

OBJECT_ID sym_int_symbol::object_id()
{
	return ATOM_SYM_INT_SYMBOL;
}

/*
 *	CLASS sym_int_const
 */

sym_int_const::sym_int_const(UINT v)
{
	value = v;
}
 
bool sym_int_const::compare_eq(sym_int_token *t)
{
	sym_int_const *s = (sym_int_const*)t;
	return (value == s->value);
}

object *sym_int_const::duplicate()
{
	return new sym_int_const(value);
}

bool sym_int_const::evaluate(UINT *i)
{
	*i = value;
	return true;	
}
 
int sym_int_const::nstrfy(char *buf, int n)
{
/* FIXME: use n */
	return sprintf(buf, "0x%x", value);
}
 
OBJECT_ID sym_int_const::object_id()
{
	return ATOM_SYM_INT_CONST;
}

/*
 *	CLASS sym_int_token_rec
 */
 
class sym_int_token_rec: public ht_data {
public:
	u_op uop;
	b_op bop;
	sym_int_token *token;

	sym_int_token_rec(u_op u, b_op b, sym_int_token *t)
	{
		uop = u;
		bop = b;
		token = t;		
	}

	~sym_int_token_rec()
	{
		token->done();
		delete token;
	}

	object *duplicate()
	{
		return new sym_int_token_rec(uop, bop, (sym_int_token*)token->duplicate());
	}
}; 

/*
 *	CLASS sym_int
 */

static b_op **output_op_prec = c_op_prec;

sym_int::sym_int()
{
	tokens = new ht_clist();
	((ht_clist*)tokens)->init();
}

sym_int::~sym_int()
{
	tokens->destroy();
	delete tokens;
}

bool sym_int::comp_eq(sym_int_token *a, sym_int_token *b)
{
	if (a->object_id() == b->object_id()) {
		return a->compare_eq(b);
	}
	return false;
}

void sym_int::b_operate(b_op bop, sym_int_token *t)
{
	tokens->insert(new sym_int_token_rec(u_null, bop, t));
}

void sym_int::clear()
{
	tokens->destroy();
	delete tokens;

	tokens = new ht_clist();
	((ht_clist*)tokens)->init();
}

object *sym_int::duplicate()
{
	sym_int *p = new sym_int();
	p->tokens = (ht_list*)tokens->duplicate();
	return p;
}

bool sym_int::evaluate(UINT *i)
{
	int c = tokens->count();
	UINT l = 0;
	for (int j = 0; j < c; j++) {
		UINT k;
		sym_int_token_rec *r = (sym_int_token_rec*)tokens->get(j);
		if (!r->token->evaluate(&k)) return false;
		switch (r->uop) {
			case u_null: break;
			case u_minus: k = -k; break;
			case u_not: k = ~k; break;
		}
		switch (r->bop) {
			case b_invalid: break;
			case b_mul: l *= k; break;
			case b_div: if (k) l /= k; else return false; break;
			case b_mod: if (k) l %= k; else return false; break;
			case b_add: l += k; break;
			case b_sub: l -= k; break;
			case b_and: l &= k; break;
			case b_or:  l |= k; break;
			case b_xor: l ^= k; break;
		}
	}
	*i = l;
	return true;
}

int sym_int::nstrfy(char *buf, int n)
{
	b_op lbop;
	int l = 0;
	int c = tokens->count();
	lbop = b_invalid;
	UINT para_count = 0;
	for (int i = 0; i < c; i++) {
		sym_int_token_rec *r = (sym_int_token_rec*)tokens->get(i);
		bool para = ((lbop != b_invalid) && (r->bop != b_invalid) &&
			(get_op_prec(r->bop, output_op_prec) > get_op_prec(lbop,
			output_op_prec)));
		if (para) para_count++;
		lbop = r->bop;
	}
	for (UINT i = 0; i < para_count; i++) buf[l++] = '(';
	lbop = b_invalid;
	for (int i = 0; i < c; i++) {
		sym_int_token_rec *r = (sym_int_token_rec*)tokens->get(i);
		bool para = ((lbop != b_invalid) && (r->bop != b_invalid) &&
			(get_op_prec(r->bop, output_op_prec) > get_op_prec(lbop,
			output_op_prec)));

		if (para) buf[l++] = ')';

		switch (r->uop) {
			case u_null: break;
			case u_minus: buf[l++] = '-'; break;
			case u_not: buf[l++] = '~'; break;
		}

		switch (r->bop) {
			case b_invalid: break;
			case b_mul: buf[l++] = '*'; break;
			case b_div: buf[l++] = '/'; break;
			case b_mod: buf[l++] = '%'; break;
			case b_add: buf[l++] = '+'; break;
			case b_sub: buf[l++] = '-'; break;
			case b_and: buf[l++] = '&'; break;
			case b_or:  buf[l++] = '|'; break;
			case b_xor: buf[l++] = '^'; break;
		}
		l += r->token->nstrfy(buf+l, n-l);

		lbop = r->bop;
	}
	buf[l] = 0;
	return l;
}

OBJECT_ID sym_int::object_id()
{
	return ATOM_SYM_INT;
}

void sym_int::replace(sym_int_token *token, sym_int_token *by)
{
	int c = tokens->count();
	for (int i = 0; i < c; i++) {
		sym_int_token_rec *r = (sym_int_token_rec*)tokens->get(i);
		if (r->token->object_id() == object_id()) {
			((sym_int*)r->token)->replace(token, by);
		} else if (comp_eq(r->token, token)) {
			r->token->done();
			delete r->token;
			r->token = (sym_int_token*)by->duplicate();
		}
	}		
}

void sym_int::set(sym_int_token *t)
{
	clear();
	tokens->insert(new sym_int_token_rec(u_null, b_invalid, t));
}

void sym_int::u_operate(u_op uop)
{
}

/*
 *	CLASS sym_bool_symbol
 */

sym_bool_symbol::sym_bool_symbol(char *n)
{
	name = strdup(n);
}

sym_bool_symbol::~sym_bool_symbol()
{
	free(name);
}

bool sym_bool_symbol::compare_eq(sym_bool_token *t)
{
	sym_bool_symbol *s = (sym_bool_symbol*)t;
	return (strcmp(name, s->name) == 0);
}

object *sym_bool_symbol::duplicate()
{
	sym_bool_symbol *p = new sym_bool_symbol(name);
	return p;
}

bool sym_bool_symbol::evaluate(bool *i)
{
	return false;
}

int sym_bool_symbol::nstrfy(char *buf, int n)
{
/* FIXME: use n */
	return sprintf(buf, "%s", name);
}

OBJECT_ID sym_bool_symbol::object_id()
{
	return ATOM_SYM_BOOL_SYMBOL;
}

/*
 *	CLASS sym_bool_token
 */

sym_bool_token::sym_bool_token()
{
}

bool sym_bool_token::compare_eq(sym_bool_token *t)
{
	return false;
}

bool sym_bool_token::evaluate(bool *i)
{
	return false;
}

int sym_bool_token::nstrfy(char *buf, int n)
{
	return 0;
}

/*
 *	CLASS sym_bool_const
 */

sym_bool_const::sym_bool_const(bool v)
{
	value = v;
}

bool sym_bool_const::compare_eq(sym_bool_token *t)
{
	sym_bool_const *s = (sym_bool_const*)t;
	return (value == s->value);
}

bool sym_bool_const::evaluate(bool *i)
{
	*i = value;
	return true;
}

int sym_bool_const::nstrfy(char *buf, int n)
{
	return sprintf("%s", value ? "true" : "false");
}

/*
 *	CLASS sym_bool_intcmp
 */

sym_bool_intcmp::sym_bool_intcmp(sym_int_token *i1, c_op c, sym_int_token *i2)
{
	int1 = i1;
	cop = c;
	int2 = i2;
}

bool sym_bool_intcmp::compare_eq(sym_bool_token *t)
{
	sym_bool_intcmp *s = (sym_bool_intcmp*)t;
/* FIXME: i1 == si2 && i2 == si1 */
	return (int1->compare_eq(s->int1) && int2->compare_eq(s->int2) && cop == s->cop);
}

object *sym_bool_intcmp::duplicate()
{
	return new sym_bool_intcmp((sym_int_token*)int1->duplicate(), cop, (sym_int_token*)int2->duplicate());
}

bool sym_bool_intcmp::evaluate(bool *i)
{
	UINT e1, e2;
	bool e = false;
	if (!int1->evaluate(&e1)) return false;
	if (!int2->evaluate(&e2)) return false;
	switch (cop) {
		case c_invalid: break;
		case c_eq: e = (e1 == e2); break;
		case c_ne: e = (e1 != e2); break;  
		case c_gt: e = (e1 > e2); break;  
		case c_ge: e = (e1 >= e2); break;  
		case c_lt: e = (e1 < e2); break;  
		case c_le: e = (e1 <= e2); break;  
	}
	*i = e;
	return true;
}

int sym_bool_intcmp::nstrfy(char *buf, int n)
{
	int i = 0;
	i += int1->nstrfy(buf+i, n-i);
	switch (cop) {
		case c_invalid: break;
		case c_eq:  strcpy(buf+i, "=="); i+=2; break;
		case c_ne:  strcpy(buf+i, "!="); i+=2; break;
		case c_gt:  buf[i++] = '>'; break;
		case c_ge:  strcpy(buf+i, ">="); i+=2; break;
		case c_lt:  buf[i++] = '<'; break;
		case c_le:  strcpy(buf+i, "<="); i+=2; break;
	}
	i += int2->nstrfy(buf+i, n-i);
	return i;
}

OBJECT_ID sym_bool_intcmp::object_id()
{
	return ATOM_SYM_BOOL_INTCMP;
}

/*
 *	CLASS sym_bool_token_rec
 */
 
class sym_bool_token_rec: public ht_data {
public:
	n_op nop;
	l_op lop;
	sym_bool_token *token;
	
	sym_bool_token_rec(n_op n, l_op l, sym_bool_token *t)
	{
		nop = n;
		lop = l;
		token = t;		
	}
	
	~sym_bool_token_rec()
	{
		token->done();
		delete token;
	}
	
	int nstrfy(char *buf, int n)
	{
		int i = 0;
		switch (nop) {
			case n_null: break;
			case n_not: buf[i++] = '!'; break;
		}
		switch (lop) {
			case l_invalid: break;
			case l_and: strcpy(buf+i, "&&"); i+=2; break;
			case l_or:  strcpy(buf+i, "||"); i+=2; break;
			case l_eq:  strcpy(buf+i, "=="); i+=2; break;
			case l_ne:  strcpy(buf+i, "!="); i+=2; break;
			case l_gt:  buf[i++] = '>'; break;
			case l_ge:  strcpy(buf+i, ">="); i+=2; break;
			case l_lt:  buf[i++] = '<'; break;
			case l_le:  strcpy(buf+i, "<="); i+=2; break;
		}
		return i + token->nstrfy(buf+i, n-i);
	}
	
	object *duplicate()
	{
		return new sym_bool_token_rec(nop, lop, (sym_bool_token*)token->duplicate());
	}
};

/*
 *	CLASS sym_bool
 */

sym_bool::sym_bool()
{
	tokens = new ht_clist();
	((ht_clist*)tokens)->init();
}

sym_bool::~sym_bool()
{
	tokens->destroy();
	delete tokens;
}

object *sym_bool::duplicate()
{
	sym_bool *p = new sym_bool();
	p->tokens = (ht_list*)tokens->duplicate();
	return p;
}

void sym_bool::clear()
{
	tokens->destroy();
	delete tokens;

	tokens = new ht_clist();
	((ht_clist*)tokens)->init();
}

bool sym_bool::evaluate(bool *i)
{
	int c = tokens->count();
	bool l = false;
	for (int j = 0; j < c; j++) {
		bool k;
		sym_bool_token_rec *r = (sym_bool_token_rec*)tokens->get(j);
		if (!r->token->evaluate(&k)) return false;
		switch (r->nop) {
			case n_null: break;
			case n_not: k = !k; break;
		}
		switch (r->lop) {
			case l_invalid: break;
			case l_and: l = (l && k); break;
			case l_or: l = (l || k); break;
			case l_eq: l = (l == k); break;
			case l_ne: l = (l != k); break;
			case l_gt: l = (l > k); break;
			case l_ge: l = (l >= k); break;
			case l_lt: l = (l < k); break;
			case l_le: l = (l <= k); break;
		}
	}
	*i = l;
	return true;
}

void sym_bool::l_operate(l_op l, sym_bool_token *t)
{
	tokens->insert(new sym_bool_token_rec(n_null, l, t));
}

int sym_bool::nstrfy(char *buf, int n)
{
	int l = 0;
	int c = tokens->count();
	for (int i = 0; i < c; i++) {
		sym_bool_token_rec *r = (sym_bool_token_rec*)tokens->get(i);
		l += r->nstrfy(buf+l, n-l);
	}
	return l;
}

void sym_bool::set(sym_bool_token *t)
{
	clear();
	tokens->insert(new sym_bool_token_rec(n_null, l_invalid, t));
}

void sym_bool::n_operate(n_op n)
{
}

OBJECT_ID sym_bool::object_id()
{
	return ATOM_SYM_BOOL;
}

