// -*- C++ -*-
// --------------------------------------------------------------------
// The group object.
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef IPEGROUP_H
#define IPEGROUP_H

#include "ipeobj.h"

// --------------------------------------------------------------------

class IPE_EXPORT IpeGroup : public IpeFillable {
private:
  typedef std::vector<const IpeObject *> List;

public:
  typedef List::const_iterator const_iterator;

  explicit IpeGroup();
  explicit IpeGroup(IpeRepository *rep, const IpeXmlAttributes &attr);
  IpeGroup(const IpeGroup &rhs);
  virtual ~IpeGroup();
  IpeGroup &operator=(const IpeGroup &rhs);
  virtual IpeObject *Clone() const;

  virtual IpeGroup *AsGroup();

  virtual TPinned pinned() const;

  virtual void Accept(IpeVisitor &visitor) const;
  virtual void SaveAsXml(IpePainter &painter, IpeStream &stream,
			 IpeString layer) const;
  virtual void Draw(IpePainter &painter) const;
  virtual void AddToBBox(IpeRect &box, const IpeMatrix &m) const;
  virtual double Distance(const IpeVector &v, const IpeMatrix &m,
			  double bound) const;
  virtual void SnapVtx(const IpeVector &mouse, const IpeMatrix &m,
		       IpeVector &pos, double &bound) const;
  virtual void SnapBnd(const IpeVector &mouse, const IpeMatrix &m,
		       IpeVector &pos, double &bound) const;

  //! Return number of component objects.
  inline int size() const { return iImp->iObjects.size(); }
  //! Return iterator for first object.
  inline const_iterator begin() const { return iImp->iObjects.begin(); }
  //! Return iterator for end of sequence.
  inline const_iterator end() const { return iImp->iObjects.end(); }

  void push_back(const IpeObject *);

  void SaveComponentsAsXml(IpePainter &painter, IpeStream &stream) const;

  virtual void CheckStyle(const IpeStyleSheet *sheet,
			   IpeAttributeSeq &seq) const;

  //! Return font size of text objects in the group.
  inline IpeAttribute TextSize() const { return iTextSize; }
  //! Return size of mark objects in the group.
  inline IpeAttribute MarkSize() const { return iMarkSize; }
  //! Return shape of mark objects in the group.
  inline int MarkShape() const { return iMarkShape; }

  //! Does this group contain any text objects?
  inline bool HasTexts() const { return iImp->iHasTexts; }
  //! Does this group contain any mark objects?
  inline bool HasMarks() const { return iImp->iHasMarks; }

  void SetTextSize(IpeAttribute size);
  void SetMarkSize(IpeAttribute size);
  void SetMarkShape(int shape);

private:
  void SaveGroupAttributesAsXml(IpePainter &painter, IpeStream &stream) const;

private:
  IpeAttribute iTextSize;
  IpeAttribute iMarkSize;
  int iMarkShape;

  struct Imp {
    List iObjects;
    bool iHasTexts;
    bool iHasMarks;
    int iRefCount;
    TPinned iPinned;
  };

  Imp *iImp;
};

// --------------------------------------------------------------------
#endif
