// --------------------------------------------------------------------
// AppUi
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipedoc.h"
#include "ipepage.h"
#include "ipestyle.h"
#include "ipeiml.h"
#include "ipepath.h"
#include "ipeutils.h"

#include "ipemodel.h"
#include "ipeversion.h"
#include "ipeq.h"

#include "ipeoverlay.h"
#include "ipecreatetext.h"
#include "ipeeditpath.h"

// Windows is just such a horrible mess
#ifdef MessageBox
#undef MessageBox
#endif

#include "appui.h"
#include "dialogs.h"
#include "props.h"
#include "styles.h"
#include "widgets.h"

// in main.cpp
extern void StartManual(QString url);

// --------------------------------------------------------------------

const char * aboutText =
"<qt><h1>%1</h1>"
"<p>(c) 1993-2007 Otfried Cheong</p>"
"<p>The extensible drawing editor Ipe creates figures "
"in Postscript and PDF format, "
"using LaTeX to format the text in the figures.</p>"
"<p>Ipe relies on the following fine pieces of software:"
"<ul>"
"<li> <b>PdfLaTeX</b> (www.pdftex.org)"
"<li> Some code from <b>Xpdf</b> (www.foolabs.com/xpdf)"
"<li> The GUI toolkit <b>Qt %3</b> (www.trolltech.com)"
"<li> The font rendering library <b>freetype 2</b> (www.freetype.org)"
"<li> The compression library <b>zlib</b> (www.gzip.org/zlib)"
"</ul>"
"<p>%2 is released under the GNU Public License.</p>"
"<p>See <tt>http://tclab.kaist.ac.kr/ipe</tt> for details.</p>"
"</qt>";

extern QPixmap penguinIcon(int width);
extern QPixmap ipeIcon(const char* name);

// --------------------------------------------------------------------

#define SIZE 14

QPixmap ColorPixmap(IpeAttribute sym, const IpeStyleSheet *sheet)
{
  QPixmap pixmap(SIZE, SIZE);
  if (sym.IsVoid()) {
    pixmap.fill(Qt::white);
    QPainter p(&pixmap);
    p.drawRect(0, 0, SIZE-1, SIZE-1);
    p.drawLine(0, 0, SIZE-1, SIZE-1);
    p.drawLine(0, SIZE-1, SIZE-1,  0);
  } else {
    IpeColor col = sheet->Repository()->ToColor(sheet->Find(sym));
    QColor color(int(col.iRed * 255),
		 int(col.iGreen * 255),
		 int(col.iBlue * 255));
    pixmap.fill(color);
  }
  return pixmap;
}

// --------------------------------------------------------------------

//! All windows closed, terminate Ipe.
void AppUi::closeEvent(QCloseEvent* ce)
{
  IpePreferences *prefs = IpePreferences::Static();
  prefs->iToolbarState = saveState();
  prefs->Save();

  if (!iDoc->IsEdited()) {
    ce->accept();
    return;
  }

  switch (QMessageBox::information
	  (this, QLatin1String("Ipe"),
	   tr("The document has been changed since the last save."),
	   tr("Save now"), tr("Cancel"), tr("Leave Anyway"), 0, 1)) {
  case 0:
    if (Save())
      ce->accept();
    else
      ce->ignore();
    break;
  case 1:
  default: // just for sanity
    ce->ignore(); break;
  case 2:
    ce->accept(); break;
  }
}

bool AppUi::eventFilter(QObject *obj, QEvent *e)
{
  // Event filtering on the widgets in the main window is not currently used
  if (obj != this)
    return false;

#if 0
  if (e->type() != QEvent::KeyPress)
    return false;
  QKeyEvent *ev = (QKeyEvent *) e;
  int key = ev->key();
  ipeDebug("Key press event %x on UI element %x (%s) (%x, %x)",
	   uint(e), uint(obj),
	   obj->metaObject()->className(),
	   key, int(ev->modifiers()));
  // only a few keys are permitted
  if (('0' <= key && key <= '9') ||
      key == Qt::Key_Escape || key == Qt::Key_Enter ||
      key == Qt::Key_Backspace ||
      key == Qt::Key_Period || key == Qt::Key_Return ||
      key == Qt::Key_Tab || key == Qt::Key_Backtab ||
      key == Qt::Key_Left || key == Qt::Key_Right ||
      key == Qt::Key_Up || key == Qt::Key_Down)
    // dispatch normally
    return false;
  // Other keys need to be forwarded to application,
  // so that the shortcut is activated
  ipeDebug("Forwarding event");
  QApplication::sendEvent(this, ev);
  return true;
#endif

#if 0
  // other things I've tried
  if (e->type() == QEvent::KeyPress) {
    // move focus to canvas
    iCanvas->setFocus(Qt::OtherFocusReason);
    QKeyEvent a(QEvent::ShortcutOverride, key, ev->modifiers(), ev->text());
    QApplication::sendEvent(this, &a);
    return true;
  }
#endif

  // Main Window: look at shortcuts only
  IpeOverlay *ov = iCanvas->Overlay();

  if (e->type() != QEvent::ShortcutOverride)
    return false;

  QKeyEvent *ev = (QKeyEvent *) e;
  int key = ev->key();

  ipeDebug("Shortcut override event type %d (%x, %x) on main window",
	   e->type(), key, int(ev->modifiers()));

  if (!ov)
    return false;

  if (key == Qt::Key_Escape) {
    iCanvas->FinishOverlay();
    return true;
  }

  ov->KeyPress(ev);
  ipeDebug("KeyPress passed to overlay, accepted == %d", ev->isAccepted());
  return ev->isAccepted();
}

void AppUi::aboutToShowEditMenu()
{
#if 0
  bool objs = Page()->HasSelection();
  // bool istext = false;
  bool isgroup = false;
  IpePage::iterator it = Page()->PrimarySelection();
  if (it != Page()->end()) {
    // istext = (it->Object()->AsText() != 0);
    isgroup = (it->Object()->AsGroup() != 0);
  }
  // TODO: move this functionality into Ipelib
  bool pathObjs = true;
  bool singlePathObjs = true;
  int objCount = 0;
  for (IpePage::iterator it = Page()->begin(); it != Page()->end(); ++it) {
    if (it->Select()) {
      ++objCount;
      IpePath *p = it->Object()->AsPath();
      // must be a single open subpath
      if (!p) {
	pathObjs = false;
	singlePathObjs = false;
      } else if (p->NumSubPaths() > 1 || p->SubPath(0)->Closed())
	singlePathObjs = false;
    }
  }
  iCutAction->setEnabled(objs);
  iCopyAction->setEnabled(objs);
  iDeleteAction->setEnabled(objs);
  iGroupAction->setEnabled(objs);
  iFrontAction->setEnabled(objs);
  iBackAction->setEnabled(objs);
  iDuplicateAction->setEnabled(objs);

  iUngroupAction->setEnabled(isgroup);
  iComposeAction->setEnabled(objs && pathObjs && objCount > 1);
  iJoinAction->setEnabled(objs && singlePathObjs && objCount > 1);
#endif
}

void AppUi::aboutToShowPropertiesMenu()
{
  QList<QAction *> list = iMarkShapeMenu->actions();
  for (int i = 0; i < list.size(); ++i)
    list[i]->setChecked(i + 1 == iAttributes.iMarkShape);
  list = iTransformableMenu->actions();
  for (int i = 0; i < list.size(); ++i)
    list[i]->setChecked(bool(i) == iAttributes.iTransformable);
  list = iTextStyleMenu->actions();
  for (int i = 0; i < list.size(); ++i)
    list[i]->setChecked(iSyms[IpeAttribute::ETextStyle][i] ==
			iAttributes.iTextStyle);
  list = iLabelVerticalAlignmentMenu->actions();
  for (int i = 0; i < list.size(); ++i)
    list[i]->setChecked(i == iAttributes.iVerticalAlignment);
  list = iLabelHorizontalAlignmentMenu->actions();
  for (int i = 0; i < list.size(); ++i)
    list[i]->setChecked(i == iAttributes.iHorizontalAlignment);
  list = iPinnedMenu->actions();
  for (int i = 0; i < list.size(); ++i)
    list[i]->setChecked(i == iAttributes.iPinned);
  int join = iAttributes.iStrokeStyle.Join() ?
    iAttributes.iStrokeStyle.Join().Index() + 1 : 0;
  int cap = iAttributes.iStrokeStyle.Cap() ?
    iAttributes.iStrokeStyle.Cap().Index() + 1 : 0;
  int wind = iAttributes.iStrokeStyle.WindRule() ?
    iAttributes.iStrokeStyle.WindRule().Index() + 1 : 0;
  list = iLineJoinMenu->actions();
  for (int i = 0; i < list.size(); ++i)
    list[i]->setChecked(i == join);
  list = iLineCapMenu->actions();
  for (int i = 0; i < list.size(); ++i)
    list[i]->setChecked(i == cap);
  list = iWindRuleMenu->actions();
  for (int i = 0; i < list.size(); ++i)
    list[i]->setChecked(i == wind);
}

void AppUi::aboutToShowMouseUnitsMenu()
{
  QList<QAction *> list = iMouseUnitsMenu->actions();
  for (int i = 0; i < list.size(); ++i)
    list[i]->setChecked(i == iMouseIn);
}

// --------------------------------------------------------------------
// Public slots
// --------------------------------------------------------------------

//! This handles all Ipe commands that operate on an IpePage.
void AppUi::Cmd(int cmd)
{
  // no document loaded yet, ignore UI events
  if (!iDoc)
    return;

  ipeDebug("Command %d", cmd);
  // refuse to execute commands when an overlay is active
  if (iCanvas->Overlay())
    return;

  statusBar()->clearMessage();
  IpePage *page = Page();
  IpeLayer layer = page->Layer(iLayer);

  // check whether command requires a select object
  switch (cmd) {
  case ECmdGroup:
  case ECmdUngroup:
  case ECmdCut:
  case ECmdCopy:
  case ECmdFront:
  case ECmdBack:
  case ECmdForward:
  case ECmdBackward:
  case ECmdDuplicate:
  case ECmdDelete:
  case ECmdComposePaths:
  case ECmdJoinPaths:
  case ECmdDecomposePath:
    if (!page->HasSelection()) {
      statusBar()->showMessage(tr("No selected object"));
      return;
    }
  default:
    break;
  }

  IpeAutoPtr<IpePage> originalPage(new IpePage(*page));
  page->SetEdited(false);
  QString undo;
  bool doUndo = false;

  switch (cmd) {
  case ECmdSelectAll:
    page->SelectAll(iVno);
    undo = tr("select all");
    // doUndo = true;
    break;
  case ECmdFirstView:
    iViewNumber->setValue(1);
    break;
  case ECmdLastView:
    iViewNumber->setValue(page->CountViews());
    break;
  case ECmdNewView:
  case ECmdNewLayerNewView:
    {
      IpeView view;
      undo = tr("view creation");
      if (cmd == ECmdNewLayerNewView) {
	undo = tr("layer and view creation");
	iLayer = page->NewLayer(iLayer + 1);
	for (int i = 0; i < page->CountLayers(); ++i) {
	  if (page->View(iVno).HasLayer(page->Layer(i).iName))
	    view.AddLayer(page->Layer(i).Name());
	}
	view.AddLayer(page->Layer(iLayer).Name());
      }
      view.SetActive(page->Layer(iLayer).Name());
      page->AddView(view, ++iVno);
      ViewChanged();
      break;
    }
  case ECmdDeleteView:
    if (page->CountViews() > 1) {
      page->DeleteView(iVno);
      if (iVno > 0 && iVno >= page->CountViews())
	--iVno;
      undo = tr("view deletion");
      ViewChanged();
    }
    break;
  case ECmdGroup:
    page->Group(iLayer);
    undo = tr("grouping");
    break;
  case ECmdUngroup:
    if (!page->Ungroup(iLayer))
      return;
    undo = tr("ungrouping");
    break;
  case ECmdCut:
  case ECmdCopy:
    {
      IpeString data;
      IpeStringStream stream(data);
      page->Copy(stream, iDoc->StyleSheet());
      if (cmd == ECmdCut)
	page->Delete();
      else
	statusBar()->showMessage(tr("Copied"));
      QClipboard *cb = QApplication::clipboard();
      // cb->setData(new IpeDrag(data, this));
      // cb->setText(QIpe(data), QClipboard::Selection);
      cb->setText(QIpe(data), QClipboard::Clipboard);
      undo = tr("cut");
      break;
    }
  case ECmdPaste:
    {
      QClipboard *cb = QApplication::clipboard();
      QString data = cb->text(QClipboard::Clipboard);
      // ipeDebug("Paste: %s", data.latin1());
      if (data.length() == 0) {
	if (iPasteBitmapAction && !cb->image().isNull()) {
	  // paste image
	  iPasteBitmapAction->trigger();
	} else
	  statusBar()->showMessage(tr("Nothing to paste"));
      } else if (data.left(14) != QLatin1String("<ipeselection>")) {
	page->DeselectAll();
	IpeVector pos = 0.5 * iDoc->layout().iFrameSize;
	IpeText *obj = new IpeText(iAttributes, IpeQ(data), pos,
				   IpeText::ELabel);
	page->push_back(IpePgObject(IpePgObject::EPrimary, iLayer, obj));
	page->SetEdited(true);
      } else {
	QStringDataSource source(data);
	if (!page->Paste(iLayer, source, iDoc->Repository()))
	  ErrMsg(tr("Cannot parse Ipe objects in clipboard"));
      }
      undo = tr("paste");
      break;
    }
  case ECmdFront:
    page->Front();
    undo = tr("move to front");
    break;
  case ECmdBack:
    page->Back();
    undo = tr("move to back");
    break;
  case ECmdForward:
    page->forward();
    undo = tr("move forward");
    break;
  case ECmdBackward:
    page->backward();
    undo = tr("move backward");
    break;
  case ECmdBefore:
    page->movePrimaryBeforeSecondary();
    undo = tr("move just before");
    break;
  case ECmdBehind:
    page->movePrimaryBehindSecondary();
    undo = tr("move just behind");
    break;
  case ECmdDuplicate:
    page->Duplicate(iLayer);
    undo = tr("duplication");
    statusBar()->showMessage(tr("Object duplicated"));
    break;
  case ECmdDelete:
    page->Delete();
    undo = tr("deletion");
    break;
  case ECmdNewLayer:
    {
      iLayer = page->NewLayer(iLayer + 1);
      page->View(iVno).AddLayer(page->Layer(iLayer).Name());
      page->SetEdited(true);
      UpdateLayers();
    }
    undo = tr("adding layer");
    break;
  case ECmdLineWidth:
    page->SetLineWidth(iAttributes.iLineWidth);
    undo = tr("setting line width");
    break;
  case ECmdDashStyle:
    page->SetDashStyle(iAttributes.iDashStyle);
    undo = tr("setting dash style");
    break;
  case ECmdTextSize:
    page->SetTextSize(iAttributes.iTextSize);
    undo = tr("setting text size");
    break;
  case ECmdMarkShape:
    page->SetMarkShape(iAttributes.iMarkShape);
    undo = tr("setting mark shape");
    break;
  case ECmdTextStyle:
    page->setTextStyle(iAttributes.iTextStyle);
    undo = tr("setting text style");
    break;
  case ECmdVerticalAlign:
    page->setLabelVerticalAlignment
      (IpeText::TVerticalAlignment(iAttributes.iVerticalAlignment));
    undo = tr("setting vertical alignment");
    break;
  case ECmdHorizontalAlign:
    page->setLabelHorizontalAlignment
      (IpeText::THorizontalAlignment(iAttributes.iHorizontalAlignment));
    undo = tr("setting horizontal alignment");
    break;
  case ECmdTransformableText:
    page->setTransformable(iAttributes.iTransformable);
    undo = tr("setting transformability");
    break;
  case ECmdPinned:
    page->setPinned(IpeObject::TPinned(iAttributes.iPinned));
    undo = tr("setting pinned status");
    break;
  case ECmdLineJoin:
    page->setLineJoin(iAttributes.iStrokeStyle.Join());
    undo = tr("setting line join");
    break;
  case ECmdLineCap:
    page->setLineCap(iAttributes.iStrokeStyle.Cap());
    undo = tr("setting line cap");
    break;
  case ECmdWindRule:
    page->setWindRule(iAttributes.iStrokeStyle.WindRule());
    undo = tr("setting wind rule");
    break;
  case ECmdMarkSize:
    page->SetMarkSize(iAttributes.iMarkSize);
    undo = tr("setting mark size");
    break;
  case ECmdSetArrows:
    page->SetArrows(iAttributes.iForwardArrow,
		    iAttributes.iBackwardArrow,
		    iAttributes.iArrowSize);
    undo = tr("setting arrows");
    break;
  case ECmdArrowSize:
    page->SetArrowSize(iAttributes.iArrowSize);
    undo = tr("setting arrow size");
    break;
  case ECmdStroke:
    page->SetStroke(iAttributes.iStroke);
    undo = tr("setting stroke");
    break;
  case ECmdFill:
    page->SetFill(iAttributes.iFill);
    undo = tr("setting fill");
    break;
  case ECmdComposePaths:
    if (!page->ComposePaths(iLayer)) {
      statusBar()->showMessage(tr("Only path objects can be composed"));
      break;
    }
    undo = tr("path composition");
    break;
  case ECmdJoinPaths:
    if (!page->JoinPaths(iLayer)) {
      statusBar()->showMessage(tr("Incorrect objects in selection"));
      break;
    }
    undo = tr("joining paths");
    break;
  case ECmdDecomposePath:
    if (!page->DecomposePath(iLayer)) {
      statusBar()->showMessage(tr("Primary selection is not a path object"));
      break;
    }
    undo = tr("path decomposition");
    break;
  }
  bool isEdited = page->IsEdited();
  if (originalPage->IsEdited())
    page->SetEdited(true);
  if (doUndo || isEdited) {
    // page has been modified, save in Undo stack
    AddUndo(new IpeUndoPageEdit(iPno, originalPage.Take(), IpeQ(undo)));
  }
  iCanvas->Update();
  SetCaption();
}

void AppUi::AddUndo(IpeUndoItem *item)
{
  iUndo.Add(item);
  UpdateUndo();
}

// update what's shown on the Undo/Redo actions
void AppUi::UpdateUndo()
{
  if (iUndo.CanUndo()) {
    // iUndoAction->setEnabled(true);
    iUndoAction->setText(tr("Undo %1").arg(QIpe(iUndo.UndoText())));
  } else {
    // iUndoAction->setEnabled(false);
    iUndoAction->setText(tr("Undo"));
  }

  if (iUndo.CanRedo()) {
    // iRedoAction->setEnabled(true);
    iRedoAction->setText(tr("Redo %1").arg(QIpe(iUndo.RedoText())));
  } else {
    // iRedoAction->setEnabled(false);
    iRedoAction->setText(tr("Redo"));
  }
}

void AppUi::UndoCmd(int cmd)
{
  // no document loaded yet, ignore UI events
  if (!iDoc)
    return;
  statusBar()->clearMessage();
  int pno = -1;
  switch (cmd) {
  case ECmdUndo:
    if (iUndo.CanUndo())
      pno = iUndo.Undo(iDoc);
    else {
      statusBar()->showMessage(tr("Nothing to undo"));
      return;
    }
    break;
  case ECmdRedo:
    if (iUndo.CanRedo())
      pno = iUndo.Redo(iDoc);
    else {
      statusBar()->showMessage(tr("Nothing to redo"));
      return;
    }
    break;
  }
  if (pno >= 0)
    iPno = pno;
  if (iPno >= iDoc->pages())
    iPno = iDoc->pages() - 1;
  iDoc->SetEdited(true);
  PageChanged();
  UpdateUndo();
}

void AppUi::SnapCmd(int id)
{
  // no document loaded yet, ignore UI events
  if (!iDoc)
    return;

  statusBar()->clearMessage();

  // compute snapped mouse position WITHOUT angular snapping
  IpeVector pos = iCanvas->UnsnappedPos();
  iSnapData.SimpleSnap(pos, Page(), iSnapData.iSnapDistance / iCanvas->Zoom());

  switch (id) {
  case ECmdHere:
    iCanvas->SetPan(iCanvas->Pos());
    iCanvas->Update();
    return;
  case ECmdSetOriginSnap:
    iSnapData.iSnap |= IpeSnapData::ESnapAngle;
    iSnapAction[4]->setChecked(true);
  case ECmdSetOrigin:
    iSnapData.iWithAxes = true;
    iSnapData.iOrigin = pos;
    iCanvas->SetSnap(iSnapData);
    iCanvas->Update();
    return;
  case ECmdResetOrigin:
    iSnapData.iWithAxes = false;
    iSnapData.iSnap &= ~IpeSnapData::ESnapAngle;
    iSnapAction[4]->setChecked(false);
    iCanvas->SetSnap(iSnapData);
    iCanvas->Update();
    return;
  case ECmdSetDirectionSnap:
    iSnapData.iSnap |= IpeSnapData::ESnapAngle;
    iSnapAction[4]->setChecked(true);
  case ECmdSetDirection:
    iSnapData.iWithAxes = true;
    iSnapData.iDir = (pos - iSnapData.iOrigin).Angle();
    iCanvas->SetSnap(iSnapData);
    iCanvas->Update();
    return;
  case ECmdResetDirection:
    iSnapData.iDir = 0;
    iCanvas->SetSnap(iSnapData);
    iCanvas->Update();
    return;
  case ECmdSetLineSnap:
    iSnapData.iSnap |= IpeSnapData::ESnapAngle;
    iSnapAction[4]->setChecked(true);
  case ECmdSetLine:
    if (!iSnapData.SetEdge(pos, Page())) {
      statusBar()->showMessage(tr("Mouse position is not on an edge"));
    } else {
      iSnapData.iWithAxes = true;
      iCanvas->SetSnap(iSnapData);
      iCanvas->Update();
    }
    break;
  }
}

// --------------------------------------------------------------------

void AppUi::EditObject()
{
  IpePage::iterator it = Page()->PrimarySelection();
  if (it == Page()->end()) {
    statusBar()->showMessage(tr("No object selected"));
    return;
  }
  IpePgObject original(*it);
  if (it->Object()->AsText() &&
      IpeCreateText::Edit(it->Object()->AsText(), iDoc->StyleSheet())) {
    OvSvcAddUndoItem(it, Page(), original, tr("text object edit"));
    Page()->SetEdited(true);
    iCanvas->Update();
  } else if (it->Object()->AsPath()) {
    iCanvas->SetOverlay(new IpeEditPath(iCanvas, it->Object()->AsPath(),
					Page(), this, it));
  } else
    statusBar()->showMessage(tr("No editable object selected"));
}

void AppUi::changeTextWidth()
{
  IpePage::iterator it = Page()->PrimarySelection();
  if (it == Page()->end()) {
    statusBar()->showMessage(tr("No object selected"));
    return;
  }
  IpeText *obj = it->Object()->AsText();
  if (obj && obj->isMinipage())
    iCanvas->SetOverlay(new IpeTextWidthChanger(iCanvas, obj));
  else
    statusBar()->showMessage(tr("Primary selection is not a minipage object"));
}

// --------------------------------------------------------------------

void AppUi::NewWindow()
{
  AppUi *appui = new AppUi;
  appui->NewDoc();
  if (IpePreferences::Static()->iMaximize)
    appui->showMaximized();
  else
    appui->show();
  appui->FitPage();
  statusBar()->clearMessage();
}

// --------------------------------------------------------------------

void AppUi::NextView()
{
  if (iVno + 1 >= Page()->CountViews()) {
    if (iPno + 1 < iDoc->pages()) {
      iPageNumber->stepUp();
    }
  } else
    iViewNumber->stepUp();
}

void AppUi::PreviousView()
{
  if (iVno == 0) {
    if (iPno > 0) {
      iPageNumber->stepDown();
      Cmd(ECmdLastView);
    }
  } else
    iViewNumber->stepDown();
}

void AppUi::FirstPage()
{
  iPageNumber->setValue(1);
  statusBar()->clearMessage();
}

void AppUi::LastPage()
{
  iPageNumber->setValue(iDoc->pages());
  statusBar()->clearMessage();
}

static double AdjustPan(double cmin, double cmax,
			double omin, double omax,
			double pmin, double pmax)
{
  double dx = 0;

  // if objects stick out on both sides, there is nothing we can do
  if (omin <= cmin &&  omax >= cmax)
    return dx;

  if (omax > cmax && omin > cmin) {
    // we can see more objects if we shift canvas right
    dx = IpeMin(omin - cmin, omax - cmax);
  } else if (omin < cmin && omax < cmax) {
    // we can see more objects if we shift canvas left
    dx = -IpeMin(cmin - omin, cmax - omax);
  }

  // shift canvas
  cmin += dx;
  cmax += dx;

  // if canvas fully contained in media, done
  if (pmin <= cmin && pmax >= cmax)
    return dx;
  // if media contained in canvas, can't improve
  if (cmin < pmin && pmax < cmax)
    return dx;

  if (pmin > cmin) {
    // improvement possible by shifting canvas right
    if (omin > cmin)
      dx += IpeMin(omin - cmin, IpeMin(pmin - cmin, pmax - cmax));
  } else {
    // improvement possible by shifting canvas left
    if (omax < cmax)
      dx -= IpeMin(cmax - omax, IpeMin(cmax - pmax, cmin - pmin));
  }
  return dx;
}

//! Change resolution to 72 dpi and maximize interesting visible area.
/*! As suggested by Rene:

1) scale to the proper size, with the center of the canvas as the
   origin of the scaling.
2) If there is a horizontal and/or vertical translation that makes a
   larger part of the *bounding box* of the objects visible, then
   translate (and maximize the part of the bounding box that is
   visible).
3) If there is a horizontal and/or vertical translation that makes a
   larger part of the paper visible, then translate (and maximize the
   part of the paper that is visible), under the restriction that no
   part of the bounding box of the objects may be moved `out of sight'
   in this step. (Note that there may be objects outside the paper).
*/
void AppUi::NormalSize()
{
  iResolution->setValue(72000);

  IpeRect media = iDoc->layout().paper();

  IpeRect bbox;
  for (IpePage::const_iterator it = Page()->begin();
       it != Page()->end(); ++it) {
    bbox.AddRect(it->BBox());
  }

  // size of canvas in user coordinates
  IpeVector s = (1.0 / iCanvas->Zoom()) *
    IpeVector(iCanvas->width(), iCanvas->height());

  IpeRect canvas(iCanvas->Pan() - 0.5 * s, iCanvas->Pan() + 0.5 * s);

  IpeVector pan;
  pan.iX = AdjustPan(canvas.Min().iX, canvas.Max().iX,
		     bbox.Min().iX, bbox.Max().iX,
		     media.Min().iX, media.Max().iX);
  pan.iY = AdjustPan(canvas.Min().iY, canvas.Max().iY,
		     bbox.Min().iY, bbox.Max().iY,
		     media.Min().iY, media.Max().iY);
  iCanvas->SetPan(iCanvas->Pan() + pan);
  statusBar()->clearMessage();
}

void AppUi::FitBox(const IpeRect &box)
{
  if (box.IsEmpty())
    return;
  double xfactor = box.Width() > 0.0 ?
    (iCanvas->width() / box.Width()) : 20.0;
  double yfactor = box.Height() > 0.0 ?
    (iCanvas->height() / box.Height()) : 20.0;
  double zoom = (xfactor > yfactor) ? yfactor : xfactor;
  int resolution = int(zoom * 72000.0 + 0.5);
  if (resolution > iResolution->maximum())
    resolution = iResolution->maximum();
  else if (resolution < iResolution->minimum())
    resolution = iResolution->minimum();
  iCanvas->SetPan(0.5 * (box.Min() + box.Max()));
  iResolution->setValue(resolution);
  statusBar()->clearMessage();
}

//! Do standard zoom (after loading new document)
void AppUi::DefaultZoom()
{
  if (iDoc->pages() > 1)
    FitPage();
  else
    NormalSize();
}

//! Change resolution so that page is displayed fully.
void AppUi::FitPage()
{
  IpeRect box = iDoc->layout().paper();
  FitBox(box);
}

void AppUi::FitObjects()
{
  IpeRect bbox;
  for (IpePage::const_iterator it = Page()->begin();
       it != Page()->end(); ++it) {
    bbox.AddRect(it->BBox());
  }
  if (bbox.IsEmpty()) {
    bbox.AddPoint(IpeVector::Zero);
    bbox.AddPoint(iDoc->layout().iFrameSize);
  }
  FitBox(bbox);
}

void AppUi::FitSelection()
{
  IpeRect bbox;
  for (IpePage::const_iterator it = Page()->begin();
       it != Page()->end(); ++it) {
    if (it->Select() != IpePgObject::ENone)
      bbox.AddRect(it->BBox());
  }
  FitBox(bbox);
}

//! Slot to create a new page after the current one.
void AppUi::CreatePage()
{
  ++iPno;
  IpePage *page = iDoc->newPage(iSnapData.iGridSize);
  iDoc->addPage(iPno, page);
  PageChanged();
  iDoc->SetEdited(true);
  statusBar()->clearMessage();
  AddUndo(new IpeUndoPageIns(iPno, IpeQ(tr("page creation"))));
}

//! Slot to delete current page --- if it is empty.
void AppUi::DeletePage()
{
  if (iDoc->pages() == 1) {
    statusBar()->showMessage(tr("Cannot delete the only page."));
    return;
  }
  IpePage *page = Page();
  if (page->size() > 0) {
    statusBar()->showMessage(tr("Page is not empty, not deleted."));
    return;
  }
  // undo stack takes ownership of page
  AddUndo(new IpeUndoPageDel(iPno, iDoc->removePage(iPno),
			     IpeQ(tr("page deletion"))));
  if (iPno == iDoc->pages())
    --iPno;
  PageChanged();
  iDoc->SetEdited(true);
  statusBar()->clearMessage();
}

void AppUi::CutPage()
{
  CopyPage(true);
}

void AppUi::CopyPage(bool cut)
{
  if (cut && iDoc->pages() == 1) {
    statusBar()->showMessage(tr("Cannot cut the only page."));
    return;
  }
  IpeString data;
  IpeStringStream stream(data);
  Page()->CopyPage(stream, iDoc->StyleSheet());
  QClipboard *cb = QApplication::clipboard();
  // cb->setData(new IpeDrag(data, this));
  cb->setText(QIpe(data));

  if (cut) {
    IpePage *original = iDoc->removePage(iPno);
    if (iPno == iDoc->pages())
      --iPno;
    PageChanged();
    iDoc->SetEdited(true);
    // undo stack takes ownership of original page
    AddUndo(new IpeUndoPageDel(iPno, original, IpeQ(tr("cut page"))));
  } else
    statusBar()->showMessage(tr("Page copied"));
}

void AppUi::PastePage()
{
  QClipboard *cb = QApplication::clipboard();
  QString data = cb->text();
  // if (!IpeDrag::decode(cb->data(), data) ||
  if (data.left(9) != QLatin1String("<ipepage>")) {
    statusBar()->showMessage(tr("No Ipe page to paste"));
    return;
  }
  QStringDataSource source(data);
  IpeImlParser parser(source, iDoc->Repository());
  IpePage *page = parser.ParsePageSelection();
  if (!page) {
    ErrMsg(tr("Could not parse page on clipboard"));
    return;
  }
  ++iPno;
  iDoc->addPage(iPno, page);
  PageChanged();
  iDoc->SetEdited(true);
  AddUndo(new IpeUndoPageIns(iPno, IpeQ(tr("paste page"))));
  statusBar()->clearMessage();
}

//! Slot to edit document properties.
void AppUi::EditDocProps()
{
  IpeDocument::SProperties props = iDoc->Properties();
  DialogDocumentProperties *dialog =
    new DialogDocumentProperties(this, props, iDoc->StyleSheet());
  if (dialog->exec() == QDialog::Accepted) {
    if (props.iPreamble != iDoc->Properties().iPreamble)
      iNeedLatex = true;
    iDoc->SetProperties(props);
    // paper size may have changed
    PageChanged();
  }
  statusBar()->clearMessage();
}

void AppUi::SetMouseDisplayIn(int mode)
{
  iMouseIn = mode;
}

// --------------------------------------------------------------------

void AppUi::InsertTextBox()
{
  // need to use media and margins
  IpeRect r = Page()->TextBox(iDoc->StyleSheet());
  IpeCreateText::New(0, iCanvas, this, IpeCreateText::EMinipage, &r);
}

//! Slot to set mode (connected to object buttons and menu).
void AppUi::SetCreator(QAction *action)
{
  statusBar()->clearMessage();
  for (iMode = 0; iMode < IpeOverlayFactory::Num; iMode++) {
    if (action == iModeAction[iMode])
      return;
  }
}

// --------------------------------------------------------------------

//! Slot to run Latex on text objects.
/*! Returns true if document can now be saved to PDF.  Does not run
  Latex if \a force is \c false and no text objects have been
  modified.
 */
bool AppUi::RunLatex(bool force)
{
  statusBar()->showMessage(tr("Running Latex ..."));
  QStringList sd;
  sd << QDir::current().path();
  if (!iFileName.isNull()) {
    QFileInfo fi(iFileName);
    sd << fi.dir().path();
  }
  QString logFile;
  switch (IpeModel::RunLatex(iDoc, force ? true : iNeedLatex, sd, logFile)) {
  case IpeModel::ErrNoText:
    statusBar()->showMessage(tr("No text objects in document, "
			    "no need to run Pdflatex."));
    iNeedLatex = false;
    return true;
  case IpeModel::ErrNoDir:
    ErrMsg(tr("<qt>Directory '%1' does not exist and cannot be created.</qt>")
	   .arg(IpePreferences::Static()->iLatexDir));
    statusBar()->clearMessage();
    return false;
  case IpeModel::ErrWritingSource:
    ErrMsg(tr("<qt>Error writing Latex source</qt>"));
    statusBar()->clearMessage();
    return false;
  case IpeModel::ErrOldPdfLatex:
    ErrMsg(tr("<qt>Your installed version of Pdflatex is too old."
	      "Please install a more recent version (at least version 0.14f)."
	      "Until you do so, Ipe cannot convert text objects.</qt>"));
    statusBar()->clearMessage();
    return false;
  case IpeModel::ErrLatexOutput:
    ErrMsg(tr("<qt>Something is wrong with the PDF file generated by "
	      "Pdflatex.  "
#ifdef WIN32
	      "<hr/>You may also perform a <em>Refresh Filename Database</em> "
	      "in <u>MikTeX Options</u> and try again."
#else
	      "Please consult the stderr output to identify the problem."
#endif
	      "</qt>"));
    statusBar()->clearMessage();
    return false;
  case IpeModel::ErrLatex:
    {
      DialogLatexError *dialog = new DialogLatexError(0, logFile);
      dialog->exec();
    }
    statusBar()->clearMessage();
    return false;
  case IpeModel::ErrNone:
    statusBar()->showMessage(tr("Latex run completed successfully"));
    iCanvas->SetFontPool(iDoc->FontPool());
    iCanvas->Update();
    iNeedLatex = false;
    return true;
  case IpeModel::ErrAlreadyHaveForm:
    statusBar()->clearMessage();
    return true;
  default:
    assert(false);
    return false;
  };
}

// --------------------------------------------------------------------

//! Slot to edit Ipe preferences.
void AppUi::EditPrefs()
{
  statusBar()->clearMessage();
  IpePreferences *prefs = IpePreferences::Static();

  DialogPreferences *dialog =
    new DialogPreferences(this, *prefs, iSnapData.iGridSize);
  if (dialog->exec() == QDialog::Accepted)
    prefs->Save(); // will cause call to PreferencesChanged
}

void AppUi::PreferencesChanged()
{
  IpePreferences *prefs = IpePreferences::Static();
  iSnapData.iSnapDistance = prefs->iSnapDistance;
  iSnapData.iSelectDistance = prefs->iSelectDistance;
  iCanvas->SetSnap(iSnapData);
  IpeDialogCreateText::SetEditorFont(prefs->iFont);
  iCanvas->SetAntiAlias(prefs->iAntiAlias);
  iCanvas->Update();
  statusBar()->clearMessage();
}

void AppUi::EditBookmarks()
{
  statusBar()->clearMessage();
  IpePage *page = Page();
  QStringList sections;
  sections += QIpe(page->title());
  sections += QIpe(page->section(0));
  sections += QIpe(page->section(1));
  bool useTitle[2];
  useTitle[0] = page->sectionUsesTitle(0);
  useTitle[1] = page->sectionUsesTitle(1);
  DialogPage *dialog = new DialogPage(this, sections, useTitle);
  if (dialog->exec() == QDialog::Accepted) {
    // AddUndo!!
    page->setTitle(IpeQ(sections[0]));
    page->setSection(0, useTitle[0], IpeQ(sections[1]));
    page->setSection(1, useTitle[1], IpeQ(sections[2]));
    PageChanged();
  }
}

void AppUi::EditEffects()
{
  statusBar()->clearMessage();
  IpePage *page = Page();
  IpeView view = page->View(iVno);
  DialogEffects *dialog = new DialogEffects(this, view);
  if (dialog->exec() == QDialog::Accepted) {
    // AddUndo(new IpeUndoViews(iPno, page->Views(), IpeQ("view change")));
    page->SetView(iVno, view);
    PageChanged();
  }
}

//! Slot to show Ipe manual.
void AppUi::Manual()
{
  QDir dir(IpePreferences::Static()->iDocDir);
  QString url = dir.filePath(QLatin1String("manual.html"));
  StartManual(url);
  statusBar()->clearMessage();
}

void AppUi::StyleSheets()
{
  statusBar()->clearMessage();
  DialogStyles *dialog = new DialogStyles(this, iDoc);
  dialog->exec();
  // need to force update of canvas information (i.e. stylesheet)
  PageChanged();
  ShowStyleInUi();
  // AddUndo(new IpeUndoViews(iPno, page->Views(),
  // IpeQ("style sheet change")));
}

void AppUi::UpdateStyleSheets()
{
  statusBar()->clearMessage();
  if (iFileName.isNull()) {
    ErrMsg(tr("No filename known for document, cannot find style sheets"));
    return;
  }
  QFileInfo fi(iFileName);
  IpeString log = IpeModel::UpdateStyleSheets(iDoc, fi.dir());
  // need to force update of canvas information (i.e. stylesheet)
  PageChanged();
  ShowStyleInUi();
  QMessageBox::information(this, QLatin1String("Ipe"), QIpe(log),
			   tr("Dismiss"));
}

// --------------------------------------------------------------------

void AppUi::RunIpelet(int id)
{
  statusBar()->clearMessage();
  int ipelet = id >> 12;
  int function = id & 0xfff;

  IpePage *original = new IpePage(*Page());
  iIpeletNoUndo = false;
  iIpeletMaster[ipelet]->Run(function, Page(), this);
  Page()->SetEdited(true);
  if (iIpeletNoUndo) {
    delete original;
    iUndo.Clear();
    UpdateUndo();
  } else
    AddUndo(new IpeUndoPageEdit(iPno, original, IpeQ(tr("Ipelet result"))));
  iCanvas->Update();
  UpdateLayers();
  SetCaption();
}

// --------------------------------------------------------------------

//! Slot to display about window.
void AppUi::About()
{
  QMessageBox about(tr("About Ipe"),
		    tr(aboutText).arg(QLatin1String(IPE_VERSION)).
		    arg(QLatin1String(IPE_VERSION))
		    .arg(QLatin1String(QT_VERSION_STR)),
		    QMessageBox::NoIcon,
		    QMessageBox::Ok|QMessageBox::Default,
		    Qt::NoButton,
		    Qt::NoButton);
  about.setIconPixmap(penguinIcon(100));
  about.setButtonText(QMessageBox::Ok, tr("Dismiss"));
  about.exec();
  statusBar()->clearMessage();
}

void AppUi::AboutIpelets()
{
  QString s;
  s = tr("<qt><h1>About the available ipelets</h1><ul>");
  for (uint i = 0; i < iIpeletMaster.size(); ++i) {
    Ipelet *ipelet = iIpeletMaster[i];
    const char *about = ipelet->About();
    if (about) {
      s += QString(QLatin1String("<li>%1<br/>%2</li>\n"))
	.arg(QString::fromUtf8(ipelet->Label()))
	.arg(QString::fromUtf8(about));
    }
  }
  s += tr("</ul>Remaining ipelets are written by Otfried Cheong.</qt>");
  QMessageBox about(tr("About ipelets"), s,
		    QMessageBox::NoIcon,
		    QMessageBox::Ok|QMessageBox::Default,
		    Qt::NoButton,
		    Qt::NoButton);
  about.setIconPixmap(penguinIcon(100));
  about.setButtonText(QMessageBox::Ok, tr("Dismiss"));
  about.exec();
  statusBar()->clearMessage();
}

// --------------------------------------------------------------------
// Private slots
// --------------------------------------------------------------------

void AppUi::AbsoluteAttributes(bool abs)
{
  iAbsoluteAttributes = abs;
  SwitchAttributeUi();
  statusBar()->clearMessage();
  if (iAbsoluteAttributes) {
    // when switching to absolute, set absolute values from symbolic ones
    const IpeStyleSheet *sheet = iDoc->StyleSheet();
    iAttributes.iStroke = sheet->Find(iAttributes.iStroke);
    iAbsStrokeColor->setIcon(ColorPixmap(iAttributes.iStroke, sheet));
    iAttributes.iFill = sheet->Find(iAttributes.iFill);
    iAbsFillColor->setIcon(ColorPixmap(iAttributes.iFill, sheet));
    ConvertAbsolute(iAbsLineWidth, iAttributes.iLineWidth);
    ConvertAbsolute(iAbsMarkSize, iAttributes.iMarkSize);
    ConvertAbsolute(iAbsArrowSize, iAttributes.iArrowSize);
    IpeAttribute ts = iDoc->StyleSheet()->Find(iAttributes.iTextSize);
    if (ts.IsNumeric())
      iAbsTextSize->set(ts.Number().Internal());
    else
      iAbsTextSize->set(10000); // 10 pt
  } else {
    // when switching to symbolic, use previous symbolic settings
    // and update iAttributes
    iAttributes.iStroke =
      iSyms[IpeAttribute::EColor][iStrokeColor->currentIndex() + 1];
    iAttributes.iFill =
      iSyms[IpeAttribute::EColor][iFillColor->currentIndex()];
    iAttributes.iLineWidth =
      iSyms[IpeAttribute::ELineWidth][iLineWidth->currentIndex()];
    iAttributes.iTextSize =
      iSyms[IpeAttribute::ETextSize][iTextSize->currentIndex()];
    iAttributes.iMarkSize =
      iSyms[IpeAttribute::EMarkSize][iMarkSize->currentIndex()];
    iAttributes.iArrowSize =
      iSyms[IpeAttribute::EArrowSize][iArrowSize->currentIndex()];
  }
}

void AppUi::ToggleAbsoluteAttributes()
{
  iAbsoluteAttributesAction->toggle();
  AbsoluteAttributes(iAbsoluteAttributesAction->isChecked());
}

void AppUi::AbsoluteSnapping(bool abs)
{
  iAbsoluteSnapping = abs;
  SwitchSnapUi();
  statusBar()->clearMessage();
  if (iAbsoluteSnapping) {
    // when switching to absolute,
    // set absolute values from current setting
    iAbsGridSize->set(iSnapData.iGridSize);
    iAbsAngleSize->set(iSnapAngle.Internal());
  } else {
    // when switching to symbolic, use previous symbolic settings
    iSnapData.iGridSize = AbsValue(iSyms[IpeAttribute::EGridSize]
				   [iGridSize->currentIndex()]).ToInt();
    iSnapAngle = AbsValue(iSyms[IpeAttribute::EAngleSize]
			  [iAngleSize->currentIndex()]);
    iSnapData.iAngleSize = IpeAngle::Degrees(iSnapAngle.ToDouble());
    iCanvas->SetSnap(iSnapData);
    if (iSnapData.iWithAxes || iSnapData.iGridVisible)
      iCanvas->Update();
  }
}

void AppUi::PageChanged(int no)
{
  iPno = no - 1;
  PageChanged();
}

void AppUi::ViewChanged(int no)
{
  iVno = no - 1;
  ViewChanged();
}

void AppUi::ResolutionChanged(int resolution)
{
  // compute zoom factor from resolution: 72 dpi = value 72000 = 1.0
  double zoom = resolution / 72000.0;
  iCanvas->SetZoom(zoom);
  statusBar()->clearMessage();
}

void AppUi::LineWidthChanged(int id)
{
  iAttributes.iLineWidth = iSyms[IpeAttribute::ELineWidth][id];
  Cmd(ECmdLineWidth);
}

void AppUi::AbsLineWidthChanged(int val)
{
  iAttributes.iLineWidth =
    iDoc->Repository()->ToAttribute(IpeAttribute::ELineWidth,
				    IpeFixed::FromInternal(val));
  Cmd(ECmdLineWidth);
}

void AppUi::DashStyleChanged(int id)
{
  iAttributes.iDashStyle = iSyms[IpeAttribute::EDashStyle][id];
  Cmd(ECmdDashStyle);
}

void AppUi::TextSizeChanged(int id)
{
  iAttributes.iTextSize = iSyms[IpeAttribute::ETextSize][id];
  Cmd(ECmdTextSize);
}

void AppUi::AbsTextSizeChanged(int val)
{
  iAttributes.iTextSize =
    iDoc->Repository()->ToAttribute(IpeAttribute::ETextSize,
				    IpeFixed::FromInternal(val));
  Cmd(ECmdTextSize);
}

void AppUi::setMarkShape(int id)
{
  iAttributes.iMarkShape = id;
  Cmd(ECmdMarkShape);
}

void AppUi::setTransformable(int id)
{
  iAttributes.iTransformable = bool(id);
  Cmd(ECmdTransformableText);
}

void AppUi::setHorizontalAlignment(int id)
{
  iAttributes.iHorizontalAlignment = IpeText::THorizontalAlignment(id);
  Cmd(ECmdHorizontalAlign);
}

void AppUi::setVerticalAlignment(int id)
{
  iAttributes.iVerticalAlignment = IpeText::TVerticalAlignment(id);
  Cmd(ECmdVerticalAlign);
}

void AppUi::setTextStyle(int id)
{
  iAttributes.iTextStyle = iSyms[IpeAttribute::ETextStyle][id];
  Cmd(ECmdTextStyle);
}

void AppUi::setPinned(int id)
{
  iAttributes.iPinned = IpeObject::TPinned(id);
  Cmd(ECmdPinned);
}

void AppUi::setLineJoin(int id)
{
  iAttributes.iStrokeStyle.SetJoin(id - 1);
  Cmd(ECmdLineJoin);
}

void AppUi::setLineCap(int id)
{
  iAttributes.iStrokeStyle.SetCap(id - 1);
  Cmd(ECmdLineCap);
}

void AppUi::setWindRule(int id)
{
  iAttributes.iStrokeStyle.SetWindRule(id - 1);
  Cmd(ECmdWindRule);
}

void AppUi::MarkSizeChanged(int id)
{
  iAttributes.iMarkSize = iSyms[IpeAttribute::EMarkSize][id];
  Cmd(ECmdMarkSize);
}

void AppUi::AbsMarkSizeChanged(int val)
{
  iAttributes.iMarkSize =
    iDoc->Repository()->ToAttribute(IpeAttribute::EMarkSize,
				    IpeFixed::FromInternal(val));
  Cmd(ECmdMarkSize);
}

void AppUi::ArrowChanged(bool)
{
  iAttributes.iForwardArrow = iForwardArrow->isChecked();
  iAttributes.iBackwardArrow = iBackwardArrow->isChecked();
  Cmd(ECmdSetArrows);
}

void AppUi::ArrowSizeChanged(int id)
{
  iAttributes.iArrowSize = iSyms[IpeAttribute::EArrowSize][id];
  Cmd(ECmdArrowSize);
}

void AppUi::AbsArrowSizeChanged(int val)
{
  iAttributes.iArrowSize =
    iDoc->Repository()->ToAttribute(IpeAttribute::EArrowSize,
				    IpeFixed::FromInternal(val));
  Cmd(ECmdArrowSize);
}

void AppUi::GridSizeChanged(int id)
{
  iSnapData.iGridSize = AbsValue(iSyms[IpeAttribute::EGridSize][id]).ToInt();
  iCanvas->SetSnap(iSnapData);
  if (iSnapData.iGridVisible)
    iCanvas->Update();
  if (Page())
    Page()->SetGridSize(iSnapData.iGridSize);
  statusBar()->clearMessage();
}

void AppUi::AbsGridSizeChanged(int val)
{
  iSnapData.iGridSize = val;
  iCanvas->SetSnap(iSnapData);
  if (iSnapData.iGridVisible)
    iCanvas->Update();
  if (Page())
    Page()->SetGridSize(iSnapData.iGridSize);
  statusBar()->clearMessage();
}

void AppUi::AngleSizeChanged(int id)
{
  iSnapAngle = AbsValue(iSyms[IpeAttribute::EAngleSize][id]);
  iSnapData.iAngleSize = IpeAngle::Degrees(iSnapAngle.ToDouble());
  iCanvas->SetSnap(iSnapData);
  if (iSnapData.iWithAxes)
    iCanvas->Update();
  statusBar()->clearMessage();
}

void AppUi::AbsAngleSizeChanged(int val)
{
  iSnapAngle = IpeFixed::FromInternal(val);
  iSnapData.iAngleSize = IpeAngle::Degrees(iSnapAngle.ToDouble());
  iCanvas->SetSnap(iSnapData);
  if (iSnapData.iWithAxes)
    iCanvas->Update();
  statusBar()->clearMessage();
}

//! Snapping was changed.
void AppUi::SnapChanged(bool)
{
  iSnapData.iSnap = 0;
  for (int i = 0; i < 6; ++i) {
    if (iSnapAction[i]->isChecked())
      iSnapData.iSnap |= (1 << i);
  }
  if (iSnapData.iSnap & IpeSnapData::ESnapAngle)
    iSnapData.iWithAxes = true;
  iCanvas->SetSnap(iSnapData);
  statusBar()->clearMessage();
}

void AppUi::GridVisible(bool vis)
{
  iSnapData.iGridVisible = vis;
  iCanvas->SetSnap(iSnapData);
  iCanvas->Update();
  statusBar()->clearMessage();
}

void AppUi::CoordinatesVisible(bool vis)
{
  if (vis)
    iMouse->show();
  else
    iMouse->hide();
}

//! Slot for the absolute stroke color button.
void AppUi::SetStrokeColor()
{
  QColor stroke = QColorDialog::getColor(Qt::red, 0);
  if (stroke.isValid()) {
    SetColorFromQColor(iAttributes.iStroke, stroke, iAbsStrokeColor);
    Cmd(ECmdStroke);
  }
}

//! Slot for the absolute fill color button.
void AppUi::SetFillColor()
{
  QColor fill = QColorDialog::getColor(Qt::red, 0);
  if (fill.isValid()) {
    SetColorFromQColor(iAttributes.iFill, fill, iAbsFillColor);
    Cmd(ECmdFill);
  }
}

//! Slot for the stroke color comboBox.
void AppUi::SetStrokeColorName(int item)
{
  iAttributes.iStroke = iSyms[IpeAttribute::EColor][item + 1]; // + 1 for void
  Cmd(ECmdStroke);
}

//! Slot for the fill color comboBox.
void AppUi::SetFillColorName(int item)
{
  iAttributes.iFill = iSyms[IpeAttribute::EColor][item];
  Cmd(ECmdFill);
}

// ----------------------------------------------------------
// Private utility functions
// ----------------------------------------------------------

void AppUi::PageChanged()
{
  iVno = 0;
  iLayer = 0;
  iPageNumber->set(iPno + 1, iDoc->pages());
  SetGridSizeFromPage();
  ViewChanged();

  iBookmarks->clear();
  for (int i = 0; i < iDoc->pages(); ++i) {
    IpeString s = iDoc->page(i)->section(0);
    IpeString ss = iDoc->page(i)->section(1);
    if (!s.empty()) {
      QListWidgetItem *item = new QListWidgetItem(QIpe(s));
      iBookmarks->addItem(item);
    }
    if (!ss.empty()) {
      QListWidgetItem *item =
	new QListWidgetItem(QLatin1String("    ") + QIpe(ss));
      item->setTextColor(Qt::blue);
      iBookmarks->addItem(item);
    }
  }
  if (!iBookmarkTools->isVisible() && iBookmarks->count() > 0
      && !iBookmarksShown) {
    iBookmarkTools->show();
    iBookmarksShown = true;
  }
}

void AppUi::BookmarkSelected(QListWidgetItem *item)
{
  int index = iBookmarks->row(item);
  int count = 0;
  for (int i = 0; i < iDoc->pages(); ++i) {
    IpeString s = iDoc->page(i)->section(0);
    IpeString ss = iDoc->page(i)->section(1);
    if (!s.empty()) {
      if (count == index)
	iPageNumber->setValue(i + 1);
      ++count;
    }
    if (!ss.empty()) {
      if (count == index)
	iPageNumber->setValue(i + 1);
      ++count;
    }
  }
}

void AppUi::ViewChanged()
{
  IpePage *page = Page();
  iViewNumber->set(iVno + 1, page->CountViews());
  int i = page->FindLayer(page->View(iVno).Active());
  if (i >= 0)
    iLayer = i;
  page->DeselectNotInView(iVno);
  page->EnsurePrimarySelection();
  iCanvas->SetPage(page, iVno, iDoc->StyleSheet(), PageColor());
  UpdateLayers();
  SetCaption();
  if (iDoc->totalViews() > 1)
    iPageTools->show();
  else
    iPageTools->hide();
  statusBar()->clearMessage();
}

void AppUi::UpdateLayers()
{
  IpePage *page = Page();
  // make sure it doesn't try to change layers
  iLayerList->disconnect(this);
  iLayerList->clear();
  for (int i = 0; i < page->CountLayers(); ++i) {
    const IpeLayer &l = page->Layer(i);
    QString s;
    if (l.IsDimmed())
      s += QLatin1String("D");
    if (l.IsLocked())
      s += QLatin1String("L");
    if (!l.IsSnapping())
      s += QLatin1String("S");
    if (!s.isEmpty())
      s += QLatin1String(" ");
    s += QIpe(l.iName);
    QListWidgetItem *item = new QListWidgetItem(s);
    if (i == iLayer)
      item->setBackgroundColor(Qt::yellow);
    item->setFlags(Qt::ItemIsUserCheckable|Qt::ItemIsEnabled);
    item->setCheckState(page->View(iVno).HasLayer(l.iName) ?
			Qt::Checked : Qt::Unchecked);
    iLayerList->addItem(item);
  }
  // reconnect signals
  connect(iLayerList,
	  SIGNAL(rightButtonPressed(QListWidgetItem *, const QPoint &)),
	  this, SLOT(LayerRightPress(QListWidgetItem *, const QPoint &)));
  connect(iLayerList, SIGNAL(itemChanged(QListWidgetItem *)),
	  this, SLOT(LayerChanged(QListWidgetItem *)));
  // insert layerlist in layer menu
  iSelectInLayerMenu->clear();
  iMoveToLayerMenu->clear();
  for (int i = 0; i < page->CountLayers(); ++i) {
    QAction *a1 = new IpeAction(i, QIpe(page->Layer(i).Name()), this);
    connect(a1, SIGNAL(triggered(int)), SLOT(selectInLayer(int)));
    iSelectInLayerMenu->addAction(a1);
    QAction *a2 = new IpeAction(i, QIpe(page->Layer(i).Name()), this);
    connect(a2, SIGNAL(triggered(int)), SLOT(moveToLayer(int)));
    iMoveToLayerMenu->addAction(a2);
  }
}

void AppUi::selectInLayer(int id)
{
  statusBar()->clearMessage();
  if (id < 0)
    id = iLayer;
  Page()->SelectAllInLayer(id);
  iCanvas->Update();
}

void AppUi::moveToLayer(int id)
{
  if (!Page()->HasSelection()) {
    statusBar()->showMessage(tr("No selected object"));
    return;
  }
  statusBar()->clearMessage();
  if (id < 0)
    id = iLayer;
  IpePage *originalPage = new IpePage(*Page());
  Page()->MoveToLayer(id);
  AddUndo(new IpeUndoPageEdit(iPno, originalPage, IpeQ(tr("move to layer"))));
  iCanvas->Update();
}

void AppUi::LayerRightPress(QListWidgetItem *item, const QPoint &pos)
{
  QAction *activateAct = new QAction(tr("Set as active"), this);
  QAction *deleteAct = new QAction(tr("Delete"), this);
  QAction *dimAct = new QAction(tr("Dim"), this);
  QAction *undimAct = new QAction(tr("Undim"), this);
  QAction *lockAct = new QAction(tr("Lock layer"), this);
  QAction *unlockAct = new QAction(tr("Unlock layer"), this);
  QAction *snapAct = new QAction(tr("Enable snapping"), this);
  QAction *unsnapAct = new QAction(tr("Disable snapping"), this);
  QAction *renameAct = new QAction(tr("Rename"), this);

  IpePage *page = Page();

  int index = iLayerList->row(item);
  IpeLayer &layer = page->Layer(index);

  QMenu *menu = new QMenu;
  if (index != iLayer) {
    if (!layer.IsLocked())
      menu->addAction(activateAct);
    menu->addAction(deleteAct);
  }
  if (layer.IsDimmed())
    menu->addAction(undimAct);
  else
    menu->addAction(dimAct);
  if (layer.IsLocked())
    menu->addAction(unlockAct);
  else if (index != iLayer && !page->IsLayerActiveInView(index))
    menu->addAction(lockAct);
  if (layer.IsSnapping())
    menu->addAction(unsnapAct);
  else
    menu->addAction(snapAct);
  menu->addAction(renameAct);

  IpeAutoPtr<IpePage> originalPage = new IpePage(*page);

  QAction *a = menu->exec(pos);

  if (a == activateAct) {
    iLayer = index;
    if (page->CountViews() > 0) {
      page->View(iVno).SetActive(page->Layer(iLayer).Name());
      page->SetEdited(true);
      AddUndo(new IpeUndoPageEdit(iPno, originalPage.Take(),
				  IpeQ(tr("active layer change"))));
    }
    // return immediately, don't use layer undo below
    UpdateLayers();
    return;
  } else if (a == deleteAct) {
    if (page->CountLayers() > 1 && index != iLayer) {
      IpePage::const_iterator it = page->begin();
      while (it != page->end() && it->Layer() != index)
	++it;
      if (it != page->end()) {
	statusBar()->showMessage(tr("Layer is not empty, not deleted"));
	return;
      }
      if (iLayer > index)
	--iLayer;
      page->DeleteLayer(index);
      AddUndo(new IpeUndoPageEdit(iPno, originalPage.Take(),
				  IpeQ(tr("deleting layer"))));
      UpdateLayers();
    }
    // return immediately, don't use layer undo below
    return;
  } else if (a == dimAct) {
    layer.SetDimmed(true);
  } else if (a == undimAct) {
    layer.SetDimmed(false);
  } else if (a == lockAct) {
    layer.SetLocked(true);
    page->DeselectLayer(index); // no object in this layer must be selected
  } else if (a == unlockAct) {
    layer.SetLocked(false);
  } else if (a == snapAct) {
    layer.SetSnapping(true);
  } else if (a == unsnapAct) {
    layer.SetSnapping(false);
  } else if (a == renameAct) {
    bool ok = false;
    QString text = QInputDialog::getText
      (this, tr("Rename layer"), tr("Enter new layer name"),
       QLineEdit::Normal, QIpe(layer.Name()), &ok);
    if (!ok || text.isEmpty())
      return;
    text.replace(QLatin1Char(' '), QLatin1String("_"));
    if (page->FindLayer(IpeQ(text)) >= 0) {
      statusBar()->showMessage(tr("Layer '%1' already exists").arg(text));
      return;
    }
    page->renameLayer(layer.Name(), IpeQ(text));
  }

  page->SetEdited(true);
  AddUndo(new IpeUndoPageEdit(iPno, originalPage.Take(),
			      IpeQ(tr("layer modification"))));
  UpdateLayers();
  iCanvas->Update();
  SetCaption();
}

void AppUi::LayerChanged(QListWidgetItem *)
{
  IpePage *page = Page();
  IpePage *original = new IpePage(*page);
  IpeView &view = page->View(iVno);
  view.ClearLayers();
  for (int i = 0; i < page->CountLayers(); ++i) {
    if (iLayerList->item(i)->checkState() == Qt::Checked)
      view.AddLayer(page->Layer(i).Name());
  }
  view.SetActive(page->Layer(iLayer).Name());
  AddUndo(new IpeUndoPageEdit(iPno, original, IpeQ(tr("view modification"))));
  page->DeselectNotInView(iVno);
  page->EnsurePrimarySelection();
  page->SetEdited(true);
  iCanvas->Update();
  statusBar()->showMessage(tr("View modified"));
  SetCaption();
}

IpeFixed AppUi::AbsValue(IpeAttribute attr)
{
  IpeAttribute abs = iDoc->StyleSheet()->Find(attr);
  return iDoc->Repository()->ToScalar(abs);
}

void AppUi::ConvertAbsolute(DecimalSpinBox *spin, IpeAttribute &attr)
{
  IpeAttribute abs = iDoc->StyleSheet()->Find(attr);
  IpeFixed value = iDoc->Repository()->ToScalar(abs);
  spin->set(value.Internal());
  attr = abs;
}

//! Switch to displaying absolute or symbolic values
void AppUi::SwitchAttributeUi()
{
  bool a = (iAbsoluteAttributes ? 1 : 0);
  iStrokeColorStack->setCurrentIndex(a);
  iFillColorStack->setCurrentIndex(a);
  iLineWidthStack->setCurrentIndex(a);
  iArrowSizeStack->setCurrentIndex(a);
  iTextSizeStack->setCurrentIndex(a);
  iMarkSizeStack->setCurrentIndex(a);
}

//! Switch to displaying absolute or symbolic snap values
void AppUi::SwitchSnapUi()
{
  bool a = (iAbsoluteSnapping ? 1 : 0);
  iGridSizeStack->setCurrentIndex(a);
  iAngleSizeStack->setCurrentIndex(a);
}

void AppUi::ResetCombo(IpeKind kind,
		       QComboBox *combo,
		       IpeAttribute &attr,
		       QPixmap *pixmap)
{
  combo->clear();
  for (IpeAttributeSeq::const_iterator it = iSyms[kind].begin();
       it != iSyms[kind].end(); ++it) {
    if (pixmap)
      combo->addItem(*pixmap, QIpe(iDoc->Repository()->String(*it)));
    else
      combo->addItem(QIpe(iDoc->Repository()->String(*it)));
  }
  combo->setCurrentIndex(0);
  attr = iSyms[kind][0];
}

void AppUi::ShowStyleInUi()
{
  iDoc->StyleSheet()->AllNames(iSyms);

  iFillColor->clear();
  iStrokeColor->clear();
  for (IpeAttributeSeq::const_iterator
	 it = iSyms[IpeAttribute::EColor].begin();
       it != iSyms[IpeAttribute::EColor].end(); ++it) {
    if (*it != IpeAttribute::Void())
      InsertColor(iStrokeColor, *it);
    InsertColor(iFillColor, *it);
  }
  iStrokeColor->setCurrentIndex(0);
  iAttributes.iStroke = iSyms[IpeAttribute::EColor][1];
  iAbsStrokeColor->setIcon(ColorPixmap(iAttributes.iStroke,
					 iDoc->StyleSheet()));

  iFillColor->setCurrentIndex(0);
  iAttributes.iFill = iSyms[IpeAttribute::EColor][0];
  iAbsFillColor->setIcon(ColorPixmap(iAttributes.iFill, iDoc->StyleSheet()));

  iDashStyle->clear();
  for (IpeAttributeSeq::const_iterator
	 it = iSyms[IpeAttribute::EDashStyle].begin();
       it != iSyms[IpeAttribute::EDashStyle].end(); ++it) {
    iDashStyle->addItem(QIpe(iDoc->Repository()->String(*it)));
  }
  iDashStyle->setCurrentIndex(0);
  iAttributes.iDashStyle = iSyms[IpeAttribute::EDashStyle][0];

  QPixmap lwIcon = ipeIcon("lineWidth");
  QPixmap arrowIcon = ipeIcon("arrow");
  QPixmap abcIcon = ipeIcon("paragraph");
  QPixmap markIcon = ipeIcon("marks");

  ResetCombo(IpeAttribute::ELineWidth, iLineWidth,
	     iAttributes.iLineWidth, &lwIcon);
  ResetCombo(IpeAttribute::ETextSize, iTextSize,
	     iAttributes.iTextSize, &abcIcon);
  ResetCombo(IpeAttribute::EArrowSize, iArrowSize,
	     iAttributes.iArrowSize, &arrowIcon);
  ResetCombo(IpeAttribute::EMarkSize, iMarkSize,
	     iAttributes.iMarkSize, &markIcon);

  iTextStyleMenu->clear();
  int count = 0;
  for (IpeAttributeSeq::const_iterator it =
	 iSyms[IpeAttribute::ETextStyle].begin();
       it != iSyms[IpeAttribute::ETextStyle].end(); ++it) {
    QAction *a = new IpeAction(count++, QIpe(iDoc->Repository()->String(*it)),
			       this);
    a->setCheckable(true);
    connect(a, SIGNAL(triggered(int)), SLOT(setTextStyle(int)));
    iTextStyleMenu->addAction(a);
  }
  iAttributes.iTextStyle = iSyms[IpeAttribute::ETextStyle][0];

  IpeAttribute grid;
  ResetCombo(IpeAttribute::EGridSize, iGridSize, grid);
  iSnapData.iGridSize = AbsValue(grid).ToInt();

  IpeAttribute angle;
  ResetCombo(IpeAttribute::EAngleSize, iAngleSize, angle);
  iSnapAngle = AbsValue(angle);
  iSnapData.iAngleSize = IpeAngle::Degrees(iSnapAngle.ToDouble());
  iCanvas->SetSnap(iSnapData);
}

void AppUi::InsertColor(QComboBox *combo, IpeAttribute sym)
{
  combo->addItem(ColorPixmap(sym, iDoc->StyleSheet()),
		 QIpe(iDoc->Repository()->String(sym)));
}

// Set gridsize box from page information
void AppUi::SetGridSizeFromPage()
{
  int gs = Page()->GridSize();
  if (gs > 0) {
    // find correct symbolic gridsize
    int sym = -1;
    for (uint i = 0; i < iSyms[IpeAttribute::EGridSize].size(); ++i) {
      if (AbsValue(iSyms[IpeAttribute::EGridSize][i]).ToInt() == gs) {
	sym = i;
	break;
      }
    }
    // change absolute snapping mode
    if (iAbsoluteSnapping && sym >= 0 || !iAbsoluteSnapping && sym < 0) {
      iAbsoluteSnapAction->setChecked(!iAbsoluteSnapping);
    }
    if (iAbsoluteSnapping) {
      iAbsGridSize->set(gs);
    } else
      iGridSize->setCurrentIndex(sym);
    iSnapData.iGridSize = gs;
    iCanvas->SetSnap(iSnapData);
    if (iSnapData.iGridVisible)
      iCanvas->Update();
  }
}

/*! Set \c colorToSet from \c qColor, and update the color in the \c
  button.
*/
void AppUi::SetColorFromQColor(IpeAttribute &colorToSet, QColor &qColor,
			       QPushButton *button)
{
  QPixmap pixmap(14, 14);
  pixmap.fill(qColor);
  button->setIcon(pixmap);
  int r, g, b;
  qColor.getRgb(&r, &g, &b);
  colorToSet = iDoc->Repository()->
    ToAttribute(IpeColor(r / 255.0, g / 255.0, b / 255.0));
}

// --------------------------------------------------------------------

void AppUi::ErrMsg(QString str)
{
  QMessageBox::warning(this, QLatin1String("Ipe"),
		       QLatin1String("<qt>") + str + QLatin1String("</qt>"),
		       tr("Dismiss"));
}

QKeySequence AppUi::Key(IpeString function) const
{
  // ipeDebug("Key: %s", function);
  std::map<IpeString, IpeString>::const_iterator it =
    keyAssignment.find(function);
  if (it != keyAssignment.end())
    return QKeySequence(QIpe(it->second));
  return QKeySequence();
}

// --------------------------------------------------------------------

std::map<IpeString, IpeString> AppUi::keyAssignment;

class KeyParser : public IpeXmlParser {
public:
  explicit KeyParser(IpeDataSource &source,
		     std::map<IpeString, IpeString> &map) :
    IpeXmlParser(source), iMap(map) { /* nothing */ }
  bool ParseKeys();
  std::map<IpeString, IpeString> &iMap;
};

// Parse key assignment.
bool KeyParser::ParseKeys()
{
  IpeXmlAttributes att;
  IpeString tag = ParseToTag();

  if (tag == "?xml") {
    if (!ParseAttributes(att, true))
      return false;
    tag = ParseToTag();
  }

  if (tag != "ipekeys")
    return false;
  if (!ParseAttributes(att))
    return false;
  tag = ParseToTag();

  while (tag == "key") {
    if (!ParseAttributes(att))
      return false;
    IpeString f = att["function"];
    IpeString k = att["key"];
    if (k == "none") {
      iMap[f] = k;
    } else if (k != "") {
      iMap[f] = k;
    }
    tag = ParseToTag();
  }

  if (tag != "/ipekeys")
    return false;
  return true;
}


// --------------------------------------------------------------------

void AppUi::ReadKeyAssignment()
{
  IpePreferences *prefs = IpePreferences::Static();
  QFile file1(prefs->iStdKeysFileName);
  if (file1.open(QIODevice::ReadOnly)) {
    QDataSource source(&file1);
    KeyParser parser(source, keyAssignment);
    parser.ParseKeys();
    file1.close();
  } else
    qDebug("Could not open key assignment file '%s'",
	   prefs->iStdKeysFileName.toLatin1().constData());

  QFile file2(IpePreferences::Static()->iKeysFileName);
  if (file2.open(QIODevice::ReadOnly)) {
    ipeDebug("Reading personal key assignment from file '%s'",
	     prefs->iKeysFileName.toLatin1().constData());
    QDataSource source(&file2);
    KeyParser parser(source, keyAssignment);
    parser.ParseKeys();
    file2.close();
  }
}

IpeAttribute AppUi::PageColor()
{
  if (IpePreferences::Static()->iWhitePaper)
    return IpeAttribute::White();
  return iDoc->Repository()->ToAttribute(IpeColor(1.0, 1.0, 0.5));
}

// --------------------------------------------------------------------
// IpeCanvasServices interface
// --------------------------------------------------------------------

IpeBuffer AppUi::StandardFont(IpeString fontName)
{
  return IpePreferences::Static()->StandardFont(fontName);
}

void AppUi::CvSvcRequestOverlay(QMouseEvent *ev)
{
  IpeOverlayFactory f(ev, iCanvas, Page(), iVno, this);
  // remember current mouse position
  iMouseBase = iCanvas->Pos();
  // need to select right Overlay
  if (ev->button() == Qt::RightButton) {
    if (IpePreferences::Static()->iRightMouseSelects &&
	!(ev->modifiers() & Qt::ControlModifier)) {
      f.CreateOverlay(IpeOverlayFactory::ESelecting);
    } else {
      double d = IpePreferences::Static()->iSelectDistance / iCanvas->Zoom();
      if (Page()->UpdateCloseSelection(iCanvas->Pos(), d, true, iVno)) {
	AttributePopup *pop =
	  new AttributePopup(Page()->PrimarySelection(), Page(),
			     iCanvas, iLayer,
			     iDoc->StyleSheet(), iSyms, this);
	pop->Exec();
      }
      SetCaption();
    }
  } else if (ev->button() == Qt::MidButton) {
    if ((ev->modifiers() & Qt::ControlModifier) &&
	(ev->modifiers() & Qt::ShiftModifier))
      f.CreateOverlay(IpeOverlayFactory::EMoving);
    else if (ev->modifiers() & Qt::ControlModifier)
      f.CreateOverlay(IpeOverlayFactory::ERotating);
    else if (ev->modifiers() & (Qt::AltModifier|Qt::ShiftModifier))
      f.CreateOverlay(IpeOverlayFactory::EPanning);
    else
      f.CreateOverlay(IpeOverlayFactory::EMoving);
  } else {
    // Left button
    if (ev->modifiers() & (Qt::AltModifier|Qt::ControlModifier))
      f.CreateOverlay(IpeOverlayFactory::EStretching);
    else
      f.CreateOverlay(iMode);
  }
}

static void enableActions(QMenu *menu, bool mode)
{
  menu->setEnabled(!mode);
  QListIterator<QAction *> it(menu->actions());
  while (it.hasNext())
    it.next()->setEnabled(!mode);
}

void AppUi::CvSvcSetDrawingMode(bool mode)
{
  enableActions(iFileMenu, mode);
  enableActions(iEditMenu, mode);
  enableActions(iModeMenu, mode);
  enableActions(iLayerMenu, mode);
  enableActions(iViewMenu, mode);
  enableActions(iPageMenu, mode);
  enableActions(iIpeletMenu, mode);

  iPageTools->setEnabled(!mode);
  iFileTools->setEnabled(!mode);
  iModeActionGroup->setEnabled(!mode);
}

void AppUi::CvSvcWheelZoom(int delta)
{
  if (delta > 0)
    iResolution->stepUp();
  else
    iResolution->stepDown();
}

inline void Adjust(IpeScalar &x, int mode)
{
  if (IpeAbs(x) < 1e-12)
    x = 0.0;
  switch (mode) {
  case 1: // mm
    x = (x / 72.0) * 25.4;
    break;
  case 2: // in
    x /= 72;
    break;
  default:
    break;
  }
}

static const char * const mouseUnits[] = { "", " mm", " in" };

void AppUi::CvSvcMousePosition(const IpeVector &pos)
{
  if (iMouse->isVisible()) {
    IpeVector v = pos;
    if (iSnapData.iWithAxes) {
      v = v - iSnapData.iOrigin;
      v = IpeLinear(-iSnapData.iDir) * v;
    }
    Adjust(v.iX, iMouseIn);
    Adjust(v.iY, iMouseIn);
    const char *units = mouseUnits[iMouseIn];
    QString s;
    s.sprintf("%g%s,%g%s", v.iX, units, v.iY, units);
    if (!iFileTools->isEnabled()) {
      IpeVector u = pos - iMouseBase;
      if (iSnapData.iWithAxes)
	u = IpeLinear(-iSnapData.iDir) * u;
      Adjust(u.iX, iMouseIn);
      Adjust(u.iY, iMouseIn);
      QString r;
      r.sprintf(" (%+g%s,%+g%s)", u.iX, units, u.iY, units);
      s += r;
    }
    iMouse->setText(s);
  }
}

void AppUi::CvSvcClearMessage()
{
  statusBar()->clearMessage();
}

// --------------------------------------------------------------------
// IpeOverlayServices interface
// --------------------------------------------------------------------

void AppUi::OvSvcAddObject(IpeObject *obj)
{
  IpePage *page = Page();
  page->DeselectAll();
  bool sel = page->View(iVno).HasLayer(page->Layer(iLayer).Name());
  page->push_back(IpePgObject((sel ? IpePgObject::EPrimary :
			       IpePgObject::ENone),
			      iLayer, obj));
  page->SetEdited(true);
  AddUndo(new IpeUndoObjInsertion(iPno, IpeQ(tr("object insertion"))));
  SetCaption();
  if (!sel)
    ErrMsg(tr("Active layer is not visible, "
	      "so your new object will be invisible."));
}

const IpeAllAttributes &AppUi::OvSvcAttributes()
{
  return iAttributes;
}

const IpeStyleSheet *AppUi::OvSvcStyleSheet()
{
  return iDoc->StyleSheet();
}

void AppUi::OvSvcAddUndoItem(IpePage *page, QString s)
{
  AddUndo(new IpeUndoPageEdit(iPno, page, IpeQ(s)));
}

void AppUi::OvSvcAddUndoItem(IpePage::iterator it, IpePage *page,
			     const IpePgObject &original, QString s)
{
  AddUndo(new IpeUndoObjectEdit(iPno, it, page, original, IpeQ(s)));
}

void AppUi::OvSvcMessage(QString s)
{
  statusBar()->showMessage(s);
}

// --------------------------------------------------------------------
// IpeletHelper interface
// --------------------------------------------------------------------

void AppUi::Message(const char *msg)
{
  statusBar()->showMessage(QString::fromUtf8(msg));
}

int AppUi::MessageBox(const char *text, const char *button1,
		      const char *button2, const char *button3)
{
  return (QMessageBox::information(this, tr("Ipelet"),
				   QLatin1String("<qt>") +
				   QString::fromUtf8(text) +
				   QLatin1String("</qt>"),
				   QString::fromUtf8(button1),
				   QString::fromUtf8(button2),
				   QString::fromUtf8(button3)));
}

bool AppUi::GetString(const char *prompt, IpeString &str)
{
  bool ok = false;
  QString text = QInputDialog::getText(this, tr("Ipelet"),
				       QString::fromUtf8(prompt),
				       QLineEdit::Normal,
				       QIpe(str), &ok);
  if (ok)
    str = IpeQ(text);
  return ok;
}

const IpeStyleSheet *AppUi::StyleSheet()
{
  return iDoc->StyleSheet();
}

const IpeDocument *AppUi::Document()
{
  return iDoc;
}

IpeDocument *AppUi::EditDocument()
{
  iDoc->SetEdited(true);
  iIpeletNoUndo = true;
  return iDoc;
}

int AppUi::CurrentPage() const
{
  return iPno;
}

int AppUi::CurrentView() const
{
  return iVno;
}

int AppUi::CurrentLayer() const
{
  return iLayer;
}

const IpeAllAttributes &AppUi::Attributes() const
{
  return iAttributes;
}

const IpeSnapData &AppUi::SnapData() const
{
  return iSnapData;
}

// --------------------------------------------------------------------
