// --------------------------------------------------------------------
// Ipelet maintaining and calling
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "master.h"
#include "ipeprefs.h"
#include "ipeq.h"

#ifdef WIN32
#include <windows.h>
#else
#include <dlfcn.h>
#endif

#include <qdir.h>

typedef Ipelet *(*PNewIpeletFn)();

// --------------------------------------------------------------------

/*! \class IpeletMaster
  \brief Maintains the set of Ipelets.
*/

//! Load all Ipelets.
IpeletMaster::IpeletMaster()
{
  QStringList dirs = IpePreferences::Static()->iIpeletPath;
  for (int i = 0; i < dirs.count(); ++i) {
    QDir dir(dirs[i]);
    dir.setFilter(QDir::Files);
    QStringList s;
    s << QLatin1String("*.dll") << QLatin1String("*.so*")
      << QLatin1String("*.dylib");
    dir.setNameFilters(s);
    // dir.setNameFilters(QStringList(QLatin1String(("*.dll;*.so*;*.dylib"))));
    ipeDebug("Ipelet directory '%s'", dir.path().toLatin1().data());
    // if directory does not exist, simply skip it
    if (!dir.isReadable())
      continue;
    QStringList dlls = dir.entryList();
    for (int j = 0; j < dlls.count(); ++j) {
      QString fn = dir.filePath(dlls[j]);
      QFileInfo finfo(fn);
      if (finfo.isSymLink())
	continue;
      ipeDebug("Loading dll '%s'", fn.toLatin1().data());
      PNewIpeletFn pIpelet = 0;
#ifdef WIN32
      HMODULE hMod =
	LoadLibraryA(QFile::encodeName(QDir::convertSeparators(fn)));
      if (hMod) {
	pIpelet = (PNewIpeletFn) GetProcAddress(hMod, "NewIpelet");
	if (!pIpelet)
	  pIpelet = (PNewIpeletFn) GetProcAddress(hMod, "_NewIpelet");
      }
#else
      void *handle = dlopen(QFile::encodeName(fn), RTLD_NOW);
      if (handle) {
	pIpelet = (PNewIpeletFn) dlsym(handle, "NewIpelet");
	if (!pIpelet)
	  pIpelet = (PNewIpeletFn) dlsym(handle, "_NewIpelet");
      }
#endif
      if (pIpelet) {
	Ipelet *ipelet = pIpelet();
	assert(ipelet != 0);
	ipeDebug("Success, ipelet '%s'", ipelet->Label());
	push_back(ipelet);
      } else
#ifdef WIN32
	ipeDebug("Error loading DLL");
#else
      qDebug("Error loading Ipelet '%s': %s", fn.toLatin1().data(), dlerror());
#endif
    }
  }
}

//! Destroys all Ipelets.
IpeletMaster::~IpeletMaster()
{
  for (iterator it = begin(); it != end(); ++it)
    delete (*it);
}

// --------------------------------------------------------------------
