// --------------------------------------------------------------------
// Overlay for creating IpeText objects
// --------------------------------------------------------------------
/*

    This file is part of the extensible drawing editor Ipe.
    Copyright (C) 1993-2007  Otfried Cheong

    Ipe is free software; you can redistribute it and/or modify it
    under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    As a special exception, you have permission to link Ipe with the
    CGAL library and distribute executables, as long as you follow the
    requirements of the Gnu General Public License in regard to all of
    the software in the executable aside from CGAL.

    Ipe is distributed in the hope that it will be useful, but WITHOUT
    ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
    or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
    License for more details.

    You should have received a copy of the GNU General Public License
    along with Ipe; if not, you can find it at
    "http://www.gnu.org/copyleft/gpl.html", or write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#include "ipecreatetext.h"
#include "ipecanvas.h"

#include <QEvent>
#include <QPainter>
#include <QString>
#include <QLabel>
#include <QMouseEvent>
#include <QPaintEvent>
#include <QTextDocumentFragment>
#include <QTextCharFormat>
#include <QTextCursor>

inline IpeString IpeQ(const QString &str)
{
  return IpeString(str.toUtf8());
}

inline QString QIpe(const IpeString &str)
{
  return QString::fromUtf8(str.CString());
}

// --------------------------------------------------------------------

/*! \class IpeDialogCreateText
  \brief Dialog to edit a text paragraph
*/

//! The argument \c data is only updated if the dialog is accepted.
/*! This dialog will include the combobox for the text style. */
IpeDialogCreateText::IpeDialogCreateText(QWidget* /*parent*/, QString caption,
					 QString label, QString &data,
					 const IpeStyleSheet *sheet,
					 IpeAttribute &style,
					 IpeAttribute &size)
  : QDialog(), iData(data), iCurrentStyle(&style), iCurrentSize(&size)
{
  init(caption, label);
  sheet->AllNames(IpeAttribute::ETextStyle, iStyles);
  // make sure "default" is entry number 0
  for (IpeAttributeSeq::iterator it = iStyles.begin();
       it != iStyles.end(); ++it) {
    IpeString s = sheet->Repository()->String(*it);
    if (s == "default") {
      IpeAttribute t = *it;
      *it = iStyles[0];
      iStyles[0] = t;
      break;
    }
  }
  for (IpeAttributeSeq::const_iterator it = iStyles.begin();
       it != iStyles.end(); ++it) {
    IpeString s = sheet->Repository()->String(*it);
    iStyle->addItem(QIpe(s));
    if (*it == *iCurrentStyle)
      iStyle->setCurrentIndex(iStyle->count() - 1);
  }
  sheet->AllNames(IpeAttribute::ETextSize, iSizes);
  for (IpeAttributeSeq::const_iterator it = iSizes.begin();
       it != iSizes.end(); ++it) {
    IpeString s = sheet->Repository()->String(*it);
    iSize->addItem(QIpe(s));
    if (*it == *iCurrentSize)
      iSize->setCurrentIndex(iSize->count() - 1);
  }
}

//! The argument \c data is only updated if the dialog is accepted.
/*! This dialog will not include the text style selection or the text size. */
IpeDialogCreateText::IpeDialogCreateText(QWidget* /*parent*/, QString caption,
					 QString label, QString &data)
  : QDialog(), iData(data), iCurrentStyle(0), iCurrentSize(0)
{
  init(caption, label);
  iStyle->hide();
  iSize->hide();
}

void IpeDialogCreateText::init(const QString &caption, const QString &label)
{
  setupUi(this);
  setWindowTitle(caption);
  connect(buttonOk, SIGNAL(clicked()), this, SLOT(UpdateData()));
  iLabel->setText(label);
  QTextDocument *doc = new QTextDocument(iEditor);
  QTextCharFormat format;
  // format.setBackground(Qt::yellow);
  // format.setFont(editorFont);
  QTextCursor cursor(doc);
  cursor.setBlockCharFormat(format);
  cursor.insertFragment(QTextDocumentFragment::fromPlainText(iData));
  iEditor->setDocument(doc);
  iEditor->setTextCursor(cursor);
  iEditor->setFocus();
}

void IpeDialogCreateText::UpdateData()
{
  iData = iEditor->toPlainText();
  if (iCurrentStyle) {
    if (iStyle->currentIndex() == 0)
      *iCurrentStyle = IpeAttribute(); // null style
    else
      *iCurrentStyle = iStyles[iStyle->currentIndex()];
  }
  if (iCurrentSize)
    *iCurrentSize = iSizes[iSize->currentIndex()];
}

void IpeDialogCreateText::SetEditorFont(const QString &font)
{
  /*
  if (!font.isEmpty()) {
    editorFont.fromString(font);
  } else
    editorFont = QApplication::font();
  */
}

// QFont IpeDialogCreateText::editorFont;

// --------------------------------------------------------------------

/*! \class CreateText
  \brief Overlay for creating IpeText objects.
*/

//! Constructor starts selection.
IpeCreateText::IpeCreateText(QMouseEvent *, IpeCanvas *canvas,
			     IpeOverlayServices *services)
  : IpeOverlay(canvas), iServices(services)
{
  iV[0] = iCanvas->Pos();
  iV[1] = iV[0];
}

void IpeCreateText::New(QMouseEvent *, IpeCanvas *canvas,
			IpeOverlayServices *services, TMode mode,
			IpeRect *rect)
{
  IpeVector pos = canvas->Pos();
  QString text;
  IpeAttribute currentStyle = services->OvSvcAttributes().iTextStyle;
  IpeAttribute currentSize = services->OvSvcAttributes().iTextSize;
  IpeDialogCreateText *dialog = 0;
  if (rect && mode == EMinipage) {
    dialog = new
      IpeDialogCreateText(canvas, QObject::tr("Create text object"),
			  QObject::tr("Enter text (LaTeX source)"), text,
			  services->OvSvcStyleSheet(),
			  currentStyle, currentSize);
  } else {
    dialog = new
      IpeDialogCreateText(canvas, QObject::tr("Create text object"),
			  (mode == EMath ?
			   QObject::tr("Enter formula (LaTeX source)") :
			   QObject::tr("Enter text (LaTeX source)")), text);
  }

  if (dialog->exec() == QDialog::Accepted) {
    text = text.trimmed();
    if (!text.isEmpty()) {
      IpeString data = IpeQ(text);
      if (mode == EMath)
	data = IpeString("$") + data + "$";
      IpeText *obj;
      if (rect && mode == EMinipage) {
	obj = new IpeText(services->OvSvcAttributes(),
			  data, rect->TopLeft(),
			  IpeText::EMinipage, rect->Width());
	obj->SetStyle(currentStyle);
	obj->SetSize(currentSize);
	obj->SetTransformable(false);
	obj->setPinned(IpeObject::EHorizontalPin);
      } else
	obj = new IpeText(services->OvSvcAttributes(),
			  data, pos, IpeText::ELabel);
      services->OvSvcAddObject(obj);
      canvas->Update();
    }
  }
}

bool IpeCreateText::Edit(IpeText *obj, const IpeStyleSheet *sheet)
{
  QString text;
  text = QIpe(obj->Text());
  IpeDialogCreateText *dialog = 0;
  IpeAttribute style = obj->Style();
  IpeAttribute size = obj->Size();
  if (obj->isMinipage()) {
    dialog = new IpeDialogCreateText(0, QObject::tr("Edit text object"),
				     QObject::tr("Edit LaTeX source"), text,
				     sheet, style, size);
  } else
    dialog = new IpeDialogCreateText(0, QObject::tr("Edit text object"),
				     QObject::tr("Edit LaTeX source"), text);
  if (dialog->exec() == QDialog::Accepted) {
    text = text.trimmed();
    if (!text.isEmpty()) {
      obj->SetText(IpeQ(text));
      obj->SetStyle(style);
      obj->SetSize(size);
      return true;
    }
  }
  return false;
}

void IpeCreateText::Draw(QPaintEvent *, QPainter *qPainter) const
{
  qPainter->setPen(CreateColor());
  IpeOverlayPainter painter(iCanvas->StyleSheet(), qPainter);
  painter.Transform(iCanvas->CanvasTfm());
  IpeVector d(0, -20.0);
  painter.NewPath();
  painter.MoveTo(iV[0]);
  painter.LineTo(iV[1]);
  painter.LineTo(iV[1] + d);
  painter.LineTo(iV[0] + d);
  painter.ClosePath();
  painter.DrawPath();
}

void IpeCreateText::MouseMove(QMouseEvent *)
{
  iV[1].iX = iCanvas->Pos().iX;
  iCanvas->UpdateOverlay();
}

void IpeCreateText::MousePress(QMouseEvent *)
{
  IpeScalar wid = iCanvas->Pos().iX - iV[0].iX;
  if (wid < 0) {
    iV[0].iX += wid;
    wid = -wid;
  }
  QString text;
  IpeAttribute style;
  IpeAttribute size;
  IpeDialogCreateText *dialog =
    new IpeDialogCreateText(iCanvas, QObject::tr("Create text object"),
			    QObject::tr("Enter LaTeX source"), text,
			    iServices->OvSvcStyleSheet(), style, size);
  if (dialog->exec() == QDialog::Accepted) {
    text = text.trimmed();
    if (!text.isEmpty()) {
      IpeText *obj = new IpeText(iServices->OvSvcAttributes(),
				 IpeQ(text), iV[0], IpeText::EMinipage, wid);
      obj->SetStyle(style);
      obj->SetSize(size);
      iServices->OvSvcAddObject(obj);
    }
  }
  iCanvas->FinishOverlay();
}

// --------------------------------------------------------------------

IpeTextWidthChanger::IpeTextWidthChanger(IpeCanvas *canvas, IpeText *text)
  : IpeOverlay(canvas), iObj(text)
{
  iV[0] = iObj->Matrix() * iObj->Position();
  iV[1] = iV[0];
  iV[1].iX = iV[0].iX + iObj->Width();
  iDragging = false;
}

void IpeTextWidthChanger::Draw(QPaintEvent *, QPainter *qPainter) const
{
  qPainter->setPen(Qt::magenta);
  IpeOverlayPainter painter(iCanvas->StyleSheet(), qPainter);
  painter.Transform(iCanvas->CanvasTfm());
  IpeVector d(0, -20.0);
  painter.NewPath();
  painter.MoveTo(iV[0]);
  painter.LineTo(iV[1]);
  painter.LineTo(iV[1] + d);
  painter.LineTo(iV[0] + d);
  painter.ClosePath();
  painter.DrawPath();
}

void IpeTextWidthChanger::MouseRelease(QMouseEvent *)
{
  IpeScalar wid = IpeAbs(iCanvas->Pos().iX - iV[0].iX);
  iObj->SetWidth(wid);
  iCanvas->FinishOverlay();
}

void IpeTextWidthChanger::MousePress(QMouseEvent *)
{
  iMouseDown = iCanvas->Pos();
  iDragging = true;
  iCanvas->UpdateOverlay();
}

void IpeTextWidthChanger::MouseMove(QMouseEvent *)
{
  if (iDragging) {
    IpeScalar d = iCanvas->Pos().iX - iMouseDown.iX;
    iV[1].iX = iV[0].iX + iObj->Width() + d;
    iCanvas->UpdateOverlay();
  }
}

// --------------------------------------------------------------------
