/*
 *   Jackbeat - JACK sequencer
 *    
 *   Copyright (c) 2004-2005 Olivier Guilyardi <olivier@samalyse.com>
 *    
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#ifndef JACKBEAT_SEQUENCE_H
#define JACKBEAT_SEQUENCE_H

#include <jack/jack.h>
#include "sample.h"

#define SEQUENCE_SINC 1
#define SEQUENCE_LINEAR 2

typedef struct sequence_t sequence_t;

typedef enum sequence_track_type_t 
{
  EMPTY,
  SAMPLE,
  NESTED
} sequence_track_type_t;

/* Sequence object construction and destruction */
sequence_t *  sequence_new (char *name, int *error);
void          sequence_destroy (sequence_t *sequence);

/* Global settings and informations */
char *          sequence_get_name (sequence_t *sequence);
void            sequence_set_auto_connect (sequence_t *sequence, char status);
char            sequence_is_auto_connecting (sequence_t *sequence);
jack_nframes_t  sequence_get_framerate (sequence_t *sequence);
void            sequence_set_resampler_type (sequence_t * sequence, int type);
int             sequence_get_resampler_type (sequence_t *sequence);

/* Transport control */
void    sequence_set_transport (sequence_t *sequence, char respond, char query);
void    sequence_start (sequence_t *sequence);
void    sequence_stop (sequence_t *sequence);
void    sequence_rewind (sequence_t *sequence);
int     sequence_is_playing (sequence_t *sequence);

/* Dimensions, BPM and looping */
void    sequence_resize (sequence_t *sequence, int tracks_num, int beats_num, 
                         int measure_len, int duplicate_beats);
int     sequence_get_tracks_num (sequence_t *sequence);
int     sequence_get_beats_num (sequence_t *sequence);
int     sequence_get_measure_len (sequence_t *sequence);
float   sequence_get_bpm (sequence_t *sequence);
void    sequence_set_bpm (sequence_t *sequence, float bpm);
void    sequence_set_looping (sequence_t *sequence);
void    sequence_unset_looping (sequence_t *sequence);
int     sequence_is_looping(sequence_t *sequence);


/* Track informations */
char *  sequence_get_track_name (sequence_t *sequence, int track);
void    sequence_set_track_name (sequence_t *sequence, int track, char *name);
int     sequence_track_name_exists (sequence_t *sequence, char *name);
sequence_track_type_t sequence_get_track_type (sequence_t *sequence, int track);

/* Samples */
sample_t *  sequence_get_sample (sequence_t *sequence, int track);
void        sequence_set_sample (sequence_t *sequence, int track, sample_t *sample);
int         sequence_get_sample_usage (sequence_t *sequence, sample_t *sample);

/* Track volume, pitch and smoothing */
void    sequence_set_pitch (sequence_t *sequence, int track, double pitch);
double  sequence_get_pitch (sequence_t *sequence, int track);
void    sequence_set_volume (sequence_t *sequence, int track, double volume);
double  sequence_get_volume (sequence_t *sequence, int track);
void    sequence_mute_track (sequence_t *sequence, char status, int track);
char    sequence_track_is_muted (sequence_t *sequence, int track);
void    sequence_set_smoothing (sequence_t *sequence, int track, int status);
int     sequence_get_smoothing (sequence_t *sequence, int track);

/* Track-masking */
void    sequence_enable_mask (sequence_t *sequence, int track);
void    sequence_disable_mask (sequence_t *sequence, int track);
int     sequence_is_enabled_mask (sequence_t *sequence, int track);
void    sequence_set_mask_beat (sequence_t * sequence, int track, int beat, char status);
char    sequence_get_mask_beat (sequence_t * sequence, int track, int beat);
int     sequence_get_active_mask_beat (sequence_t *sequence, int track);

/* Beat operations */
void    sequence_set_beat (sequence_t *sequence,int track, int beat, char status);
char    sequence_get_beat (sequence_t *sequence, int track, int beat);
int     sequence_get_active_beat (sequence_t *sequence, int track);

#endif
