#ifndef K3DSDK_FRAMES_H
#define K3DSDK_FRAMES_H

// K-3D
// Copyright (c) 1995-2004, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\brief Declares the k3d::frames class, useful for working with ranges of numbered files
		\author Timothy M. Shead (tshead@k-3d.com)
*/

#include <boost/filesystem/path.hpp>
#include <string>

namespace k3d
{

/////////////////////////////////////////////////////////////////////////////
// frames

/// A utility class for working with ranges of numbered files
class frames
{
public:
	/// Initializes a range of frames based on a single filesystem sample
	frames(const boost::filesystem::path& BaseFileName);
	/// Initializes a range of frames from a user-specified basefilename
	frames(const boost::filesystem::path& BaseFileName, const long FirstFrame, const long LastFrame);
	/// Initialize a range of frames from scratch
	frames(const std::string PreDigits, const std::string PostDigits, const long FirstFrame, const long LastFrame, const long FrameDigits);

	/// Generates a filename for the given frame number
	bool frame(const long FrameNumber, boost::filesystem::path& Result) const;

	/// Returns the largest possible frame number, based on the number of digits
	long max_frame() const;

	/// Returns the text that precedes frame numbers
	const std::string& pre_digits() const;
	/// Returns the text that follows frame numbers
	const std::string& post_digits() const;
	/// Returns the number of digits in the frame number
	const unsigned long frame_digits() const;
	/// Returns the first frame number
	long first_frame() const;
	/// Returns the last frame number
	long last_frame() const;

protected:
	/// Stores the text that precedes frame numbers
	std::string m_pre_digits;
	/// Stores the text that follows frame numbers
	std::string m_post_digits;
	/// Stores the first frame number
	long m_first_frame;
	/// Stores the last frame number
	long m_last_frame;
	/// Stores the number of digits in the frame number
	unsigned long m_frame_digits;
};

} // namespace k3d

#endif // K3DSDK_FRAMES_H


