// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Anders Dahnielson (anders@dahnielson.com)
*/

#include "simple_bitmap_modifier.h"

#include <k3dsdk/i18n.h>
#include <k3dsdk/module.h>

namespace libk3dbitmap
{

/////////////////////////////////////////////////////////////////////////////
// bitmap_gamma

class bitmap_gamma :
	public simple_bitmap_modifier
{
	typedef simple_bitmap_modifier base;

public:
	bitmap_gamma(k3d::idocument& Document) :
		base(Document),
		m_value(init_owner(*this) + init_name("value") + init_label(_("Gamma value")) + init_description(_("Apply gamma value to each pixel")) + init_value(1.0))
	{
		m_value.changed_signal().connect(make_update_bitmap_slot());
	}

	void on_update_bitmap(const k3d::bitmap& Input, k3d::bitmap& Output)
	{
		const double value = m_value.value();
		const double gamma_value = value ? 1.0 / value : 1.0;

		k3d::bitmap::const_iterator input = Input.begin();
		const k3d::bitmap::const_iterator end = Input.end();

		k3d::bitmap::iterator output = Output.begin();

		for(; input != end; ++input, ++output)
		{
			output->red = std::pow(static_cast<double>(input->red), gamma_value);
			output->green = std::pow(static_cast<double>(input->green), gamma_value);
			output->blue = std::pow(static_cast<double>(input->blue), gamma_value);
			output->alpha = input->alpha;
		}
	}

	k3d::iplugin_factory& factory()
	{
		return get_factory();
	}

	static k3d::iplugin_factory& get_factory()
	{
		static k3d::plugin_factory<
			k3d::document_plugin<bitmap_gamma>,
			k3d::interface_list<k3d::ibitmap_source,
			k3d::interface_list<k3d::ibitmap_sink> > > factory(
				k3d::uuid(0xa2ff08c9, 0x96e54063, 0x907ad507, 0xec36dd1c),
				"BitmapGamma",
				_("Apply gamma value to each pixel"),
				"Bitmap",
				k3d::iplugin_factory::STABLE);

		return factory;
	}

private:
	k3d_data(double, immutable_name, change_signal, with_undo, local_storage, no_constraint, writable_property, with_serialization) m_value;
};

/////////////////////////////////////////////////////////////////////////////
// bitmap_gamma_factory

k3d::iplugin_factory& bitmap_gamma_factory()
{
	return bitmap_gamma::get_factory();
}

} // namespace libk3dbitmap

