/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : Specialized list to display a list of photos
 *
 * Copyright 2005 by M. Asselstine
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */
#include "photolistview.h"

#include <qdom.h>
#include <qpen.h>
#include <qrect.h>
#include <qmime.h>
#include <qfile.h>
#include <qevent.h>
#include <qstring.h>
#include <qwidget.h>
#include <qheader.h>
#include <klocale.h>
#include <qwmatrix.h>
#include <qpainter.h>
#include <qpalette.h>
#include <qpopupmenu.h>
#include <qscrollbar.h>
#include <kiconloader.h>
#include <kmainwindow.h>
#include <kapplication.h>
#include <qapplication.h>
#include <kxmlguiclient.h>
#include <kstandarddirs.h>
#include <kio/previewjob.h>
#include <kxmlguifactory.h>

#include "photo.h"
#include "kflickrpart.h"
#include "photolistviewtooltip.h"


PhotoListView::PhotoListView(QWidget *parent, const char *name)
    : KListView(parent, name)
{
    // disable sorting
    setSorting(-1);

    // enable draging for sorting purposes
    setDragEnabled(TRUE);

    // replace default tooltip with our own
    QToolTip::remove(this);
    new PhotoListViewToolTip(this);

    // hide header
    header()->hide();

    // Fixup look and sizing
    setVScrollBarMode(QScrollView::AlwaysOn);
    header()->setStretchEnabled(true, -1);

    // load the border pixmap
    KStandardDirs *dirs = KApplication::kApplication()->dirs();
    m_border = QPixmap( dirs->findResource( "data", "kflickr/border.png" ) );

    // Signal - Slot connections
    connect(&m_previewTimer, SIGNAL(timeout()), SLOT(startPreviewJob()));

    connect(this, SIGNAL(itemAdded(QListViewItem *)), SLOT(addPreviewRequest(QListViewItem *)));
    connect(this, SIGNAL(rightButtonClicked(QListViewItem *, const QPoint &, int)), SLOT(showRMBMenu(QListViewItem *, const QPoint &, int)));
}

PhotoListView::~PhotoListView()
{
}

int PhotoListView::numSelected() const
{
    return selectedItems().count();
}

void PhotoListView::removeSelected()
{
    QPtrList<QListViewItem> list = selectedItems();
    for( QListViewItem* item = list.first(); item; item= list.next() )
    {
        delete item;
    }

    // We have to do this as it does not seem to be emitted by default
    emit selectionChanged();
}

void PhotoListView::doBackup(const QString& filename)
{
    // Ensure a filename is given
    if( filename == QString::null )
        return;

    QFile f(filename);

    // Remove the file if there is no session to backup
    if( childCount() <= 0 )
    {
        if( f.exists() )
        {
            f.remove();
        }
        return;
    }

    // Create our backup XML data
    QDomDocument doc;
    QDomProcessingInstruction instr = doc.createProcessingInstruction("xml", "version='1.0' encoding='UTF-8'");
    doc.appendChild(instr);
    QDomElement root = doc.createElement("backup");
    doc.appendChild(root);

    PhotoListViewItem *item = dynamic_cast<PhotoListViewItem*>(firstChild());
    while( item )
    {
        item->photo().save2DOMDoc(doc, root);
        item = dynamic_cast<PhotoListViewItem*>(item->itemBelow());
    }

    // Attempt to open the file for writing
    if( !f.open(IO_WriteOnly) )
        return;

    // Write out our backup XML data
    QTextStream stream(&f);
    stream.setEncoding(QTextStream::UnicodeUTF8);
    stream << doc.toString();
}

void PhotoListView::doRestore(const QString& filename)
{
    // Ensure a filename is given
    if( filename == QString::null )
        return;

    // Attempt to open the file for reading
    QFile f(filename);
    if( !f.open(IO_ReadOnly) )
        return;

    // setup the DOM
    QDomDocument doc;
    if( !doc.setContent(&f) )
        return;

    QDomElement root = doc.documentElement();
    if( root.isNull() || root.tagName() != "backup" )
        return;

    // get going with the restore
    PhotoListViewItem *newPhotoItem = 0L;
    for( QDomNode n = root.firstChild(); !n.isNull(); n = n.nextSibling())
    {
        QDomElement e = n.toElement();
        if( e.isNull() || e.tagName() != "photo" ) continue;

        /** TODO: Maybe est to ensure url is still valid. It works fine as is but
         *  maybe removing the photo is best?!?
         */
        KURL url(e.attribute("url"));

        newPhotoItem = new PhotoListViewItem(this, url, newPhotoItem);
        newPhotoItem->photo().restoreFromDOM(e);
    }
}

QDragObject* PhotoListView::dragObject()
{
    if (!currentItem())
        return 0;

    return new QStoredDrag("application/x-photolistviewitem", viewport());
}

void PhotoListView::resizeEvent(QResizeEvent * e)
{
    // Call base class implementation
    KListView::resizeEvent(e);

    // If list is empty, invalidate everything to display
    // the information text as a nice touch
    if( !firstChild() )
    {
        viewport()->repaint(viewport()->rect());
    }
}

void PhotoListView::viewportPaintEvent(QPaintEvent *e)
{
    // Call base class implementation
    KListView::viewportPaintEvent(e);

    // If list is empty, invalidate everything to display
    // the information text as a nice touch
    if( !firstChild() )
    {
        QString str;
        QRect rect( 0, 0, 200, 170 );


        // Initialize painter device
        QPainter painter(viewport());

        // Setup the pen
        QPen pen(colorGroup().mid(), 4);
        painter.setPen(pen);

        // position our rect
        rect.moveTopLeft(viewport()->rect().center() - QPoint(rect.width()/2, rect.height()/2));

        // draw our text
        str = i18n("KFlickr");
        QFont font("Helvetica", 30, QFont::Bold);
        painter.setFont(font);
        QRect brect = painter.boundingRect(rect, AlignHCenter|AlignTop, str);
        painter.drawText(rect, AlignHCenter|AlignTop, str);

        str = i18n("A Flickr.com Uploader for KDE\nhttp://kflickr.sourceforge.net");
        font.setPointSize(6);
        painter.setFont(font);
        painter.drawText(rect, AlignHCenter|AlignVCenter, str);

        str = i18n("(Drag-n-Drop your photos here)");
        painter.drawText(rect, AlignHCenter|AlignBottom, str);
    }
}

bool PhotoListView::acceptDrag( QDropEvent *event ) const
{
    bool acceptable = false;

    if( acceptDrops() && itemsMovable() )
    {
        if( event->provides( "text/plain" ) ||  event->provides( "text/uri-list") )
            acceptable = true;
        else if( event->provides("application/x-photolistviewitem") && event->source() == viewport() )
            acceptable = true;
    }
    return acceptable;
}

void PhotoListView::selectNext()
{
    if( numSelected() == 0 && firstChild() )
    {
        setSelected(firstChild(), TRUE);
    }
    else
    {
        QListViewItemIterator it(this, QListViewItemIterator::Selected);
        QListViewItem* next = it.current()->itemBelow();
        if( next )
        {
            clearSelection();
            setSelected(next, TRUE);
        }
    }
}

void PhotoListView::selectPrevious()
{
    if( numSelected() == 0 && lastItem() )
    {
        setSelected(lastItem(), TRUE);
    }
    else
    {
        QListViewItemIterator it(this, QListViewItemIterator::Selected);
        QListViewItem* previous = it.current()->itemAbove();
        if( previous )
        {
            clearSelection();
            setSelected(previous, TRUE);
        }
    }
}

void PhotoListView::showRMBMenu(QListViewItem*, const QPoint &pt, int)
{
    if( childCount() > 0)
    {
        KXMLGUIClient* gui = static_cast<KXMLGUIClient*>(kflickrPart::Instance());
        QPopupMenu *pop = (QPopupMenu*)gui->factory()->container("listitem_popup", gui);
        pop->popup(pt);
    }
}

void PhotoListView::addPreviewRequest(QListViewItem* item)
{
    // Reset the aggregation timer as some new photos have been added
    m_previewTimer.start(250, TRUE);

    // We have to store the items as QListViewItem pointers as this signal is emitted in
    // the KListViewItem's constructor and therefore if we try to cast to a PhotoListViewItem
    // the cast will fail. After the timer triggers the object's initialization is complete and
    // we can therefore do the cast safely in that function as we have done.
    m_previewItems.append(item);
}

void PhotoListView::startPreviewJob()
{
    KIO::PreviewJob *job;
    KURL::List previewURLs;

    // As stated in the addPreviewRequest() func we must now perform the cast
    // allowing us access to the URLs we need to pass to the preview job.
    for( QListViewItem* item = m_previewItems.first(); item; item = m_previewItems.next() )
    {
        PhotoListViewItem* i = dynamic_cast<PhotoListViewItem*>(item);
        if( i )
        {
            previewURLs.append(i->photo().URL());
        }
    }
    m_previewItems.clear();

    // start the IO slave
    job = KIO::filePreview(previewURLs, 140, 90, 0, 0, true, false);
    job->setIgnoreMaximumSize(TRUE);

    // make notification connections
    connect(job, SIGNAL(result(KIO::Job*)), this, SLOT(jobResult(KIO::Job*)));
    connect(job, SIGNAL(gotPreview(const KFileItem*, const QPixmap&)), SLOT(gotPreview(const KFileItem*, const QPixmap&)));
    connect(job, SIGNAL(failed(const KFileItem*)), this, SLOT(jobFailed(const KFileItem*)));

    // add to outstanding jobs DB
    m_jobs.append(job);
}

void PhotoListView::jobResult(KIO::Job* job)
{
    // do some house cleaning
    if( m_jobs.contains(job) )
        m_jobs.remove(job);
}

void PhotoListView::jobFailed(const KFileItem* fi)
{
    // find list item to match fi
    PhotoListViewItem *item = dynamic_cast<PhotoListViewItem*>(firstChild());

    while( item )
    {
        if( item->photo().URL() == fi->url() )
        {
            // show broken image icon
            item->photo().preview(SmallIcon("file_broken", KIcon::SizeHuge, KIcon::DefaultState));
            repaintItem(item);

            // Don't put a break here, although probably not used this would
            // prevent having two of the same photos with the same filename.
        }
        item = dynamic_cast<PhotoListViewItem*>(item->itemBelow());
    }
}

void PhotoListView::gotPreview(const KFileItem* fi, const QPixmap& pm)
{
    // find list item to match fi
    PhotoListViewItem *item = dynamic_cast<PhotoListViewItem*>(firstChild());

    while( item )
    {
        if( item->photo().URL() == fi->url() )
        {
            item->photo().preview(pm);
            repaintItem(item);

            // Don't put a break here, although probably not used this would
            // prevent having two of the same photos with the same filename.
        }
        item = dynamic_cast<PhotoListViewItem*>(item->itemBelow());
    }
}

PhotoListViewItem::PhotoListViewItem(PhotoListView *parent, const KURL& url)
    : QObject()
    , KListViewItem(parent)
    , m_photo(*(new Photo(url)))
{
    init();

    // Always add to bottom of list if not adding after a specific item
    QListViewItem* lastItem = parent->lastItem();
    if( lastItem )
    {
        moveItem(lastItem);
    }
}

PhotoListViewItem::PhotoListViewItem(PhotoListView *parent, const KURL& url, PhotoListViewItem* after)
    : QObject()
    , KListViewItem(parent, after)
    , m_photo(*(new Photo(url)))
{
    init();
}

void PhotoListViewItem::init( )
{
    connect(&m_photo, SIGNAL(update(Q_UINT16)), SLOT(updateItem(Q_UINT16)));
}

PhotoListViewItem::~PhotoListViewItem()
{
    delete &m_photo;
}

void PhotoListViewItem::setHeight( int height )
{
    height = 115;
    KListViewItem::setHeight( height );
}

void PhotoListViewItem::paintCell( QPainter *p, const QColorGroup &cg, int column, int width, int alignment )
{
    QRect cr = QRect( 0, 0, width, height() );

    p->setPen( cg.foreground() );
    p->setBackgroundColor( backgroundColor(0) );

    if( isSelected() )
        p->setBackgroundColor( cg.highlight() );

    p->eraseRect( cr );

    if( column == 0 )
    {
        QSize previewSize = QSize(m_photo.preview().width(), m_photo.preview().height()).boundedTo(QSize(140,90));

        int borderWidth = static_cast<PhotoListView*>(listView())->getBorderImage().width();
        p->drawPixmap( cr.width() / 2 - borderWidth / 2, 0, static_cast<PhotoListView*>(listView())->getBorderImage() );
        p->drawPixmap( cr.width() / 2 - previewSize.width() / 2,
                       cr.height() / 2 - previewSize.height() / 2,
                       m_photo.preview(), 0, 0, 140, 90 );
    }
    else
    {
        KListViewItem::paintCell(p, cg, column, width, alignment);
    }
}

void PhotoListViewItem::setSelected(bool s)
{
    KListViewItem::setSelected(s);

    m_photo.selected(s);
}

int PhotoListViewItem::width( const QFontMetrics &fm, const QListView *lv, int c ) const
{
    if( c == 0 )
        return 175;

    return KListViewItem::width( fm, lv, c );
}

void PhotoListViewItem::updateItem(Q_UINT16 flag)
{
    if( flag & PHOTO_PREVIEW )
    {
        repaint();
    }
}

#include "photolistview.moc"
