/* -*- c++ -*-
 *
 * statspagerates.cpp
 *
 * Copyright (C) 2003 Sebastian Sauer <mail@dipe.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include <qlabel.h>
#include <qvbox.h>
#include <qpainter.h>

#include <kdebug.h>
#include <klocale.h>

#include "kmldonkey.h"
#include "donkeyprotocol.h"
#include "prefs.h"

#include "statspagerates.h"
//#include "statspagerates.moc"

StatsWidget::StatsWidget(QWidget *parent, long defaultmaxvalue) : QFrame(parent)
{
    setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Expanding);
    setBackgroundColor(Qt::white);
    maxvalue = defaultmaxvalue;
    minvalue = defaultmaxvalue;
    pixmap = 0;
}

StatsWidget::~StatsWidget()
{
}

void StatsWidget::clear()
{
    stats.clear();
    update();
}

void StatsWidget::addStat(const QString &statid)
{
    stats.replace(statid, new StatsWidgetObject());
}

void StatsWidget::removeStat(const QString& statid)
{
    //delete stats[statid];
    stats.remove(statid);
}

void StatsWidget::setStatEnable(const QString &statid, const bool on)
{
    if (! stats[statid]) return;
    stats[statid]->enable = on;
}

void StatsWidget::setStatColor(const QString &statid, const QColor& color)
{
    if (! stats[statid]) return;
    stats[statid]->color = color;
}

void StatsWidget::addStatValue(const QString &statid, long value)
{
    if (! stats[statid]) return;

    if (value < 0) value = 0; // negativ values arn't evaluated
    else if (value > maxvalue) maxvalue = value;

    if (value < minvalue) minvalue = value;
    if (maxvalue - minvalue <= 0) minvalue = 0; // we don't want divide something with negativ numbers or 0

    uint maxv = 1024; // default is to remember enough items to fill a 2048-Screen
    if (width() > (int)(maxv * 2)) maxv = (uint)(width() + 1 / 2); // if we've more space to fill
    while (stats[statid]->values.count() > maxv) // remove older values
        stats[statid]->values.pop_back();

    stats[statid]->values.insert(stats[statid]->values.begin(), value);
    update();
}

void StatsWidget::paintEvent(QPaintEvent*)
{
    if(! pixmap)
        pixmap = new QPixmap(width(), height());
    else if(pixmap->width() != width() || pixmap->height() != height())
        pixmap->resize(width(), height());

    QPainter painter(pixmap);
    painter.fillRect(0, 0, width(), height(), Qt::white); // clear the painting-area

    int w, pos, prevpos;
    int h = height() - 4;
    int steps = 2;

    int vrange = maxvalue - minvalue;
    if(vrange >= 3) { // paint grid
        int vsteps = vrange / 10;
        if (vsteps < 1) vsteps = 1;
        int psteps = (int)( (float)h / vrange * vsteps );
        int r = maxvalue;

        painter.setPen(Qt::lightGray);
        //painter.setPen( QPen(Qt::lightGray, 0, Qt::DotLine) );

        for(int i = psteps; i < height(); i += psteps) {
            r -= vsteps;
            painter.drawLine(0, i, width(), i);
            painter.drawText(3, i - 1, KGlobal::locale()->formatNumber(r, 0) );
        }
    }

    StatsWidgetObject *so;
    QMap<QString, StatsWidgetObject*>::iterator statsit;
    QValueList<long>::iterator it;
    for (statsit = stats.begin(); statsit != stats.end(); ++statsit) {
        so = statsit.data();
        if (! so) continue;

        w = width();
        prevpos = -1;
        painter.setPen(so->color);
        for (it = so->values.begin(); it != so->values.end() && w >= 0; ++it) {
            pos = h - (int)( (float)h / (maxvalue - minvalue) * ((*it) - minvalue) );

            if (prevpos >= 0)
                painter.drawLine(w, pos, w + steps, prevpos);

            prevpos = pos;
            w = (w > steps) ? w - steps : (w > 0) ? 0 : -1;
        }
    }

    bitBlt(this, 0, 0, pixmap);
}

StatsPageRates::StatsPageRates(QWidget* parent) : QVBox(parent)
{
    bool ok;
    int i;
    int maxvalue = 15; // 15k
    QMap<QString, QString> options = KMLDonkey::App->donkey->optionsList();

    i = options["max_hard_download_rate"].toInt(&ok);
    if (ok && i > maxvalue) maxvalue = i;

    i = options["max_hard_upload_rate"].toInt(&ok);
    if (ok && i > maxvalue) maxvalue = i;

    statswidget = new StatsWidget(this, maxvalue * 1000);

    statswidget->addStat("tdl");
    statswidget->addStat("tul");
}

