/*
	actions.cpp - Actions
	Copyright (C) 2005  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "actions.h"
#include "configuration.h"
#include "confirmation.h"
#include "extras.h"
#include "miscutils.h"
#include "mmainwindow.h"
#include "msystemtray.h"
#include "systemconfig.h"

#include <dcopclient.h>
#include <kdebug.h>
#include <kiconloader.h>
#include <klocale.h>
#include <kmessagebox.h>

Action *Action::_instance = 0;

Action::~Action()
{
}

QString Action::actionToConfigGroup(const Type action) const
{
	switch (action)
	{
		case ShutDown: return "Shut Down";
		case Reboot: return "Reboot";
		case LockScreen: return "Lock Screen";
		case Logout: return "Logout";
		default: return QString::null;
	}
}

// TODO: 0.9: KDE 3.5(?) GDM support
// TODO: 0.9: force application exit (without asking for data save)
bool Action::endSession(const KApplication::ShutdownType type, const Type action)
{
	_totalExit = false;

	// test mode
	if (_testMode)
	{
		MiscUtils::showTestMessage(getName(action));

		return true;
	}

	ks_main->setCaption(i18n("Please wait..."));
	_totalExit = true;
	if (
		!kapp->requestShutDown(
			KApplication::ShutdownConfirmNo,
			type,
			KApplication::ShutdownModeForceNow
		)
	)
	{
		ks_main->setCaption("");
		KMessageBox::error(
			0,
			i18n(
				"Could not logout properly.\n" \
				"The session manager cannot be contacted."
			)
		);
		_totalExit = false;

		return false; // error
	}

	return true; // ok
}

bool Action::exec(const Type action, const bool stopTimer)
{
	if (stopTimer)
		ks_main->cancel();

	_totalExit = false;

	Method method = Method_KDE;
	QString command;

	if (!isEnabled(action))
		return false; // error

	// kdDebug() << "Action::exec: " << action << endl;

	switch (action)
	{
		// nothing
		case Nothing:
			return false; // error

		// shut down
		case ShutDown:
			_totalExit = true;
			MiscUtils::closeCDTray();
			getMethod(action, method, command);
			MiscUtils::runCommandBeforeAction("Shut Down");

			if (method == Method_KDE)
				return endSession(KApplication::ShutdownTypeHalt, action);

			break;

		// reboot
		case Reboot:
			_totalExit = true;
			MiscUtils::closeCDTray();
			getMethod(action, method, command);
			MiscUtils::runCommandBeforeAction("Reboot");

			if (method == Method_KDE)
				return endSession(KApplication::ShutdownTypeReboot, action);

			break;

		// lock screen
		case LockScreen:
			getMethod(action, method, command);
			MiscUtils::runCommandBeforeAction("Lock Screen");
			if (method == Method_KDE)
			{
				// test mode
				if (_testMode)
				{
					MiscUtils::showTestMessage(getName(action));

					return true; // ok
				}

				// this is a modified "void Lockout::lock()"
				// from "Lock/Logout Applet" (lockout.cpp [3.1.4])
				QCString kdesktop("kdesktop");
				int kshutdown_screen_number = qt_xscreen();
				if (kshutdown_screen_number)
					kdesktop.sprintf("kdesktop-screen-%d", kshutdown_screen_number);

				if (
					!kapp->dcopClient()->send(
						kdesktop,
						"KScreensaverIface",
						"lock()",
						""
					)
				)
				{
					KMessageBox::error(0, i18n("kdesktop: DCOP call failed!"));

					return false; // error
				}
				// hide window
				if (MSystemTray::mode() == MSystemTray::Always)
					ks_main->hide();

				return true; // ok
			}

			break;

		// logout
		case Logout:
			_totalExit = true;
			MiscUtils::closeCDTray();
			getMethod(action, method, command);
			MiscUtils::runCommandBeforeAction("Logout");

			if (method == Method_KDE)
				return endSession(KApplication::ShutdownTypeNone, action);

			break;

		// "extras" action
		case Extras:
			// test mode
			if (_testMode)
			{
				MiscUtils::showTestMessage(
					ks_extras->getActionDescription() + "\n" +
					ks_extras->fileToExecute()
				);

				return true; // ok
			}

			return ks_extras->execAction();
	}

	// test mode
	if (_testMode)
	{
		MiscUtils::showTestMessage(i18n("Command: %1").arg(command));

		return true; // ok
	}

	// run default or user command
// TODO: 0.9: add function to change /sbin/* permissions (e.g. /sbin/poweroff)
// TODO: 0.9: save session before /sbin/* execution
	if (MiscUtils::runCommand(command))
	{
// TODO: 0.9: MMainWindow::stealth
		// hide window
		if ((action == LockScreen) && (MSystemTray::mode() == MSystemTray::Always))
			ks_main->hide();

		return true; // ok
	}

	_totalExit = false;

	return false; // error
}

bool Action::execConfirm(const Type action)
{
	if (!Confirmation::confirm(action))
		return false;

	return exec(action, action != LockScreen);
}

bool Action::execCurrent()
{
	return exec(_current);
}

QString Action::getCurrentName() const
{
	return getName(_current);
}

QPixmap Action::getIcon(const Type action) const
{
	return SmallIcon(getIconName(action));
}

QString Action::getIconName(const Type action) const
{
	// compatible with the standard KDE logout dialog
	switch (action)
	{
		case Nothing: return "misc";
		case ShutDown: return "exit";
		case Reboot: return "reload";
		case LockScreen: return "lock";
		case Logout: return "undo";
		case Extras: return "bookmark";
	}

	return "misc";
}

QString Action::getMethod(const Action::Type action, Method &method, QString &command)
{
	QString group = actionToConfigGroup(action);

	if (group.isNull())
	{
		// kdDebug() << "Action::getMethod: No group in config for action " << action << endl;

		return QString::null;
	}

	QString defaultCommand;
	switch (action)
	{
		case ShutDown:
			defaultCommand = DEFAULT_SHUT_DOWN_COMMAND;
			break;
		case Reboot:
			defaultCommand = DEFAULT_REBOOT_COMMAND;
			break;
		case LockScreen:
			defaultCommand = DEFAULT_LOCK_SCREEN_COMMAND;
			break;
		case Logout:
			defaultCommand = DEFAULT_LOGOUT_COMMAND;
			break;
		default:
			defaultCommand = QString::null;
	}

	KConfig *conf = kshutdownrc->config();
	if (!conf->hasGroup(group))
	{
		method = Method_KDE;
		command = defaultCommand;
		// kdDebug() << "Action::getMethod: No group in config for action " << action << endl;

		return defaultCommand;
	}

	conf->setGroup(group);

	// read method
	method = (Method)conf->readNumEntry("Method", Method_KDE);
	if ((method < Method_KDE) || (method > Method_UserCommand))
		method = Method_KDE;

	switch (method)
	{
		case Method_KDE:
			command = conf->readEntry("Command", defaultCommand);
			break;
		case Method_DefaultCommand:
			command = defaultCommand;
			break;
		case Method_UserCommand:
			command = conf->readEntry("Command", defaultCommand);
			if (command.isEmpty())
				method = Method_KDE;
			break;
	}

	return defaultCommand;
}

void Action::setMethod(const QString &group, const Method method, const QString &command) const
{
	KConfig *conf = kshutdownrc->config();
	conf->setGroup(group);
	if (method == Method_UserCommand)
		conf->writeEntry("Command", command);
	conf->writeEntry("Method", method);
}

QString Action::getName(const Type action) const
{
	switch (action)
	{
		case Nothing: return i18n("Nothing");
		case ShutDown: return i18n("Turn Off Computer");
		case Reboot: return i18n("Restart Computer");
		case LockScreen: return i18n("Lock Session");
		case Logout: return i18n("End Current Session");
		case Extras: return ks_extras->getActionDescription();
	}

	return i18n("Unknown");
}

bool Action::isEnabled(const Type action)
{
	Method m = Method_KDE;
	QString c = QString::null;
	switch (action)
	{
		case Nothing:
			return false;
		case ShutDown:
			getMethod(action, m, c);
			return
				!MiscUtils::isRestricted("action_shutdown") &&
				!((m == Method_KDE) && !SystemConfig::canShutDown());
		case Reboot:
			getMethod(action, m, c);
			return
				!MiscUtils::isRestricted("action_reboot") &&
				!((m == Method_KDE) && !SystemConfig::canShutDown());
		case LockScreen:
			return !MiscUtils::isRestricted("action_lockscreen");
		case Logout:
			return !MiscUtils::isRestricted("action_logout");
		case Extras:
			return !MiscUtils::isRestricted("action_extras");
	}

	return false;
}

void Action::totalExec()
{
	if (!execCurrent())
		KMessageBox::sorry(0, i18n("Action failed! (%1)").arg(_current));
}

Action::Action()
	: QObject(ks_main),
	_active(false),
	_testMode(false),
	_totalExit(false),
	_current(Nothing)
{
}
