/*
	actions.h - Actions
	Copyright (C) 2005  Konrad Twardowski <kdtonline@poczta.onet.pl>

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef __ACTIONS_H__
#define __ACTIONS_H__

#include <kapplication.h>

// default external commands
const QString
	DEFAULT_LOCK_SCREEN_COMMAND("xscreensaver-command -lock"),
// TODO: 0.9: logout command for GNOME (?)
	DEFAULT_LOGOUT_COMMAND(""),
// TODO: 0.9: /usr/bin/reboot, /usr/bin/poweroff
	DEFAULT_REBOOT_COMMAND("/sbin/reboot"),
	DEFAULT_SHUT_DOWN_COMMAND("/sbin/poweroff");

#define ks_actions Action::getInstance()

/** @short An action manager. */
class Action: public QObject
{
public:

	/**
	 * Methods. Don't change these values!
	 */
	enum Method {
		Method_KDE = 0, /**< Standard KDE/KDM method. (default) */
		Method_DefaultCommand = 1, /**< Default external command. (e.g. /sbin/reboot) */
		Method_UserCommand = 2 /**< User command. Any command that can be executed by @c KRun::run. */
	};

	/**
	 * Actions.
	 */
	enum Type {
		Nothing = 0, /**< Nothing. */
		ShutDown = 1, /**< System shut down. */
		Reboot = 2, /**< System reboot. */
		LockScreen = 3, /**< Screen lock. */
		Logout = 4, /**< End current session. */
		Extras = 5 /**< Extras. Execute an item selected from the "Extras..." menu. */
	};

	/** Destructor. */
	virtual ~Action();

	/**
	 * Converts @p action value to the configuration group name.
	 */
	QString actionToConfigGroup(const Type action) const;

	/**
	 * Returns @c true if the current action is active.
	 */
	inline bool active() const { return _active; }

	/**
	 * Activates the current action if @p yes is @c true.
	 */
	inline void setActive(const bool yes) { _active = yes; }

	/** Returns the currently selected action. */
	inline Type current() const { return _current; }

	/**
	 * Sets current action to @p action.
	 */
	inline void setCurrent(const Type action) { _current = action; }

	/**
	 * Ends current session.
	 * This function is a wrapper for the @c kapp->requestShutDown().
	 * @param type A shut down type
	 * @param action An action to execute
	 * @return @c true if successful; otherwise @c false
	 */
	bool endSession(const KApplication::ShutdownType type, const Type action);

	/**
	 * Executes an action.
	 * @param action An action to execute
	 * @param stopTimer If @c true, the current active action and its timer are stopped
	 * @return @c true if successful; otherwise @c false
	 */
	bool exec(const Type action, const bool stopTimer = true);

	/**
	 * Executes an action.
	 * Displays confirmation message before execution.
	 * @param action An action to execute
	 */
	bool execConfirm(const Type action);

	/**
	 * Executes the current action without confirmation.
	 */
	bool execCurrent();

	/**
	 * Returns the current action name as text.
	 */
	QString getCurrentName() const;

	/**
	 * Returns an icon for @p action.
	 */
	QPixmap getIcon(const Type action) const;

	/**
	 * Returns an icon name for @p action.
	 */
	QString getIconName(const Type action) const;

	/**
	 * Initializes and return singleton instance of the @c Action class.
	 */
	inline static Action *getInstance()
	{
		if (!_instance)
			_instance = new Action();

		return _instance;
	}

	/**
	 * Reads method for action from the configuration file.
	 * @param action An action
	 * @retval method A method to use
	 * @retval command A command to run
	 * @return A default command
	 */
	QString getMethod(
		const Type action,
		Method &method,
		QString &command
	);

	/**
	 * Changes the action method.
	 * @param group A configuration group (in @b kshutdownrc file)
	 * @param method A new method
	 * @param command A new command
	 */
	void setMethod(const QString &group, const Method method, const QString &command) const;

	/**
	 * Returns an action name as text.
	 * @param action An action
	 */
	QString getName(const Type action) const;

	/** Returns @c true if @p action is @b not disabled by KIOSK. */
	bool isEnabled(const Type action);

	/** Returns @c true if test mode is enabled. */
	inline bool testMode() const { return _testMode; }

	/**
	 * Enables/disables the <b>test mode</b>.
	 * @param yes @c true to enable test mode
	 */
	inline void setTestMode(const bool yes) { _testMode = yes; }

	/**
	 * Executes the current action,
	 * and displays an error message if action failed.
	 */
	void totalExec();

	/**
	 * Returns @c true if application/system is shutting down.
	 */
	inline bool totalExit() const { return _totalExit; }

	/**
	 * Sets <b>total exit</b> info to @p yes.
	 */
	inline void setTotalExit(const bool yes) { _totalExit = yes; }
private:
	static Action *_instance;
	bool
		_active,
		_testMode,
		_totalExit;
	Type _current;
	Action();
};

#endif // __ACTIONS_H__
