// =============================================================================
//
//      --- kvi_memmove.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#include "kvi_debug.h"
#include "kvi_settings.h"

#define _KVI_MEMMOVE_CPP_

#ifdef COMPILE_i386_ASM_CODE

void *kvi_memmove(void *dst_ptr, const void *src_ptr, int len)
{
	__range_valid(dst_ptr);
	__range_valid(src_ptr);
	__range_valid(len >= 0);
	// Save pointer registers
	asm("	pushl %esi");                     // Save %esi
	asm("	pushl %edi");                     // Save %edi
	// Load arguments
	asm("	movl 16(%ebp),%ecx");             // %ecx = len
	asm("	movl 12(%ebp),%esi");             // %esi = src
	asm("	movl 8(%ebp),%edi");              // %edi = dst
	// Compare src and dest
	asm("	cmpl %esi,%edi");                 // %edi - %esi
	asm("	jbe move_from_bottom_to_top");    // If( %edi < %esi ) jump to move_from_bottom_to_top
	// dst_ptr > src_ptr
	asm("	addl %ecx,%esi");                 // %esi += %ecx (src_ptr += len);
	asm("	addl %ecx,%edi");                 // %edi += %ecx (dst_ptr += len);
	asm("	decl %esi");                      // %esi--; (src_ptr--);
	asm("	decl %edi");                      // %edi--; (dst_ptr--);
	asm("	std");                            // Set direction flag (decrement esi and edi in movsb)
	// Optimization : check for non-odd len (1, 3, 5, 7...)
	asm("	shr $1,%ecx");                    // %ecx >> 1, shifted bit -> CF
	asm("	jnc move_two_bytes_top_to_bottom_directly");  // If !carry (CF == 0) skip this move
	// Move the first byte (non-odd)
	asm("	movsb %ds:(%esi),%es:(%edi)");    // *dst-- = *src-- if DF else *dst++ = *src++
	asm("move_two_bytes_top_to_bottom_directly:");
	asm("	decl %esi");                      // %esi--; (src_ptr--);
	asm("	decl %edi");                      // %edi--; (dst_ptr--);
	asm("move_two_bytes_top_to_bottom:");
	asm("	shr $1,%ecx");                    // %ecx >> 1, shifted bit -> CF
	asm("	jnc move_the_rest_top_to_bottom_directly"); // If !carry (CF == 0) skip this move
	// Move the next two bytes
	asm("	movsw %ds:(%esi),%es:(%edi)");    // *((word *)dst)-- = *((word)src)-- if DF else *((word *)dst)++ = *((word)src)++
	asm("move_the_rest_top_to_bottom_directly:");
	asm("	subl $2,%esi");                   // %esi -= 2; (src -= 2);
	asm("   subl $2,%edi");                   // %edi -= 2; (dst -= 2);
	asm("   jmp move_the_rest");              // Call last repnz movsl
	// dst_ptr <= src_ptr
	asm("move_from_bottom_to_top:");
	asm("	cld");                            // Clear direction flag (increment esi and edi in movsb)
	// Optimization : check for non-odd len (1, 3, 5, 7...)
	asm("	shr $1,%ecx");                    // %ecx >> 1, shifted bit -> CF
	asm("	jnc move_two_bytes");             // If !carry (CF == 0) skip this move
	// Move the first byte (non-odd)
	asm("	movsb %ds:(%esi),%es:(%edi)");    // *dst-- = *src-- if DF else *dst++ = *src++
	// Optimization : pass 2, check for %2 and %3
	asm("move_two_bytes:");
	asm("	shr $1,%ecx");                    // %ecx >> 1, shifted bit -> CF
	asm("	jnc move_the_rest");              // If !carry (CF == 0) skip this move
	// Move the next two bytes
	asm("	movsw %ds:(%esi),%es:(%edi)");    // *((word *)dst)-- = *((word)src)-- if DF else *((word *)dst)++ = *((word)src)++
	// Main move remaining part
	asm("move_the_rest:");
	asm("	repnz; movsl %ds:(%esi),%es:(%edi)"); // Loop moving 4 bytes at once (increment or decrement as above)
	// Restore pointer registers
	asm("	popl %edi");                      // Restore %edi
	asm("	popl %esi");                      // Restore %esi

	return dst_ptr;
}

#else // COMPILE_i386_ASM_CODE

void *kvi_memmove(void *dst_ptr, const void *src_ptr, int len)
{
	__range_valid(dst_ptr);
	__range_valid(src_ptr);
	__range_valid(len >= 0);
	register char *dst;
	register char *src;
	if( dst_ptr > src_ptr ) {
		dst = (char *) dst_ptr + len - 1;
		src = (char *) src_ptr + len - 1;
		while( len-- ) *dst-- = *src--;
	} else {
		// It is valid even if dst_ptr == src_ptr
		dst = (char *) dst_ptr;
		src = (char *) src_ptr;
		while( len-- ) {
			*dst = *src;
			dst++;
			src++;
		}
	}
	return dst_ptr;
}

void kvi_fastmove(void *dst_ptr, const void *src_ptr, int len)
{
	__range_valid(dst_ptr);
	__range_valid(src_ptr);
	__range_valid(len >= 0);
	register const char *src = (const char *) src_ptr;
	register char *dst = (char *) dst_ptr;
	while( len-- ) {
		*dst = *src;
		dst++;
		src++;
	}
}

#endif // COMPILE_i386_ASM_CODE
