//LabPlot : Line.cc

#include <math.h>
#include <iostream>
#include <kdebug.h>
#include "Line.h"

using namespace std;

Line::Line(Point s, Point e, QColor c,int w) 
	: start(s), end(e), color(c), width(w)
{
	sarrow = new Arrow();
	earrow = new Arrow();
	sarrow->setLocation(0);
	earrow->setLocation(1);
}

QStringList Line::Info() {
	QStringList s;

	// no lines
	if(start.X()==end.X() && start.Y()==end.Y())
		return s;

        s << QString::number(start.X())+"/"+QString::number(start.Y());
        s << QString::number(end.X())+"/"+QString::number(end.Y());
	s << QString::number(width);
        s << color.name();
	s += sarrow->Info();
	s += earrow->Info();
	
	return s;
}

// draw the line on the worksheet
void Line::draw(QPainter *p, int X, int Y) {
//	kdDebug()<<"DRAW LINE from "<<start.X()*X<<'/'<<start.Y()*Y<<" to "<<end.X()*X<<'/'<<end.Y()*Y<<endl;
//	kdDebug()<<"X="<<X<<" Y="<<Y<<" start="<<start.X()<<'/'<<start.Y()<<endl;

	if(fabs(end.X()-start.X())>1.0e-6 || fabs(end.Y()-start.Y())>1.0e-6) {
		p->setPen(QPen(color,width));
		p->drawLine( (int)(X*start.X()),(int)(Y*start.Y()), (int)(X*end.X()), (int)(Y*end.Y()));

		// draw arrows
		sarrow->draw(p,start,end,(int)X,(int)Y);
		earrow->draw(p,start,end,(int)X,(int)Y);
	}
}

// save
void Line::save(QTextStream *t) {
	*t<<start.X()<<' '<<start.Y()<<' '<<end.X()<<' '<<end.Y()<<endl;
	*t<<color.name()<<' '<<width<<endl;
	sarrow->save(t);
	earrow->save(t);
}

// open
void Line::open(QTextStream *t,int version) {
	double sx,sy,ex,ey;
	QString colorstring;
	
	*t>>sx>>sy>>ex>>ey;

	start.setPoint(sx,sy);
	end.setPoint(ex,ey);
	
	*t>>colorstring>>width;
	color=QColor(colorstring);
	sarrow->open(t,version);
	earrow->open(t,version);
}

QDomElement Line::saveXML(QDomDocument doc) {
	QDomElement linetag = doc.createElement( "Line" );
	
	QDomElement tag = doc.createElement( "Start" );
	tag.setAttribute("x",start.X());
	tag.setAttribute("y",start.Y());
    	linetag.appendChild( tag );
	tag = doc.createElement( "End" );
	tag.setAttribute("x",end.X());
	tag.setAttribute("y",end.Y());
    	linetag.appendChild( tag );

	tag = doc.createElement( "Color" );
    	linetag.appendChild( tag );
  	QDomText t = doc.createTextNode( color.name() );
    	tag.appendChild( t );
	tag = doc.createElement( "Width" );
    	linetag.appendChild( tag );
  	t = doc.createTextNode( QString::number(width) );
    	tag.appendChild( t );

	tag = sarrow->saveXML(doc,"StartArrow");
    	linetag.appendChild( tag );
	tag = earrow->saveXML(doc,"EndArrow");
    	linetag.appendChild( tag );

	return linetag;
}

void Line::openXML(QDomNode node) {
	while(!node.isNull()) {
		QDomElement e = node.toElement();
		kdDebug()<<"LINE TAG = "<<e.tagName()<<endl;
		kdDebug()<<"LINE TEXT = "<<e.text()<<endl;
		
		if(e.tagName() == "Start")
			start.setPoint(e.attribute("x").toDouble(),e.attribute("y").toDouble());
		else if(e.tagName() == "End")
			end.setPoint(e.attribute("x").toDouble(),e.attribute("y").toDouble());
		else if(e.tagName() == "Color")
			color = QColor(e.text());
		else if(e.tagName() == "Width")
			width = e.text().toInt();
		else if(e.tagName() == "StartArrow")
			sarrow->openXML(e.firstChild());		
		else if(e.tagName() == "EndArrow")
			earrow->openXML(e.firstChild());		
	
		node = node.nextSibling();
	}
}

bool Line::inside(int x, int y, int w, int h) {
	double sx = w*start.X();
	double sy = h*start.Y();
	double ex = w*end.X();
	double ey = h*end.Y();
	double d = sqrt((sx-x)*(sx-x) + (sy-y)*(sy-y) - 
		pow((ex-sx)*(-sx+x) + (ey-sy)*(-sy+y),2)/((ex-sx)*(ex-sx) + (ey-sy)*(ey-sy)));

//	kdDebug()<<"Distance : "<<fabs(d)<<endl;

	// distance to line
	if(d<10) {
		// only if inside line
		if(x>=sx-5 && x<=ex+5 && y>=sy-5 && y<=ey+5)
			return true;
		if(x>=ex-5 && x<=sx+5 && y>=sy-5 && y<=ey+5)
			return true;	
	}

	return false;
}
