/*
 *
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Kinis L. Meyer with M. D. McNally 
 *          and Andrew Lumsdaine
 *
 * This file is part of the Notre Dame LAM implementation of MPI.
 *
 * You should have received a copy of the License Agreement for the
 * Notre Dame LAM implementation of MPI along with the software; see
 * the file LICENSE.  If not, contact Office of Research, University
 * of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.  
 *
 * Additional copyrights may follow.
 *
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	NJN
 *
 *	$Id: wcreate.c,v 1.0 1999/07/21 19:00:20 jsquyres Exp $
 *
 *	Function:	- create a new window
 *	Accepts:	- window base
 *			- window size
 *			- window displacement unit
 *			- info
 *			- communicator
 *			- new window (out)
 */

#include <errno.h>
#include <stdlib.h>

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <rpisys.h>


int
MPI_Win_create(base, size, disp_unit, info, comm, newwin)

void			*base;
MPI_Aint		size;
int			disp_unit;
MPI_Info		info;
MPI_Comm		comm;
MPI_Win			*newwin;

{
	struct _attrkey *keystate;		/* key state */
	MPI_Comm	wincomm;		/* window communicator */
	MPI_Win		win;			/* new window */
	int		err;

	lam_initerr_m();
	lam_setfunc(BLKMPIWINCREATE);
/*
 * Check arguments.
 */
	if (comm == MPI_COMM_NULL) {
		return(lam_err_comm(MPI_COMM_NULL,
				MPI_ERR_COMM, 0, "null handle"));
	}

	if (newwin == 0 || size < 0 || disp_unit <= 0) {
		return(lam_err_comm(comm, MPI_ERR_ARG, 0, ""));
	}

	if ((err = MPI_Comm_dup(comm, &wincomm)) != MPI_SUCCESS) {
		return(err);
	}
/*
 * Create the new window.
 */
	win = (MPI_Win) malloc(sizeof(struct _window));
	if (win == 0) {
		return(lam_err_comm(comm, MPI_ERR_OTHER, errno, ""));
	}

	win->w_pstate = (int *) calloc(comm->c_group->g_nprocs, sizeof(int));
	if (win->w_pstate == 0) {
		return(lam_err_comm(comm, MPI_ERR_OTHER, errno, ""));
	}

	win->w_comm = wincomm;
	win->w_base = base;
	win->w_size = size;
	win->w_disp_unit = disp_unit;
	win->w_errhdl = MPI_ERRORS_ARE_FATAL;
	win->w_flags = LAM_WFNOOUT;
	win->w_f77handle = -1;
	win->w_keys = 0;
	win->w_ngroup = 0;
	win->w_ncomplete = 0;
	win->w_nfence = comm->c_group->g_nprocs;
	win->w_req = MPI_REQUEST_NULL;
/*
 * Set the pre-defined attributes.  Toggle pre-defined state since these
 * are marked as pre-defined in MPI_Init().
 */
	if ((keystate = lam_getattr(MPI_WIN_BASE)) == 0) {
		return(lam_errfunc(comm,
			BLKMPIWINCREATE, lam_mkerr(MPI_ERR_INTERN, 0)));
	}
	keystate->ak_flags &= ~LAM_PREDEF;

	err = MPI_Win_set_attr(win, MPI_WIN_BASE, &win->w_base);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIWINCREATE, err));
	}
	keystate->ak_flags |= LAM_PREDEF;

	if ((keystate = lam_getattr(MPI_WIN_SIZE)) == 0) {
		return(lam_errfunc(comm,
			BLKMPIWINCREATE, lam_mkerr(MPI_ERR_INTERN, 0)));
	}
	keystate->ak_flags &= ~LAM_PREDEF;

	err = MPI_Win_set_attr(win, MPI_WIN_SIZE, (void *) win->w_size);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIWINCREATE, err));
	}
	keystate->ak_flags |= LAM_PREDEF;

	if ((keystate = lam_getattr(MPI_WIN_DISP_UNIT)) == 0) {
		return(lam_errfunc(comm,
			BLKMPIWINCREATE, lam_mkerr(MPI_ERR_INTERN, 0)));
	}
	keystate->ak_flags &= ~LAM_PREDEF;

	err = MPI_Win_set_attr(win, MPI_WIN_DISP_UNIT,
				(void *) win->w_disp_unit);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIWINCREATE, err));
	}
	keystate->ak_flags |= LAM_PREDEF;
/*
 * Create and link in the background protocol header receive quest.
 */
	err = _mpi_req_build(&win->w_header, 1, lam_osdhdr_type, MPI_ANY_SOURCE,
		LAM_OSD_HEADER, win->w_comm, LAM_RQIRECV, &win->w_req);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIWINCREATE, err));
	}

	win->w_req->rq_marks |= LAM_RQFPERSIST;
	win->w_req->rq_hdlr = lam_osd_handler;
	win->w_req->rq_extra = win;

	err = _mpi_req_start(win->w_req);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIWINCREATE, err));
	}

	_mpi_req_add(win->w_req);

	wincomm->c_window = win;
	*newwin = win;

	lam_resetfunc_m(BLKMPIWINCREATE);
	return(MPI_SUCCESS);
}
