/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	GDB
 *
 *	$Id: fctl.c,v 6.7 2000/10/12 00:21:47 jsquyres Exp $
 *
 *	Function:	- command to control certain filed functions
 *	Accepts:	- see the "usage" variable
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/param.h>
#include <sys/types.h>
#include <unistd.h>

#include <args.h>
#include <freq.h>
#include <net.h>
#include <priority.h>
#include <terror.h>
#include <typical.h>

#ifndef MAXPATHLEN
#define MAXPATHLEN	1024
#endif

/*
 * local functions
 */
static void		help();

/*
 * local variables
 */
static char		*usage =
		"fctl [-hS] [-s <descno>] [<directory>] <nodes>\n";
static char		pwd[MAXPATHLEN];


int
main(argc, argv)

int   			argc;    
char   			*argv[];

{
	int		n_index;		/* index of node entry */
	int		nodeid;			/* value of node entry */
	int		n_flags;		/* extra node entry info */
	int		tfd;			/* filed descriptor # */

	/* Ensure that we are not root */

	if (getuid() == 0 || geteuid() == 0) {
	  show_help(NULL, "deny-root", NULL);
	  exit(EACCES);
	}

/*
 * Parse the command line.
 */
	validopts("Ssh");
	followed("s");
	exclusive("Ss");

	if (do_args(&argc, argv)) {
		fprintf(stderr, usage);
		exit(errno);
	}
/*
 * Check for help request.
 */
	if (opt_taken('h')) {
		help();
		exit(0);
	}
/*
 * Attach to kernel.
 */
	if (kinit(PRCMD)) {
	  show_help(NULL, "no-lamd", "fctl", NULL);
	  exit(errno);
	}

	if ((nid_parse(&argc, argv)) || (errno = (argc <= 2) ? 0 : EUSAGE)) {
		fprintf(stderr, usage);
		kexit(errno);
	}
/*
 * Loop through all specified nodes.
 */
	nid_get(&n_index, &nodeid, &n_flags);

	if (n_index < 0) {
		fprintf(stderr, usage);
		kexit(EUSAGE);
	}

	do {
/*
 * Clean up file descriptors in filed.
 */
		if (opt_taken('S')) {
			if (lam_rfrmfd(nodeid, NOTFD)) lamfail("fctl (rfrmfd)");
		} else if (opt_taken('s')) {
		 	intparam('s', &tfd);
			if (lam_rfrmfd(nodeid, tfd)) lamfail("fctl (rfrmfd)");
		}
/*
 * Print the current working directory.
 */
		else if (argc == 1) {
			if (lam_rfgetwd(nodeid, pwd) == 0)
					lamfail("fctl (rfgetwd)");
			printf("%s: %s\n", nid_fmt(nodeid), pwd);
		}
/*
 * Change current working directory.
 */
		else if (argc == 2) {
			if (lam_rfchdir(nodeid, argv[1]))
					lamfail("fctl (rfchdir)");
		}

		nid_get(&n_index, &nodeid, &n_flags);
	} while (n_index);

	kexit(0);
	return(0);
}

/*
 *	  help
 *
 *	  Function:	- prints helpful information on the fctl command
 */
static void
help()

{
	printf("\nSynopsis:	fctl [options] <nodes> [<dir>]\n");
	printf("\nDescription:	Control file daemon operations.\n");
	printf("\nOptions:	-h	Print this help message.\n");
	printf("\t\t-S	Blow away all open descriptors.\n");
	printf("\t\t-s <#>	Blow away one specific descriptor.\n");
	printf("\t\t<dir>	Change to new working directory.\n");
	printf("\t\tnone	Print current working directory.\n");
	nid_usage();
	printf("\t\th (local), o (origin), N (all)\n");
	printf("\nExample:	fctl h -S\n");
	printf("\t\t\t\"Clean up (reset) the file daemon.\"\n");
}
