/*
 * Copyright 1998-2001, University of Notre Dame.
 * Authors: Jeffrey M. Squyres, Arun Rodrigues, and Brian Barrett with
 *          Kinis L. Meyer, M. D. McNally, and Andrew Lumsdaine
 * 
 * This file is part of the Notre Dame LAM implementation of MPI.
 * 
 * You should have received a copy of the License Agreement for the Notre
 * Dame LAM implementation of MPI along with the software; see the file
 * LICENSE.  If not, contact Office of Research, University of Notre
 * Dame, Notre Dame, IN 46556.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted subject to the conditions specified in the
 * LICENSE file.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR ``AS IS'' AND ANY EXPRESS OR
 * IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY DIRECT,
 * INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT,
 * STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING
 * IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 * 
 * Additional copyrights may follow.
 * 
 *	Ohio Trollius
 *	Copyright 1997 The Ohio State University
 *	RBD/GDB
 *
 *	$Id: iprobe.c,v 6.7 1999/12/31 21:27:31 jsquyres Exp $
 *
 *	Function:	- non-blocking check of incoming messages
 *			- message is not received
 *	Accepts:	- source rank
 *			- message tag
 *			- communicator
 *			- flag (out)
 *			- status (out)
 *	Returns:	- MPI_SUCCESS or error code
 */

#include <blktype.h>
#include <mpi.h>
#include <mpisys.h>
#include <mpitrace.h>
#include <rpisys.h>
#include <terror.h>


/*@
    MPI_Iprobe - Nonblocking test for a message

Input Parameters:
+ src - source rank, or  'MPI_ANY_SOURCE' (integer) 
. tag - tag value or  'MPI_ANY_TAG' (integer) 
- comm - communicator (handle) 

Output Parameter: 
+ flag - 1 if the message is ready to be received, 0 if it is not
(logical)
- stat - status object (Status), which may be the MPI constant
'MPI_STATUS_IGNORE'

Notes:

This function does not actually receive a message; it only indicates
that a message matching the signature specified is ready to be
received.

.N fortran

.N Errors
.N MPI_SUCCESS
.N MPI_ERR_ARG
.N MPI_ERR_COMM
.N MPI_ERR_TAG
.N MPI_ERR_RANK

.N ACK
@*/
int MPI_Iprobe(int src, int tag, MPI_Comm comm, 
	       int *flag, MPI_Status *stat)
{
	struct _req	request_storage;	/* request structure */
	MPI_Request	req;			/* request */
	int		err;			/* error code */
	int		fl_trace;		/* do tracing? */
	int		r;
	double		startt = 0.0;		/* start time */

	lam_initerr_m();
	lam_setfunc_m(BLKMPIIPROBE);
/*
 * Check additional arguments.
 */
	if (flag == 0) {
		return(lam_errfunc(comm, BLKMPIIPROBE,
				lam_mkerr(MPI_ERR_ARG, 0)));
	}

	if ((tag < 0 && tag != MPI_ANY_TAG) || tag > LAM_MAXTAG) {
		return(lam_errfunc(comm, BLKMPIIPROBE,
				lam_mkerr(MPI_ERR_TAG, 0)));
	}
/*
 * Handle the trivial case.
 */
	if (src == MPI_PROC_NULL) {
		if (stat != MPI_STATUS_IGNORE) {
			stat->MPI_ERROR = MPI_SUCCESS;
			stat->MPI_SOURCE = MPI_PROC_NULL;
			stat->MPI_TAG = MPI_ANY_TAG;
			stat->st_length = 0;
		}
		
		*flag = 1;
		lam_resetfunc_m(BLKMPIIPROBE);
		return(MPI_SUCCESS);
	}

	if ((fl_trace = LAM_TRACE_TOP())) startt = MPI_Wtime();
/*
 * Advance the system to give the probe a better chance of success.
 */
	_mpi_req_blkclr();

	err = _mpi_req_advance();
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIPROBE, err));
	}

	LAM_ZERO_ME(request_storage);
	req = &request_storage;
	err = _mpi_req_build((char *) 0, 0, MPI_BYTE, src, tag, comm,
			LAM_RQIPROBE, &req);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIPROBE, err));
	}

	err = _mpi_req_start(req);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIPROBE, err));
	}

	r = RPI_SPLIT(_rpi_lamd_iprobe, _rpi_c2c_iprobe, (req));
	if (r < 0) {
		return(lam_errfunc(comm, BLKMPIIPROBE,
				lam_mkerr(MPI_ERR_INTERN, errno)));
	}

	err = _mpi_req_end(req);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIPROBE, err));
	}
/*
 * Copy the status info if synchronized.
 */
	if (r == 1) {
		*flag = 1;
		if (stat != MPI_STATUS_IGNORE) {
			*stat = req->rq_status;
		}
	} else {
		*flag = 0;
	}

	err = _mpi_req_destroy(&req);
	if (err != MPI_SUCCESS) {
		return(lam_errfunc(comm, BLKMPIIPROBE, err));
	}
/*
 * Generate a run time trace.
 */
	if (fl_trace) {
		lam_tr_msg(TRTNOIO, startt, LAM_S2US(MPI_Wtime() - startt),
			0, src, tag, comm, 0, 0, 0, 0, 0, LAM_RQIPROBE);
	}

	lam_resetfunc_m(BLKMPIIPROBE);
	return(MPI_SUCCESS);
}
