/*
 *  printer.c: Postscript generator - high level routines calling gnome-print
 *
 *  Copyright (C) 1997 John Coppens
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2 of
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <stdio.h>
#include <libgnomeprint/gnome-print.h>
#include <libgnomeprint/gnome-print-job.h>
#include "printer.h"
#include "global.h"
#include "main.h"

GnomePrintJob		*job;
GnomePrintContext	*gpc;
GnomePrintConfig	*config;

double		prt_last_lw = 0.1;
int		prt_last_color = 0;

GnomeFont	*prt_last_font = NULL;
double		prt_last_font_size = 12.0;
int		prt_last_font_color = 0;

void
print_new_page(char *fn)
{
  if (debug & DBG_PRINTING)
    fprintf(stderr, "[prt start] File: %s\n", fn);

  job = gnome_print_job_new(NULL);
  gpc = gnome_print_job_get_context(job);
  config = gnome_print_job_get_config (job);
  
  gnome_print_config_set(config, "Printer", "GENERIC");
  gnome_print_job_print_to_file(job, fn);
  gnome_print_config_set(config, GNOME_PRINT_KEY_PAPER_SIZE, "A4");

  gnome_print_beginpage(gpc, "1");
  gnome_print_scale(gpc, 72.0/25.4, 72.0/25.4);
  gnome_print_translate(gpc, pref.prt_mleft, pref.prt_mbottom);
}


void
print_close_page(void)
{
  if (debug & DBG_PRINTING)
    fprintf(stderr, "[prt done]\n");

  gnome_print_showpage(gpc);

  gnome_print_job_close(job);
  gnome_print_job_print(job);
  g_object_unref(G_OBJECT(config));
  g_object_unref(G_OBJECT(gpc));
  g_object_unref(G_OBJECT(job));
}


void
print_setrgbcolor(int color)
{
  double r = ((color >> 24) & 0xff)/255.0,
         g = ((color >> 16) & 0xff)/255.0,
         b = ((color >>  8) & 0xff)/255.0;

  if (debug & DBG_PRINTING)
    fprintf(stderr, "[prt color] r:%f g:%f b:%f\n", r, g, b);

  gnome_print_setrgbcolor(gpc, r, g, b);
}


void
print_do_arc(double x, double y, double arcb, double arce, double rad)
{
  double ctrx, ctry, begx, begy, arcrad,
	 chrad = pref.prt_chartsize * 0.5;

  if (debug & DBG_PRINTING)
    fprintf(stderr, "[src] x:%f y:%f beg:%f end:%f rad:%f\n",
    		    x, y, arcb, arce, rad);

  arcrad = chrad * rad;
  ctrx = x * chrad;
  ctry = (1 + y) * chrad;
  begx = ctrx + arcrad * cos(arcb);
  begy = ctry + arcrad * sin(arcb);

  gnome_print_newpath(gpc);
  gnome_print_setlinewidth(gpc, prt_last_lw);
  print_setrgbcolor(prt_last_color);
  gnome_print_moveto(gpc, begx, begy);
  gnome_print_arcto(gpc, ctrx, ctry, arcrad,
		    arcb*180.0/M_PI, arce*180.0/M_PI, 0);
  gnome_print_stroke(gpc);
}


void
print_do_line(double x1, double y1, double x2, double y2)
{
  double chrad = pref.prt_chartsize * 0.5;

  if (debug & DBG_PRINTING)
    fprintf(stderr, "[line] x1:%f y1:%f x2:%f y2:%f\n",
    		    x1, y1, x2, y2);
  
  gnome_print_newpath(gpc);
  print_setrgbcolor(prt_last_color);
  gnome_print_setlinewidth(gpc, prt_last_lw);
  gnome_print_moveto(gpc, chrad * x1, chrad * (1 + y1));
  gnome_print_lineto(gpc, chrad * x2, chrad * (1 + y2));
  gnome_print_stroke(gpc);
}


void
print_do_string(char *str, double x, double y, 
		double size, double angle)
{
  double chrad = pref.prt_chartsize * 0.5;

  if (debug & DBG_PRINTING)
    fprintf(stderr, "[string] str:[%s] x:%f y:%f size:%f angle:%f\n",
                    str, x, y, size, angle);

  print_setrgbcolor(prt_last_font_color);

  gnome_print_gsave(gpc);
  gnome_print_translate(gpc, chrad * x, chrad * (y + 1));
  gnome_print_rotate(gpc, angle * 180 / M_PI);
  gnome_print_moveto(gpc, size/8, size/8);
  gnome_print_show(gpc, str);
  gnome_print_grestore(gpc);
}


void
print_point(double x, double y, int pstyle)
{
  double r = pref.prt_diam[pstyle] / 2,
	 chrad = pref.prt_chartsize * 0.5;

  if (debug & DBG_PRINTING) {
    fprintf(stderr, "[prt pt] x:%f y:%f r:%f\n", x, y, r);
    fprintf(stderr, "         bc:%08x fc:%08x\n", 
			pref.prt_fill_color[pstyle],
			pref.prt_brdr_color[pstyle]);
  }
  gnome_print_newpath(gpc);
  gnome_print_moveto(gpc, x * chrad + r, (1 + y) * chrad);
  gnome_print_arcto (gpc, x * chrad,     (1 + y) * chrad, r, 0, 359, 0);
  gnome_print_closepath(gpc);
  gnome_print_gsave(gpc);
  print_setrgbcolor(pref.prt_fill_color[pstyle]);
  gnome_print_fill(gpc);
  gnome_print_grestore(gpc);
  print_setrgbcolor(pref.prt_brdr_color[pstyle]);
  gnome_print_setlinewidth(gpc, pref.prt_brdr[pstyle]);
  gnome_print_stroke(gpc);
}


void
print_setlinestyle(double lw, int color)
{
  prt_last_lw = lw;
  prt_last_color = color;
}


gboolean
print_setfontstyle(char *font, double size, int color)
{
  prt_last_font_size = size;
  prt_last_font_color = color;

  if (prt_last_font) gnome_font_unref(prt_last_font);

  prt_last_font = gnome_font_find_closest(font, prt_last_font_size);
  if (!prt_last_font || 
      (strcmp(gnome_font_get_name(prt_last_font), font) != 0)) {
    g_print("%s: %s\n", _("Could not get font"), font);
    return FALSE;
  }  

  if (debug & DBG_PRINTING) {
    fprintf(stderr, "[prt fontstyle] req:%f found:%f  %08x\n", 
		size, gnome_font_get_size(prt_last_font), color);
  }
  gnome_print_setfont(gpc, prt_last_font);
  return TRUE;
}


void
print_test(void)
{
  print_new_page("test_page.ps");
  print_setfontstyle(pref.prt_font_name, pref.prt_font_size, 0xff00ff00);
  print_do_string("01234.56789 - 'ere we are", 0, 0, 20, 0);
  print_close_page();
}


void
print_load_font_list(GtkCombo *combo)
{
  GList *flist = gnome_font_list();

  gtk_combo_set_popdown_strings(combo, flist);
  gnome_font_list_free(flist);
}
